/* @TITLE "signal: Handle signals and shutdown gracefully" */
/* 
 * signal - We try to catch SIGTERM and SIGINT and dump elogs, 
 * flush stdout, etc. This is useful in debugging.
 * There is no overhead when not debugging.
 * 
 * Seems to work when killed as "kill %1" from csh to a background job,
 * or "kill pid1 pid2 ... pidn-1 pid0" where the pids are the process id's
 * of all processes. Unfortunately, the complicated mechanism used here for 
 * pid 0 to catch the signal and send it to the others does not seem to work. 
 * The elog never quite finishes being dumped. Perhaps the US interferes. 
 * 
 * Send a SIGUSR1 (30) to call RT_dump on this process. May not 
 * return gracefully, but it tries. 
 */

#include <stdio.h>
#include <signal.h>
#include "usdfk.h"
#include "rapidelog.h"
#include "rapid.h"
#include "synch.h"
#include "driver.h"

static char rcsid[] = "$Id: signal.c,v 7.1 91/05/09 19:34:39 dfk Tape2 $";

static int *proc;			/* list of pids for all procs */
static short *count;		/* number still alive */

static void SignalInitForProc(); /* initialize on each proc */
static void catch();			/* catch for proc 0 */
static void all_catch();		/* catch for all other procs */
static void dodump();		/* do an RT_dump call */

/* @SUBTITLE "SignalInit: Initialize signal handling" */
void
SignalInit()				/* overall init */
{
    proc = (int *)UsAllocLocal(sizeof(int) * ProcsInUse());
    Share(&proc);

    AllocateShare(count, short, ProcsInUse());

    GenTaskForEachProc(SignalInitForProc);
}

/* @SUBTITLE "SignalInitForProc: Init on each processor" */
static void
SignalInitForProc()
{
    if (UsProc_Node > 0) {
	   signal(SIGINT, all_catch);
	   signal(SIGTERM, all_catch);
	   proc[UsProc_Node] = getpid();
    } else {				/* proc 0 is different */
	   signal(SIGINT, catch);
	   signal(SIGTERM, catch);
    }

    signal(SIGUSR1, dodump);
}

/* @SUBTITLE "catch: catch signal on processor 0" */
/* We then pass the signal on to other processors */

static void
catch(sig)
{
    int i;
    int p = ProcsInUse();

    printf("Node 0 caught signal %d and killing others...\n", sig);

    for (i = 1; i < p; i++)	/* note: not proc 0 */
	 kill(sig, proc[i]); 

    all_catch(sig);
}

/* @SUBTITLE "all_catch: catch signal on other processors" */
/* Also called directly from proc 0, when it is ready.
 * Note that although we may get multiple signals of this type,
 * they are automatically blocked while we are in here.
 */
static void				/* catch function for all other procs */
all_catch(sig)
	int sig;
{
    printf("\nNode %d caught signal %d; exiting\n", UsProc_Node, sig);

    ELOG_LOG(E_ABORT, sig);
    ELOG_OUTPUT();
    fflush(stdout);

#ifdef ELOG
    if (Atomic_add(count, -1)-1 == 0) {
 	   printf("Aborted Elog will be called %s\n", RTELOG_ABORTNAME);
	   rename(RTELOG, RTELOG_ABORTNAME);
    }
#endif

    exit(-sig);
}

/* @SUBTITLE "dodump: do a RT_dump then continue" */
/* We then pass the signal on to other processors */

static void
dodump()
{
    RT_dump(my_rpd);
}

