/* @TITLE "neighbor.c - SYNCH_NEIGHBOR synchronization" */
/* 
 * neighbor.c: Implement synchronization with two neighbors in a linear
 * processor array. Once your neighbors are "done", you can go on. 
 * When you synchronize, you tell your neighbors that you are done, 
 * then wait for them to finish. You need not wait for them to synchronize
 * with THEIR neighbors.
 *
 * David Kotz, May 1989
 */

static char rcsid[] = "$Id: neighbor.c,v 7.1 91/05/09 19:34:39 dfk Tape2 $"; 

#include <stdio.h>
#include <usdfk.h>

/* for SYNCH_NEIGHBOR */
typedef struct neighbor NBOR_FLAG;
struct neighbor {
    TICS up_ready;			/* when neighbor number-1 was ready */
    TICS down_ready;		/* when neighbor number+1 was ready */
    short up_flag;			/* neighbor number-1 status */
    short down_flag;		/* neighbor number+1 status */
};
#define WAIT 0				/* Wait! I haven't seen the ready value yet */
#define SAWIT 1			/* Ok, I've seen the ready value. */
#define GONE 2				/* I'm gone, don't wait for me. */

static int MyNumber;		/* our number */
static int NeighborCount = 0;	/* number of procs involved */
static short *NCounter;
static NBOR_FLAG **Neighbors;	/* pointers to all neighbor structures */
static NBOR_FLAG *Here; /* our neighbor structure */
static NBOR_FLAG *Up, *Down;	/* our neighbors' structure */

static void NeighborInitWorker();

/* Statistics */
TICS *TotalDelay;			/* total time wasted in synch */
unsigned short *Count;		/* number of synchs */

/* @SUBTITLE "NeighborInit: Set up for SYNCH_NEIGHBOR style" */
void
NeighborInit()
{
    int i;

    /* free old structure if not the same size */
    if (NeighborCount > 0 && NeighborCount != ProcsInUse()) {
	   for (i=0; i < NeighborCount; i++)
		UsFree(Neighbors[i]);
	   UsFree(Neighbors);
    }
    
    AllocateShareZero(NCounter, short);

    NeighborCount = ProcsInUse();
    Share(&NeighborCount);

    Neighbors = (NBOR_FLAG **)AllocGlobal(sizeof(NBOR_FLAG *) * NeighborCount);
    bzero (Neighbors, (sizeof(NBOR_FLAG *) * NeighborCount));
    Share(&Neighbors);
    GenTaskForEachProc(NeighborInitWorker, 0);

    UsFree(NCounter);

    if (Count) {
	   *TotalDelay = 0;
	   *Count = 0;
    } else {
	   AllocateShareZero(TotalDelay, TICS);
	   AllocateShareZero(Count, unsigned short);
    }
}

static void
NeighborInitWorker()
{
    int up, down;			/* indices of our neighbors */

    Here = (NBOR_FLAG *)AllocLocal(sizeof(NBOR_FLAG));
    MyNumber = Atomic_add(NCounter, 1);
    Here->up_ready = 0;
    Here->up_flag = SAWIT;
    Here->down_ready = 0;
    Here->down_flag = SAWIT;
    Neighbors[MyNumber] = Here;

    up = (MyNumber-1 + NeighborCount) % NeighborCount;
    down = (MyNumber+1) % NeighborCount;

    /* store pointers to our neighbors */
    do {
	   Up = Neighbors[up];
    } while(Up == NULL);
    do {
	   Down = Neighbors[down];
    } while(Down == NULL);
}

/* @SUBTITLE "SynchronizeNeighbor: Synchronize for neighbor style" */
void
SynchronizeNeighbor(worker, arg)
	int (*worker)();
	int arg;
{
    boolean notready;
    TICS delay;

    /* mark neighbors to say that we are ready */
    while (Here->up_flag == WAIT)
	 UsWait(10);
    if (Here->up_flag == SAWIT) {
	   Here->up_flag = WAIT;
	   Up->down_ready = rtc;
    }

    while (Here->down_flag == WAIT)
	 UsWait(10);
    if (Here->down_flag == SAWIT) {
	   Here->down_flag = WAIT;
	   Down->up_ready = rtc;
    }

    /* wait for them to mark us */
    notready = FALSE;
    while (   (Here->down_flag != GONE && Here->down_ready == 0)
		 || (Here->up_flag != GONE && Here->up_ready == 0)) {
	   notready = TRUE;
	   if (worker != NULL)
		(*worker)(arg);
	   else
		UsWait(10);
    }
    
    /* compute waste (time from last one ready until now) */
    if (notready) {
	   delay = rtc - max(Here->down_ready, Here->up_ready);
	   Atomic_add_long(TotalDelay, delay);
	   Atomic_add(Count, 1);
    } else				/* delay is 0 */
	 Atomic_add(Count, 1);

    /* now reset our wait flags */
    Here->up_ready = 0;
    Here->down_ready = 0;

    /* and tell them that we've seen that round of flags */
    Up->down_flag = SAWIT;
    Down->up_flag = SAWIT;
}


/* @SUBTITLE "SynchOutNeighbor: Check out, for neighbor style" */
void
SynchOutNeighbor()
{
    Up->down_flag = GONE;
    Down->up_flag = GONE;
    /* now they'll ignore us */
}


/* @SUBTITLE "NBSynchTotal: statistics about synchronization" */
/* return the avg delay time taken and clear it */
float
NBSynchTotal()
{
    float time = (float) *TotalDelay / (float) *Count;
    
/*     printf("Delay %u, count %u, avg %g\n", *TotalDelay, *Count, time); */

    *TotalDelay = 0;
    *Count = 0;
    return (time);
}

/* return the avg max delay time and clear it */
float
NBSynchMax()
{
    /* this measure is meaningless for this style */
    return (0);
}

