/* @TITLE "neighbor.c - SYNCH_NEIGHBOR synchronization" */
/* 
 * neighbor.c: Implement synchronization with two neighbors in a linear
 * processor array. Once your neighbors are "done", you can go on. 
 * When you synchronize, you tell your neighbors that you are done, 
 * then wait for them to finish. You need not wait for them to synchronize
 * with THEIR neighbors.
 *
 * David Kotz, May 1989
 */

static char rcsid[] = "$Id: nbordebug.c,v 6.1 91/05/09 19:38:52 dfk Tape2 $"; 

#include <stdio.h>
#include <usdfk.h>

/* for SYNCH_NEIGHBOR */
typedef struct neighbor NBOR_FLAG;
struct neighbor {
    TICS up_ready;			/* when neighbor number-1 was ready */
    TICS down_ready;		/* when neighbor number+1 was ready */
    int count;
};

static int MyNumber;		/* our number */
static int NeighborCount = 0;	/* number of procs involved */
static short *NCounter;
static NBOR_FLAG **Neighbors;	/* pointers to all neighbor structures */
static NBOR_FLAG *Here; /* our neighbor structure */
static NBOR_FLAG *Up, *Down;	/* our neighbors' structure */

static void NeighborInitWorker();

/* Statistics */
TICS *TotalDelay;			/* total time wasted in synch */
unsigned short *Count;		/* number of synchs */

/* @SUBTITLE "NeighborInit: Set up for SYNCH_NEIGHBOR style" */
void
NeighborInit()
{
    int i;

    /* free old structure if not the same size */
    if (NeighborCount > 0 && NeighborCount != ProcsInUse()) {
	   for (i=0; i < NeighborCount; i++)
		UsFree(Neighbors[i]);
	   UsFree(Neighbors);
    }
    
    AllocateShareZero(NCounter, short);

    NeighborCount = ProcsInUse();
    Share(&NeighborCount);

    Neighbors = (NBOR_FLAG **)AllocGlobal(sizeof(NBOR_FLAG *) * NeighborCount);
    bzero (Neighbors, (sizeof(NBOR_FLAG *) * NeighborCount));
    Share(&Neighbors);
    GenTaskForEachProc(NeighborInitWorker, 0);

/*    SharePtrAndBlk(&Neighbors, sizeof(NBOR_FLAG *) * NeighborCount); */

    UsFree(NCounter);

    if (Count) {
	   *TotalDelay = 0;
	   *Count = 0;
    } else {
	   AllocateShareZero(TotalDelay, TICS);
	   AllocateShareZero(Count, unsigned short);
    }
}

static void
NeighborInitWorker()
{
    int up, down;			/* indices of our neighbors */

    Here = (NBOR_FLAG *)AllocLocal(sizeof(NBOR_FLAG));
    MyNumber = Atomic_add(NCounter, 1);
    Here->up_ready = 0;
    Here->down_ready = 0;
    Here->count = 0;
    Neighbors[MyNumber] = Here;

    up = (MyNumber-1 + NeighborCount) % NeighborCount;
    down = (MyNumber+1) % NeighborCount;

    /* store pointers to our neighbors */
    do {
	   Up = Neighbors[up];
    } while(Up == NULL);
    do {
	   Down = Neighbors[down];
    } while(Down == NULL);
}

/* @SUBTITLE "SynchNeighbor: Synchronize for neighbor style" */
void
SynchronizeNeighbor(worker, arg)
	int (*worker)();
	int arg;
{
    Here->count++;

    /* mark neighbors to say that we are ready */
    while (Up->down_ready != 0)
	 UsWait(10);
    Up->down_ready = rtc;

    while (Down->up_ready != 0)
	 UsWait(10);
    Down->up_ready = rtc;

    /* wait for them to mark us */
    while (Here->down_ready == 0 || Here->up_ready == 0) {
	   UsWait(10);
    }

    /* now reset our wait flags */
    Here->up_ready = 0;
    Here->down_ready = 0;
}

/* @SUBTITLE "NBSynchTotal: statistics about synchronization" */
/* return the avg delay time taken and clear it */
float
NBSynchTotal()
{
    float time = (float) *TotalDelay / (float) *Count;
    
/*     printf("Delay %u, count %u, avg %g\n", *TotalDelay, *Count, time); */

    *TotalDelay = 0;
    *Count = 0;
    return (time);
}

/* return the avg max delay time and clear it */
float
NBSynchMax()
{
    /* this measure is meaningless for this style */
    return (0);
}

