/* @TITLE "final.c - cleanup and print statistics" */
/* final.c: cleanup after each trial for driver.
 * This is separated mostly to isolate the statistics from the 
 * driver.
 */

static char rcsid[] = "$Id: final.c,v 7.1 91/05/09 19:34:39 dfk Tape2 $"; 

/* INCLUDES */

#include <stdio.h>
#include <usdfk.h>

#include "rapid.h"
#include "rapidelog.h"
#include "stats.h"
#include "synch.h"
#include "driver.h"

extern double sqrt();

/* @SUBTITLE "Definitions and Variables" */
TICS TotalTime;		/* measured time of true Body */

TICS *SynchTime;	/* total time spent at each synchronization pt */
short *Nsynchs;			/* number of proc-syncs in program */

/* a generic statistic, with room for variance and average */
typedef struct statvar STATV;
struct statvar {
    float sum;
    float sumsq;
    unsigned int n;
    float average;
    float variance;
};

static STATV NReads;		/* total pages read */
static STATV ReadTime;		/* total time reading */
static STATV NWrites;		/* total pages written */
static STATV WriteTime;		/* total time writing */
static STATV Demand_fetched;	/* total pages demand fetched */
static STATV Prefetched;		/* total pages prefetched */
static STATV Prefetch_used;	/* total pages used that were prefetched */
static STATV Correct;		/* total correct predictions */
static STATV Mistake;		/* total mistaken predictions */
static STATV Wasted;		/* total unused prefetches */
static STATV UsedMistake;	/* total mistakes that were used */
static STATV Scanq;		/* total swapouts that waited for read I/O */
static STATV Buffer_hits;	/* total pages used that were cached */
static STATV Unready_hits;	/* total hit that were not ready */
static STATV Hitwait_time;	/* total wait time for unready hits */
static STATV PrefetchTime;	/* total time used for prefetching */
static STATV Prefetch_avg;	/* avg time per call used for prefetching */
static STATV PFBufhit;		/* total delay from prefetch when bufhit */
static STATV PFBufhit_avg;	/* avg delay from prefetch when bufhit */
static STATV PFDemand;		/* total delay from prefetch when demand */
static STATV PFDemand_avg;	/* avg delay from prefetch when demand */
static STATV NotifyTime;		/* total predict notify time */
static STATV Rereads;		/* number of reads after writeback */
static STATV Rewrites;		/* number of writes after writeback */
static STATV DiskReads;		/* number of disk reads */
static STATV DiskWrites;		/* number of disk writes */

#define sq(x) ((x)*(x))
#define StatInit(v) (v.sum = v.sumsq = v.n = 0)
#define StatPoint(v, x) {\
    v.sum += (x); \
    v.sumsq += sq(x); \
    v.n++; \
}

#define StatAverage(v) \
  (v.average = (v.n != 0) ? v.sum / v.n : 0)
#define StatVariance(v) \
  (v.variance = (v.n > 1) ? \
   sqrt(v.sumsq / (v.n-1) - sq(v.sum) / v.n / (v.n-1)) \
   : 0)

static void SummarizeStats();	/* make a summary of all statistics */
static void FinalforProc();
static void OutputElogs();

/* @SUBTITLE "Final: process and print statistics after trial" */
/* ARGSUSED */
void
Final ()
{
    float TotalDelay;		/* avg delay at synch point */
    float MaxDelay;			/* avg max delay at synch point */
    float readtime;			/* average time for RT_read call */
    float writetime;		/* average time for RT_write call */
    float diskutil;			/* disk utilization (in [0,1]) */
    float diskidle;			/* disk utilization (in msec) */
    float diskwait;			/* disk wait time (in msec) */
    
    /* close the file */
    RT_close(&my_rpd);

    /* process-specific shutdown code */
    GenTaskForEachProc(FinalforProc);

    /* now get I/O statistics from I/O module */
    io_getstat(&diskutil, &diskidle, &diskwait);
    diskutil *= 100.;		/* convert to percentage */
    diskidle /= ndisks;		/* compute per-disk idle */
    
    /* synchronization statistics */
    if (SynchStyle == SYNCH_NEIGHBOR) {
	   TotalDelay = NBSynchTotal(); /* avg delay at synch point */
	   MaxDelay = NBSynchMax();	/* avg max delay at synch point */
    } else {
	   TotalDelay = SynchTotal(SynchCounter); /* avg delay at synch point */
	   MaxDelay = SynchMax(SynchCounter);	/* avg max delay at synch point */
    }

    /* Output Elogs, if doing elogging */
#ifdef ELOG
    if (trial <= ElogDumpLimit) {
	   printf("Elog output... ");
	   fflush(stdout);
    }
    Share(&trial);
    GenTaskForEachProc(OutputElogs, 0); /* outputs and resets */
    if (trial <= ElogDumpLimit) {
	   sprintf(ElogFile_name, "%s.%1d.elog",
			 ElogData_name, trial);
	   if (rename(RTELOG, ElogFile_name) == -1)
		perror( "ERROR: Elog not renamed");
	   printf("Done\n");
    }
#endif
    
    /* Summarize the statistics */
    SummarizeStats();
    
    /* compute average read time */
    readtime = NReads.sum > 0 ? ReadTime.sum / NReads.sum : 0.0;
    writetime = NWrites.sum > 0 ? WriteTime.sum / NWrites.sum : 0.0;

    /* write report to screen */
    printf("\n");
    printf(
		 "Trial %d: [%d] Time: %.1f secs.  Synchs: %d. \n\
 Avg Synch Time: %.1f tics.  Avg Delay: %.1f tics. Avg Max Delay: %.1f tics.\n\
 Avg Read time: %.1f msec.  Disk wait time: %.1f msec. \n\
 Disk utilization: %5.1f%%.  Avg Disk idle: %.1f msec.\n\
 Demanded %d (%.1f). Prefetched %d (%.1f) (used %d (%.1f))\n\
 Hits %d (%.1f).\n\
 Prefetch Time %.1f msec (%.1f), %.1f tics per call\n\
 Overrun when Demand %.1f tics per processor, %.1f per access.\n\
 Overrun when Bufhit %.1f tics per processor, %.1f per access.\n\
 Unready %d; average wait time %.3f msec\n\
 Correct %d; Mistakes %d; Complete Waste %d, Used Mistakes %d\n\
 Prediction notify time %.1f msec total\n\
 Unready queue scans %d\n\
 Avg Write Time: %.1f msec.  Rereads: %d  Rewrites: %d\n\
 Disk Reads: %d  Disk Writes: %d\n",
		 trial, Nprocs,
		 (float)TotalTime * SECperTICK,
		 *Nsynchs,
		 *Nsynchs ? (float)*SynchTime / (float)*Nsynchs : 0.,
		 *Nsynchs ? TotalDelay : 0.,
		 *Nsynchs ? MaxDelay : 0.,
		 readtime, diskwait, diskutil, diskidle,
		 (int)Demand_fetched.sum, Demand_fetched.variance,
		 (int)Prefetched.sum, Prefetched.variance,
		 (int)Prefetch_used.sum, Prefetch_used.variance,
		 (int)Buffer_hits.sum, Buffer_hits.variance,
		 PrefetchTime.average, PrefetchTime.variance,
		 Prefetch_avg.average,
		 PFDemand.average, PFDemand_avg.average,
		 PFBufhit.average, PFBufhit_avg.average,
		 (int)Unready_hits.sum,
		 (int)Unready_hits.sum
		 ? (Hitwait_time.sum / Unready_hits.sum) : 0,
		 (int)Correct.sum, (int)Mistake.sum, 
		 (int)Wasted.sum, (int)UsedMistake.sum,
		 NotifyTime.sum,
		 (int)Scanq.sum,
		 writetime,
		 (int)Rereads.sum,
		 (int)Rewrites.sum,
		 (int)DiskReads.sum,
		 (int)DiskWrites.sum);
    
    /* write data to data file too */
    if (DataFile) {
	   fprintf(DataFile, "%d %d %.2f %d %.1f %.1f %.1f %.1f %.1f %5.1f %.1f %d %.1f %d %.1f %d %.1f %d %.1f %g %g %g %g %g %g %g %d %.3f %d %d %d %d %.1f %d %.1f %d %d %d %d\n",
			 trial, Nprocs,
			 (float)TotalTime * SECperTICK,
			 *Nsynchs,
			 *Nsynchs ? (float)*SynchTime / (float)*Nsynchs : 0.,
			 *Nsynchs ? TotalDelay : 0.,
			 *Nsynchs ? MaxDelay : 0.,
			 readtime, diskwait, diskutil, diskidle,
			 (int)Demand_fetched.sum, Demand_fetched.variance,
			 (int)Prefetched.sum, Prefetched.variance,
			 (int)Prefetch_used.sum, Prefetch_used.variance,
			 (int)Buffer_hits.sum, Buffer_hits.variance,
			 PrefetchTime.average, PrefetchTime.variance,
			 Prefetch_avg.average,
			 PFDemand.average, PFDemand_avg.average,
			 PFBufhit.average, PFBufhit_avg.average,
			 (int)Unready_hits.sum,
			 (int)Unready_hits.sum
			 ? (Hitwait_time.sum / Unready_hits.sum) : 0,
			 (int)Correct.sum, (int)Mistake.sum,
			 (int)Wasted.sum, (int)UsedMistake.sum,
			 NotifyTime.sum,
			 (int)Scanq.sum,
			 writetime,
			 (int)Rereads.sum,
			 (int)Rewrites.sum,
			 (int)DiskReads.sum,
			 (int)DiskWrites.sum);

	   fflush(DataFile);
    }
}

/* @SUBTITLE "FinalforProc: worker for Final" */
/* (parallel) Per-process finish-up work. 
 * Releases the buffer that InitforProc created.
 */
static void
FinalforProc ()
{
    free(my_buf);
}

/* @SUBTITLE "SummarizeStats: compile stats into summary" */
static void
SummarizeStats()
{
    int i;
    STAT *s;

    StatInit(NReads);
    StatInit(ReadTime);
    StatInit(NWrites);
    StatInit(WriteTime);
    StatInit(Demand_fetched);
    StatInit(Prefetched);
    StatInit(Prefetch_used);
    StatInit(Correct);
    StatInit(Mistake);
    StatInit(Wasted);
    StatInit(UsedMistake);
    StatInit(Scanq);
    StatInit(Buffer_hits);
    StatInit(Unready_hits);
    StatInit(Hitwait_time);
    StatInit(PrefetchTime);
    StatInit(Prefetch_avg);
    StatInit(PFDemand);
    StatInit(PFDemand_avg);
    StatInit(PFBufhit);
    StatInit(PFBufhit_avg);
    StatInit(NotifyTime);
    StatInit(Rereads);
    StatInit(Rewrites);
    StatInit(DiskReads);
    StatInit(DiskWrites);
    
    for (i = 0; i < ProcsInUse(); i++) {
	   s = RT_stats[i];
	   StatPoint(NReads, s->nread);
	   StatPoint(ReadTime, s->readtime * MSECperTICK);
	   StatPoint(NWrites, s->nwrite);
	   StatPoint(WriteTime, s->writetime * MSECperTICK);
	   StatPoint(Demand_fetched, s->demand);
	   StatPoint(Prefetched, s->prefetch);
	   StatPoint(Prefetch_used, s->prefetch_used);
	   StatPoint(Correct, s->correct);
	   StatPoint(Mistake, s->mistake);
	   StatPoint(Wasted, s->wasted);
	   StatPoint(UsedMistake, s->usedmistake);
	   StatPoint(Scanq, s->scanq);
	   StatPoint(Unready_hits, s->unready);
	   StatPoint(Hitwait_time, s->hitwait * MSECperTICK);
	   StatPoint(Buffer_hits, s->bufferhit);
	   StatPoint(PrefetchTime, s->prefetchtime * MSECperTICK);
	   StatPoint(NotifyTime, s->notify * MSECperTICK);
	   StatPoint(Rereads, s->rereads);
	   StatPoint(Rewrites, s->rewrites);
	   StatPoint(DiskReads, s->reads);
	   StatPoint(DiskWrites, s->writes);
	   if (s->prefetchcount != 0)
		StatPoint(Prefetch_avg, 
				(float) s->prefetchtime
				/ (float) s->prefetchcount);
	   StatPoint(PFDemand, s->pfdemand_delay);
	   if (s->prefetchdemand != 0)
		StatPoint(PFDemand_avg,
				(float)s->pfdemand_delay 
				/ (float)s->prefetchdemand);
	   StatPoint(PFBufhit, s->pfbufhit_delay);
	   if (s->prefetchbufhit != 0)
		StatPoint(PFBufhit_avg,
				(float)s->pfbufhit_delay 
				/ (float)s->prefetchbufhit);
    }

    /* now compute averages and variances for some */
    StatAverage(NReads);
    StatAverage(ReadTime);
    StatVariance(Demand_fetched);
    StatVariance(Prefetched);
    StatVariance(Prefetch_used);
    StatVariance(Buffer_hits);
    StatAverage(PrefetchTime);
    StatVariance(PrefetchTime);
    StatAverage(Prefetch_avg);
    StatAverage(PFDemand);
    StatAverage(PFDemand_avg);
    StatAverage(PFBufhit);
    StatAverage(PFBufhit_avg);
    StatAverage(NotifyTime);
}

/* @SUBTITLE "OutputElogs: output the elog on each processor" */
/* parallel */
static void
OutputElogs ()
{
    if (trial <= ElogDumpLimit)
	 ELOG_OUTPUT();
}
