/* @TITLE "body.c - main test loop for driver" */
/* body.c - main test loop for driver
 *
 * Body of driver for testing RAPIDFILEs. This uses the reference
 * string my_refs to drive accesses to the file. The string may be either
 * local or global, and is treated separately. All synchronization types
 * are implemented here.
 */

static char rcsid[] = "$Id: body.c,v 7.1 91/05/09 19:34:39 dfk Tape2 $"; 

#include <stdio.h>
#include <usdfk.h>

#include "rapid.h"
#include "rapidelog.h"
#include "refs.h"
#include "pattern.h"
#include "io.h"
#include "synch.h"
#include "driver.h"

static void BodyForProc();
static void ResetCount();
static void Start();
static void End();

static int readingchunk;		/* used by EXACT predictor through my_refs */

static SYNCH *startend_synch;
static TICS *start_time;
static TICS *end_time;
static short *flush_index;

/* for SYNCH_TOTAL */
static short *allcount;

/* @SUBTITLE "BodyInit: set up body data structures" */
void
BodyInit()
{
    AllocateShare(allcount, short, SynchCount);
    AllocateShareZero(start_time, TICS);
    AllocateShareZero(end_time, TICS);
    AllocateShareZero(flush_index, short);
    
    startend_synch = UsMakeSynch(ProcsInUse());
    Share(&startend_synch);
}

/* @SUBTITLE "Body/Start/End: Generate and measure tasks" */
TICS						/* returns elapsed time */
Body()
{
    if (SynchStyle == SYNCH_PORTION && my_refs->portions == NULL) {
	   printf("Portion synchronization not available for this pattern.\n");
	   return;
    }
    
    if (SynchStyle == SYNCH_TOTAL)
	 *allcount = SynchCount;
    
    if (SynchStyle == SYNCH_NEIGHBOR)
	 NeighborInit();
    else
	 SetSynch(startend_synch, ProcsInUse());
    
    *flush_index = 0;		/* reset for end-of-pattern flush */

    /* start_time set in Start, from synchronization on startend_synch */
    GenTaskForEachProc(BodyForProc, 0);

    return(*end_time - *start_time);
}

/* Now that everyone is here, we can start timers */
static void
Start()
{
    io_clearstat();			/* set timers in io.c  */
    *start_time = rtc;
}

/* Now that everyone is done flushing, we can stop timer */
static void
End()
{
    *end_time = rtc;
}

/* @SUBTITLE "BodyForProc: Worker for Body" */
/* parallel */
static void
BodyForProc()
{
    int count = SynchCount;
    ONEREF chunk;
    int chunks = my_refs->nchunks;
    int s = 0;				/* index in my_refs->chunks[] */
    int p = 0;				/* index in my_refs->portions[] */
    
    readingchunk = -1;
    my_refs->readingchunk = &readingchunk;

    SynchronizeReset(startend_synch, Start, 0);
    
    ELOG_LOG(E_BEGIN, my_refs->nchunks);
    ELOG_LOG(E_CHUNKSIZE, my_refs->chunksize);
    
    while (TRUE) {
	   if (my_refs->global)
		s = Atomic_add_long(&(my_refs->next), 1);
	   else
		s = my_refs->next++;
	   if (s >= chunks)
		break;			/* LOOP EXIT */
	   readingchunk = s;	/* remember current chunk number */

	   /* Should we stop working now for a synchronization point? */
        if (SynchStyle == SYNCH_TOTAL)
		if (Atomic_add(allcount, -1) <= 0)
		  SynchPoint(ResetCount);
	   
	   /* access a chunk */
	   chunk = my_refs->chunks[s]; /* make local copy of ONEREF */
	
	   if (chunk.write) {
		  ELOG_LOG(E_WRITE, chunk.offset);
		  RT_lseek(my_rpd, chunk.offset, 0);
		  RT_write(my_rpd, my_buf, chunk.length);
	   } else {
		  ELOG_LOG(E_READ, chunk.offset);
		  RT_lseek(my_rpd, chunk.offset, 0);
		  RT_read(my_rpd, my_buf, chunk.length);
	   }

	   ELOG_LOG(E_COMP, chunk.offset);
    	   if (chunk.comptime != 0)
		UsWait(chunk.comptime);
	   
	   /* End of the portion */
	   if (my_refs->portions && s >= my_refs->cur_portion_end) {
		  my_refs->cur_portion_end = my_refs->portions[++p];
		  
		  if (SynchStyle == SYNCH_PORTION)
		    SynchPoint(NULL);
	   }
	   
	   /* Should we stop working now for a synchronization point? */
	   if (SynchStyle == SYNCH_EACH)
		if (--count == 0) {
		    SynchPoint(NULL);
		    count = SynchCount;
		}

	   /* Should we stop working now for a synchronization point? */
	   if (SynchStyle == SYNCH_NEIGHBOR)
		if (--count == 0) {
		    SynchPointNeighbor();
		    count = SynchCount;
		}
    }
    
    if (SynchStyle == SYNCH_TOTAL)
	 SynchPoint(NULL);
    
    if (SynchStyle == SYNCH_NEIGHBOR)
	 SynchOutNeighbor();
    else
	 SynchOut();
    
    Synchronize(startend_synch); /* wait for all to finish */
    RT_flush(my_rpd, flush_index);	/* flush all I/O, in parallel */
    SynchronizeReset(startend_synch, End, 0); /* wait for all to finish */

    ELOG_LOG(E_END, 0);
}

/* reset counter for TOTAL synch style */
static void
ResetCount()
{
    *allcount = SynchCount - ProcsInUse();
}
