/* @TITLE "dualq.c - simulate dual queues" */
/* dualq.c - simulate Chrysalis dual queue interface
 *
 * We do this with a very simple circular queue with monolithic lock.
 * I will not implement the "dual" nature of the queue 
 *
 * David Kotz 
 */

static char rcsid[] = "$Id: dualq.c,v 6.0 89/11/06 15:49:05 dfk clean1 Locker: dfk $"; 

#include <stdio.h>
#include <usdfk.h>
#include "queue.h"
#include "error.h"

/* QH is defined already, by public.h, as follows:
 * typedef struct dual_queue *QH;
 
 * head==tail implies empty queue 
 */
struct dual_queue {
    short size;			/* max number of entries */
    short lock;			/* lock on queue */
    short head;			/* next place to add to back */
    short tail;			/* front of queue */
    long queue[1];			/* the entries [0..size-1] */
};

#define MAX_QUEUE_ENTRIES (1 << 14 - 1)
  
/* @SUBTITLE "Make_a_queue: make a new queue" */
/* Create a dual queue that will fit the specified number of entries.
 * We allocate room for one extra item, due to the implementation.
 * The "lock" feature, specified with lock_flag=TRUE, is not implemented.
 */
/* ARGSUSED */
QH
Make_a_queue(length, lock_flag)
	int length;			/* the number that need to fit */
	int lock_flag;			/* ignored - FALSE assumed */
{
    QH new;
    
    if (length <= 0 || length > MAX_QUEUE_ENTRIES)
	 return(NULL);
    
    /* Allocate an object bigger than just the header */
    /* The queue[] array extends past end of defined header */
    new = (QH) AllocGlobal(sizeof(struct dual_queue) + length * sizeof(long));

    new->size = length + 1; /* since header contains one entry */
    new->lock = 0;
    new->head = 0;
    new->tail = 0;
    TouchBlock(new->queue, new->size * sizeof(long), TRUE);

    return(new);
}

/* @SUBTITLE "TouchQueue: touch memory in a queue" */
/* Touch all memory involved with a queue */
void
TouchQueue(q)
	QH q;
{
    TouchBlock(q->queue, q->size * sizeof(long), FALSE);
}

/* @SUBTITLE "DQ_nitems: return fullness of queue" */
/* Returns the number of items currently in the queue. */
int
DQ_nitems(q)
	QH q;
{
    if (q == NULL)
	 error("Invalid Dual Queue handle", ERR_DUMP);
    return((q->head + q->size - q->tail) % q->size);
}

/* @SUBTITLE "Enq_DualQ: add item to queue" */
/* Insert an entry into the dual queue.
 * The second argument specifies where the entry goes. Both TRUE
 * (front) and FALSE (back) are supported.
 */
void
Enq_DualQ(q, datum, front)
	QH q;
	long datum;
	int front;			/* TRUE places at front of queue */
{
    int n;
    
    LOCK(&(q->lock),10);
    n = DQ_nitems(q);
    if (n+1 == q->size) {
	   UNLOCK(&(q->lock));
	   error("Dual queue was full", ERR_DUMP);
    }
    if (front) {		/* put on front (like stack) */
	   q->tail = (q->tail - 1 + q->size) % q->size;
	   q->queue[q->tail] = datum;
    } else {			/* put on back (like queue) */
	   q->queue[q->head] = datum;
	   q->head = (q->head + 1) % q->size;
    }
    UNLOCK(&(q->lock));
}

/* @SUBTITLE "Wait_DualQ: remove an entry from the queue" */
/*  Remove an entry from the dual queue.
 * The second argument specifies the priority. This is ignored.
 */
/* ARGSUSED */
long						/* datum */
Wait_DualQ(q, priority)
	QH q;
	int priority;			/* ignored */
{
    int n;
    long datum;
    
    LOCK(&(q->lock),10);
    n = DQ_nitems(q);
    if (n == 0) {
	   UNLOCK(&(q->lock));
	   error("Dual queue was empty", ERR_DUMP);
    } else {
	   datum = q->queue[q->tail];
	   q->tail = (q->tail + 1) % q->size;
    }
    UNLOCK(&(q->lock));
    
    return(datum);
}

/* @SUBTITLE "***  TEST PROGRAM  ***" */
/* This allows this module to be tested alone 
 * compile with -DTEST to get this 
 * It is NOT a parallel test 
 */

#ifdef TEST

main()
{
    QH q;
    int i;
    
    InitializeUs();
    
    q = Make_a_queue(10,FALSE);
    if (q == NULL)
	 error("Make_a_queue failed", ERR_DIE);
    if (DQ_nitems(q) != 0)
	 error("Don't have 0 at start", ERR_DUMP);
    
    /* put on one item and remove */
    Enq_DualQ(q, 999, FALSE);
    if (DQ_nitems(q) != 1)
	 error("Don't have 1 after one enqueue", ERR_DUMP);
    if (Wait_DualQ(q, TRUE) != 999)
	 error("Dequeue wrong value", ERR_DUMP);
    
    /* fill and empty, as a queue */
    for (i=0; i < 5; i++) {
	   Enq_DualQ(q, 900+i, FALSE);
	   if (DQ_nitems(q) != i+1)
		error("Don't have right count in Enq loop", ERR_DUMP);
    }
    for (i=0; i < 5; i++) {
	   if (Wait_DualQ(q, TRUE) != 900+i)
		error("Dequeue wrong value in loop", ERR_DUMP);
	   if (DQ_nitems(q) != 5-i-1)
		error("Don't have right count in DQ loop", ERR_DUMP);
    }
    
    /* fill and empty, as a stack */
    for (i=0; i < 5; i++) {
	   Enq_DualQ(q, 900+i, TRUE);
	   if (DQ_nitems(q) != i+1)
		error("Don't have right count in Push loop", ERR_DUMP);
    }
    for (i=4; i >= 0; i--) {
	   if (Wait_DualQ(q, TRUE) != 900+i)
		error("Dequeue wrong value in loop", ERR_DUMP);
	   if (DQ_nitems(q) != i)
		error("Don't have right count in Pop loop", ERR_DUMP);
    }
    
#ifdef EMPTY
    printf("This dequeue should underflow the queue and dump core.\n");
    (void) Wait_DualQ(q, TRUE);
    
    error("Queue underflowed and no error caught\n", ERR_DUMP);
#else
    for (i=0; i < 10; i++) {
	   Enq_DualQ(q, 900+i, FALSE);
	   if (DQ_nitems(q) != i+1)
		error("Don't have right count in Filling loop", ERR_DUMP);
    }
    
    printf("This enqueue should over-fill the queue and dump core.\n");
    Enq_DualQ(q, 999, FALSE);
    
    error("Queue overfilled and no error caught\n", ERR_DUMP);
#endif EMPTY
}
#endif TEST

