/* @TITLE "coroutine.c - coroutine package" */
/*
 ******************************************************************************
 *
 * Butterfly Chrysalis High-Level Coroutine Package
 *
 * This package provides a convenient set of functions for using coroutines
 * in a Chrysalis program.
 *
 * Rick LaRowe,  Duke University
 * ported to Mach by David Kotz
 *
 ******************************************************************************
 */

static char rcsid[] = "$Id: coroutine.c,v 6.10 90/11/14 14:30:44 dfk Write1 Locker: dfk $";

#include <stdio.h>
#include "coroutine.h"

#define bool int
#define TRUE  ((bool)1)
#define FALSE ((bool)0)

typedef int error_t;

typedef struct ccb_s {
	int ccb_cid;			/* coroutine id */
	int ccb_status;			/* ccb status word */
 	char *ccb_stk;			/* stack ptr */
	int *ccb_ptr;			/* coroutine ptr */
	struct ccb_s *ccb_next;		/* next ccb */
	struct ccb_s *ccb_prev;		/* previous ccb */
} ccb_t;

struct {
	int c_cnt;			/* current number of coroutines */
	ccb_t *c_head;			/* head of the list */
	ccb_t *c_cur;			/* current coroutine */
}cpkg = {0, NULL, NULL};

extern char *malloc();

/* @SUBTITLE "initialize_coroutines: set up coroutine package" */
/*
 * This function initializes the coroutine package.
 *
 * Returns - error_t
 */

error_t
initialize_coroutines()
{
	ccb_t *ptr;

	if (cpkg.c_cnt != 0) return (CERR_ALRDY_RUNNING);

	/* create the very first ccb structure for this process */

	ptr = (ccb_t *) malloc(sizeof(ccb_t));
	cpkg.c_cnt = 1;
	cpkg.c_head = ptr;
	cpkg.c_cur = cpkg.c_head;

	/* initialize it */

	ptr->ccb_cid = cpkg.c_cnt;		/* this is the first one */
	ptr->ccb_stk = NULL;			/* process stack is used */
	ptr->ccb_ptr = NULL;			/* not yet known */
	ptr->ccb_status = CRTN_RUNNING;		/* running */
	ptr->ccb_next = cpkg.c_head;		/* short cirular list */
	ptr->ccb_prev = cpkg.c_head;

	return (CERR_OK);
}

/* @SUBTITLE "create_coroutine: create a new coroutine" */
/*
 * This function is used to create a new coroutine.
 *
 *  Arguments:
 *	int stk_sz - size of stack to create for this coroutine
 *	int func() - function for this coroutine to execute
 *	int arg    - argument to pass this function
 *
 *  Returns:
 *	CID if successful, FALSE (zero) otherwise
 */

int
create_coroutine(stk_sz, func, arg)
    int stk_sz;
    int (*func)();
    int arg;
{
	ccb_t *new_ccb;

	if (cpkg.c_cnt == 0) return (FALSE);

	/* first create a new ccb entry */

	new_ccb = (ccb_t *) malloc(sizeof(ccb_t));
	new_ccb->ccb_cid = cpkg.c_cnt+1;

	/* create a stack for this coroutine */

        new_ccb->ccb_stk = malloc(stk_sz);

	/* create the new coroutine */

        Make_Coroutine(new_ccb->ccb_stk,
		       stk_sz, func, arg,
		       &(new_ccb->ccb_ptr));

	/* hook it up */

	new_ccb->ccb_status = CRTN_READY;	/* it's ready */
	new_ccb->ccb_next = cpkg.c_cur->ccb_next;
	new_ccb->ccb_prev = cpkg.c_cur;
	new_ccb->ccb_next->ccb_prev = new_ccb;
	cpkg.c_cur->ccb_next = new_ccb;

	cpkg.c_cnt++;
	return(new_ccb->ccb_cid);
}

/* @SUBTITLE "delete_coroutine: delete a coroutine" */
/*
 * Deletes the coroutine specified and frees up its stack space.
 *
 *  Arguments:
 *	int cid - the coroutine id
 *
 *  Returns:
 *	error_t
 */

error_t
delete_coroutine(id)
    int id;
{
	ccb_t *ptr;

	if (cpkg.c_cnt == 0) return(CERR_NO_INIT);

	if (id == 1) return(CERR_DEL_INIT);	/* cannot delete initial */

	/* find the desired coroutine in our list */

	ptr = cpkg.c_head->ccb_next;
	while (ptr->ccb_cid != id) {
	    ptr = ptr->ccb_next;
	    if (ptr == cpkg.c_head)
		return(CERR_NO_EXIST); /* id does not exist */
	}

	/* remove this ccb from the list */

	ptr->ccb_prev->ccb_next = ptr->ccb_next;
	ptr->ccb_next->ccb_prev = ptr->ccb_prev;
	cpkg.c_cnt--;

	/* free this guy's stack */

	free (ptr->ccb_stk);

	/* free this ccb entry */

	free(ptr);

	return(CERR_OK);
}

/* @SUBTITLE "schedule_coroutine: change to new coroutine" */
/*
 * Switches context to another coroutine, if one is ready.
 *
 *  Notes:
 *	We use simple round-robin scheduling.  Since normally there will
 *	be a very small number of coroutines (probably just two), there
 *	is absolutely no sense in using any fancy algorithms.
 */

void
schedule_coroutine()
{
	ccb_t *ptr;
	ccb_t *cur;

	if (cpkg.c_cnt == 0) return;

	cur = cpkg.c_cur;
	ptr = cur->ccb_next;
	while (ptr->ccb_status != CRTN_READY) {
	    ptr = ptr->ccb_next;
	    if (ptr == cpkg.c_cur) return;	/* no one else is ready */
        }

	/* update the status of these two coroutines */

	if (cur->ccb_status == CRTN_RUNNING)
	    cur->ccb_status = CRTN_READY;
	ptr->ccb_status = CRTN_RUNNING;
	cpkg.c_cur = ptr;

	/* now transfer control */

	Transfer_Coroutine(&(cur->ccb_ptr), &(ptr->ccb_ptr));
}

/* @SUBTITLE "transfer_coroutine: switch to specific coroutine" */
/*
 * Switches context to a specific coroutine
 *
 *  Arguments:
 *	int id - the coroutine to switch to.
 *
 *  Returns:
 *	true if successful, false otherwise.
 */

error_t
transfer_coroutine(id)
    int id;
{
	ccb_t *ptr;
	ccb_t *cur;

	if (cpkg.c_cnt == 0) return(CERR_NO_INIT);

	if (id == cpkg.c_cur->ccb_cid) return (CERR_OK);

	cur = cpkg.c_cur;
	ptr = cur->ccb_next;
	while (ptr->ccb_cid != id) {
	    ptr = ptr->ccb_next;
	    if (ptr == cpkg.c_cur) return (CERR_NO_EXIST);
        }

	if (ptr->ccb_status != CRTN_READY) return (CERR_NOT_RDY);

	/* update the status of these two coroutines */

	cur->ccb_status = CRTN_READY;
	ptr->ccb_status = CRTN_RUNNING;
	cpkg.c_cur = ptr;

	/* now transfer control */

	Transfer_Coroutine(&(cur->ccb_ptr), &(ptr->ccb_ptr));
	return (CERR_OK);
}

/* @SUBTITLE "block_coroutine: block this coroutine" */
/*
 * Provides a means for the user to block a coroutine indefinitely.
 *
 *  Arguments:
 *	int id - the coroutine to block.
 *
 *  Returns:
 *	error_t
 */

error_t
block_coroutine(id)
    int id;
{
	ccb_t *ptr;

	if (cpkg.c_cnt == 0) return(CERR_NO_INIT);

	if ((id == 0) || (cpkg.c_cur->ccb_cid == id)) {	/* blocking self */
	    ptr = cpkg.c_cur->ccb_next;
	    while (ptr->ccb_status != CRTN_READY) {
		ptr = ptr->ccb_next;
		if (ptr == cpkg.c_cur)
		    return (CERR_ONLY_THRD);	/* can't block only one */
	    }

	    /* someone can run, so block this guy */

	    cpkg.c_cur->ccb_status = CRTN_BLOCKED;
	    schedule_coroutine();
	    return(CERR_OK);
	}

	/* blocking another coroutine - find it in the list */

	ptr = cpkg.c_cur->ccb_next;
	while (ptr->ccb_cid != id) {
	    ptr = ptr->ccb_next;
	    if (ptr == cpkg.c_cur) return(CERR_NO_EXIST); /* doesn't exist */
	}

	ptr->ccb_status = CRTN_BLOCKED;

	return(CERR_OK);
}

/* @SUBTITLE "unblock_coroutine: unblock a blocked routine" */
/*
 * Unblock a previously blocked coroutine.
 *
 *  Arguments:
 *	int id - the coroutine to unblock.
 *
 *  Returns:
 *	error_t
 */

error_t
unblock_coroutine(id)
    int id;
{
	ccb_t *ptr;

	if (cpkg.c_cnt == 0) return(CERR_NO_INIT);

	ptr = cpkg.c_cur->ccb_next;
	while (ptr->ccb_cid != id) {
	    ptr = ptr->ccb_next;
	    if (ptr == cpkg.c_cur) return (CERR_NO_EXIST);
	}

	if (ptr->ccb_status != CRTN_BLOCKED) return(CERR_NOT_BLKD);

	ptr->ccb_status = CRTN_READY;
	return(CERR_OK);
}

/* @SUBTITLE "coroutine_id: return current coroutine's id" */
/*
 * Returns the current coroutine's id.
 *
 *  Returns:
 *	the current coroutine's id.
 */

int
coroutine_id()
{
	if (cpkg.c_cnt == 0) return(0);
	return(cpkg.c_cur->ccb_cid);
}

/* @SUBTITLE "coroutine_status: status of a coroutine" */
/*
 * Returns the status of the indicated coroutine.
 *
 *  Arguments:
 *	int id - id of the coroutine in question.
 *	          Note that zero indicates the current coroutine.
 *
 *  Returns:
 *	The status of the specified coroutine, or zero if that
 *	coroutine cannot be found (i.e. it doesn't exist).
 */

int
coroutine_status(id)
    int id;
{
	ccb_t *ptr;

	if ((id == cpkg.c_cur->ccb_cid) || (id == 0))
	    return (cpkg.c_cur->ccb_status);

	/* not the current one so we'll have to look for it */

	ptr=cpkg.c_cur->ccb_next;
	while(ptr->ccb_cid != id) {
	    ptr = ptr->ccb_next;
	    if (ptr=cpkg.c_cur) return (0);
	}

	return (ptr->ccb_status);
}
