/* @TITLE "port: OPRA - Offline PRefetch Analyzer" */
/* 
 * port - PORTion identifier
 * 
 * David Kotz 1989
 *
 * This algorithm tries to notice regular sequential portions, and
 * identify regularity in the portion length and portion skip. This
 * then tries to do OBL as long as it thinks it's inside a portion.
 *
 * Immediate rereference counts as a correct prediction.
 * 
 * $Id: port.c,v 1.4 89/12/01 13:07:50 dfk Exp $
 */

#include <stdio.h>
#include "dfk.h"
#include "opra.h"

/* #define DEBUG */
/* #define PDEBUG */

/* idea is to use 2,2,2 for these parms */
/* same as BEST, for lps, if use 1,1,1 for these parms */
/* same as BEST, for lpr, if use 1,INF,INF for these parms */
/* same as OBL if use 1,INF,INF for these parms */
static int MinRun = 2;		/* number of blocks in a row to assume seq */
static int RunRep = 2;		/* number of runs in a row to assume regular */
static int SkipRep = 2;		/* number of skips in a row to assume regular */

/* use 1 for OBL-like behavior */
static int MaxDist = 5;		/* number of blocks ahead to predict */

static struct {
    /* current run */
    int first;				/* start of current run */
    int last;				/* last block read */
    int next;				/* next block we think will be read */
    int prefetch;			/* the last block prefetched or read */
    int nextpref;			/* the next block to prefetch */
    int distance;			/* number of blocks between last and prefetch */
    int runlen;			/* if regular, else 0 */
    int skiplen;			/* is regular, else 0 */
    /* previous runs */
    int same_runlen;		/* number of runs with same len */
    int prev_runlen;		/* length of previous run */
    int same_skiplen;		/* number of skips with same len */
    int prev_skiplen;		/* length of previous skip */
} port;					/* data for this algorithm */

static int predict();
#define NOPREDICT (-100)		/* value used for no prediction */

static int mistakes();
static void printnext();

/* @SUBTITLE "ParmsPORT: command-line parameters" */
void
ParmsPORT(argc, argv)
	int argc;
	char **argv;
{
    if (argc != 4) {
	   fprintf(stderr, "%s: PORT requires 4 parameters\n", program);
	   exit(EXIT_USAGE);
    }
    MinRun = atoi(argv[0]);
    RunRep = atoi(argv[1]);
    SkipRep = atoi(argv[2]);
    MaxDist = atoi(argv[3]);
    if (MinRun <= 0) {
	   fprintf(stderr, "%s: PORT must have MinRun > 0\n", program);
	   exit(EXIT_PARM);
    }
    if (MaxDist <= 0) {
	   fprintf(stderr, "%s: PORT must have MaxDist > 0\n", program);
	   exit(EXIT_PARM);
    }
#ifdef DEBUG
    printf("MinRun = %d, RunRep = %d, SkipRep = %d, MaxDist = %d\n",
		 MinRun, RunRep, SkipRep, MaxDist);
#endif
}

/* @SUBTITLE "ResetPORT: Reset data structures" */
void
ResetPORT()
{
    port.first = port.last = port.next = NOPREDICT;
    port.prefetch = port.last;
    port.nextpref = NOPREDICT;
    port.distance = 0;
    port.runlen = port.skiplen = 0;
    port.same_runlen = port.same_skiplen = 0;
    port.same_runlen = port.same_skiplen = 0;
    port.prev_runlen = port.prev_skiplen = 0;
}

/* @SUBTITLE "RefPORT: Process block Reference" */
boolean					/* did we correctly predict block */
RefPORT(block, mistake)
	int block;			/* (input) block number of reference */
	int *mistake;			/* (output) how many mistakes */
{
    int runlen;
    int skiplen;
    boolean predicted;

    if (port.last < 0) {
	   /* first reference in this sequence */
	   port.first = port.last = block;
	   *mistake = 0;
	   predicted = FALSE;
    } else if (block == port.last) {    /* immediate rereference */
	   /* port.next does not change */
	   predicted = TRUE;
	   *mistake = 0;
    } else {				/* not a rereference */
	   /* how did our prediction go? */
	   predicted = (block == port.next);
	   if (!predicted)
		*mistake = mistakes();
	   else
		*mistake = 0;

	   /* out-of-order reference, update run/skip lengths */
	   if (block != port.last + 1) {
		  runlen = port.last - port.first + 1;
		  skiplen = block - port.last;
		  port.first = port.last = block;
		  if (runlen == port.prev_runlen) {
			 /* same run length */
			 port.same_runlen++;
			 if (skiplen == port.prev_skiplen)
			   /* same run length, same skip length */
			   port.same_skiplen++;
			 else 
			   /* same run length, different skip length */
			   port.same_skiplen = 1;
		  } else {
			 /* different run length, any skip length */
			 port.same_runlen = 1;
			 port.same_skiplen = 1;
		  }
		  port.prev_runlen = runlen;
		  port.prev_skiplen = skiplen;
#ifdef DEBUG
		  fprintf(stderr, "run length %d, skip %d\n", runlen, skiplen);
#endif DEBUG
		  port.runlen 
		    = (port.same_runlen >= RunRep) ? port.prev_runlen : 0;
		  port.skiplen
		    = (port.runlen > 0 && port.same_skiplen >= SkipRep) ?
			 port.prev_skiplen : 0;
	   } else
		port.last = block;
    }

    if (predicted) {
	   if (port.distance > 0)
		port.distance--;
    }
    if (port.distance == 0)
	 port.prefetch = port.last;

#ifdef PDEBUG
    if (!predicted)
	 fprintf(stderr, "incorrectly predicted %d and got %d\n", 
		    port.next, block);
#endif PDEBUG
    
    /* predict the next block to be referenced */
    port.next = 
	 predict(port.first, 
		    port.runlen,
		    port.skiplen,
		    port.last,
		    port.last,	/* pretend none prefetched */
		    0);			/* pretend none prefetched */

#ifdef PDEBUG
    printnext();
#endif PDEBUG

    /* and return status of last prediction */
    return(predicted);
}

/* @SUBTITLE "predict: predict next block for prefetch" */
/* parameters give the current state */
/* see notebook for 11/23/89 */
static int				/* next block to be prefetched */
predict(s, r, h, last, p, d)
	int s;				/* start of current run */
	int r;				/* runlen if regular else 0 */
	int h;				/* skiplen if regular else 0 */
	int last;				/* last block referenced */
	int p;				/* last block prefetched */
	int d;				/* number of blocks between p and last */
{
    /* limit our prefetching to a certain distance */
    /* for irregular runs, we use the distance based on current runlen */
    /* for regular runs, we predict up to MaxDist ahead */
    if ((r == 0 && d >= distance(last-s+1)) || (r > 0 && d >= MaxDist))
	 return(NOPREDICT);

    if (h == 0)
	 /* no regular skip known */
	 if (r == 0 || (p-s+1) < r)
	   return(p+1);		/* still in run */
	 else
	   return(NOPREDICT);	/* finished run */
    else 
	 /* regular run, skip known */
	 if ((d+last-s+1) % r > 0)
	   return(p+1);		/* still in run */
	 else
	   return(p+h);		/* time for a skip */

}

/* @SUBTITLE "distance: how far ahead do we predict?" */
/* This is the most tunable portion of the prediction algorithm */
static int 
distance(runlen)
	int runlen;
{
    if (runlen < MinRun)		/* too short - none */
	 return(0);
    else if (runlen >= MinRun + MaxDist) /* very long - cut off */
	 return(MaxDist);
    else					/* linear guess in between */
	 return (runlen - MinRun + 1);
}

/* @SUBTITLE "mistakes: process mistakes" */
/* assume all prefetches to be mistaken */
static int
mistakes()
{
    int s = port.first;		/* first in run */
    int r = port.runlen;	/* run length, if regular */
    int h = port.skiplen;	/* skip length, if regular */
    int last = port.last;	/* last reference */
    int p;				/* block number for loop */
    int d;				/* loop distance variable */
    int count = 0;			/* count of mistakes */

    p = last;
    d = 0;

    do {
	   p = predict(s, r, h, last, p, d);
	   d++;
	   if (p != NOPREDICT)
		count++;
    } while(p != NOPREDICT);

    /* now reset prediction status */
    port.distance = 0;
    port.prefetch = port.last;
    port.nextpref = NOPREDICT;

    return(count);
}

/* @SUBTITLE "printnext: print out predictions" */
static void
printnext()
{
    int s = port.first;		/* first in run */
    int r = port.runlen;	/* run length, if regular */
    int h = port.skiplen;	/* skip length, if regular */
    int last = port.last;	/* last reference */
    int p;				/* block number for loop */
    int d;				/* loop distance variable */

    p = last;
    d = 0;

    printf("prediction is ");
    do {
	   printf("%d, ", p);
	   p = predict(s, r, h, last, p, d);
	   d++;
    } while(p != NOPREDICT);
    printf("\n");
}

/* @SUBTITLE "LastPORT: What happens at end of string" */
void
LastPORT(mistake)
	int *mistake;			/* how many mistakes? */
{
    *mistake = mistakes();
}

