/* @TITLE "main: OPRA - Offline PRefetch Analyzer" */
/* 
 * opra - Offline PRefetch Analysis program. 
 * 
 * David Kotz 1989
 *
 * This program is intended to analyze different prefetch algorithms 
 * for their prediction capabilities. Any analysis of overhead or 
 * realistic hit ratios needs to be done in real time. This program 
 * may be extended to provide maximum hit rates, or mistake rates.
 * 
 * This handles local patterns only.
 *
 * Interprocess locality is not considered.
 *
 * Usage: opra [-alg] pattern [parms]
 * where
 *    alg is one of the prefetch algorithm names (default none);
 *    pattern is the pattern description file. 
 *    parms are optional parameters to the algorithm
 * Output
 *  five columns: procnum nrefs #reref #correct #mistakes
 *  (no output when nrefs==0)
 */

#include <stdio.h>
#include "dfk.h"
#include "opra.h"

static void usage();
static enum alg GetAlg();
static void ParseParms();
static void AnalyzeOneProc();
static void ResetPrefetch();
static boolean Reference();
static void LastPrefetch();

enum alg {NONE, BEST, OBL, PORT, ADAPT} algorithm = NONE;

char *program;				/* name of program */

/* @SUBTITLE "main - startup code" */

main(argc, argv)
	int argc;
	char **argv;
{
    char *filename;			/* name of pattern file */
    FILE *pattern;			/* pattern file */
    int proc;				/* processor number */
    int nrefs;				/* number of references in pattern */
    int correct;			/* number of correct predictions */
    int rereference;		/* number of immediate rereferences */
    int mistakes;			/* number of mistake-prefetches */

    program = argv[0];		/* remember name of program */
    argc--; argv++;

    if (argc == 0)
	 usage();
    if (*argv[0] == '-') {
	   algorithm = GetAlg(++argv[0]);
	   argc--;
	   argv++;
    }
    if (argc < 1)
	 usage();
    filename = argv[0];
    argc--; argv++;

    pattern = fopen(filename, "r"); /* open the file */
    if (pattern == (FILE *)NULL) {
	   fprintf(stderr, "%s: cannot open file '%s'\n", program, filename);
	   exit(EXIT_PARM);
    }

    if (argc > 0)
	 ParseParms(argc, argv);

    /* cycle through processors */
    for (proc = 0; !feof(pattern); proc++) {
	   AnalyzeOneProc(pattern, &nrefs, &correct, &rereference, &mistakes);
	   if (nrefs > 0)
		printf("%d %d %d %d %d\n", proc, nrefs,
			  rereference, correct, mistakes);
    }

    fclose(pattern);
}

/* @SUBTITLE "usage: complain about argument usage" */
static void
usage()
{
    fprintf(stderr, "usage: %s [-alg] patternfile\n", program);
    exit(EXIT_USAGE);
}

/* @SUBTITLE "GetAlg: parse the algorithm name" */
static enum alg
GetAlg(name)
	char *name;
{
    if (strcmp(name, "NONE") == 0)
	 return(NONE);
    else if (strcmp(name, "BEST") == 0)
	 return(BEST);
    else if (strcmp(name, "OBL") == 0)
	 return(OBL);
    else if (strcmp(name, "PORT") == 0)
	 return(PORT);
    else if (strcmp(name, "ADAPT") == 0)
	 return(ADAPT);
    else {
	   fprintf(stderr,
			 "%s: valid algorithms are NONE, BEST, OBL, PORT, and ADAPT\n",
			 program);
	   exit(EXIT_PARM);
    }
}

/* @SUBTITLE "ParseParms: Read Parameters for pattern" */
static void
ParseParms(argc, argv)
	int argc;
	char **argv;
{
    switch(algorithm) {
	   case NONE: fprintf(stderr, "Extraneous parameters for NONE\n"); break;
	   case BEST: fprintf(stderr, "Extraneous parameters for BEST\n"); break;
	   case OBL: fprintf(stderr, "Extraneous parameters for OBL\n"); break;
	   case PORT: ParmsPORT(argc, argv); break;
	   case ADAPT:ParmsADAPT(argc, argv); break;
	   default: {
		  fprintf(stderr, "%s: unknown algorithm in ParseParms()\n", 
				program);
		  exit(EXIT_INTERR);
	   }
    }	   
}

/* @SUBTITLE "AnalyzeOneProc: analyze one processor's pattern" */
/* Read a set of references from the file, for one processor. 
 * This processor's set is terminated by a negative reference or
 * end-of-file.
 */
static void
AnalyzeOneProc(file, nrefs, correct, rereference, mistakes)
	FILE *file;
	int *nrefs;
	int *correct;
	int *rereference;
	int *mistakes;
{
    int block;
    int last = -100;
    int mistake;

    *nrefs = 0;
    *correct = 0;
    *rereference = 0;
    *mistakes = 0;

    ResetPrefetch();

    while (fscanf(file, "%d", &block) == 1 && block >= 0) {
	   (*nrefs)++;
	   if (block == last)
		(*rereference)++;
	   if (Reference(block, &mistake))
		(*correct)++;
	   (*mistakes) += mistake;
	   last = block;
    }

    LastPrefetch(&mistake);
    (*mistakes) += mistake;
}

/* @SUBTITLE "ResetPrefetch: reset the prefetch algorithm" */
static void
ResetPrefetch()
{
    switch(algorithm) {
	   case NONE: ResetNONE(); break;
	   case BEST: ResetBEST(); break;
	   case OBL: ResetOBL(); break;
	   case PORT: ResetPORT(); break;
	   case ADAPT: ResetADAPT(); break;
	   default: {
		  fprintf(stderr, "%s: unknown algorithm in ResetPrefetch()\n", 
				program);
		  exit(EXIT_INTERR);
	   }
    }	   
}

/* @SUBTITLE "Reference: reset the prefetch algorithm" */
static boolean
Reference(block, mistake)
	int block;
	int *mistake;
{
    switch(algorithm) {
	   case NONE: return(RefNONE(block, mistake));
	   case BEST: return(RefBEST(block, mistake));
	   case OBL: return(RefOBL(block, mistake));
	   case PORT: return(RefPORT(block, mistake));
	   case ADAPT: return(RefADAPT(block, mistake));
	   default: {
		  fprintf(stderr, "%s: unknown algorithm in Reference()\n", 
				program);
		  exit(EXIT_INTERR);
	   }
    }	   
}

/* @SUBTITLE "LastPrefetch: what happens at EOF" */
static void
LastPrefetch(mistake)
	int *mistake;
{
    switch(algorithm) {
	   case NONE: LastNONE(mistake); break;
	   case BEST: LastBEST(mistake); break;
	   case OBL: LastOBL(mistake); break;
	   case PORT: LastPORT(mistake); break;
	   case ADAPT: LastADAPT(mistake); break;
	   default: {
		  fprintf(stderr, "%s: unknown algorithm in LastPrefetch()\n", 
				program);
		  exit(EXIT_INTERR);
	   }
    }	   
}

