/* @TITLE "GAPS -- Global-Access-Pattern Sequentiality" */
/* 
 * GAPS: Global-Access-Pattern Sequentiality.
 * 
 * This is a method to detect sequentiality in global access 
 * patterns. See notebook for 2/27/90. Assumes 20 processes in pattern.
 * This further tests the continuation phase of sequentiality.
 * See notebook for 3/2-3/4. Switched from slope-near-1 to 
 * coefficient-of-determination-near-1, since slope may be as low 
 * as 1/NPROCS. Don't calculate slope/coef once in continuation mode;
 * we use maxjump. We flush queue of that process on a jumpback.
 * We separate the mode to a per-process basis. Implemented as a module.
 * Extended Watch() to keep track of portion start when the portion grows
 * beyond the context size. Added code to use regularity of portion 
 * length and skip to ride GPS with no loss of sequentiality.
 * Capped the value of maxjump to 3*nprocs; large values come from 
 * procs reading large chunks, which are really locally sequential portions. 
 * So we need not handle them here.
 *
 * David Kotz March 1990
 */

/* $Id: gaps.c,v 1.8 90/03/29 15:36:47 dfk Exp $ */

#include <stdio.h>
#include "dfk.h"
#include "queuestack.h"

static char *format="%2d %4d %d %d %d %d %.3f  %c\n";
# define Trace(node, block, n, cutoff, sample, usedcount, coef, seq) \
  printf(format, \
		(node), (block), (n), (cutoff), (sample), (usedcount), (coef), (seq))

#define NPROCS 20			/* how many procs are in the pattern */

#define CONTEXTSIZE 100		/* examine this many references */
#define FILESIZE 8000		/* maximum size of file in blocks */
#define MINSAMPLE 4			/* must have a sample of at least this */
#define MAXSLOPE 3.0		/* to put a limit on maxjump */

#define HUGEINT 9999999999	/* bigger than any sector number I'll ever use */

static int RunRep = 2;		/* number of runs in a row to assume regular */
static int SkipRep = 2;		/* number of skips in a row to assume regular */

/* This structure contains the entire state of the GAPS predictor. */
static struct gapsdata {
    int nprocs;			/* == NPROCS */
    int last[NPROCS];		/* last block number used by each proc */
    boolean sequential[NPROCS]; /* for each node, is it sequential? */
    int num_sequential;		/* number of nodes thinking sequential */
    QUEUE *context;			/* the most recent blocks; FIFO queue of REFs */

    float coef;			/* the coefficient of determination */
#define TOLERANCE (0.8)		/* coef of determination must be greater */
    int maxjump;			/* maximum allowed jump past maxlast */

    int start;
    int minlast;
    int maxlast;

    boolean portion1;		/* true if first portion */
    int same_runlen;		/* number of runs with same len */
    int prev_runlen;		/* length of previous run */
    int same_skiplen;		/* number of skips with same len */
    int prev_skiplen;		/* length of previous skip */
    int prev_end;			/* end of previous portion */
    int exp_end;			/* expected end of next portion */

    int runlen;			/* run length if regular, else 0 */
    int skiplen;			/* skip length if regular, else 0 */
} gaps;

/* One of the entries in the context queue */
typedef struct ref_struct REF;
struct ref_struct {
    int block;				/* block number */
    int node;				/* processor node number */
};

static struct sme {
    int lastcount;			/* how many nodes share this as last */
    int usedcount;			/* how many uses for this block */
} map[FILESIZE];

/* @PAGE */
/* MACROS */
#define sqr(x) ((x)*(x))

/* FUNCTIONS */
void InitGAPS();
void NotifyGAPS();
void DoneGAPS();

static boolean Watch();
static boolean CheckComplete();
static boolean CheckSlope();
static void JumpBack();
static boolean Contin();
static void ToSequential();
static void FromSequential();
static void Finalize();
static void NewPortion();
static boolean ExtendPortion();
static int JumpGap();
static void EndPortion();
static REF *NewRef();

extern double fabs();
extern double ceil();
extern char *malloc();

/* @SUBTITLE "InitGAPS - initialize data structures" */
/* This function allocates and initializes the GAPS data structures. Only 
 * one process will do most of the work (process 0). The others wait
 * until the gaps pointer is ready, and then do per-process initialization.
 */
void
InitGAPS()
{
    int i;

    gaps.nprocs = NPROCS;
    gaps.context = MakeQueue(CONTEXTSIZE);

    for (i = 0; i < gaps.nprocs; i++) {
	   gaps.last[i] = -1;
	   gaps.sequential[i] = FALSE;
    }

    gaps.num_sequential = 0;
    gaps.coef = 0;
    gaps.maxjump = 0;
    gaps.start = -1;
    gaps.minlast = -1;
    gaps.maxlast = 0;

    gaps.portion1 = TRUE;
    gaps.same_runlen = 0;  
    gaps.prev_runlen = 0;  
    gaps.same_skiplen = 0; 
    gaps.prev_skiplen = 0; 
    gaps.prev_end = 0;     
    gaps.exp_end = 0;

    gaps.runlen = 0;
    gaps.skiplen = 0;

    bzero(map, FILESIZE * sizeof(struct sme));
}

/* @SUBTITLE "NotifyGAPS - update for a new reference" */
/* This is called with each sector that is accessed. Currently, this function 
 * is protected with a FIFO entry lock so that only one process will be 
 * in the code at a time. The whole predictor is thus serialized. 
 * The return value has to be a boolean for consistency with others,
 * but the return value is irrelevant unless WANTED is defined. This 
 * is not allowed (see above), so our return value never means anything.
 *  If this process is in sequential mode, Contin() is called. Otherwise
 * Watch() is called. Rereferences are ignored.
 */
void
NotifyGAPS(block, node)
	int block;
	int node;
{
    if (block == gaps.last[node]) /* ignore rereferences by same node */
	 return;

    if (gaps.sequential[node]) {
	   /* check sequentiality */
	   if (!Contin(block, node))
		FromSequential(block, node);
    } else {
	   if (Watch(block, node))
		ToSequential();
    }
}

/* @SUBTITLE "Watch: watch the pattern for sequentiality" */
/* This function is for tracking blocks when no sequentiality has been 
 * detected. The blocks are entered into the context queue, which
 * retains the most recent references for each processor, that may be 
 * combined to form a sequential portion. Once all processors have
 * contributed to the queue, two conditions are evaluated to check for
 * sequentiality: 1) completeness, when all of the queue entries less 
 * than the current minlast are contiguous, and 2) slope, when the 
 * ordered queue entries closely approximate a line. 
 */
static boolean
Watch(block, node)
	int block;
	int node;
{
    int n;
    REF *r;
    REF *dummy;

    /* Jumpback? If so, we remove all of this process's entries 
	* from the queue. Orderedness is am important criteria for sequentiality. 
	*/
    if (block < gaps.last[node]) {
	   JumpBack(node);
	   gaps.minlast = -1;
	   gaps.start = -1;
    }

    /* update this node's last block */
    gaps.last[node] = block;
	   
    /* Add to queue. If queue is full, flush the oldest block. */
    r = NewRef(block, node);
    if (Enqueue(gaps.context, r) == QS_OVERFLOW) {
	   Dequeue(gaps.context, &dummy);
	   Enqueue(gaps.context, r);
    }

    /* All processors must be participating. */
    n = Inqueue(gaps.context);
    if (n <= gaps.nprocs || gaps.num_sequential > 0) {
	   Trace(node, block, n, 0, 0, 0, 0.0, 'n');
	   return(FALSE);
    }
    /* note here that gaps.sequential[i] is FALSE for all i */

    /* Now we check for completeness: all of the blocks in the queue
	* that are less than minlast are contiguous in the file.
	*/
    if (!CheckComplete(node, block)) {
	   /* trace output already done */
	   return(FALSE);
    }

    /* So, completeness has been established. It is possible for 
	* the reference pattern to be complete without having a definite 
	* direction. So now we check the slope (or, more accurately, the
	* coefficient of determination). The block sequence is fit
	* to a straight line; the quality of this fit must be above a certain 
	* tolerance to be considered sequential.
	*/
    if (!CheckSlope()) {
	   Trace(node, block, n, 0, 0, 0, gaps.coef, 'n');
	   return(FALSE);
    }

    /* Now, we have confirmed that the pattern is sequential. 
	* We will enter Continuation mode, instead of Watching mode, 
	* to follow the progress of the pattern.
	*/
    Trace(node, block, n, 0, 0, 0, gaps.coef, 'y');

    return(TRUE);
}

/* @SUBTITLE "CheckComplete: Check for completeness." */
static boolean
CheckComplete(node, block)
	int node;				/* processor */
	int block;			/* block in this reference */
{
    int b;				/* a block number */
    int sample = 0;			/* size of sample <= cutoff */
    int used[CONTEXTSIZE];	/* used counts in sample area */
    int usedcount = 0;		/* total contiguous usedcount */
    int cutoff;			/* minlast */
    boolean complete;		/* is it complete? */
    REF *r;				/* a reference off queue */
    int n;				/* number in queue */
    int i;				/* loop index */

    /* Compute minlast as cutoff. */
    cutoff = HUGEINT;
    for (i = 0; i < gaps.nprocs; i++) {
	   if (gaps.last[i] < cutoff)
		cutoff = gaps.last[i];
    }

    if (cutoff <= 0) {
	   Trace(node, block, n, cutoff, 0, 0, 0.0, 'n');
	   return(FALSE);
    }

    /* Now we check for completeness. If completeness was noticed before, 
	* the cutoff was stored as gaps->minlast. If the cutoff has not changed, 
	* completeness still holds. Otherwise, we may have to update the
	* completeness.
	*/
    if (cutoff == gaps.minlast) {
	   complete = TRUE;
    } else {
	   /* initialize used table */
	   bzero(used, sizeof(int) * CONTEXTSIZE);
	   
	   complete = TRUE;
	   
	   /* We look through the entire queue, checking each block against the
	    * cutoff. If the block is less than the new cutoff, but larger 
	    * than the new cutoff, it is part of the sample that is interesting.
	    * This sample may not fit in the used[] array; if so, then it
	    * cannot be complete, since used[] can hold the whole queue if
	    * it were all one complete sample. Otherwise, we count the occurrences
	    * of each block in the sample.
	    */
	   n = Inqueue(gaps.context);
	   for (i = 0; i < n; i++) {
		  Peekqueue(gaps.context, i, &r);
		  b = r->block;
		  if (b <= cutoff && b > gaps.minlast) {
			 sample++;
			 if (cutoff-b < CONTEXTSIZE)
			   used[cutoff-b]++;
			 else {
				complete = FALSE;
				break;
			 }
		  }
	   }
	   
	   /* If it is still complete, then all blocks in the sample were 
	    * able to fit into used[]. However, it is possible that there
	    * is still a gap in the blocks in the sample. So here we 
	    * trace backward from the cutoff to the first unused block,
	    * counting occurrences. If this total is the same as the sample 
	    * size, we have a complete set. Otherwise this unused block
	    * represents a gap before which there were other used blocks.
	    * If we decide it is complete, then we can record the start
	    * of the portion to be the start of the complete area. If 
	    * we are simply updating completeness, then the start was
	    * already recorded and we need only record the cutoff. 
	    */
	   if (complete)
		if (gaps.minlast >= 0 || sample >= MINSAMPLE) {
		    /* verify completeness more carefully */
		    for (i = 0, usedcount = 0; i < CONTEXTSIZE && used[i] > 0; i++)
			 usedcount += used[i];
		    if (usedcount == sample) {
			   complete = TRUE;
			   if (gaps.start < 0)
				gaps.start = cutoff-i+1;
			   gaps.minlast = cutoff;
		    } else
			 complete = FALSE;
		} else
		  complete = FALSE;
    }

    if (!complete) {
	   Trace(node, block, n, cutoff, sample, usedcount, 0.0, 'n');
	   gaps.minlast = -1;
	   gaps.start = -1;
    }

    return(complete);
}

/* @SUBTITLE "JumpBack: delete refs from  context queue" */
/* When in watching mode, if a processor exhibits a jump-back
 * (referencing a block whose number is lower than that processor's 
 * previous reference), we must remove all of that processor's 
 * references from the context queue. This is because, a globally sequential
 * portion does not have processors doing jumpbacks. Since we are looking 
 * for portions, we can safely assume the references before the jumpback
 * were for another portion and can be removed. 
 */
static void
JumpBack(node)
	int node;
{
    REF *r;
    int n;
    
    n = Inqueue(gaps.context);

    while (n-- > 0) {
	   Dequeue(gaps.context, &r);
	   if (r->node == node)
		free(r);
	   else
		Enqueue(gaps.context, r);
    }
}

/* @SUBTITLE "CheckSlope: update the recent-blocks slope" */
/* We treat the context queue as a function, with x represented by count,
 * and y represented by the block number. We are interested in the slope 
 * of this function, hopefully a line. Although we compute the slope,
 * we check the coefficient of determination (see below) to determine
 * whether the function fits the line well.
 */
static boolean				/* passes as sequential? */
CheckSlope()
{
    int n;				/* number in queue */
    int block;				/* a block number */
    int sumx, sumx2, sumxy, sumy; /* significant sums */
    float slope;			/* the slope */
    float inter;			/* the intercept */
    float y;				/* predicted y value */
    float ymean;			/* the mean block number */
    float sum1, sum2;		/* two parts to the coef of determination */
    REF *r;				/* a reference */
    int i;

    n = Inqueue(gaps.context);

    /* not enough data */
    if (n < gaps.nprocs) {
	   gaps.coef = 0;
	   gaps.maxjump = 0;
	   return(FALSE);
    }

    /* find significant sums */
    sumx = sumx2 = sumxy = sumy = 0;
    for (i = 0; i < n; i++) {
	   Peekqueue(gaps.context, i, &r);
	   block = r->block;
	   sumxy += i*block;
	   sumy += block;
    }
    sumx = (n-1) * n / 2;	/* sum of i, i=0,1,...(n-1) */
    sumx2 = (n-1)*n*(2*n-1)/6; /* sum of sqr(i), i=0,1,...(n-1) */

    /* find the slope */
    slope = (float)(n * sumxy - sumx*sumy) / (n * sumx2 - sqr(sumx));

    /* negative slope means we are NOT sequentially increasing */
    if (slope < 0) {
	   gaps.coef = 0;
	   gaps.maxjump = 0;
	   return(FALSE);
    }

    /* used later in Continuation mode */
    gaps.maxjump = ceil(min(slope, MAXSLOPE) * gaps.nprocs);

    /* @PAGE */
    /* Compute the coefficient of determination:
	* A measure of the deviation from the linear fit.
	* See Trivedi Pages 546--548.
	*/
    ymean = (float) sumy / n;
    inter = ymean - slope * sumx / n;
    y = inter;				/* start at (0,inter) */
    sum1 = sum2 = 0;
    for (i = 0; i < n; i++) {
	   Peekqueue(gaps.context, i, &r);
	   block = r->block;
	   sum1 += sqr(y - ymean);
	   sum2 += sqr(block - ymean);
	   y += slope;
    }
    gaps.coef = sum1/sum2;

    /* 0 <= Coef <= 1, and is better near 1. */
    return(gaps.coef >= TOLERANCE);
}

/* @SUBTITLE "ToSequential: change into sequential mode" */
/* When the Watcher notices sequentiality, this function is called to 
 * send all processors into sequential mode. This also involves
 * breaking down the old data structure (the queue) and building the
 * new one (in the sector map table, conveniently). 
 */
static void
ToSequential()
{
    int i;
    int block;
    REF *r;
    int maxlast;
    int last;

    /* 'start' and 'minlast' were set by watch() */

    /* Set the last counters in the table. The last count indicates
	* how many processes have that block as their most-recently-referenced
	* block. The count will be decremented as the processes move on.
	*/
    maxlast = -1;
    for (i = 0; i < gaps.nprocs; i++) {
	   last = gaps.last[i];
	   map[last].lastcount++;
	   if (last > maxlast)
		maxlast = last;

	   gaps.sequential[i] = TRUE;
    }
    gaps.maxlast = maxlast;

    gaps.num_sequential = gaps.nprocs;

    /* Copy queue to used bits in the table. This indicates which blocks
	* have been used. These will be cleared as the minlast moves through
	* the table, and are used for detecting gaps in the sequence, and
	* for determining which blocks might be prefetched. 
	*/
    while(Dequeue(gaps.context, &r) == QS_OK) {
	   block = r->block;
	   if (block >= gaps.minlast)
		map[block].usedcount++;
    }

    printf("-> starting sequential from %d, maxjump %d\n", 
		 gaps.start, gaps.maxjump);

    /* Record the new portion beginning now. */
    NewPortion();
}


/* @SUBTITLE "Contin: watch the pattern as it continues to be sequential" */
/*   This is Continuation mode. Once sequentiality is noted, we shift modes and
 * track the sequentiality in a much more efficient manner. Thus, Notify
 * calls this function instead of Watch(). We return FALSE here if we 
 * fall out of sequentiality.
 *   The efficiency is possible since we know the pattern will be active
 * in a compact area, and we don't need the ordering information from the
 * queue so much as easy access to minlast/maxlast and completeness
 * information.
 *   The sector map table is used to store our information. We record
 * the lastcount, a count for each block of the number of processes that
 * have that block as their most-recently accessed, and a "used" bit for each 
 * block.
 *   We watch for jumpbacks, sure signs of sequentiality breaking, and 
 * forward jumps that are too far to be explained. In the process of
 * checking the slope we obtained 'maxjump', the furthest any process
 * should extend the maxlast at any one jump. Forward jumps longer
 * than maxjump past maxlast are also taken to be a loss in sequentiality
 * (this is modified when the portions are highly regular, and we can take
 * jumps over portion skip in stride; see ExtendPortion). 
 *   We must also maintain minlast and maxlast in order to check for
 * completeness failures (which arise due to small jumps and portion skips). 
 * Minlast is updated whenever the lastcount of a block goes to zero, and
 * that block was minlast. To find the new minlast, we sequentially scan
 * the table until another block with a nonzero lastcount is found. In the
 * process, we can look for blocks that were not used. Any found are
 * a completeness failure. We let JumpGap handle this case.
 */
static boolean
Contin(block, node)
	int block;
	int node;
{
    int i;
    int oldlast;
    
    oldlast = gaps.last[node];
    gaps.last[node] = block;

    map[oldlast].lastcount--;
    
    if (block < oldlast) {
	   printf("-> %d jumpback from %d on node %d\n", 
			block, oldlast, node);
	   return(FALSE);		/* a jump-back: cancel sequential */
    }

    map[block].usedcount++;

    if (block > oldlast) {	/* we ignore block==oldlast */
	   if (block > gaps.maxlast) {
		  /* update maxlast */
		  if (ExtendPortion(oldlast, block))
		    gaps.maxlast = block;
		  else {
			 printf("-> %d jump too large %d > %d\n",
				   block, block-gaps.maxlast, gaps.maxjump);
			 return(FALSE);
		  }
	   }
	   map[block].lastcount++;
	   if (gaps.minlast == oldlast && map[oldlast].lastcount == 0) {
		  /* update minlast */
		  for (i = oldlast; map[i].lastcount == 0; ) {
			 if (map[i].usedcount == 0) {
				/* completeness failure */
				i = JumpGap(i);
			 } else
			   map[i++].usedcount = 0; /* reset */
		  }
		  gaps.minlast = i;
	   }
    }	   

    printf("%2d %4d %d %d  y\n",
		 node, block, gaps.minlast, gaps.maxlast);

    return(TRUE);
}

/* @SUBTITLE "FromSequential: change out of sequential mode" */
/* This function handles any break from sequentiality. It is called for
 * each processor as it leaves the portion, either due to a jumpback or 
 * a too-long jump forward. [Of course, if not all processors leave the 
 * portion in this way, we're stuck.]  First this we do is to limit the
 * prefetch to maxlast; prefetching to their is safe. Otherwise, 
 * we set our sequentiality flag to FALSE, and enqueue this block on the 
 * context queue to be part of a future portion.
 */
static void
FromSequential(block, node)
	int block;
	int node;
{
    int n;

    gaps.sequential[node] = FALSE;
    printf("-> process %d ending sequential\n", node);

    if (--gaps.num_sequential == 0)
	 Finalize();

    /* Add this latest block and start over. */
    Enqueue(gaps.context, NewRef(block, node));

    n = Inqueue(gaps.context);
    printf("%2d %4d %d x x x x  n\n", node, block, n);
}

/* @SUBTITLE "Finalize: Finish off this sequential portion" */
/* This function makes the final transition from Continuation mode to 
 * Watching mode, as the last processor leaves continuation mode.
 * We have to clear the table of all the used bits and last counts,
 * process all mistakes, stop prefetching, and end the portion. 
 * We also reset all state to be ready for Watching mode.
 */
static void
Finalize()
{
    int i;
    int end;

    /* clean out the array */
    end = gaps.minlast;
    while (end <= gaps.maxlast && map[end].usedcount > 0) {
	   map[end].usedcount = 0;
	   map[end].lastcount = 0;
	   end++;
    }
    for (i = end; i <= gaps.maxlast; i++) {
	   map[i].usedcount = 0;
	   map[i].lastcount = 0;
    }
    end--;				/* first loop overshoots by one */

    EndPortion(end);

    gaps.minlast = -1;
    gaps.start = -1;
}

/* @SUBTITLE "NewPortion: Starting a new portion" */
/* This is the first of several functions for identifying and
 * tracking portions. We record the skip from the previous portion, 
 * if any, and look for regularity. Also, since this occurs at the 
 * start of sequentiality, we use any regularity in the portion
 * length or skip to set up limits for prefetching.
 */
static void
NewPortion()
{
    int skip = gaps.start - gaps.prev_end;

    /* the portion started in gaps.start */

    /* check skip */
    if (gaps.same_skiplen == 0 || skip != gaps.prev_skiplen) {
	   gaps.same_skiplen = 1; 
	   gaps.prev_skiplen = skip;
    } else
	 gaps.same_skiplen++;

    /* Default is irregular portions */
    gaps.runlen = 0;
    gaps.skiplen = 0;

    /* now look for regularity */
    if (gaps.same_runlen >= RunRep) {
	   /* run length is regular */
	   gaps.runlen = gaps.prev_runlen;
	   gaps.exp_end = gaps.start + gaps.runlen - 1;
	   if (gaps.same_skiplen >= SkipRep) {
		  /* skip length is regular */
		  gaps.skiplen = gaps.prev_skiplen;
		  printf("-> skip is regular %d\n", skip);
	   }
    }
}

/* @SUBTITLE "ExtendPortion: Record this sequential portion" */
/*  If the portion length and skip are regular, then we allow
 * jumps over the skip if they still meet the maxjump criteria. 
 * The skip must be positive and regular, and the length must be regular. 
 * The previous reference must be in the current portion, and the
 * current reference in the expected next portion. 
 *  Then, we allow a jump of maxjump plus the skip length.
 *  If the current reference is in the area we expect to skip, the 
 * regularity is assumed broken and we reset the flag. 
 *  If we do not use the regularity, we do a simple maxjump check on the
 * jump.
 */

static boolean				/* is this a valid extension? */
ExtendPortion(last, block)
	int last;				/* last block ref'd by this proc */
	int block;			/* the most recent reference */
{
    if (gaps.skiplen > 0)	/* regular, and positive skip */
	 if (block > gaps.exp_end) {
		if (last <= gaps.exp_end)
		  if (block >= gaps.exp_end + gaps.skiplen)
		    if (block - gaps.maxlast < gaps.maxjump + gaps.skiplen)
			 return(TRUE);
		    else
			 return(FALSE);
		  else
		    gaps.skiplen = 0; /* regularity broken */
	 }


    if (block - gaps.maxlast <= gaps.maxjump)
	 return(TRUE);
    else
	 return(FALSE);
}

/* @SUBTITLE "JumpGap: jump over an unused gap in the portion" */
/* This function is used to jump over non-used gaps in the table.
 * These gaps are either a completeness failure or the skip
 * between portions, depending on how you look at it. The gap must 
 * define a new portion, so one thing we do is call EndPortion and
 * NewPortion. We also scan the gap looking for mistakes and the
 * beginning of the next portion (the end of the gap).  It is
 * considered a completeness failure only if it was unexpected.
 */
static int
JumpGap(block)
	int block;			/* the first unused block number */
{
    int end = block - 1;		/* end of the previous portion */

    EndPortion(end);

    /* it would be nice to find a way to optimize when we expect regularity */
    while (map[block].usedcount == 0)
	 block++;
    /* block is now the next used block */

    gaps.start = block;

    NewPortion();

    if (gaps.skiplen == 0)
	 printf("-> completeness failure: skipping blocks %d-%d\n", 
		   end+1, block-1);

    return(block);
}

/* @SUBTITLE "EndPortion: Record this sequential portion" */
/* Here we record the end of a portion, remembering its length
 * and comparing the length to previous portions.
 */
static void
EndPortion(end)
	int end;
{
    int length, skip;

    length = end - gaps.start + 1;
    skip = gaps.start - gaps.prev_end;
    gaps.prev_end = end;
    gaps.portion1 = FALSE;
    printf("-> finished portion: %d to %d length %d skip %d\n", 
		 gaps.start, end, length, skip);

    if (gaps.same_runlen == 0 || length != gaps.prev_runlen) {
	   gaps.same_runlen = 1;		/* reset length */
	   gaps.prev_runlen = length;
	   gaps.same_skiplen = 0;	/* reset skip */
	   gaps.prev_skiplen = 0;
    } else {
	   gaps.same_runlen++;		/* same length */
    }

    if (gaps.same_runlen >= RunRep) {
	   printf("-> portion length is regular %d\n", length);
    }

    /* we look at the skip at the start of the portion */
}

/* @SUBTITLE "DoneGAPS: Finished with reference pattern" */
void
DoneGAPS()
{
    if (gaps.num_sequential > 0)
	 Finalize();
    gaps.num_sequential = 0;

    free(gaps.context);
}

/* @SUBTITLE "NewRef: allocate a new REF structure" */
static REF *
NewRef(block, node)
	int block;
	int node;
{
    REF *r;

    r = (REF *)malloc(sizeof(REF));
    if (r == (REF *)NULL) {
	   fprintf(stderr, "out of memory\n");
	   exit(1);
    }
    r->block = block;
    r->node = node;

    return(r);
}
