/* @TITLE "stats.c: DMCACHE statistics"*/
/* 
 * This initializes the metrics and stats used by DMCACHE.
 * variables:
 *   stats
 * function:
 *   InitStats
 *
 * Procedure for adding a metric:
 *   add an entry to dmcache_stats in stats.h
 *   add a line in InitStats, in stats.c
 *   use it as desired in code
 *   put it somewhere in GraphFile
 * Procedure for adding a parameter:
 *   add a #definition in the appropriate parameter file *.param
 *   add a line in InitStats, in stats.c
 *   add a line to the Parameters Summary MetricsTable in Graphfile
 *   add it to Configfile
 *   add it to ParamHelp
 *   use it where needed; re-run domake mydepend
 *
 * Part of 
 *           The STARFISH Parallel file-system simulator
 *      (Simulation Tool for Advanced Research in File Systems)
 *
 *                              David Kotz
 *                          Dartmouth College
 *                             Version 3.0
 *                             October 1996
 *                         dfk@cs.dartmouth.edu
 */

/* $Id: stats.c,v 3.0 1996/10/18 06:05:51 dfk RELEASE3 dfk $ */

#include "dmcache.h"		/* which includes dmcache.param */
#include "stats.h"
#include "cpfs.param"
#include "iopfs.param"
#include "disk.h"
#include "diskmodel.param"
#include "lu.param"
#include "file.h"
#include "pattern.h"

/* Define the stats structure */
struct dmcache_stats stats;

/* @SUBTITLE "Assign numbers to stage types" */
/* This is necessary because of the way config does parameters */

  /* No buffering at either compute proc or I/O proc. */
  /* No prefetching, but intra-request parallelism. */
#if defined(CP_NONE) && defined(IOP_NONE) 
#  define STAGENUM (1)

  /* No buffering at the compute proc, single buffering at the I/O proc. */
  /* No prefetching, but intra-request parallelism. */
#elif defined(CP_NONE) && defined(IOP_BUFFER) && !defined(IOP_AHEAD_BEHIND) 
#  define STAGENUM (2)

  /* No buffering at the compute proc, multiple buffering at the I/O proc. */
  /* No prefetching, but intra-request parallelism. */
#elif defined(CP_NONE) && defined(IOP_BUFFER) && defined(IOP_AHEAD_BEHIND) 
#  define STAGENUM (3)

  /* Single buffering at the compute proc, multiple buffering at I/O proc. */
  /* Simple OBL prefetching (only when the single buffer is drained). */
#elif defined(CP_SINGLE) && defined(IOP_BUFFER) && defined(IOP_AHEAD_BEHIND) 
#  define STAGENUM (4)

  /* Double buffering at the compute proc, multiple buffering at I/O proc. */
  /* OBL prefetching (double buffering) in IOP. */
#elif defined(CP_DOUBLE) && defined(IOP_BUFFER) && defined(IOP_AHEAD_BEHIND) && defined(CONSISTENCY_NONE) 
#  define STAGENUM (5) 

  /* Multithreaded compute proc, multiple buffering at I/O proc. */
  /* OBL prefetching (double buffering) in IOP. */
#elif defined(CP_THREAD) && defined(IOP_BUFFER) && defined(IOP_AHEAD_BEHIND) && defined(CONSISTENCY_NONE) 
#  define STAGENUM (5.5)		/* temporary stage number */

  /* Double buffering at the compute proc, caching at the I/O proc. */
  /* OBL prefetching (double buffering) in IOP. */
#elif defined(CP_DOUBLE) && defined(IOP_CACHE) && defined(IOP_AHEAD_BEHIND) && defined(CONSISTENCY_NONE) 
#  define STAGENUM (6)

  /* Double buffering at the compute proc, buffering or caching at the IOP. */
  /* Sprite consistency control. */
#elif defined(CP_DOUBLE) && (defined(IOP_CACHE) || defined(IOP_BUFFER)) && defined(CONSISTENCY_SPRITE) 
#  define STAGENUM (7)

  /* Double buffering at the compute proc, buffering or caching at the IOP. */
  /* Write-combining consistency control. */
#elif defined(CP_DOUBLE) && (defined(IOP_CACHE) || defined(IOP_BUFFER)) && defined(CONSISTENCY_COMBINE) 
#  define STAGENUM (8)

  /* Some unrecognized combination, including GENERAL */
#else 
#  define STAGENUM (0)
#endif

/* @SUBTITLE "Assign numbers to cpfs, and iopfs types" */
/* This is necessary because of the way config does parameters */

#ifdef CP_NONE
# define CPFS 0
#endif
#ifdef CP_SINGLE
# define CPFS 1
#endif
#ifdef CP_DOUBLE
# define CPFS 2
#endif
#ifdef CP_THREAD
# define CPFS 3
#endif
#ifdef CP_DIRECT
# define CPFS 4
#endif

#ifdef IOP_NONE
# define IOPFS 0
#endif
#ifdef IOP_BUFFER
# define IOPFS 1
#endif
#ifdef IOP_CACHE
# define IOPFS 2
#endif
#ifdef IOP_DIRECT	      /* no longer used */
# define IOPFS 3
#endif
#ifdef IOP_GENERAL
# define IOPFS 4
#endif

#ifdef CONSISTENCY_NONE
# define CONSISTENCY 0
#endif
#ifdef CONSISTENCY_SPRITE
# define CONSISTENCY 1
#endif
#ifdef CONSISTENCY_COMBINE
# define CONSISTENCY 2
#endif

#ifdef NET_SCATTER_GATHER
#define NSG 1
#else
#define NSG 0
#endif 

#ifdef IOP_AHEAD_BEHIND
#define IAB 1
#else
#define IAB 0
#endif 

#ifdef TRIVIAL_DISK
# define DISK_TYPE 0
#endif

#ifdef MODEL_DISK
# ifdef HP97560
#  define DISK_TYPE 1
# endif
#endif

#ifdef LAYOUT_CONTIGUOUS
#define DISK_LAYOUT 0
#endif

#ifdef LAYOUT_RANDOM_SECTORS
#define DISK_LAYOUT 1
#endif

#ifdef LAYOUT_RANDOM_TRACKS
#define DISK_LAYOUT 2
#endif

#ifdef DISK_SORT
#define DSORT 1
#else
#define DSORT 0
#endif

#ifdef DISK_FCFS
#define DFCFS 1
#else
#define DFCFS 0
#endif

#ifdef USE_QUEUE_REQUESTS
#define UQR 1
#else
#define UQR 0
#endif

#ifdef CP_MEMGET_WRITES
#define MEMGET 1
#else
#define MEMGET 0
#endif

#ifdef REAL_DATA
#define RD 1
#else
#define RD 0
#endif

#ifdef TWO_PHASE
#define TP 1
#else
#define TP 0
#endif

#ifdef MEMPUT_QUEUED
#define MEMQ 1
#else
#define MEMQ 0
#endif

#ifdef DISK_RAND_ROT
# define DRR 1
#else
# define DRR 0
#endif

/* @SUBTITLE "InitStats: initialize the statistics" */
/* All parameters must be defined before calling this. */
void
InitStats(void)
{
    /* compile-time parameters, so I can make a table out of them */
    (void) new_metric("[STAGE] Stage number", (double)STAGENUM);
    (void) new_metric("[CPFS] CPFS type", (double)CPFS);
    (void) new_metric("[IOPFS] IOPFS type", (double)IOPFS);
    (void) new_metric("[TWO_PHASE] Two-phase I/O", (double)TP);
    (void) new_metric("[MEMQ] Memputq and Memgetq", (double)MEMQ);
    (void) new_metric("[CONSISTENCY] Consistency type", (double)CONSISTENCY);
    (void) new_metric("[AHEAD_BEHIND] IOP ahead/behind", (double)IAB);
    (void) new_metric("[TICSperSEC] TICS per second", (double)TICSperSEC);
    (void) new_metric("[BYTESperSEC] Bytes per second", (double)BYTESperSEC);
    (void) new_metric("[BYTESperFLIT] Bytes per flit", (double)BYTESperFLIT);
    (void) new_metric("[BLOCK_SIZE] in bytes", (double)BLOCK_SIZE);
    (void) new_metric("[NO_OF_DISKS]", (double)NO_OF_DISKS);
    (void) new_metric("[DISK_READ_TIME] in msec", (double)DISK_READ_TIME);
    (void) new_metric("[DISK_WRITE_TIME] in msec", (double)DISK_WRITE_TIME);
    (void) new_metric("[FILE_BLOCKS] in blocks", (double)FILE_BLOCKS);
    (void) new_metric("[FILE_SIZE] in bytes", (double)FILE_SIZE);
    (void) new_metric("[RECORD_SIZE] in bytes", (double)RECORD_SIZE);
    (void) new_metric("[NUM_RECORDS]", (double)NUM_RECORDS);
    (void) new_metric("[NET_SCATTER_GATHER]", (double)NSG);
    (void) new_metric("[DISK_TYPE]", (double)DISK_TYPE);
    (void) new_metric("[DISK_LAYOUT]", (double)DISK_LAYOUT);
    (void) new_metric("[DISK_SORT] Sort disk requests", (double)DSORT);
    (void) new_metric("[DISK_FCFS] Use FCFS disk schedule", (double)DFCFS);
    (void) new_metric("[DISK_RAND_ROT] Random initial rotation", (double)DRR);
    (void) new_metric("[USE_QUEUE_REQUESTS]", (double)UQR);
    (void) new_metric("[CP_MEMGET_WRITES]", (double)MEMGET);
    (void) new_metric("[MAX_OUTSTANDING]", (double)MAX_OUTSTANDING);
    (void) new_metric("[REAL_DATA]", (double)RD);
    (void) new_metric("[FILTER_PERCENT]", (double)FILTER_PERCENT);
    (void) new_metric("[LU_MATRIX_SIZE]", (double)LU_MATRIX_SIZE);
    (void) new_metric("[LU_SLAB_SIZE] in cols", (double)LU_SLAB_SIZE);

    /* run-time parameters */
    (void) new_metric("[Pattern] Access pattern", (double)Pattern);
    (void) new_metric("[Nio] Number of I/O procs", (double)Nio);
    (void) new_metric("[Ncomp] Number of compute procs", (double)Ncomp);
    (void) new_metric("[trial] Trial number", (double)proteusSeed);

    /* actual statistics */
    stats.total 	= new_metric("[total] Total execution time", 0.);
    stats.tread 	= new_metric("[tread] Record read time, avg MSEC", 0.);
    stats.tread_sd	= new_metric("[tread_sd] Record read time, stddev MSEC",
				     0.);
    stats.twrite 	= new_metric("[twrite] Record write time, avg MSEC", 0.);
    stats.twrite_sd	= new_metric("[twrite_sd] Record write time, stddev MSEC", 0.);
    stats.permute 	= new_metric("[permute] Permutation time, avg MSEC", 0.);

    stats.ThreadRequest
      = new_metric("[ThreadRequest] send time, avg TICS", 0.);
    stats.ThreadRequesth
      = new_metric("[ThreadRequesth] recv time, avg TICS", 0.);
    stats.ThreadRequesth_sd
      = new_metric("[ThreadRequesths] recv time, stddev TICS", 0.);
    stats.QueueRequest
      = new_metric("[QueueRequest] send time, avg TICS", 0.);
    stats.QueueRequesth
      = new_metric("[QueueRequesth] recv time, avg TICS", 0.);
    stats.QueueRequesth_sd
      = new_metric("[QueueRequesths] recv time, stddev TICS", 0.);
    stats.InstantRequest 
      = new_metric("[InstantRequest] send time, avg TICS", 0.);
    stats.Reply = new_metric("[Reply] send time, avg TICS", 0.);
    stats.Replyh = new_metric("[Replyh] recv time, avg TICS", 0.);
    stats.Replyh_sd = new_metric("[Replyhs] recv time, stddev TICS", 0.);

    stats.IOPread = new_metric("[IOPread] read time, avg MSEC", 0.);
    stats.IOPread_sd = new_metric("[IOPread_sd] read time, stddev MSEC", 0.);
    stats.IOPwrite = new_metric("[IOPwrite] write time, avg MSEC", 0.);
    stats.IOPwrite_sd = new_metric("[IOPwrite_sd] write time, stddev MSEC", 0.);

    stats.CPread_n = new_metric("[CPread_n] FS*read call count", 0.);
    stats.CPwrite_n = new_metric("[CPwrite_n] FS*write call count", 0.);
    stats.CPread_bytes = new_metric("[CPread_bytes] FS*read byte count", 0.);
    stats.CPwrite_bytes = new_metric("[CPwrite_bytes] FS*write byte count", 0.);

    stats.IOPread_n = new_metric("[IOPread_n] IOPread call count", 0.);
    stats.IOPwrite_n = new_metric("[IOPwrite_n] IOPwrite call count", 0.);
    stats.IOPread_bytes = new_metric("[IOPread_bytes] IOPread byte count", 0.);
    stats.IOPwrite_bytes = new_metric("[IOPwrite_bytes] IOPwrite byte count", 0.);

    stats.IOPputget = new_metric("[IOPputget] IOP tics spent in memput/memget activity", 0.);

    stats.EmptyBlock = new_metric("[EmptyBlock] time, avg TICS", 0.);
    stats.FillBlock = new_metric("[FillBlock] time, avg TICS", 0.);

    stats.Redirects = new_metric("[Redirects] Redirects, fraction successful", 0.);
    stats.messages = new_metric("[messages] Number of messages", 0.);
    stats.messageBytes = new_metric("[messageBytes] Number of bytes in messages", 0.);
    stats.LUtime = new_metric("[LUtime] LU decomposition, in msec", 0.);
    stats.LUbytes = new_metric("[LUbytes] LU decomposition, bytes moved", 0.);

    stats.totalThruput	= new_metric("[tthruput] Total thruput in KBps", 0.);

    stats.innerThruput	= new_array_metric
      ("[ithruput] Inner thruput in KBps, per CP", Ncomp, 0.);
    stats.outerThruput	= new_array_metric
      ("[othruput] Outer thruput in KBps, per CP", Ncomp, 0.);
    stats.miss	 	= new_array_metric
      ("[miss] Cache misses, per proc", NO_OF_PROCESSORS, 0.);
    stats.hit	 	= new_array_metric
      ("[hit] Cache hits, per proc", NO_OF_PROCESSORS, 0.);
    stats.prefetch 	= new_array_metric
      ("[prefetch] Prefetches, per proc", NO_OF_PROCESSORS, 0.);
    stats.prefetchused 	= new_array_metric
      ("[prefetchused] Prefetches used, per proc", NO_OF_PROCESSORS, 0.);
    stats.diskmiss 	= new_array_metric
      ("[diskmiss] Disk cache misses, per disk", NO_OF_DISKS, 0.);
    stats.diskhit 	= new_array_metric
      ("[diskhit] Disk cache hits, per disk", NO_OF_DISKS, 0.);
    stats.diskread	= new_array_metric
      ("[diskread] Disk reads, per disk", NO_OF_DISKS, 0.);
    stats.diskwrite	= new_array_metric
      ("[diskwrite] Disk writes, per disk", NO_OF_DISKS, 0.);
    stats.diskidle	= new_array_metric
      ("[diskidle] Disk idle time, per disk", NO_OF_DISKS, 0.);
    stats.diskwait	= new_array_metric
      ("[diskwait] Disk wait time, avg", NO_OF_DISKS, 0.);
    stats.diskwait_sd = new_array_metric
      ("[diskwait_sd] Disk wait time, std dev", NO_OF_DISKS, 0.);
    stats.diskmoves	= new_array_metric
      ("[diskmoves] Disk movements, per disk", NO_OF_DISKS, 0.);
    stats.bustrans	= new_array_metric
      ("[bustrans] Bus transfers, per bus", Nio, 0.);
    stats.buswait	= new_array_metric
      ("[buswait] Bus wait time, per bus, total", Nio, 0.);

    stats.diskqlength	= new_array_metric
      ("[diskqlength] Diskq length, avg per disk", NO_OF_DISKS, 0.);
    stats.diskqpasses	= new_array_metric
      ("[diskqpasses] Diskq passes, per disk", NO_OF_DISKS, 0.);
    stats.diskqempty	= new_array_metric
      ("[diskqempty] Deq's on empty diskq, per disk", NO_OF_DISKS, 0.);
}

