/* @TITLE "pattern.h: Define the access patterns"*/
/* 
 * The access patterns are currently defined as a run-time parameter in
 * driver.c, and then used to call the write function in cp.ca.
 * 
 * variables:
 *   Pattern
 *   PatternNames
 *
 * Part of 
 *           The STARFISH Parallel file-system simulator
 *      (Simulation Tool for Advanced Research in File Systems)
 *
 *                              David Kotz
 *                          Dartmouth College
 *                             Version 3.0
 *                             October 1996
 *                         dfk@cs.dartmouth.edu
 */

/* $Id: pattern.h,v 3.0 1996/10/18 06:05:51 dfk RELEASE3 dfk $ */

#ifndef PATTERN_H
#define PATTERN_H

#include "dmcache.h"

/* ADD ALL NEW PATTERNS AT THE BOTTOM.
 * Changing this requires changing NUMPATTERNS below,
 * and PatternNames[] in pattern.c 
 * and PatternIsWrite[] in pattern.c 
 * and PatternFuncs in cp.ca
 * and run/bin/map/Pattern
 */
enum pattern_e {
    NULLCOMM_TEST,	      /* not a real pattern; a communications test */
    DATACOMM_TEST,	      /* not a real pattern; a communications test */
    MEMGET_TEST,	      /* not a real pattern; a Memget test */
    COPY_TEST,		      /* not a real pattern; a word-copy test */
    DISK_TEST,		      /* not a real pattern; a disk device test */

    LUINIT_PATTERN,	      /* special pattern; initialize for LU */
    LU_PATTERN,		      /* special pattern, LU decomposition */

    RECORD_READ_LW,
    RECORD_READ_LPS,
    BLOCK_READ_LW,
    RECORD_WRITE_LW,
    BLOCK_WRITE_LW,
    WHOLE_READ_LW,
    WHOLE_READ_LW1,
    WHOLE_WRITE_LW1,
    WHOLE_READ_SEG,
    WHOLE_WRITE_SEG,

    /* Read, 1-D distributions */
    WHOLE_READ_NONE,
    WHOLE_READ_BLOCK,
    WHOLE_READ_CYCLIC,

    /* Read, 2-D DISTRIBUTION(NONE, ...) */
    WHOLE_READ_NONE_NONE,
    WHOLE_READ_NONE_BLOCK,
    WHOLE_READ_NONE_CYCLIC,

    /* Read, 2-D DISTRIBUTION(BLOCK, ...) */
    WHOLE_READ_BLOCK_NONE,
    WHOLE_READ_BLOCK_BLOCK,
    WHOLE_READ_BLOCK_CYCLIC,

    /* Read, 2-D DISTRIBUTION(CYCLIC, ...) */
    WHOLE_READ_CYCLIC_NONE,
    WHOLE_READ_CYCLIC_BLOCK,
    WHOLE_READ_CYCLIC_CYCLIC,

    /* Write, 1-D distributions */
    WHOLE_WRITE_NONE,
    WHOLE_WRITE_BLOCK,
    WHOLE_WRITE_CYCLIC,

    /* Write, 2-D DISTRIBUTION(NONE, ...) */
    WHOLE_WRITE_NONE_NONE,
    WHOLE_WRITE_NONE_BLOCK,
    WHOLE_WRITE_NONE_CYCLIC,

    /* Write, 2-D DISTRIBUTION(BLOCK, ...) */
    WHOLE_WRITE_BLOCK_NONE,
    WHOLE_WRITE_BLOCK_BLOCK,
    WHOLE_WRITE_BLOCK_CYCLIC,

    /* Write, 2-D DISTRIBUTION(CYCLIC, ...) */
    WHOLE_WRITE_CYCLIC_NONE,
    WHOLE_WRITE_CYCLIC_BLOCK,
    WHOLE_WRITE_CYCLIC_CYCLIC,

    /* new special patterns */
    WHOLE_READ_SUBSET,	      /* special pattern, read subset of file */
    WHOLE_READ_HASH	      /* special pattern, hash-based distribution */

    /* ADD ALL NEW PATTERNS HERE */

    /* don't forget to update NUMPATTERNS, below, pattern.c, and cp.ca */
};
#define NUMPATTERNS 43

typedef enum pattern_e PATTERN;

/* large enough for a string to hold a pattern name */
#define PATNAMELENGTH 15

/* GLOBAL variables, defined in pattern.c */
extern char PatternNames[NUMPATTERNS][PATNAMELENGTH];
extern boolean PatternIsWrite[NUMPATTERNS];

/* GLOBAL PARAMETER */
extern PATTERN Pattern;		/* access pattern to use */

/* Macros/functions for working with BLOCK and CYCLIC distributions.
 * nProcs is the number of processors involved in that dimension
 * (rows or cols).
 */
extern ulong NumInBLOCK(int CPnum, int nProcs, ulong nItems);

#define NumRowsBLOCK(CPnum, nProcs, nRows) \
    NumInBLOCK((CPnum), (nProcs), (nRows))

#define NumRowsCYCLIC(CPnum, nProcs, nRows) \
    ((nRows) / (nProcs) + ((CPnum) < (nRows) % (nProcs)))

#define NumColsBLOCK(CPnum, nProcs, nCols)  \
    ((nCols) / (nProcs))     /* PLUS you MUST check that nCols % nProcs == 0 */

#define NumColsCYCLIC(CPnum, nProcs, nCols) \
    ((nCols) / (nProcs) + ((CPnum) < (nCols) % (nProcs)))

#endif /* PATTERN_H */
