/* @TITLE "pattern.c: global variables about patterns"*/
/* 
 * This file has functions and variables for patterns.
 * 
 * Functions: NumInBLOCK()
 * Variables: PatternNames[], PatternIsWrite[]
 *
 * Part of 
 *           The STARFISH Parallel file-system simulator
 *      (Simulation Tool for Advanced Research in File Systems)
 *
 *                              David Kotz
 *                          Dartmouth College
 *                             Version 3.0
 *                             October 1996
 *                         dfk@cs.dartmouth.edu
 */

/* $Id: pattern.c,v 3.0 1996/10/18 06:05:51 dfk RELEASE3 dfk $ */

#include "dmcache.h"			/* the overall include file */
#include "pattern.h"

/* ADD ALL NEW PATTERNS AT THE BOTTOM.
 * Order must correspond to the enumeration in pattern.h.
 * Also must correspond to PatternIsWrite, below.
 * Also must correspond to PatternFuncs in cp.ca.
 * Also must correspond to run/bin/map/Pattern
 */
char PatternNames[NUMPATTERNS][PATNAMELENGTH] = {
    /* test routines */
    "nullcommtest",
    "datacommtest",
    "memgettest",
    "copytest",
    "disktest",

    "luinit",		      /* initialize for LU decomposition */
    "lu",		      /* LU decomposition */

    /* plain patterns */
    "rrlw",
    "rrlps",
    "brlw",
    "rwlw",
    "bwlw",
    "wrlw",
    "wrlw1",
    "wwlw1",
    "wrseg",
    "wwseg",

    /* Read, 1-D distributions */
    "wrn",
    "wrb",
    "wrc",

    /* Read, 2-D DISTRIBUTION(NONE, ...) */
    "wrnn",
    "wrnb",
    "wrnc",

    /* Read, 2-D DISTRIBUTION(BLOCK, ...) */
    "wrbn",
    "wrbb",
    "wrbc",

    /* Read, 2-D DISTRIBUTION(CYCLIC, ...) */
    "wrcn",
    "wrcb",
    "wrcc",

    /* Write, 1-D distributions */
    "wwn",
    "wwb",
    "wwc",

    /* Write, 2-D DISTRIBUTION(NONE, ...) */
    "wwnn",
    "wwnb",
    "wwnc",

    /* Write, 2-D DISTRIBUTION(BLOCK, ...) */
    "wwbn",
    "wwbb",
    "wwbc",

    /* Write, 2-D DISTRIBUTION(CYCLIC, ...) */
    "wwcn",
    "wwcb",
    "wwcc",

    /* new special patterns */
    "subset",		      /* special distribution to read subset of file */
    "hash"		      /* special data-dependent (hash) distribution */

    /* ADD ALL NEW PATTERNS HERE */
};

/* @PAGE */

/* order must correspond to the enumeration in pattern.h */
boolean PatternIsWrite[NUMPATTERNS] = {
    FALSE,		      /* NULLCOMM_TEST */
    FALSE,		      /* DATACOMM_TEST */
    FALSE,		      /* MEMGET_TEST */
    FALSE,		      /* COPY_TEST */
    FALSE,		      /* DISK_TEST */
    TRUE,		      /* LUINIT_PATTERN */
    FALSE,		      /* LU_PATTERN (it writes, but is old file) */

    FALSE,		      /* RECORD_READ_LW */
    FALSE,		      /* RECORD_READ_LPS */
    FALSE,		      /* BLOCK_READ_LW */
    TRUE,		      /* RECORD_WRITE_LW */
    TRUE,		      /* BLOCK_WRITE_LW */
    FALSE,		      /* WHOLE_READ_LW */
    FALSE,		      /* WHOLE_READ_LW1 */
    TRUE,		      /* WHOLE_WRITE_LW1 */
    FALSE,		      /* WHOLE_READ_SEG */
    TRUE,		      /* WHOLE_WRITE_SEG */

    FALSE,		      /* WHOLE_READ_NONE */
    FALSE,		      /* WHOLE_READ_BLOCK */
    FALSE,		      /* WHOLE_READ_CYCLIC */

    FALSE,		      /* WHOLE_READ_NONE_NONE */
    FALSE,		      /* WHOLE_READ_NONE_BLOCK */
    FALSE,		      /* WHOLE_READ_NONE_CYCLIC */

    FALSE,		      /* WHOLE_READ_BLOCK_NONE */
    FALSE,		      /* WHOLE_READ_BLOCK_BLOCK */
    FALSE,		      /* WHOLE_READ_BLOCK_CYCLIC */

    FALSE,		      /* WHOLE_READ_CYCLIC_NONE */
    FALSE,		      /* WHOLE_READ_CYCLIC_BLOCK */
    FALSE,		      /* WHOLE_READ_CYCLIC_CYCLIC */

    TRUE,		      /* WHOLE_WRITE_NONE */
    TRUE,		      /* WHOLE_WRITE_BLOCK */
    TRUE,		      /* WHOLE_WRITE_CYCLIC */

    TRUE,		      /* WHOLE_WRITE_NONE_NONE */
    TRUE,		      /* WHOLE_WRITE_NONE_BLOCK */
    TRUE,		      /* WHOLE_WRITE_NONE_CYCLIC */

    TRUE,		      /* WHOLE_WRITE_BLOCK_NONE */
    TRUE,		      /* WHOLE_WRITE_BLOCK_BLOCK */
    TRUE,		      /* WHOLE_WRITE_BLOCK_CYCLIC */

    TRUE,		      /* WHOLE_WRITE_CYCLIC_NONE */
    TRUE,		      /* WHOLE_WRITE_CYCLIC_BLOCK */
    TRUE,		      /* WHOLE_WRITE_CYCLIC_CYCLIC */

    FALSE,		      /* WHOLE_READ_SUBSET */
    FALSE		      /* WHOLE_READ_HASH */
};

/* @SUBTITLE "NumInBLOCK: # items this CP gets in BLOCK distribution" */
/* Note that it is possible for some of the last CPs to get 0 items,
 * by the way HPF defines the BLOCK distribution.  The last CP before that 
 * may get fewer than all the rest of the CPs, who in turn get the full
 * amount "most". 
 */
ulong
NumInBLOCK(int CPnum, int nProcs, ulong nItems)
{
    register ulong most = ceildiv(nItems, nProcs); /* most get this many */
    register beforeUs = most * CPnum; /* how many items on CPs before us? */

    if (beforeUs >= nItems) {
	/* all items are assigned to procs [0..CPnum-1]; we get none */
	return (0);
    } else if (beforeUs + most > nItems) {
	/* we get a partial set of items, ie, the leftovers */
	return (nItems - beforeUs);
    } else {
	/* we are a typical case, and get the full amount */
	return(most);
    }
}

