/* @TITLE "driver.c: Main program for dmcache"*/
/* 
 * The driver is responsible for starting up the other processes,
 * parsing the arguments, running the experiment, and storing stats. 
 * 
 * Functions:
 *    usermain
 *    main (if not Proteus)
 *    see usage() for usage
 *
 * Part of 
 *           The STARFISH Parallel file-system simulator
 *      (Simulation Tool for Advanced Research in File Systems)
 *
 *                              David Kotz
 *                          Dartmouth College
 *                             Version 3.0
 *                             October 1996
 *                         dfk@cs.dartmouth.edu
 */

/* $Id: driver.c,v 3.0 1996/10/18 06:05:51 dfk RELEASE3 dfk $ */

#include <sys/time.h>
#include <sys/resource.h>     /* for getrusage */

#include "dmcache.h"			/* the overall include file */
#include "broadcast.h"
#include "barrier.h"
#include "ready.h"
#include "stats.h"
#include "worker.h"
#include "time.h"
#include "pattern.h"
#include "message.h"

/* parameter files for PrintParameters() */
#include "cpfs.param"
#include "disk.h"
#include "diskmodel.param"
#include "dmcache.param"
#include "dummy.param"
#include "file.h"
#include "iopfs.param"
#include "message.param"
#include "sim.param"
#include "userdata.param"

/* EXPORTED FUNCTIONS */
void usermain(int argc, char **argv);
/* and possibly main() */

/* PRIVATE FUNCTIONS */
PRIVATE boolean ParseParameters(int argc, char **argv);
PRIVATE void PrintParameters(boolean all);
PRIVATE void Initialize();
PRIVATE void RunExperiment(void);
PRIVATE void MemoryUsage(void);
PRIVATE void usage(void);

/* DEFINITIONS */
#define PROGNAME "dmcache"

/* GLOBAL PARAMETERS, not all exported */
PATTERN Pattern;			/* access pattern */
int Nio, Ncomp;		      		/* number of IOPs and CPs */
boolean FileIsNew;			/* is the file new? */
PRIVATE boolean helpflag = FALSE;

/* @SUBTITLE "usage: print usage message" */
PRIVATE void
usage(void)
{
    fprintf(stderr, "usage: %s -v help\n", PROGNAME);
    fprintf(stderr, " or    %s -v patname Nio Ncomp\n", PROGNAME);
    fprintf(stderr, "Note (Nio + Ncomp) must be <= %d\n", NO_OF_PROCESSORS-1);
    fprintf(stderr, " and (%d %% Nio) == 0\n", NO_OF_DISKS);
}

/* @SUBTITLE "main/usermain: Top-level procedure" */
#ifndef PROTEUS
main(argc, argv)
	int argc;
	char **argv;
{
    usermain(argc, argv);
}
#endif PROTEUS

/* Proteus has its own main(), and calls usermain() for us. */
void
usermain(int argc, char **argv)
{
    if (ParseParameters(argc, argv)) {
	/* Now we can really initialize */
	Initialize();
	if (helpflag) {
	    /* print ALL the parameters to screen */
	    PrintParameters(TRUE);
	    exit(0);
	} else {
	    /* print SOME of the parameters to screen */
	    PrintParameters(FALSE);
	    /* Run the experiment */
	    RunExperiment();
	    printf("\nExperiment complete\n");
	    MemoryUsage();
	    return; /* success! */
	}
    } else
      exit(1); /* failed */
}

/* @SUBTITLE "Get the parameters from the command line" */
PRIVATE boolean
ParseParameters(int argc, char **argv)
{
    /* if 1 argument, must be "help" */
    if (argc == 1 && strcmp(argv[0], "help") == 0) {
	helpflag = TRUE;
	return(TRUE);
    }

    /* 3 arguments are patname, Nio, Ncomp */
    if (argc != 3) {
	usage();
	return(FALSE);
    }

    /* argv[0] is the pattern name */
    for (Pattern = 0; Pattern < NUMPATTERNS; Pattern++) {
	if (strcmp(argv[0], PatternNames[Pattern]) == 0)
	  break;
    }
    if (Pattern == NUMPATTERNS) {
	fprintf(stderr, "%s: unrecognized pattern name '%s'\n", 
		PROGNAME, argv[0]);
	usage();
	return(FALSE);
    }

    /* argv[1] is Nio */
    if(sscanf(argv[1], "%d", &Nio) != 1 || Nio < 1) {
	fprintf(stderr, "%s: Nio = '%s' is invalid\n", PROGNAME, argv[1]);
	usage();
	return(FALSE);
    }

    if (NO_OF_DISKS % Nio != 0) {
	fprintf(stderr, 
		"%s: invalid Nio=%d (there are %d disks)\n", 
		PROGNAME, Nio, NO_OF_DISKS);
	usage();
	return(FALSE);
    }

    /* argv[2] is Ncomp */
    if(sscanf(argv[2], "%d", &Ncomp) != 1 || Ncomp < 1) {
	fprintf(stderr, "%s: Ncomp = '%s' is invalid\n", PROGNAME, argv[2]);
	usage();
	return(FALSE);
    }

    /* wants too many processors? (we need one to be a master) */
    if (1 + Nio + Ncomp > NO_OF_PROCESSORS) {
	fprintf(stderr, "%s: Nio=%d + Ncomp=%d > %d usable processors\n",
		PROGNAME, Nio, Ncomp, NO_OF_PROCESSORS-1);
	return(FALSE);
    }

    /* Find out if the file is to be considered "newly created" */
    FileIsNew = PatternIsWrite[Pattern];

    return(TRUE);	      /* success */
}

/* @SUBTITLE "Print the current set of parameters" */
PRIVATE void
PrintParameters(boolean printAll)
{
    extern char params_[];  /* the text of the current set of parameters */

    printf("\nParameters:\n");
    printf(" Access pattern code = '%s'\n", PatternNames[Pattern]);
    printf(" FileIsNew = %s\n", FileIsNew ? "TRUE" : "FALSE");
    printf(" Nproc = %d\n", NO_OF_PROCESSORS);
    printf(" Nio = %d\n", Nio);
    printf(" Ncomp = %d\n", Ncomp);
    printf(" Ndisk = %d\n", NO_OF_DISKS);

#if defined(REAL_DATA)
    printf(" REAL DATA\n");
#else
    printf(" FAKE DATA\n");
#endif
#if defined(TWO_PHASE)
    printf(" Using TWO-PHASE I/O\n");
#endif
#if defined(MEMPUT_QUEUED)
    printf(" Using MEMPUTQ and MEMGETQ\n");
#endif
    printf("\n");
    fflush(stdout);

    if (!printAll)
      return;

    /* proteus gathers all the parameters here, so print them! */
    fputs(params_, stdout);

    printf("\n");
    fflush(stdout);
    usage();
}

/* @SUBTITLE "Initialize: initialize auxiliary modules" */
PRIVATE void
Initialize()
{
    barrier_init();
    broadcast_init(NO_OF_PROCESSORS);
    ready_init(NO_OF_PROCESSORS);
    InitMessages();
    InitStats();
}

/* @SUBTITLE "Run one experiment, perhaps with several trials" */
/* This processor is a driver processor, and does not take on 
 * any role in the experiments. It starts the others working, and 
 * waits until they are done.
 */
PRIVATE void
RunExperiment(void)
{
    int nprocs;
    ELAPSED_DEFS;

    /* restrict the global operations to the necessary subset */
    nprocs = 1 + Nio + Ncomp; /* including us */
    broadcast_init(nprocs);
    ready_init(nprocs);

    /* Get them all started up */
    broadcast0(Worker);

    /* Wait until all are initialized */
    barrier(0, nprocs);

    START_CLOCK;	      /* just after barrier */

    /* Wait for all comp procs to finish the experiment */
    ready();
    wait_all_ready();
    
    STOP_CLOCK;		      /* just after all are ready */

    /* Now tell all to shut down. */
    broadcast0(Shutdown);
    ready();
    wait_all_ready();

    /* Compute elapsed time */
    SET_METRIC(stats.total, ELAPSED_SEC);
}

/* @SUBTITLE "MemoryUsage: print out overall memory usage" */
#define TEXT_BASE (unsigned)0x00400000    /* from 'man ld' */
#define DS_BASE (unsigned)0x10000000    /* from 'man ld' */
#define STACK_BASE (unsigned)0x80000000    /* from 'man ld' */

PRIVATE void 
MemoryUsage(void)
{
    struct rusage r;
    unsigned text, data, kb;
    
    extern unsigned etext;    /* from 'man end' */
    extern void *sbrk(int);
    extern void getrusage(int, struct rusage *);
    extern int getpagesize();

/*
    extern unsigned end, edata, eprol;
    printf("\n");
    printf("end     0x%08x\n", (unsigned) &end);
    printf("etext   0x%08x\n", (unsigned) &etext);
    printf("edata   0x%08x\n", (unsigned) &edata);
    printf("eprol   0x%08x\n", (unsigned) &eprol);
    printf("sbrk(0) 0x%08x\n", (unsigned) sbrk(0));
*/
    printf("\n");

    text = (unsigned) &etext - TEXT_BASE;
    printf("memusage: %u bytes text\n", text);

    data = (unsigned) sbrk(0) - DS_BASE;
    printf("memusage: %u bytes data\n", data);

    kb = ceildiv(data, 1024);
    printf("memusage: %u KB total data (not incl. stack)\n", kb);
    kb += ceildiv(text, 1024);
    printf("memusage: %u KB total text and data (not incl. stack)\n", kb);

    getrusage(RUSAGE_SELF, &r);
    printf("memusage: %g MB maxrss?\n", 
	   r.ru_maxrss * getpagesize() / 1024. / 1024.);

    printf("\n");
}

