/* @TITLE "diskrequest.c: Disk device's request management"*/
/* 
 * The disk-device module is event driven.  This module keeps the event 
 * request mechanism, which is based on proteus SimRequests, 
 * abstracted away from the bulk of the disk device code. 
 *
 * Exported functions
 *  	DDEventInit
 *  	DDEventRequest
 *  	DDEventPrint
 *  	DDEventCancel
 *
 * Part of 
 *           The STARFISH Parallel file-system simulator
 *      (Simulation Tool for Advanced Research in File Systems)
 *
 *                              David Kotz
 *                          Dartmouth College
 *                             Version 3.0
 *                             October 1996
 *                         dfk@cs.dartmouth.edu
 */

/* $Id: diskevent.c,v 3.0 1996/10/18 06:05:51 dfk RELEASE3 dfk $ */

#include "dmcache.h"
#include "diskevent.h"
#include "simcalls.h"	      /* proteus */
#include "processor.h"	      /* proteus */
#include "conf.h"	      /* proteus */

PRIVATE int DiskDevice_SimRequest; /* proteus event code */
PRIVATE void DDEventHandler(SimRequest *req);

/*
#define REQUEST_TRACE(x) {CYCLE_COUNTING_OFF; printf x; CYCLE_COUNTING_ON;}
*/
#define REQUEST_TRACE(x) {}

#define NOW GLOBALTIME

/* We need to be able to call functions in .ca files, like thread_wakeup 
 * and sched_choose, but in a way that
 *    1) we don't context-switch during the call
 *    2) we don't charge any time to the "current" processor
 *       (after all, our disk events are not attached to any particular
 *        processor, and which processor happens to be "current" is arbitrary).
 * So we steal some time from the clock, using a trick used elsewhere in
 * Proteus, and then restore the clock to exactly its value before the call.
 * In theory any statement could be put inside the PROTECT call, within the
 * limits of the macro preprocessor.
 */
#define PROTECT(s) {	\
    Time now = cycles_;	\
    SubtractTime(10000);\
    { s; }   	    	\
    cycles_ = now;  	\
}


/* @SUBTITLE DDEventInit: Initialize for disk device events" */
/* call once, on only one processor */
void
DDEventInit(void)
{
    DiskDevice_SimRequest = define_new_simcall("DiskDeviceRequest", 
					       DDEventHandler,
					       DDEventPrint);
    /* DDEventPrint type is ok, despite gcc */
}

/* @SUBTITLE DDEvent: Make a request for a future disk device event" */
/* This might be called by a thread or from a simcall handler.  In the 
 * latter case there is not really a tid to associate with the request.
 * I try to keep the originating thread's tid carried through all of the
 * requests (which will help in determining which processor caused the
 * action) by taking the processor's current tid.  When called from a thread,
 * this is actually the thread; then later notice that the handler restores
 * this value from the request before executing the handler, so new requests
 * can inherit that value.
 */
DDEvent *
DDEventRequest(REQUESTCODE event, TICS eventTime, int disk)
{
    SimRequest *req; 

    REQUEST_TRACE(("\tDDEventRequest %s for %d @%s\n", 
		   DDhandlers[event].name, disk, time_print(eventTime)));
    INVARIANT4((eventTime < NEVER), 
	       "DDEvent %d: event %s scheduled for NEVER\n", 
	       DDhandlers[event].name, disk);

    /* 
     * MAKE_REQUEST_3(DiskDevice_SimRequest, MY_STID, eventTime, 
     *      	      event, disk, proc):
     */
    req = new_request_(); 
    req->h.reqtype = DiskDevice_SimRequest; 
    req->h.tid = NO_TID;      /* or currpptr->p_tid */
    req->h.timestamp = eventTime; 
    req->h.argc = 2; 
    req->argv[0] = (Word)(event); 
    req->argv[1] = (Word)(disk); 
    enqueue_request_(req); 

    return(req);
}

/* @SUBTITLE DDEventHandler: handle a disk device request" */
/* Note that we are running in ENGINE mode, and thus the DiskDevice 
 * code will also be running in ENGINE mode. This should be ok, as it 
 * usually just make more requests, and sometimes wakeup a thread. It
 * is not cycle-counted, and in a sense is not really part of the user's 
 * thread anyway.  However, sometimes diskdevice calls "user"-mode code, 
 * so it needs to be operating in the context of a user-mode thread, at
 * least so there is a clearly-defined current processor and current thread
 * identifier.
 */
PRIVATE void
DDEventHandler(SimRequest *req)
{
    TICS eventTime = req->h.timestamp; /* what time the event happens */
    REQUESTCODE event = req->argv[0]; /* what event should happen */
    int disk = req->argv[1];  /* on what disk */

    INVARIANT4(req->h.argc == 2, 
	       "DDEventHandler has %d-argument request, code %d\n",
	       req->h.argc, event);

    (*(DDhandlers[event].handler))(disk, eventTime);
}

/* @SUBTITLE DDEventPrint: print a disk-device request" */
void
DDEventPrint(FILE *fp, void *request) /* really SimRequest*request */
{
    SimRequest *req = request;
    TICS eventTime = req->h.timestamp;
    REQUESTCODE event = req->argv[0];
    int disk = req->argv[1];

    INVARIANT4(req->h.argc == 2, 
	       "DDEventPrint has %d-argument request, code %d\n",
	       req->h.argc, event);

    fprintf(fp, "DiskDevice request %s(disk %d, time %s)\n", 
	    DDhandlers[event].name, disk, time_print(eventTime));
}

/* @SUBTITLE DDEventCancel: cancel a pending disk-device request" */
void
DDEventCancel(DDEvent *ddreq)
{
    SimRequest *req = ddreq;
    TICS eventTime = req->h.timestamp;
    REQUESTCODE event = req->argv[0];
    int disk = req->argv[1];

    REQUEST_TRACE(("\tDDEventCancel %s for disk %d, @%s\n", 
		   DDhandlers[event].name, disk, time_print(eventTime)));

    req->h.reqtype = SC_NOP; /* make it into a no-op */
}

/* @SUBTITLE DDEventWakeup: do a thread-wakeup, from within DD handler" */
/* Remember that all the DDhandler functions are not running in user mode,
 * as part of a thread, but rather as code within the simulator.  Thus,
 * some "OS" functions like thread_wakeup won't work correctly.  Indeed,
 * thread_wakeup will do its duty and put the given thread on the ready 
 * queue, but if the processor is idle (has no running threads or pending 
 * interrupts) then we need to resume something -- in particular, that 
 * newly wakened thread.   This code is adapted from _thread_yieldToScheduler.
 * So much for a modular, extensible simcall interface, Proteus. 
 */
void
DDEventWakeup(int stid, TICS when) /* note we need an stid, not tid */
{
    Thread *osptr;	      /* runtime thread info */
    Thread *newt;	      /* a new thread to run */
    Thread *sched_choose();   /* scheduler picks a new thread to run */
    int proc = processor_;    /* remember where we are */
    ProcBlk *pptr = currpptr; /* remember it so we can switch back */

    /* get OS info about this stid */
    INVARIANT3(!Invalidstid(stid),
	       "DDEventWakeup failed: Thread stid %d is invalid", stid);
    
    osptr = (Thread *)T_OSBLOCK(stid);

    INVARIANT3(osptr != NULL,
	       "DDEventWakeup failed: Thread stid %d does not exist", stid);
    
    /* From that, get the processor where it lives; */
    /* and switch "context" to that processor. */
    processor_ = osptr->t_processor; /* also redefines CURR_PROCESSOR */
    currpptr = &proc_table_[CURR_PROCESSOR];

    /* update processor-specific user pointers (for scheduler's sake) */
    set_local_pointers(CURR_PROCESSOR);
    
    REQUEST_TRACE(("DDEventWakeup for thread %d/%d @%s (now %s)\n", 
		   CURR_PROCESSOR, osptr->t_tid, time_print(when), 
		   time_print(NOW)));
    
    /* wake up the given thread */
    PROTECT(thread_wakeup(osptr->t_tid));

    /* Do nothing more unless the processor is idle, in which case
     * we need to get it working on the newly wakened thread.
     */
    if (Idle(CURR_PROCESSOR)) {
	/* Choose a new thread to run -- should be the one we woke up */
	PROTECT(newt = sched_choose());

	REQUEST_TRACE(("\tDDEventWakeup resuming thread %d/%d @%s (now %s)\n", 
		       CURR_PROCESSOR, newt->t_tid, time_print(when), 
		       time_print(NOW)));
    
	INVARIANT3(newt != NULL, "DDEventWakeup %d: no ready thread to run??",
		   CURR_PROCESSOR);

	idle_processors--;
	EVENT_BUSY(max(when, NOW), CURR_PROCESSOR);

	/* Make that thread the current thread, and resume it */
	newt->t_state = T_CURRENT;
	currpptr->p_tid = newt->t_stid;
	MAKE_RESUME_REQUEST(newt->t_stid, max(when, NOW), OK);
    } else {
	/* this processor is not idle, ie, there are other threads to run */
	/* we're not going anywhere, so put things back where they were */
	processor_ = proc; /* also redefines CURR_PROCESSOR */
	currpptr = pptr;
	set_local_pointers(CURR_PROCESSOR);
    }
}
