/* @TITLE "diskdevice.h: Definitions for disk device"*/
/* 
 * Interface for the disk-device module.
 * macros:
 *   all of the useful macros are found in diskdriver.h
 *
 * functions:
 *   DiskDeviceInit
 *   DiskDeviceMove
 *   DiskDeviceXfer
 *   DiskDeviceShape
 *
 * Part of 
 *           The STARFISH Parallel file-system simulator
 *      (Simulation Tool for Advanced Research in File Systems)
 *
 *                              David Kotz
 *                          Dartmouth College
 *                             Version 3.0
 *                             October 1996
 *                         dfk@cs.dartmouth.edu
 */

/* $Id: diskdevice.h,v 3.0 1996/10/18 06:05:51 dfk RELEASE3 dfk $ */
/* Based on disk.h,v 1.33 93/12/24 12:40:44 dfk GENERALWORKS */

#ifndef DISKDEVICE_H
#define DISKDEVICE_H

/* *********************************************************************** */
#ifdef DFK
# include "userdata.h"
#else		      /* this stuff needed for standalone version */
# include "modularize.h"
# include "heap.h"
#endif DFK

#include "queue.h"

#ifdef TRIVIAL_DISK
#ifdef MODEL_DISK
error!!!! cannot have both TRIVIAL_DISK and MODEL_DISK.
#endif
#endif

/* **********************************************************************  */

/* #include "time.h" */

/* FUNCTIONS */
/* The caller must provide a pointer to an integer, which  
 * we use to tell which disk is using the bus.  We also need a QUEUE, 
 * with room for as many disks as are attached to that bus, where disks 
 * can wait for service on the bus.   Presumably, several disks share
 * the same values; thus, this is how we handle bus contention.
 * It must not be deallocated until after DiskDeviceDone is called.
 * The busId is used for debugging and tracing, nothing functional.
 * We also need an optional file name. If this name is not NULL, the 
 * file is opened (created if needed) and used as a fake "disk" where 
 * the data will be stored.  The file is likely to have lots of holes,
 * if the disk is not used in a contiguous layout. 
 */
extern void DiskDeviceInit(int disk, 
			   int busId, int *busOwner, NCQUEUE *busWaitq,
			   char *diskFileName);

#define BUS_FREE (-1)	      /* (*busOwner == BUS_FREE) ==> bus is free */

extern void DiskDeviceDone(int disk);

/* Do a disk operation. We give it everything it needs to know: the disk
 * number, physical block number on that disk, a read/write flag,
 * and the buffer for the data.  This BLOCKS until the time when the
 * transfer would be complete.
 */
/* disk number, logical sector, number of sectors, write?, buffer */
extern void DiskDeviceTransfer(int disk, ulong lsector, ulong nsectors,
		   boolean write, UserData buffer);

/* Like DiskDeviceTransfer, but split into two parts: the first part
 * starts the disk moving to the appropriate location, and the second supplies
 * the buffer needed to complete the transaction.  The second call 
 * BLOCKs until the transfer is complete.  There should not be any other
 * requests between a Start and a corresponding Finish.
 */ 
extern void DiskDeviceStart(int disk, ulong sector, ulong nsectors, 
			    boolean write);
extern void DiskDeviceFinish(int disk, UserData buffer);
extern void DiskDeviceSync(int disk);

/* Return the physical size of the disk */
extern void DiskDeviceShape(int disk, ulong *nSectors, ulong *sectorSize,
			    ulong *nTracks, ulong *sectorsPerTrack);

#endif /* DISKDEVICE_H */
