/**************************************************************************\
 * support.c, part of simex
 *
 *  David Kotz, 1996
 *
 *  simex is a stripped-down version of stats, in which I tossed all 
 * of the GUI and graph-making stuff.  It's just enough to read events.sim
 * and print out the metrics. 
 *
 * DERIVED FROM
 *
 *                 Proteus Parallel-Architecture Simulator                
 *                Eric A. Brewer  and  Chris N. Dellarocas                
 *                     Laboratory for Computer Science                    
 *                  Massachusetts Institute of Technology                 
 *
 * Module: support.c (stats)
 *
 * Last Modified: $Date: 1996/07/04 03:30:05 $ ($Author: dfk $)
 * Revision $Revision: 1.7 $
 * 
 *     
 ****************************************************************************
 *   Copyright 1991-1994                                                  
 *   Eric A. Brewer  and  Chris N. Dellarocas                             
 *   Massachusetts Institute of Technology                                
 *                                                                        
 *   Permission to use, copy, modify, and distribute this program         
 *   for any purpose and without fee is hereby granted, provided          
 *   that this copyright and permission notice appear on all copies       
 *   and supporting documentation, the name of M.I.T. not be used         
 *   in advertising or publicity pertaining to distribution of the        
 *   program without specific prior permission, and notice be given       
 *   in supporting documentation that copying and distribution is         
 *   by permission of M.I.T.  M.I.T. makes no representations about       
 *   the suitability of this software for any purpose.  It is pro-        
 *   vided "as is" without express or implied warranty.		          
 ****************************************************************************
 *
 * $Log: support.c,v $
 * Revision 1.7  1996/07/04 03:30:05  dfk
 * this is a nice working version.
 *
 * Revision 1.6  1996/07/02 02:58:17  dfk
 * completely new command-line interface
 * detailed control over what gets printed
 * ability to use abbreviated names for metrics
 *
 * Revision 1.5  1996/07/02 02:34:59  dfk
 * completely revamped options and printing choices.
 *
 * Revision 1.4  1996/07/01 22:51:44  dfk
 * first really working version
 *
 * Revision 1.3  1996/06/30 21:33:10  dfk
 * NEW for simex
 *
 * Revision 1.2  1996/06/30 19:40:05  dfk
 * some bug fixes and changes.
 *
 * Revision 1.2  94/12/14  14:51:07  dfk
 * fixed to improve error handling, particularly when the
 * file is not complete (it used to loop infinitely).
 * 
 * Also fixcxed pinting metrics so that large integers are
 * printed as integers rather than in E format.
 * 
 * Revision 1.1  94/12/14  12:20:23  dfk
 * Initial revision
 * 
 * Revision 1.3  1993/02/26  22:14:06  brewer
 * Added support for metrics from command line
 * Added usertypes arg
 *
 * Revision 1.2  92/09/22  13:14:43  brewer
 * Fixed bug with tables in old stats; now uses multiple columns and rows
 * correctly.
 * 
 * Revision 1.1  92/09/16  13:58:55  brewer
 * Initial revision
 * 
 * Revision 1.1  92/09/14  15:41:47  brewer
 * Initial revision
 * 
 \**************************************************************************/

#include <stdio.h>
#include <ctype.h>
#include <string.h>
#include <stdlib.h>
#include <malloc.h>
#include <math.h>
#include "simex.h"

const int nameWidth = 40, numWidth = 14;

/* values for array-metric printing */
const int PRINT_SUM =    001;
const int PRINT_MIN =    002;
const int PRINT_AVG =    004;
const int PRINT_MAX =    010;
const int PRINT_NUM =    020;
const int PRINT_ALL =    037; /* all of the above */
const int PRINT_DETAIL = 040;
static int arrayOptions = 0;  /* current set of options (used by -all) */

static BOOL print_name = TRUE;
static BOOL short_name = FALSE;
static BOOL print_header = TRUE;
static char *fieldSep = "";   /* field separator */

typedef struct s_nameList {
    char *name;
    struct s_nameList *next;
    int options;	      /* or'd combo of PRINT_* */
} nameList;

/*  list of metric names to be output */
static nameList *metricsListHead, **metricsListTail;

static inline void numPrint(float val, int width);
static void PrintHelp(void);
static void PrintScalarMetric(int m, char *name);
static void PrintArrayMetric(int m, char *name, int arrayOptions);
static int WhichScalarMetric(const char *name);
static int WhichArrayMetric(const char *name);
static void PrintArrayMetricsHeader(void);

/*************************************************************************\
*  numPrint, used by PrintMetrics
\*************************************************************************/

static inline void numPrint(float val, int width)
{
  extern float ftrunc(float); /* is not, tho should be, in math.h */
  
      /* if it's an integer; print with plenty of precision */
  if (val - ftrunc(val) == 0)
    printf(" %*.0f%s", width, val, fieldSep);
  else
    printf(" %*g%s", width, val, fieldSep);
}

/*************************************************************************\
*  PrintVersion
\*************************************************************************/

GLOBAL void PrintVersion(void)
{
    printf("\nSimex Version %.2f (%s)\n", version_number, date_string);
    printf("By David Kotz, derived from Stats by Eric A. Brewer\n");
    printf("Event file: `%s'\n", event_filename);
    printf("Event types file: `%s'\n", types_filename);
    printf("User event types file: `%s'\n", usertypes_filename);
    if (simulation_title != NULL)
      printf("Simulation title: \"%s\".\n\n", simulation_title);
    else
      printf("Simulation does not have a title.\n\n");
}

/*************************************************************************\
*  Print help text
\*************************************************************************/

static void PrintHelp(void)
{
    PrintVersion();

    fprintf(stderr, "Usage: simex -help ...\n");
    fprintf(stderr, "or     simex [baseOption] [arrayOption | metricName]...\n");
    fprintf(stderr,
	" Where 'metricName' is the name of one of the metrics\n");
    fprintf(stderr,
	"   (or an abbreviation, ie, [metricName] is sufficient)\n");
    fprintf(stderr,
	" and 'baseOption' is one of the following:\n");
    fprintf(stderr,
	"\t-sim <sim file>      Defines sim file (Default: events.sim).\n");
    fprintf(stderr,
	"\t-types <type file>   Defines type file (Default: events.h).\n");
    fprintf(stderr,
	"\t-usertypes <file>    Defines user type file (Default: user-events.h).\n");
    fprintf(stderr,
	"\t-t <num>             Allocate space for <num> events initially.\n");
    fprintf(stderr,
	"\t-br                  Byte reverse event file during load.\n");
    fprintf(stderr,
	"\t-trace               Print event trace.\n");
    fprintf(stderr,
	"\t-FSsep               Output field separator is string \"sep\".\n");
    fprintf(stderr,
	"\t-noname              Do not print metric names, just values.\n");
    fprintf(stderr,
	"\t-shortname           Print only short form of metric names.\n");
    fprintf(stderr,
	"\t-noheader            Do not print column headers, just metrics.\n");
    fprintf(stderr,
	"\t-debug               Print information for debugging simex.\n");
    fprintf(stderr,
	"\t-all                 Print all metrics, ignoring metricNames;\n");
    fprintf(stderr,
	"\t                       turns on -sum -min -avg -max -num;\n");
    fprintf(stderr,
	"\t                       turns off -shortname.\n");
    fprintf(stderr,
	"\t-v                   Verbose; prints version date and number.\n");
    fprintf(stderr,
	" and 'arrayOption' is combination of the following, which toggle\n");
    fprintf(stderr,
	" on each occurrence, and affect any subsequent array metrics:\n");
    fprintf(stderr,
	"\t-sum                 Print sum of each array metric.\n");
    fprintf(stderr,
	"\t-min                 Print min of each array metric.\n");
    fprintf(stderr,
	"\t-avg                 Print average of each array metric.\n");
    fprintf(stderr,
	"\t-max                 Print max of each array metric.\n");
    fprintf(stderr,
	"\t-num                 Print number of each array metric.\n");
    fprintf(stderr,
	"\t-detail              Print detail of each array metric.\n");
}

static BOOL checkArrayOptions(char *arg)
{
  if (strcmp("-num", arg) == 0) {
    arrayOptions ^= PRINT_NUM;
    return(TRUE);
  } else if (strcmp("-sum", arg) == 0) {
    arrayOptions ^= PRINT_SUM;
    return(TRUE);
  } else if (strcmp("-min", arg) == 0) {
    arrayOptions ^= PRINT_MIN;
    return(TRUE);
  } else if (strcmp("-avg", arg) == 0) {
    arrayOptions ^= PRINT_AVG;
    return(TRUE);
  } else if (strcmp("-max", arg) == 0) {
    arrayOptions ^= PRINT_MAX;
    return(TRUE);
  } else if (strcmp("-detail", arg) == 0) {
    arrayOptions ^= PRINT_DETAIL;
    return(TRUE);
  } else
    return(FALSE);
}

/*************************************************************************\
*  Handle Command line arguments
\*************************************************************************/

GLOBAL void HandleArgs(int argc, char **argv)
{
  int i;

  /* first part: baseOptions */
  for (i = 1; i < argc; i++) {
    if (argv[i][0] == '-') {
      /* first see if it is start of arrayOptions */
      if (checkArrayOptions(argv[i])) {
	i++;
	break;
      }

      /* then look for baseOptions */
      switch (argv[i][1]) {
      case 'b':	      /* byte reverse */
	if (strcmp(argv[i], "-br") == 0)
	  byte_reverse = !byte_reverse;
	else
	  fatal("Unknown base option %s, type simex -help", argv[i]);
	break;
      case 'd':	      /* debug */
	if (strcmp(argv[i], "-d") == 0)
	  debug = !debug;
	else
	  fatal("Unknown base option %s, type simex -help", argv[i]);
	break;
      case 'F':	      /* field separator */
	if (strncmp(argv[i], "-FS", 3) == 0)
	  fieldSep = &(argv[i][3]);
	else
	  fatal("Unknown base option %s, type simex -help", argv[i]);
	break;
      case 'n':	      /* noname */
	if (strcmp(argv[i], "-noname")==0) {
	  print_name = !print_name;
	} else if (strcmp(argv[i], "-noheader")==0) {
	  print_header = !print_header;
	} else
	  fatal("Unknown base option %s, type simex -help", argv[i]);
	break;
      case 'a':	      /* all */
	if (strcmp(argv[i], "-all")==0) {
	  all_metrics = !all_metrics;
	  if (all_metrics)
	    arrayOptions = PRINT_ALL;
	} else
	  fatal("Unknown base option %s, type simex -help", argv[i]);
	break;
      case 's':	      /* file name of sim file */
	if (strcmp(argv[i], "-sim") == 0) {
	  i++;
	  if (i == argc || argv[i][0] == '-')
	    fatal("Expecting filename after `-sim' argument.");
	  else
	    event_filename = argv[i];
	} else if (strcmp(argv[i], "-shortname") == 0) {
	  short_name = !short_name;
	} else {
	  fatal("Unknown base option %s, type simex -help", argv[i]);
	}
	break;
      case 'h':	      /* help */
	if (strcmp(argv[i], "-help") == 0) {
	  PrintHelp();
	  exit(0);
	}
	break;
      case 't':	      /* allocate space for this many events  */
	if (strcmp(argv[i], "-t")==0) {
	  i++;
	  event_list_size = atol(argv[i]);
	  if (event_list_size < 1000)
	    fatal("-t specifies table size < 1000 events.\n");
	} else if (strcmp(argv[i], "-types")==0) {
	  i++;
	  if (i == argc || argv[i][0] == '-')
	    fatal("Expecting filename after `-types' argument.");
	  else
	    types_filename = argv[i];
	} else if (strcmp(argv[i], "-trace")==0) {
	  display_trace = !display_trace;
	} else {
	  fatal("Unknown base option %s, type simex -help", argv[i]);
	}
	break;
      case 'u':	      /* usertype file name */
	if (strcmp(argv[i], "-usertypes")==0) {
	  i++;
	  if (i == argc || argv[i][0] == '-')
	    fatal("Expecting filename after `-usertypes' argument.");
	  else
	    usertypes_filename = argv[i];
	}
	break;
      case 'v':	      /* verbose */
	if (strcmp(argv[i], "-v")==0) {
	  verbose = TRUE;
	} else {
	  fatal("Unknown base option %s, type simex -help", argv[i]);
	}
	break;
      default:
	fatal("Unknown base option %s, type simex -help", argv[i]);
      }
    } else {	      /* if argv[i][0] == '-' */
      /* assume this is a metric name */
      /* no more base options allowed */
      break;
    }
  }

  /* second part: mixture of metric names and arrayOptions */
  for ( ; i < argc; i++) {
    if (argv[i][0] == '-') {
      if (!checkArrayOptions(argv[i]))
	fatal("Unknown arrayOption %s, type simex -help", argv[i]);
    } else {
      /* assume this is a metric name */
      if (all_metrics) {
	fprintf(stderr,
		"Ignoring argument '%s', since -all is in effect\n", 
		argv[i]);
      } else {
	/* allocate a new list node and save this name there */
	nameList *np = (nameList *) malloc(sizeof(nameList));
	if (np == NULL) fatal("out of memory");
	  
	np->name = malloc(strlen(argv[i]) + 1);
	if (np->name == NULL) fatal("out of memory");
	  
	strcpy(np->name, argv[i]);
	np->next = NULL;
	np->options = arrayOptions;
	  
	/* add to list */
	if (metricsListTail == NULL) {
	  metricsListHead = np;
	} else {
	  *metricsListTail = np;
	}
	metricsListTail = &np->next;
      }
    }
  }
}


/*************************************************************************\
*  Which*Metric
\*************************************************************************/

/* return the index of metrics[] where we found np->name, or -1 if not found */
static int WhichScalarMetric(const char *name)
{
  int m;

  for (m = 0; m < MAX_METRICS; m++) {
    char *metricName = metrics[m].name;

    if (metricName == NULL) continue;
    /* check to see if name is an exact match for metric name */
    if (strcmp(metricName, name) == 0)
      return (m);
    /* check to see if name is an abbreviated metric name */
    if (metricName[0] == '[' && strncmp(metricName+1, name, strlen(name)) == 0)
      return (m);
  }

  return(-1);
}

/* return index of ametrics[] where we found np->name, or -1 if not found */
static int WhichArrayMetric(const char *name)
{
  int m;

  for (m = 0; m < MAX_AMETRICS; m++) {
    char *metricName = ametrics[m].name;

    if (metricName == NULL) continue;
    /* check to see if name is an exact match for metric name */
    if (strcmp(metricName, name) == 0)
      return (m);
    /* check to see if name is an abbreviated metric name */
    if (metricName[0] == '[' && strncmp(metricName+1, name, strlen(name)) == 0)
      return (m);
  }

  return(-1);
}

/*************************************************************************\
*  PrintMetrics
\*************************************************************************/

GLOBAL void PrintSelectedMetrics(void)
{
  nameList *np;

  if (print_header)
    PrintArrayMetricsHeader();

  for (np = metricsListHead; np != NULL; np = np->next) {
    int m = WhichScalarMetric(np->name); /* see if it is scalar metric */
    if (m >= 0)
      if (short_name)	      /* use their abbreviation */
	PrintScalarMetric(m, np->name);
      else		      /* use full name */
	PrintScalarMetric(m, metrics[m].name); 
    else {
      int m = WhichArrayMetric(np->name); /* see if it is array metric */
      if (m >= 0)
	if (short_name)	      /* use their abbreviation */
	  PrintArrayMetric(m, np->name, np->options);
	else		      /* use full name */
	  PrintArrayMetric(m, ametrics[m].name, np->options);
      else
	fprintf(stderr, "Unknown metric name '%s'\n", np->name);
    }
  }
}

GLOBAL void PrintAllScalarMetrics(void)
{
  int m;

  for (m = 0; m < MAX_METRICS; m++)
    if (metrics[m].name != NULL)
      PrintScalarMetric(m, metrics[m].name);
}

static void PrintScalarMetric(int m, char *name)
{
  if (print_name)
    printf("%-*s%s", nameWidth, name, fieldSep);
  numPrint(metrics[m].value, numWidth);
  printf("\n");
}

GLOBAL void PrintAllArrayMetrics(void)
{
  int m;

  if (print_header)
    PrintArrayMetricsHeader();

  for (m = 0; m < MAX_AMETRICS; m++)
    if (ametrics[m].name != NULL)
      PrintArrayMetric(m, ametrics[m].name, arrayOptions);
}

static void PrintArrayMetricsHeader(void)
{
  if (arrayOptions) {
    printf("\n");
    if (print_name)
      printf("%-*s%s", nameWidth, "Array Metrics columns", fieldSep);
    if (arrayOptions & PRINT_SUM) printf(" %*s%s", numWidth, "sum", fieldSep);
    if (arrayOptions & PRINT_MIN) printf(" %*s%s", numWidth, "min", fieldSep);
    if (arrayOptions & PRINT_AVG) printf(" %*s%s", numWidth, "avg", fieldSep);
    if (arrayOptions & PRINT_MAX) printf(" %*s%s", numWidth, "max", fieldSep);
    if (arrayOptions & PRINT_NUM) printf(" %2s%s", "n", fieldSep);
    printf("\n");
  } /* else there will be no columns printed; need no header */
}

static void PrintArrayMetric(int m, char *name, int opts)
{
  ametricType *ptr = &ametrics[m];
  
  if (print_name)
    printf("%-*s%s", nameWidth, name, fieldSep);

  /* if any of these desired, compute and print them */
  if (opts & (PRINT_SUM | PRINT_MIN | PRINT_AVG | PRINT_MAX)) {
    double sum, min, avg, max;
    int i;
    min = max = sum = ptr->value[0];
    for (i = 1; i < ptr->size; i++) {
      double value = ptr->value[i];
      sum += value;
      if (value > max) max = value;
      if (value < min) min = value;
    }
    avg = sum / ptr->size;

    if (opts & PRINT_SUM) numPrint(sum, numWidth);
    if (opts & PRINT_MIN) numPrint(min, numWidth);
    if (opts & PRINT_AVG) numPrint(avg, numWidth);
    if (opts & PRINT_MAX) numPrint(max, numWidth);
  }

  if (opts & PRINT_NUM) printf(" %2d%s", ptr->size, fieldSep);

  printf("\n");

  if (opts & PRINT_DETAIL) {
    ametricType *ptr = &ametrics[m];
    const int indexWidth = 5;
    int i;

    if (print_name)
      printf("%-*s%s", nameWidth, ptr->name, fieldSep);
    printf(" %*s%s %*s%s\n", 
	   indexWidth, "index", fieldSep, 
	   numWidth, "value", fieldSep);
    
    for (i = 0; i < ptr->size; i++) {
      if (print_name)
	printf("%-*s%s", nameWidth, ptr->name, fieldSep);
      printf(" %*d%s", indexWidth, i, fieldSep);
      numPrint(ptr->value[i], numWidth);
      printf("\n");
    }
    printf("\n");
  }
}

