
 /**************************************************************************\
 *
 *                 Proteus Parallel-Architecture Simulator
 *                Eric A. Brewer  and  Chris N. Dellarocas
 *                     Laboratory for Computer Science
 *                  Massachusetts Institute of Technology
 *
 * Module:  "snapshot" debugging routines
 *
 * Description: snapshot() and supporting routines
 *
 * Last Modified:  $Date: 94/11/27 20:20:27 $ (eab)
 *
 * Global Functions:
 *     void control_C_()
 *     void snapshot_handler_(SimRequest *ptr)
 *     void snapshot()
 *     void DisplaySimReq(FILE *fp, SimRequest *ptr)
 *
 * Global Variables:
 *     BOOL snap
 *
 * Referenced Parameters:
 *     MULT_RL, MAX_SEMAPHORES, MAX_THREADS, NO_OF_PROCESSORS, LINE_SIZE,
 *     MAX_DEFINED_IPI
 *
 ***************************************************************************
 *   Copyright 1991
 *   Eric A. Brewer  and  Chris N. Dellarocas
 *   Massachusetts Institute of Technology
 *
 *   Permission to use, copy, modify, and distribute this program
 *   for any purpose and without fee is hereby granted, provided
 *   that this copyright and permission notice appear on all copies
 *   and supporting documentation, the name of M.I.T. not be used
 *   in advertising or publicity pertaining to distribution of the
 *   program without specific prior permission, and notice be given
 *   in supporting documentation that copying and distribution is
 *   by permission of M.I.T.  M.I.T. makes no representations about
 *   the suitability of this software for any purpose.  It is pro-
 *   vided "as is" without express or implied warranty.		
 ****************************************************************************
 * $Header: /usr/wildcat/dfk/research/parallel/proteus/proteus-V3.01/engine/RCS/snapshot.c,v 1.4 94/11/27 20:20:27 dfk Exp $
 * $Log:	snapshot.c,v $
 * Revision 1.4  94/11/27  20:20:27  dfk
 * they forgot the fp parameter in DisplayReadyList.
 * 
 * Revision 1.3  94/11/22  21:15:02  dfk
 * added flush to most places to get the output to output file
 * 
 * Revision 1.2  94/01/24  00:39:40  dfk
 * Added Sanjay's support for 64-bit timers.
 * Added thread_sleep_until.
 * Fixed a few bugs in thread sleep/wakeup code (my bugs).
 * Fixed printf formats in many places.
 * 
 * Revision 1.1  94/01/23  11:35:29  dfk
 * Initial revision
 * 
 * Revision 1.16  92/12/09  17:40:30  brewer
 * added PassControl_no_interrupts to avoidTable
 * 
 * Revision 1.15  92/12/09  15:17:51  brewer
 * Updated output routines to use fp instead of stdout
 * 
 * Revision 1.14  92/11/25  17:49:19  brewer
 * Anthony added improved support for shared-memory display.
 * 
 * Revision 1.13  92/11/19  14:18:00  brewer
 * Added DisplayMessageQueue to DisplayProc
 * 
 * Revision 1.12  92/11/12  12:55:09  brewer
 * Moved DisplayTask in rt_thread.aux.c
 * Added parsing for runtime tids, i.e., #24:46 -- thread 46 on proc 24.
 * 
 * Revision 1.11  92/11/03  13:27:40  brewer
 * Moved DisplayLock and DisplayActiveLocks into sema.c
 * Added abort_on_fatal check to top of snapshot()
 * Added call to LookupThreadPC to print out RESUME requests
 * 
 * Revision 1.10  92/10/20  11:24:53  brewer
 * Added DisplayTask Loop to display ready list in DisplayProc
 * 
 * Revision 1.9  92/10/07  16:44:13  brewer
 * removed exit_statistics call from abort action since it now occurs
 * via the cleanup procedure in main.c
 * 
 * Revision 1.8  92/09/23  15:49:03  brewer
 * Now include <stdlib.h>
 * 
 * Revision 1.7  92/07/06  11:26:36  brewer
 * Changed to use print_request_ for user-defined sim calls
 * 
 * Revision 1.6  92/05/08  13:34:32  brewer
 * added initialization of FILE *fp to stdout
 * This allows non-snapshot calls to DisplaySimReq to work
 * 
 * Revision 1.5  92/05/01  17:29:59  brewer
 * Improved DisplaySimReq for SC_ROUTE_PACKET.
 * (now prints current node for hop-by-hop simulation)
 * 
 * Revision 1.4  92/04/29  17:14:55  brewer
 * Added intelligent cases for SC_SHARED_READ_FL, SC_SHARED_WRITE_FL
 * in DisplaySimReq
 * 
 * Revision 1.3  92/03/31  16:46:36  brewer
 * Changed DisplayTask to take an int tid instead of a Word (ulong)
 * 
 * Revision 1.2  92/02/12  15:08:49  brewer
 * Made DisplaySimReq global instead of static for use by req_queue.c
 * 
 * Revision 1.1  92/02/11  13:56:31  brewer
 * Initial revision
 * 
 \**************************************************************************/

#include <stdlib.h>
#include <signal.h>
#include <string.h>
#include <math.h>
#include <ctype.h>
#include "user.h"
#include "processor.h"
#include "intreq.h"
#include "simcalls.h"
#include "q.h"
#include "packet.h"
#include "net.h"
#include "event.h"   /* for exit_statistics_(), flush_event_file_() */

#define CMD_SIZE 40

static FILE *fp = stdout;
static char fnam[64];

GLOBAL BOOL snap;
static BOOL single_step = 0;
static BOOL trace_thread = 0, no_trace = 0;
static ulong trace_tid = 0;
extern BOOL abort_on_fatal_;  /* declared and set in main.c */

GLOBAL void control_C_() 
{
    snap = TRUE;
}

/**************************************************************************\
*   Display routines
\**************************************************************************/

static void DefineOutfile() 
{
    printf("\nOutput filename: ");
    if (fgets(fnam, 64, stdin) == NULL) {
	printf("input error: output file set to \"stdout\".\n");
	fp = stdout;
	strcpy(fnam, "stdout");
	return;
    }

    fnam[strlen(fnam) - 1] = 0; /* remove newline */

    if ((strcmp(fnam, "") == 0)	|| (strcmp(fnam, "stdout")==0)) {
	fp = stdout;
	strcpy(fnam, "stdout");
	printf("Output file set to \"stdout\"\n");
    } else if ((fp = fopen(fnam,"w")) == NULL) {
	printf("Cannot open file %s for writing\n", fnam);
	fp = stdout;
	strcpy(fnam, "stdout");
	printf("Output file set to \"stdout\"\n");
    } else {
	printf("Output file set to \"%s\"\n", fnam);
    }
}

/************************************/

extern void DisplayTask(FILE *fp, int stid);

#ifdef MULT_RL

static void DisplayReadyList(fp, p) 
FILE *fp;
int p;
{
    int start, end, tid;
    
    start = proc_table_[p].p_ready_head;
    end   = proc_table_[p].p_ready_tail;
    
    tid = q[start].qnext;
    while( tid != end ) {
	if (Invalidstid(tid) ) {
	    fprintf(fp, 
		 "Ready List of Processor %d corrupted. Invalid index %d.\n",
		    p, tid);
	    break;
	}
	DisplayTask(fp, tid);
	tid = q[tid].qnext;
    }

    fflush(fp);
}

#endif   

/************************************/

static void DisplayIntr(iptr) 
IntrRequest *iptr;
{
    int i, arg_count;
    
    fprintf(fp, "%5d\t%5d\t%5d (%10s)\t%s(",
	    iptr->iprio, iptr->isender,
	    iptr->isendtid, thread_table_[iptr->isendtid].t_name,
	    intr_name_[iptr->itype]);

    arg_count = MAX(iptr->argc, 5);

    for(i=0; i<arg_count; i++)
      fprintf(fp, "%s%lu", i>0 ? ", " : "" , iptr->argv[i]);

    if (arg_count < iptr->argc) 
      fprintf(fp, ", ... )\n");
    else
      fprintf(fp, ")\n");
}

static void DisplayInterrupts(p) 
int p;
{
    IntrRequest *ptr;
    extern IntrRequest *InterruptQueue[];
    
    fprintf(fp, "Interrupts pending on processor %d: %d\n\n",
	    p, intr_pending[p] );
    fprintf(fp, "Prio\tsender\tsendtid\t\ttype(args)\n\n");
    
    ptr = InterruptQueue[p];
    while( ptr != NULL ) {
	DisplayIntr(ptr);
	ptr = ptr->inextreq;
    }

    fflush(fp);
}

/************************************/

void DisplayProc(i) 
int i;
{
    ProcBlk *pptr = &proc_table_[i];
    int j;

    if (InvalidProc(i)) {
	fprintf(fp, "Invalid processor number (%d).\n", i);
	return;
    }
    
    fprintf(fp, "PROCESSOR %d\tTime %s\t%s\n",
	    i, time_print(proc_table_[i].p_time), Idle(i) ? "IDLE" : 
	    (proc_table_[i].p_ihandleractive ?
	     "SERVICING INTR" : "") );

#ifdef MULT_RL
    fprintf(fp, "Number of threads on processor : %d\n", pptr->p_numthreads);
    fprintf(fp,"Number of threads in ready list: %d\n", pptr->p_ready_threads);
#endif

    if ( NotIdle(i) ) {
	fprintf(fp, "Current task:\n");
	DisplayTask(fp,  proc_table_[i].p_tid );
    }

#ifdef MULT_RL
    fprintf(fp, "Ready List:\n");
    DisplayReadyList(fp, i);
    for(j=0; j<MAX_PROC_THREADS; j++)
      if (processor_private[i].thread_table[j].t_state == T_READY)
	DisplayTask(fp, processor_private[i].thread_table[j].t_stid);
    
#endif

    fprintf(fp, "Other Tasks:\n");
    for(j=0; j<MAX_PROC_THREADS; j++)
      if ((processor_private[i].thread_table[j].t_state == T_SUSPENDED ||
	   processor_private[i].thread_table[j].t_state == T_FORWARDER))
	DisplayTask(fp, processor_private[i].thread_table[j].t_stid);
    
    if ( intr_pending[i] > 0)
      DisplayInterrupts(i);

#ifdef SEND_RECEIVE
    DisplayMessageQueue(fp, i);
#endif

    fflush(fp);
}

static void DisplayProcessors() 
{
    int i;
    
    fprintf(fp,"\n\nPROCESSORS STATE AT TIME %s\n\n", time_print(GLOBALTIME));
    fprintf(fp,"\n Number of Processors %d\tIdle Processors %d\n",
	    NO_OF_PROCESSORS, idle_processors);
    
    for(i=0; i<NO_OF_PROCESSORS; i++) {
	DisplayProc(i); putc('\n', fp);
    }
}



/*************************************************************************
* Display Simulator Requests
*************************************************************************/

extern int LookupThreadPC_(ulong tid, char **proc_name, char **file_name,
			   ulong *avoidList, ulong count);
extern SimQuantum();
extern void PassControl_no_interrupts();

static avoidCount = 6;
static ulong avoidTable[] = {
  (ulong) ctxsw,
  (ulong) SimQuantum,
  (ulong) SimCall,
  (ulong) TrapToSimulator,
  (ulong) PassControl,
  (ulong) PassControl_no_interrupts
};


GLOBAL void DisplaySimReq(FILE *fp, SimRequest *rptr) 
{
    Word *a;
    int rc;
    char *pn, *fn;
    
    a = rptr->argv;
    
    fprintf(fp, "%10s %5d ", time_print(rptr->h.timestamp), rptr->h.tid);

    if (rptr->h.reqtype >= MIN_USER_SC) {
	(*print_request_[rptr->h.reqtype])(fp, rptr);
	return;
    }

    switch (rptr->h.reqtype) {
      case SC_RESUME:
	fprintf(fp, "Resume thread\n");
        rc = LookupThreadPC_(rptr->h.tid, &pn, &fn, avoidTable, avoidCount);
	if (rc > 0)
	  fprintf(fp, "%17s%s in \"%s\" line %d\n", "", pn, fn, rc);
	else if (rc == -4)
	  fprintf(fp, "%17sNew thread -- no stack yet\n", "");
	else
	  fprintf(fp, "%17sInvalid thread!\n", "");
	break;
      case SC_TIMER:
	fprintf(fp, "Timer interrupt, processor %d\n", (int)a[0]);
	break;
      case SC_IPI_BUS:
	fprintf(fp, "IPI (Bus), %d -> %d, priority %lu, type %s\n",
	        (int)a[0], (int)a[1], a[2], intr_name_[a[3]]);
	break;
      case SC_NOP:
	fprintf(fp, "Null Request\n");
	break;
      case SC_SEND_PACKET:
	fprintf(fp, "Send packet, %d -> %d\n", (int)a[0], (int)a[1]);
	fprintf(fp, "%17s", "");
	DisplayPacket(fp, (Packet *)a[2]);
	break;
      case SC_RECV_PACKET:
	fprintf(fp, "Receive, %d -> %d, at=%d, next=%d, line=%d\n",
		(int)a[0], (int)a[1], (int)a[4], (int)a[5], (int)a[6]);
	fprintf(fp, "%17s", "");
	DisplayPacket(fp, (Packet *)a[2]);
	break;
      case SC_ROUTE_PACKET:
#ifdef NET_EXACT
	fprintf(fp, "Route packet, %d -> %d, current = %d\n",
		(int)a[0], (int)a[1], (int)a[5]);
#else
	fprintf(fp, "Route packet, %d -> %d, (modeled network)\n",
		(int)a[0], (int)a[1]);
#endif
	fprintf(fp, "%17s", "");
	DisplayPacket(fp, (Packet *)a[2]);
	break;
      case SC_SHARED_READ_FL:
	fprintf(fp, "Shared double read (0x%lx), from (%d, 0x%lx), mode=%d\n",
		a[3], (int)a[0], a[1], (int)a[2]);
	break;
      case SC_SHARED_WRITE_FL:
	fprintf(fp,"Shared double write (0x%lx), %g to (%d, 0x%lx), mode=%d\n",
		a[3], *((double *)a[4]), (int)a[0], a[1], (int)a[2]);
	break;
      case SC_SHARED_READ:
	fprintf(fp, "Shared read (0x%lx), from (%d, 0x%lx), mode=%d\n",
		a[3], (int)a[0], a[1], (int)a[2]);
	break;
      case SC_SHARED_WRITE:
	fprintf(fp, "Shared write (0x%lx), %d to (%d, 0x%lx), mode=%d\n",
		a[3], (int)a[4], (int)a[0], a[1], (int)a[2]);
	break;
      case SC_FLUSH:
	fprintf(stderr, "Flush cache line, (%d, 0x%lx)\n",
		(int)a[0], a[1]);
	break;
      case SC_SHARED_READTAG:
	fprintf(fp, "Shared read tag (0x%lx), from (%d, 0x%lx), mode=%d\n",
		a[3], (int)a[0], a[1], (int)a[2]);
	break;
      case SC_SHARED_WRITETAG:
	fprintf(fp, "Shared write tag (0x%lx), %d to (%d, 0x%lx), mode=%d\n",
		a[3], (int)a[4], (int)a[0], a[1], (int)a[2]);
	break;
      case SC_SNAPSHOT:
	fprintf(fp, "Enter snapshot mode\n");
	break;
      case SC_SCHEDULE_INTR:
	fprintf(fp, "Schedule interrupt on %d: %s (priority %lu)\n",
		(int)a[0],
		intr_name_[a[1]],
		a[2]);
	break;
    }
}


static void DisplaySimReqs(FILE *fp) 
{
    SimRequest *ptr;
    
    fprintf(fp,"\n CONTENTS OF SIMULATOR REQUEST QUEUE AT TIME %s.\n", 
	    time_print(GLOBALTIME));
    fprintf(fp,"\n Pending Simulator Requests:\n\n");
    fprintf(fp,"    Time   Tid  Type\tTid\t\t\tType(args)\n\n");
    
    ptr = InitGenRequest_();
    while( ptr != NULL ) {
	DisplaySimReq(fp, ptr);
	ptr = NextGenRequest_();
    }

    fflush(fp);
}

/************************************/

static void DisplayShmem(FILE *fp) 
{
    char *cmd, tmp[CMD_SIZE];
    char input_string[32];
    ulong address;
    long num_words;
    char *pointer, *start;
    int i, j, k, module;
    
    for (;;) {
      printf("\nShared memory display\nOptions:\n");
      printf("\t1        Dump shared memory\n");
      printf("\t2        Print shared memory usage statistics\n");
      printf("\t3        Display shared memory buckets for a module\n");
      printf("\t4        Display shared memory buckets for all modules\n");
      printf("\t5        Perform shared memory allocation test\n");
      printf("\tQ        Return to main snapshot menu\n");

      do {
	printf("\nShared Memory Command >>>> ");
	fgets(tmp, CMD_SIZE, stdin);
	for (cmd = tmp; *cmd == ' ' || *cmd == '\t'; cmd++);
      } while ( *cmd == '\n');

      switch(toupper(*cmd)) {
      case 'Q':
	return;
      case '1':
	printf("Start address: ");
	fgets(input_string, 32, stdin);
	address = strtol(input_string, NULL, 0);
	start = pointer = (char *) address;

	if ((start < memory) || (start >= top_of_memory)) {
	  printf("Invalid address, not in shared memory\n");
	  continue;
	}
	printf("Number of words: ");
	fgets(input_string, 32, stdin);
	num_words = strtol(input_string, NULL, 0);
	
	if (num_words < 0) {
	  printf("Invalid number of words, negative\n");
	  continue;
	} else if ((start + num_words) > top_of_memory) {
	  num_words = top_of_memory - start;
	  printf("Number of words out of range of memory block, only showing %ld words\n", num_words);
	}
	for(i=0; i< num_words; i += 4) {
	  fprintf(fp, "\n%8lx: ", (ulong)start);

	  /* print hex bytes */
	  for (j = 0; j < 4; j++) {
	    for(k=0; k<4; k++) {
	      fprintf(fp, "%02x ", ((unsigned int) *pointer) & 0xFF);
	      pointer++;
	    }
	    putc(' ', fp);
	  }
	  putc('\n', fp);

	  /* print hex long words */
	  pointer = start;
	  fprintf(fp, "%8s  ", "longs");
	  for (j = 0; j < 4; j++) {
	    fprintf(fp, "%lx ", *((ulong *) pointer));
	    pointer += sizeof(ulong);
	  }
	  putc('\n', fp);

	  /* print decimal long words */
	  pointer = start;
	  fprintf(fp, "%8s  ", "longs");
	  for (j = 0; j < 4; j++) {
	    fprintf(fp, "%ld ", *((long *) pointer));
	    pointer += sizeof(long);
	  }
	  putc('\n', fp);

	  pointer = (start += 4*sizeof(long));
	  putc('\n', fp);
	}
	break;
      case '2':
	putchar('\n');
	print_memory_statistics(fp);
	break;
      case '3':
	printf("Module number: ");
	fgets(input_string, 32, stdin);
	module = strtol(input_string, NULL, 0);
	if ((module < 0) || (module > NO_OF_MODULES))
	  printf("Invalid module number, valid numbers are 0 to %d\n",
		 NO_OF_MODULES);
	else
	  dump_module(fp, module);
	break;
      case '4':
	putchar('\n');
	view_shared_memory(fp);
	break;
      case '5':
	putchar('\n');
	diagnostic_shared_memory();
	break;
      default:
	printf("Invalid command.\n");
      }

    fflush(fp);
  }
}


/**************************************************************************\
*  Snapshot routines
\**************************************************************************/

static void PrintCommandMenu()
{
    printf("\nOptions are :\n");
    printf("\t0        Define output filename (current: %s)\n", fnam);
    printf("\t1        Display detailed processors' state\n");
    printf("\t2        Display state of semaphores\n");
    printf("\t3        Display simulator request queue\n");
    printf("\t4        Shared memory display\n");
    printf("\tH<t>     Schedule halt at time <t>\n");
    printf("\t#<tid>   Display info on Thread <tid>\n");
    printf("\tP<n>     Display info on Processor <n>\n");
    printf("\tC        Continue simulation\n");
    printf("\tA        Abort simulation\n");
    printf("\tU        Execute UserDebug\n");
    printf("\tD<a>     Disassemble address/symbol <a>\n");
    printf("\tO<a>     Add overwrite check for address <a>\n");
    printf("\tS        Toggle single step mode for simulator requests\n");
    printf("\tT<tid>   Trace requests of thread <tid>\n");
    printf("\tF        Flush event file.\n");
    printf("\t?        Redisplay Menu\n");
}


GLOBAL void snapshot_handler_(ptr)
SimRequest *ptr;
{
    printf("Simulator halted at time %s by user request.\n\n",
	   time_print(ptr->h.timestamp));
    snap = 1;
}



/***************************************************************************\
*    Enter snapshot mode
\***************************************************************************/

GLOBAL void snapshot()
{
    char *cmd, tmp[CMD_SIZE], *ptr;
    int i; Time time;
    SimRequest *req;
    
    /* snap > 1 implies that we are resuming 'snap' requests,
       so decrement snap and return */
    if (snap > 1) {
	snap--;
	return;
    }
    
    req = InitGenRequest_();

    if (!no_trace && trace_thread && req != NULL)
      if (req->h.tid != trace_tid) return;

    signal(SIGINT, SIG_IGN);
    
    fp = stdout;
    strcpy(fnam, "stdout");
    
    printf("\n*******************************************************************************\n\n");
    printf("Snapshot of machine at time %s.\n", time_print(GLOBALTIME));
    printf("Total number of active threads : %d\n", numthreads);

    if (req != NULL) {
	printf("Current request :\n");
	DisplaySimReq(stdout, req);
    } else {
	printf("Request queue is empty.\n");
    }
    
    if (abort_on_fatal_) exit(1);

    PrintCommandMenu();
    fflush(stdout);
    fflush(fp);

    for (;;) { 
	
	do {
	    printf("\n>>>> ");
	    fgets(tmp, CMD_SIZE, stdin);
	    for (cmd = tmp; *cmd == ' ' || *cmd == '\t'; cmd++);
	} while ( *cmd == '\n');
	
	switch(toupper(*cmd)) {
	  case '0':
	    DefineOutfile();
	    break;
	  case '1':
	    DisplayProcessors();
	    break;
	  case '2':
	    DisplayActiveLocks(fp);
	    break;
	  case '3':
	    DisplaySimReqs(fp);
	    break;
	  case '4':
	    DisplayShmem(fp);
	    break;
	  case '#':
	    for (cmd++; *cmd==' ' || *cmd=='\t'; cmd++);
	    if (isdigit(*cmd)) {
		int proc, stid; Thread *thread;
		
		ptr = cmd;
		while (isdigit(*ptr)) ptr++;
		if (*ptr == ':') {
		    ptr++;
		    proc = atoi(cmd);
		    if (proc < 0  ||  proc >= NO_OF_PROCESSORS) {
			printf("Invalid processor number %d.\n", proc);
			break;
		    }
		    if (!isdigit(*ptr)) {
			printf("Expecting runtime tid after colon.\n");
			break;
		    }
		    i = atoi(ptr);
		    if (i<0 || i>=MAX_PROC_THREADS) {
			printf("Runtime tid number %d out of range.\n",
				i);
			break;
		    }
		    stid = processor_private[proc] .thread_table[i].t_stid;
		    thread = T_OSBLOCK(stid);
		    if (thread==NULL || thread->t_tid != i
			|| thread->t_processor != proc)
		      printf("Thread %d:%d is inactive.\n", proc, i);
		    else
		      DisplayTask(fp, stid);
		} else {
		    DisplayTask(fp, atoi(cmd));
		}
	    } else if (*cmd == '\n') {
		if (req != NULL) DisplayTask(fp, req->h.tid);
		else printf("No current request.\n");
	    } else printf("Invalid thread id.\n");
	    break;
	  case 'P':
	    for (cmd++; *cmd==' ' || *cmd=='\t'; cmd++);
	    if (isdigit(*cmd)) DisplayProc(atoi(cmd));
	    else if (*cmd == '\n') {
		if (req != NULL)
		  DisplayProc(T_OSBLOCK(req->h.tid)->t_processor);
		else printf("No current request.\n");
	    } else printf("Invalid processor number.\n");
	    break;
	  case 'C':
	    if ( fp != stdout )
	      fclose(fp);
	    for (cmd++; *cmd==' ' || *cmd=='\t'; cmd++);
	    if (*cmd == '\n') {
		snap = single_step | trace_thread;
		no_trace = single_step;
		signal(SIGINT, control_C_);
		return;
	    } else {
		i = atol(cmd);
		if (i <= 0) {
		    printf("Invalid request count.\n");
		} else {
		    snap = i;
		    no_trace = TRUE;
		    printf("Executing %d requests.\n", i);
		    signal(SIGINT, control_C_);
		    return;
		}
	    }
	    break;
	  case 'A':
	    exit(1);
	    break;
	  case 'D':
	    for (cmd++; *cmd==' ' || *cmd=='\t'; cmd++);
	    if (isalpha(*cmd)) {
		/* skip to non alpha/num char */
		for (ptr = cmd + 1; isalnum(*ptr) || (*ptr == '_'); ptr++);
		*ptr = 0; /* terminate string at first non alpha/num char */
		DisassembleSymbol(cmd);
	    } else {
		/* convert substring to address and disassemble */
		Disassemble(strtol(cmd, NULL, 0), 0);
	    }
	    break;
	  case 'U':
	    UserDebug();
	    break;
	  case '?':
	    PrintCommandMenu();
	    break;
	  case 'O':
	    for (cmd++; *cmd==' ' || *cmd=='\t'; cmd++);
	    if (*cmd == '\n') {
		ListOverwriteChecks();
	    } else {
		if (*cmd == '#') {
		    i = strtol(cmd+1, NULL, 0);
		    if (i == 0)
		      printf("Invalid tid\n");
		    else
		      AddOverwriteCheck((long *)
					stktop(thread_table_[i].t_stkblk),
					"(stack check)");
		} else {
		    i = strtol(cmd, NULL, 0);
		    if (i == 0) printf("Invalid address.\n");
		    else AddOverwriteCheck((long *)i, "(set in snapshot)");
		}
	    }
	    break;
	  case 'S':
	    single_step = !single_step;
	    printf("Single step mode is now %s.\n",
		   single_step ? "on":"off");
	    break;
	  case 'T':
	    for (cmd++; *cmd==' ' || *cmd=='\t'; cmd++);
	    trace_tid = atol(cmd);
	    if (trace_tid == 0) {
		printf("Thread tracing turned off.\n");
		trace_thread = 0;
	    } else {
		trace_thread = 1;
		printf("Tracing thread #%lu\n", trace_tid);
	    }
	    break;
	  case 'H':
	    for (cmd++; *cmd==' ' || *cmd=='\t'; cmd++);
	    time = atol(cmd);
	    if (time > GLOBALTIME)
	      enqueue_snapshot_(time);
	    else
	      printf("Invalid time.\n");
	    break;
	  case 'F':
	    flush_event_file_();
	    break;
	  default:
	    printf("Invalid command.\n");
	} /* end switch */
    } /* end for */
}
