
 /**************************************************************************\
 * 
 *                 Proteus Parallel-Architecture Simulator 
 *                Eric A. Brewer  and  Chris N. Dellarocas 
 *                     Laboratory for Computer Science     
 *                  Massachusetts Institute of Technology  
 * 
 * Module: shmem.calls.c
 *                                                       
 * Description: shared-memory calls (run in user threads)
 *                                                       
 * Last Modified:  $Date: 94/01/24 00:39:35 $
 *                                                       
 * Global Functions:
 *     Word Shared_Memory_Read(void *address, Word mode)
 *     Word Shared_Memory_Write(void *address, Word value, Word mode) 
 *     Word Shared_Memory_ReadTag(void *address, Word mode)
 *     Word Shared_Memory_WriteTag(void *address, Word value, Word mode)
 *     double Shared_Memory_Read_Fl(void *address, Word mode) 
 *     double Shared_Memory_Write_Fl(void *address, double value, Word mode)
 *     int Flush(void *address)
 *     void Fence(void)
 *     
 * Global Variables: none                                
 *                                                       
 ****************************************************************************
 *   Copyright 1991                                                         *
 *   Eric A. Brewer  and  Chris N. Dellarocas                               *
 *   Massachusetts Institute of Technology                                  *
 *                                                                          *
 *   Permission to use, copy, modify, and distribute this program           *
 *   for any purpose and without fee is hereby granted, provided            *
 *   that this copyright and permission notice appear on all copies         *
 *   and supporting documentation, the name of M.I.T. not be used           *
 *   in advertising or publicity pertaining to distribution of the          *
 *   program without specific prior permission, and notice be given         *
 *   in supporting documentation that copying and distribution is           *
 *   by permission of M.I.T.  M.I.T. makes no representations about         *
 *   the suitability of this software for any purpose.  It is pro-          *
 *   vided "as is" without express or implied warranty.		            *
 ****************************************************************************
 * $Header: /usr/wildcat/dfk/research/parallel/proteus/proteus-V3.01/engine/RCS/shmem.calls.c,v 1.2 94/01/24 00:39:35 dfk Time64bit Locker: dfk $
 * $Log:	shmem.calls.c,v $
 * Revision 1.2  94/01/24  00:39:35  dfk
 * Added Sanjay's support for 64-bit timers.
 * Added thread_sleep_until.
 * Fixed a few bugs in thread sleep/wakeup code (my bugs).
 * Fixed printf formats in many places.
 * 
 * Revision 1.1  94/01/23  11:35:25  dfk
 * Initial revision
 * 
 * Revision 1.6  92/11/03  14:41:30  brewer
 * Added livelock detection
 * changed SAFE_MODE checks to use top_of_memory
 * 
 * Revision 1.5  92/04/21  12:19:41  brewer
 * Fixed return value calculation in Shared_Memory_Read_Fl and
 * Shared_Memory_Write_Fl
 * 
 * Revision 1.4  92/04/16  14:20:38  brewer
 * Improved the error messages for the SAFE_MODE checks.
 * 
 * Revision 1.3  92/04/16  12:38:56  brewer
 * Added safety checks for incoming addresses.
 * 
 * Revision 1.2  92/04/01  11:23:34  brewer
 * cleanup: fixed gcc warnings, ANSI prototypes
 * 
 * Revision 1.1  92/02/11  13:56:24  brewer
 * Initial revision
 * 
 \**************************************************************************/

/* Shared memory call interface */

#define SAFE_MODE

/* FATAL_LIVELOCK limits the number of retries within one read or write
   before Proteus exits with a fatal livelock error */
#define FATAL_LIVELOCK  5000
#define LIVELOCK_WARNING_COUNT  1000


GLOBAL Word Shared_Memory_Read(void *address, Word mode) 
{
    Word val;
    BOOL pprev, iprev, livelock_detected = FALSE;
    int try_count = 0;
    
#ifdef SAFE_MODE
    if (((char *)address < memory) ||
	((char *)address >= top_of_memory))
      fatal("Shared_Memory_Read: address=%ld=0x%lx\n\tNOT in shared memory\n",
	    address, address);
#endif

    pprev = currosptr->t_preempt;
    currosptr->t_preempt = FALSE;
    iprev = currosptr->t_intrflag;
    currosptr->t_intrflag = FALSE;
    
    for (try_count=0; try_count < FATAL_LIVELOCK; try_count++) {
	switch(SHARED_READ_RQ(processor_, address, mode, &val)) {
	  case MEM_READY:
	    goto exit;
	  case MEM_SWITCH:
	    if ((*currpptr->p_switch_ihandler)(address, 0,
					       &val, MR_READ) == MEM_RETRY)
	      break;
	    goto exit;
	  case MEM_WAIT:
	    break;
	  case MEM_MISMATCH:
	    if ((*currpptr->p_mismatch_ihandler)(address, 0, 
						 &val, MR_READ) == MEM_RETRY)
	      break;
	    goto exit;
	  case MEM_ERROR:
	    if ((*currpptr->p_error_ihandler)(address, 0, 
					      &val, MR_READ) == MEM_RETRY)
	      break;
	    goto exit;
	}
	if (try_count == LIVELOCK_WARNING_COUNT) {
	    SemBlk *sptr;
	    /* tried LIVELOCK_WARNING_COUNT times and failed,
	       possible livelock */
	    livelock_detected = TRUE;
	    fprintf(stderr,
		    "Warning: possible cache livelock, READ from #%d, P%d\n",
		    currtid, processor_);
	    fprintf(stderr,
		    "\ttime %s, location 0x%lx (real address 0x%lx -> %d)\n",
		    time_print(CURRENTTIME),
		    (ulong)((char*)address - memory), (ulong)address,
		    *((int *)address));
	    sptr = GetSemaphoreData_(address);
	    if (sptr != NULL)
	      fprintf(stderr, "\t(congestion on semaphore %s)\n",
		      sptr->name);
	}
    }
    /* tried FATAL_LIVELOCK times and failed, assume livelock */
    fatal("%s: P%d #%d cache livelock on location 0x%lx = %lu",
	  time_print(CURRENTTIME), processor_, currtid,
	  address, address);
    
  exit:
    if (livelock_detected)
      fprintf(stderr,
	       "Livelock resolved for #%d, P%d\n", currtid, processor_);
    currosptr->t_preempt = pprev;
    currosptr->t_intrflag = iprev;
    CheckForInterrupts();
    return(val);
}


GLOBAL Word Shared_Memory_Write(void *address, Word value, Word mode) 
{
    Word val;
    BOOL pprev, iprev, livelock_detected = FALSE;
    int try_count = 0;

#ifdef SAFE_MODE
    if (((char *)address < memory) ||
	((char *)address >= top_of_memory))
      fatal("Shared_Memory_Write: address=%ld=0x%lx\n\tNOT in shared memory\n",
	    address, address);
#endif

    pprev = currosptr->t_preempt;
    currosptr->t_preempt = FALSE;
    iprev = currosptr->t_intrflag;
    currosptr->t_intrflag = FALSE;
    
    for (try_count=0; try_count < FATAL_LIVELOCK; try_count++) {
	switch(SHARED_WRITE_RQ(processor_, address, mode, &val, value))
	  {
	    case MEM_READY:
	      goto exit;
	    case MEM_SWITCH:
	      if ((*currpptr->p_switch_ihandler)(address, value,
						 &val, MR_WRITE) == MEM_RETRY)
		break;
	      goto exit;
	    case MEM_WAIT:
	      break;
	    case MEM_MISMATCH:
	      if ((*currpptr->p_mismatch_ihandler)(address, value,
						   &val, MR_WRITE)
		  == MEM_RETRY)
		break;
	      goto exit;
	    case MEM_ERROR:
	      if ((*currpptr->p_error_ihandler)(address, value,
						&val, MR_WRITE) == MEM_RETRY)
		break;
	      goto exit;
	    default:
	      fatal("Internal: unexpected case, file %s line %d\n",
		    __FILE__, __LINE__);		    
	  }
	if (try_count == LIVELOCK_WARNING_COUNT) {
	    SemBlk *sptr;
	    /* tried LIVELOCK_WARNING_COUNT times and failed,
	       probably livelock */
	    livelock_detected = TRUE;
	    fprintf(stderr,
		    "Warning: probable cache livelock, WRITE from #%d, P%d\n",
		    currtid, processor_);
	    fprintf(stderr,
		    "\ttime %s, location 0x%lx (real address 0x%lx -> %d)\n",
		    time_print(CURRENTTIME),
		    (ulong)((char*)address - memory), (ulong)address,
		    *((int *)address));
	    sptr = GetSemaphoreData_(address);
	    if (sptr != NULL)
	      fprintf(stderr, "\t(congestion on semaphore %s)\n",
		      sptr->name);
	}
    }
    /* tried FATAL_LIVELOCK times and failed, assume livelock */
    fatal("%s: P%d #%d cache livelock on location 0x%lx = %lu",
	  time_print(CURRENTTIME), processor_, currtid,
	  address, address);

  exit:
    if (livelock_detected)
      fprintf(stderr,
	       "Livelock resolved for #%d, P%d\n", currtid, processor_);
    currosptr->t_preempt = pprev;
    currosptr->t_intrflag = iprev;
    CheckForInterrupts();
    return(val);
}


GLOBAL Word Shared_Memory_ReadTag(void *address, Word mode) 
{
    Word val;
    BOOL pprev, iprev;
    
#ifdef SAFE_MODE
    if (((char *)address < memory) ||
	((char *)address >= top_of_memory))
      fatal("Shared_Memory_ReadTag: address=%ld=0x%lx\n\tNOT in shared memory\n",
	    address, address);
#endif

    pprev = currosptr->t_preempt;
    currosptr->t_preempt = FALSE;
    iprev = currosptr->t_intrflag;
    currosptr->t_intrflag = FALSE;
    
    while(1)
      switch(SHARED_READTAG_RQ(processor_, address, mode, &val))
	{
	  case MEM_READY:
	    goto exit;
	  case MEM_SWITCH:
	    if ((*currpptr->p_switch_ihandler)(address, 0, &val, MR_READTAG)
		== MEM_RETRY)
	      break;
	    goto exit;
	  case MEM_WAIT:
	    break;
	  case MEM_MISMATCH:
	    if ((*currpptr->p_mismatch_ihandler)(address, 0, &val, MR_READTAG)
		== MEM_RETRY)
	      break;
	    goto exit;
	  case MEM_ERROR:
	    if ((*currpptr->p_error_ihandler)(address, 0, &val, MR_READTAG)
		== MEM_RETRY)
	      break;
	    goto exit;
	}
  exit:
    currosptr->t_preempt = pprev;
    currosptr->t_intrflag = iprev;
    CheckForInterrupts();
    return(val);
}


GLOBAL Word Shared_Memory_WriteTag(void *address, Word value, Word mode) 
{
    Word val;
    BOOL pprev, iprev;
    
#ifdef SAFE_MODE
    if (((char *)address < memory) ||
	((char *)address >= top_of_memory))
      fatal("Shared_Memory_WriteTag: address=%ld=0x%lx\n\tNOT in shared memory\n",
	    address, address);
#endif

    pprev = currosptr->t_preempt;
    currosptr->t_preempt = FALSE;
    iprev = currosptr->t_intrflag;
    currosptr->t_intrflag = FALSE;
    
    while(1)
      switch(SHARED_WRITETAG_RQ(processor_, address, mode, &val, value))
	{
	  case MEM_READY:
	    goto exit;
	  case MEM_SWITCH:
	    if ((*currpptr->p_switch_ihandler)(address, value,
					       &val, MR_WRITETAG) == MEM_RETRY)
	      break;
	    goto exit;
	  case MEM_WAIT:
	    break;
	  case MEM_MISMATCH:
	    if ((*currpptr->p_mismatch_ihandler)(address, value,
						 &val, MR_WRITETAG)
		== MEM_RETRY)
	      break;
	    goto exit;
	  case MEM_ERROR:
	    if ((*currpptr->p_error_ihandler)(address, value,
					      &val, MR_WRITETAG) == MEM_RETRY)
	      break;
	    goto exit;
	    
	}
  exit:
    currosptr->t_preempt = pprev;
    currosptr->t_intrflag = iprev;
    CheckForInterrupts();
    return(val);
}



GLOBAL double Shared_Memory_Read_Fl(void *address, Word mode) 
{
    double val;
    BOOL pprev, iprev;
    
#ifdef SAFE_MODE
    if (((char *)address < memory) ||
	((char *)address >= top_of_memory))
      fatal("Shared_Memory_Read_Fl: address=%ld=0x%lx\n\tNOT in shared memory\n",
	    address, address);
#endif

    pprev = currosptr->t_preempt;
    currosptr->t_preempt = FALSE;
    iprev = currosptr->t_intrflag;
    currosptr->t_intrflag = FALSE;
    
    while(1)
      switch(SHARED_READ_FL_RQ(processor_, address, mode, &val))
	{
	  case MEM_READY:
	    goto exit;
	  case MEM_SWITCH:
	    if ((*currpptr->p_switch_ihandler)(address, 0, 
					       &val, MR_READFL) == MEM_RETRY)
	      break;
	    goto exit;
	  case MEM_WAIT:
	    break;
	  case MEM_MISMATCH:
	    if ((*currpptr->p_mismatch_ihandler)(address, 0, 
						 &val, MR_READFL) == MEM_RETRY)
	      break;
	    goto exit;
	  case MEM_ERROR:
	    if ((*currpptr->p_error_ihandler)(address, 0,
					      &val, MR_READFL) == MEM_RETRY)
	      break;
	    goto exit;
	}
  exit:
    currosptr->t_preempt = pprev;
    currosptr->t_intrflag = iprev;
    CheckForInterrupts();
    return(val);
}


GLOBAL double Shared_Memory_Write_Fl(void *address, double value, Word mode) 
{
    double val;
    BOOL pprev, iprev;
    
#ifdef SAFE_MODE
    if (((char *)address < memory) ||
	((char *)address >= top_of_memory))
      fatal("Shared_Memory_Write_Fl: address=%ld=0x%lx\n\tNOT in shared memory\n",
	    address, address);
#endif

    pprev = currosptr->t_preempt;
    currosptr->t_preempt = FALSE;
    iprev = currosptr->t_intrflag;
    currosptr->t_intrflag = FALSE;
    
    while(1)
      switch(SHARED_WRITE_FL_RQ(processor_, address, mode, &val, &value))
	{
	  case MEM_READY:
	    goto exit;
	  case MEM_SWITCH:
	    if ((*currpptr->p_switch_ihandler)(address, &value,
					       &val, MR_WRITEFL) == MEM_RETRY)
	      break;
	    goto exit;
	  case MEM_WAIT:
	    break;
	  case MEM_MISMATCH:
	    if ((*currpptr->p_mismatch_ihandler)(address, &value,
						 &val, MR_WRITEFL)
		== MEM_RETRY)
	      break;
	    goto exit;
	  case MEM_ERROR:
	    if ((*currpptr->p_error_ihandler)(address, &value,
					      &val, MR_WRITEFL) == MEM_RETRY)
	      break;
	    goto exit;
	}
  exit:
    currosptr->t_preempt = pprev;
    currosptr->t_intrflag = iprev;
    CheckForInterrupts();
    return(val);
}


GLOBAL int Flush(void *address) 
{
#ifdef SAFE_MODE
    if (((char *)address < memory) ||
	((char *)address >= top_of_memory))
      fatal("Flush: address=%ld=0x%lx\n\tNOT in shared memory\n",
	    address, address);
#endif

#ifndef NOCACHE
    return(FLUSH_RQ(processor_, address));
#else
    return(MEM_READY);
#endif
}


GLOBAL void Fence() 
{
#ifndef NOCACHE
    protocol_fence();
#endif
}
