
 /**************************************************************************\
 *
 *                 Proteus Parallel-Architecture Simulator
 *                Eric A. Brewer  and  Chris N. Dellarocas
 *                     Laboratory for Computer Science
 *                  Massachusetts Institute of Technology
 *
 * Module: shmem.cache.c
 *
 * Description:
 *         Shared memory operations for all machines with caches
 *
 * Last Modified:  $Date: 92/04/01 11:22:17 $
 *
 * Global Functions:
 *
 * Global Variables:
 *
 ****************************************************************************
 *   Copyright 1991
 *   Eric A. Brewer  and  Chris N. Dellarocas
 *   Massachusetts Institute of Technology
 *
 *   Permission to use, copy, modify, and distribute this program
 *   for any purpose and without fee is hereby granted, provided
 *   that this copyright and permission notice appear on all copies
 *   and supporting documentation, the name of M.I.T. not be used
 *   in advertising or publicity pertaining to distribution of the
 *   program without specific prior permission, and notice be given
 *   in supporting documentation that copying and distribution is
 *   by permission of M.I.T.  M.I.T. makes no representations about
 *   the suitability of this software for any purpose.  It is pro-
 *   vided "as is" without express or implied warranty.
 ****************************************************************************
 * $Header: /psg/proteus/RCS/shmem.cache.c,v 1.2 92/04/01 11:22:17 brewer Exp $
 * $Log:	shmem.cache.c,v $
 * Revision 1.2  92/04/01  11:22:17  brewer
 * cleanup: fixed gcc warnings
 * uniform ANSI prototypes
 * 
 * Revision 1.1  92/02/11  13:56:23  brewer
 * Initial revision
 * 
 \**************************************************************************/

#include "net.h"
#include "sema.h"
#include "simcalls.h"
#include "simreq.h"
#include "processor.h"
#include "thread.h"
#include "event.h"

#include "shmem.common.c"

#include "shmem.calls.c"

/* Shared Memory call handlers for all machines with caches */

typedef struct {
    SimHeader  h;
    Word       processor; /* argv[0] */
    Word       address;   /* argv[1] */
    Word       mode;      /* argv[2] */
    Word       valaddr;   /* argv[3] */
    Word       value;     /* argv[4] */
} MemRequest;


void shared_read_handler_(MemRequest *rptr) 
{
    Word cache_op, result;
    Word iaddr = AddrToIndex(rptr->address);
    Word laddr = RoundAddrToLine(iaddr);
    
#ifdef ATDEBUG
    printf("@ %d (%d)will Read(%x) i=%x l=%x\n",
	   rptr->tid, rptr->processor, rptr->address, iaddr, laddr);
#endif
    
    cache_op = rptr->mode & SETTAG ? 
      (rptr->mode & SOFT ? MR_WRITESOFT : MR_WRITE) : 
	(rptr->mode & SOFT ? MR_READSOFT  : MR_READ );
    
    switch( result = processor_request_pre(&rptr->h.timestamp, cache_op,
					   rptr->processor, rptr->h.tid,
					   laddr) ) {
      case MEM_READY:
	
	if ( rptr->mode & TESTTAG )
	  if ( FebitInCache(rptr->processor, iaddr) != FULL ) {
	      MAKE_RESUME_REQUEST(rptr->h.tid, rptr->h.timestamp,
				  MEM_MISMATCH);
	      return;
	  }
	
	*(Word *)rptr->valaddr = 
	  shared_memory(AddrInCache(rptr->processor, iaddr), rptr->mode);
	
	if ( rptr->mode & SETTAG )
	  FebitInCache(rptr->processor, iaddr) = EMPTY;
	
#ifdef MEMDEBUG
	printf("READ(%d) returns %d to user\n", rptr->address,
	       *(Word *)rptr->valaddr);
#endif
	(void)processor_request_post(&rptr->h.timestamp, cache_op, 
				     rptr->processor, rptr->h.tid, laddr);
      case MEM_SWITCH:
      case MEM_WAIT:
      case MEM_ERROR:
	MAKE_RESUME_REQUEST(rptr->h.tid, rptr->h.timestamp, result);
	return;
      default:
	fatal("illegal value %d returned by CACHE_PROTOCOL_PRE", result);
    }
}



void shared_readtag_handler_(MemRequest *rptr) 
{
    Word cache_op, result;
    Word iaddr = AddrToIndex(rptr->address);
    Word laddr = RoundAddrToLine(iaddr);
    
#ifdef ATDEBUG
    printf("@ %d (%d)will ReadTag(%x) i=%x l=%x\n",
	   rptr->tid, rptr->processor, rptr->address, iaddr, laddr);
#endif
    
    cache_op = (rptr->mode & SOFT ? MR_READSOFT  : MR_READ );
    
    switch( result = processor_request_pre(&rptr->h.timestamp, cache_op,
					   rptr->processor,
					   rptr->h.tid, laddr)) {
      case MEM_READY:
	
	*(Word *)rptr->valaddr = FebitInCache(rptr->processor, iaddr);
	
#ifdef MEMDEBUG
	printf("READTAG(%d) returns %d to user\n", rptr->address,
	       *(Word *)rptr->valaddr);
#endif
	(void)processor_request_post(&rptr->h.timestamp, cache_op, 
				     rptr->processor, rptr->h.tid, laddr);
      case MEM_SWITCH:
      case MEM_WAIT:
      case MEM_ERROR:
	MAKE_RESUME_REQUEST(rptr->h.tid, rptr->h.timestamp, result);
	return;
      default:
	fatal("illegal value %d returned by CACHE_PROTOCOL_PRE", result);
    }
}



void shared_write_handler_(MemRequest *rptr) 
{
    Word cache_op, result, previous;
    void *caddr;
    Word iaddr = AddrToIndex(rptr->address);
    Word laddr = RoundAddrToLine(iaddr);
#ifdef ATDEBUG
    printf("@ %d (%d)will Write(%x, %d) i=%x l=%x\n",
	   rptr->tid, rptr->processor, rptr->address, rptr->value,
	   iaddr, laddr);
#endif
    
    
    cache_op = rptr->mode & SOFT ? MR_WRITESOFT : MR_WRITE;
    
    switch( result = processor_request_pre(&rptr->h.timestamp, cache_op,
					   rptr->processor,
					   rptr->h.tid, laddr)) {
      case MEM_READY:
	
	if ( rptr->mode & TESTTAG )
	  if ( FebitInCache(rptr->processor, iaddr) == FULL ) {
	      MAKE_RESUME_REQUEST(rptr->h.tid, rptr->h.timestamp,
				  MEM_MISMATCH);
	      return;
	  }
	
	caddr = AddrInCache(rptr->processor, iaddr);
	previous = shared_memory(caddr, rptr->mode);
	/* Special case for compare & swap */
	if ( (rptr->mode & ATOMIC_MASK) == CMPSWP ) {
	    previous = (previous == ((Cmpswp *)rptr->value)->old);
	    if (previous)
	      write_shared_memory(caddr, 
				  ((Cmpswp *)rptr->value)->new, 
				  rptr->mode);
	}
	else
	  write_shared_memory(caddr, rptr->value, rptr->mode);
	
	if ( rptr->mode & SETTAG )
	  FebitInCache(rptr->processor, iaddr) = FULL;
	
	(void)processor_request_post(&rptr->h.timestamp, cache_op, 
				     rptr->processor, rptr->h.tid, laddr);
#ifdef MEMDEBUG
	printf("WRITE(%d) returns %d to user\n", rptr->address, 
	       shmem_result(previous, 
			    shared_memory(caddr, rptr->mode), 
			    rptr->mode) );
#endif
	
	*(Word *)rptr->valaddr = shmem_result(previous,
					      shared_memory(caddr,
							    rptr->mode),
					      rptr->mode);
	
      case MEM_SWITCH:
      case MEM_WAIT:
      case MEM_ERROR:
	MAKE_RESUME_REQUEST(rptr->h.tid, rptr->h.timestamp, result);
	return;
      default:
	fatal("illegal value %d returned by CACHE_PROTOCOL_PRE", result);
    }
    
}


void shared_writetag_handler_(MemRequest *rptr) 
{
    Word cache_op, result, previous;
    Word iaddr = AddrToIndex(rptr->address);
    Word laddr = RoundAddrToLine(iaddr);
#ifdef TATDEBUG
    printf("@ %d (%d)will WriteTag(%x, %d) i=%x l=%x\n", rptr->tid,
	   rptr->processor, rptr->address, rptr->value, iaddr, laddr);
#endif
    
    
    cache_op = rptr->mode & SOFT ? MR_WRITESOFT : MR_WRITE;
    
    switch( result = processor_request_pre(&rptr->h.timestamp, cache_op,
					   rptr->processor, rptr->h.tid,
					   laddr)) {
      case MEM_READY:
	
	previous = FebitInCache(rptr->processor, iaddr);
	FebitInCache(rptr->processor, iaddr) = rptr->value;
	
	(void)processor_request_post(&rptr->h.timestamp, cache_op, 
				     rptr->processor, rptr->h.tid, laddr);
	
	*(Word *)rptr->valaddr = shmem_result(previous,
					      rptr->value, rptr->mode);
	
      case MEM_SWITCH:
      case MEM_WAIT:
      case MEM_ERROR:
	
#ifdef TMEMDEBUG
	printf("WRITETAG(%d) returns %d to user\n", rptr->address, result );
#endif
	
	MAKE_RESUME_REQUEST(rptr->h.tid, rptr->h.timestamp, result);
	return;
      default:
	fatal("illegal value %d returned by CACHE_PROTOCOL_PRE", result);
    }
    
}



void flush_handler_(MemRequest *rptr) 
{
    Word result;
    Word iaddr = AddrToIndex(rptr->address);
    Word laddr = RoundAddrToLine(iaddr);
    
    result = processor_request_pre(&rptr->h.timestamp, MR_FLUSHSOFT,
				   rptr->processor, rptr->h.tid, laddr);
    MAKE_RESUME_REQUEST(rptr->h.tid, rptr->h.timestamp+CACHE_ACCESS_LATENCY,
			result);
}


 /****************************************************************************\
 *    Floating Point Memory Accesses
 \****************************************************************************/


void shared_read_fl_handler_(MemRequest *rptr) 
{
    Word cache_op, result;
    ulong iaddr = AddrToIndex(rptr->address);
    ulong laddr = RoundAddrToLine(iaddr);
    
#ifdef ATDEBUG
    printf("@ %d (%d)will Read_Fl(%x) i=%x l=%x\n", rptr->tid,
	   rptr->processor, rptr->address, iaddr, laddr);
#endif
    
    
    cache_op = rptr->mode & SETTAG ? 
      (rptr->mode & SOFT ? MR_WRITESOFT : MR_WRITE) : 
	(rptr->mode & SOFT ? MR_READSOFT  : MR_READ );
    
    
    switch( result = processor_request_pre(&rptr->h.timestamp, cache_op,
					   rptr->processor,
					   rptr->h.tid, laddr)) {
      case MEM_READY:
	
	if ( rptr->mode & TESTTAG )
	  if ( FebitInCache(rptr->processor, iaddr) != FULL ) {
	      MAKE_RESUME_REQUEST(rptr->h.tid, rptr->h.timestamp,
				  MEM_MISMATCH);
	      return;
	  }
	
	*(double *)rptr->valaddr = 
	  shared_memory_fl(AddrInCache(rptr->processor, iaddr), rptr->mode);
	
	if ( rptr->mode & SETTAG )
	  FebitInCache(rptr->processor, iaddr) = EMPTY;
	
#ifdef MEMDEBUG
	printf("READ_FL(%d) returns %f to user\n", rptr->address,
	       *(double *)rptr->valaddr);
#endif
	(void)processor_request_post(&rptr->h.timestamp, cache_op, 
				     rptr->processor, rptr->h.tid, laddr);
      case MEM_SWITCH:
      case MEM_WAIT:
      case MEM_ERROR:
	MAKE_RESUME_REQUEST(rptr->h.tid, rptr->h.timestamp, result);
	return;
      default:
	fatal("illegal value %d returned by CACHE_PROTOCOL_PRE", result);
    }
}



void shared_write_fl_handler_(MemRequest *rptr) 
{
    Word cache_op, result;
    void *caddr;
    double previous;
    Word iaddr = AddrToIndex(rptr->address);
    Word laddr = RoundAddrToLine(iaddr);
#ifdef ATDEBUG
    printf("@ %d (%d)will Write_Fl(%x, %f) i=%x l=%x\n", rptr->tid,
	   rptr->processor, rptr->address, *(double *)rptr->value,
	   iaddr, laddr);
#endif
    
    cache_op = rptr->mode & SOFT ? MR_WRITESOFT : MR_WRITE;
    
    switch( result = processor_request_pre(&rptr->h.timestamp, MR_WRITE,
					   rptr->processor,
					   rptr->h.tid, laddr)) {
      case MEM_READY:
	
	if ( rptr->mode & TESTTAG )
	  if ( FebitInCache(rptr->processor, iaddr) == FULL ) {
	      MAKE_RESUME_REQUEST(rptr->h.tid, rptr->h.timestamp,
				  MEM_MISMATCH);
	      return;
	  }
	
	caddr = AddrInCache(rptr->processor, iaddr);
	previous = shared_memory_fl(caddr, rptr->mode);
	write_shared_memory_fl(caddr, *(double *)rptr->value, rptr->mode);
	
	if ( rptr->mode & SETTAG )
	  FebitInCache(rptr->processor, iaddr) = FULL;
	
	(void)processor_request_post(&rptr->h.timestamp, MR_WRITE, 
				     rptr->processor, rptr->h.tid, laddr);
#ifdef MEMDEBUG
	printf("WRITE_FL(%d) returns %f to user\n", rptr->address,
	       shmem_result_fl(previous, *(double *)rptr->value, rptr->mode));
#endif
	
	*(double *)rptr->valaddr = shmem_result_fl(previous,
						   *(double *)rptr->value,
						   rptr->mode);
	
      case MEM_SWITCH:
      case MEM_WAIT:
      case MEM_ERROR:
	MAKE_RESUME_REQUEST(rptr->h.tid, rptr->h.timestamp, result);
	return;
      default:
	fatal("illegal value %d returned by CACHE_PROTOCOL_PRE", result);
    }
    
}

