
 /**************************************************************************\
 *                                                         
 *                 Proteus Parallel-Architecture Simulator 
 *                Eric A. Brewer  and  Chris N. Dellarocas 
 *                     Laboratory for Computer Science     
 *                  Massachusetts Institute of Technology  
 *                                                         
 * Module: resume.c                                        
 *
 * Description: Engine thread management                   
 *
 * Last Modified: $Date: 94/01/24 00:39:25 $ (eab)
 *
 * Global Functions:
 *     int resume_handler_(SimRequest *) 
 *     void TrapToSimulator(void)
 *     void PassControl(void)    
 *     void PassControl_no_interrupts(void)
 *
 * Global Variables:
 *     char *sim_regs_[REGISTERS];
 *
 ****************************************************************************
 *   Copyright 1991                                                         *
 *   Eric A. Brewer  and  Chris N. Dellarocas                               *
 *   Massachusetts Institute of Technology                                  *
 *                                                                          *
 *   Permission to use, copy, modify, and distribute this program           *
 *   for any purpose and without fee is hereby granted, provided            *
 *   that this copyright and permission notice appear on all copies         *
 *   and supporting documentation, the name of M.I.T. not be used           *
 *   in advertising or publicity pertaining to distribution of the          *
 *   program without specific prior permission, and notice be given         *
 *   in supporting documentation that copying and distribution is           *
 *   by permission of M.I.T.  M.I.T. makes no representations about         *
 *   the suitability of this software for any purpose.  It is pro-          *
 *   vided "as is" without express or implied warranty.		            *
 ****************************************************************************
 * $Header: /usr/wildcat/dfk/research/parallel/proteus/proteus-V3.01/engine/RCS/resume.c,v 1.2 94/01/24 00:39:25 dfk Time64bit Locker: dfk $
 * $Log:	resume.c,v $
 * Revision 1.2  94/01/24  00:39:25  dfk
 * Added Sanjay's support for 64-bit timers.
 * Added thread_sleep_until.
 * Fixed a few bugs in thread sleep/wakeup code (my bugs).
 * Fixed printf formats in many places.
 * 
 * Revision 1.1  94/01/23  21:55:11  dfk
 * Initial revision
 * 
 * Revision 1.6  92/12/09  15:20:46  brewer
 * Added PassControl_no_interrupts
 * 
 * Revision 1.5  92/11/19  13:44:31  brewer
 * Replaced user_pointer_table_ update with call to set_local_pointers,
 * primarily for modularity.
 * 
 * Revision 1.4  92/11/12  12:57:08  brewer
 * Changed T_INTERRUPTED from state to flag -- see rt_thread.ca log for
 * details.
 * 
 * Revision 1.3  92/10/28  11:19:50  brewer
 * Added TRACK_RUN_LENGTH code to run-length data (between ctxsw calls)
 * 
 * Revision 1.2  92/04/09  11:45:45  brewer
 * Added void PassControl() -- transfers control from user thread
 * to simulator engine (after making a resume request)
 * 
 * Revision 1.1  92/02/11  13:56:09  brewer
 * Initial revision
 * 
 \**************************************************************************/

#include "sim.h"
#include "conf.h"
#include "rt_thread_def.h"
#include "simreq.h"
#include "processor.h"
#include "simcalls.h"
#include "thread.h"
#include "event.h"
#include "intreq.h"

GLOBAL Word sim_regs_[REGISTERS]; /* referenced in SimQuantum.s */
extern void ctxsw(Word *, Word *);
extern void OverwriteCheck();


#ifdef TRACK_RUN_LENGTH
static BOOL rl_init = FALSE;
static int rl_avg, rl_sd;
#endif


/* RESUME request handler */
/* RESUME(result_value)   */

GLOBAL void resume_handler_(SimRequest *rptr) 
{
    SimRequest *newptr;
    SimRequest *build_new_request_();
    int tid;
    ThreadInfo *tptr;
    Thread *osptr;
    Time time;
    Time timer;
    extern BOOL overwrite_check;  /* defined in debug.c */

    tid  = rptr->h.tid;
    time = rptr->h.timestamp;
    
    /* Check validity of parameters */
    if ( Invalidstid(tid) )
      fatal("resume_handler: attempted resume of invalid thread %d",
	    tid);
    
    if ((osptr = (Thread *)T_OSBLOCK(tid)) == NULL || 
	!(osptr->t_state & T_ACTIVE))
      return;
    
    tptr = &thread_table_[tid];
    processor_ = osptr->t_processor;
    currpptr = &proc_table_[processor_];    
    currtid = tid;
    currtptr = tptr;
    currosptr = osptr;
    
    /* update processor-specific user pointers */
    set_local_pointers(processor_);
    
    /* Update Timer */
    timer = ReadTimer(processor_);
    if (timer > time - currpptr->p_time)
      WriteTimer(processor_,  timer - (time - currpptr->p_time));
    else
      WriteTimer(processor_,  (Time) 0); /* timer already ran out */

    quantum_ = cycles_ = sim_quantum_;
    
    expired_ = FALSE;
    
    /* Store result in a place the thread can find it */
    result_ = rptr->argv[0];
    
    currpptr->p_time = tptr->t_time = time;
    base_time_ =  time + quantum_;
    stackmin_ = currtptr->t_stkbase + STK_THRESHOLD;
    
    mode_ = USER;
    ctxsw(sim_regs_, tptr->t_regs);
    mode_ = ENGINE;
    
    /* check for overwrite of watched addresses */
    if (overwrite_check) OverwriteCheck();
    
    /* update times */
    currtptr->t_time += quantum_ - cycles_;
    if (timer > quantum_ - cycles_)
      WriteTimer(processor_, timer - (quantum_ - cycles_));
    else
      WriteTimer(processor_,  (Time) 0); /* timer already ran out */
    currpptr->p_time = currtptr->t_time;

#ifdef TRACK_RUN_LENGTH
    if (!rl_init) {
	rl_avg = new_metric("Average Run Length", 0);
	rl_sd = new_metric("Run Length Standard Deviation", 0);
	rl_init = TRUE;
    }
    AVG_METRIC(rl_avg, quantum_ - cycles_);
    STDDEV_METRIC(rl_sd, rl_avg, quantum_ - cycles_);
#endif

    /* Normally when we return here, a new request has already been 
     * inserted in request queue. The exception is when the simulator
     * is entered following a quantum expiration. In that case a ctxsw
     * is done immediately after detecting the expiration condition
     * and the expired_ flag is set. We detect this special case here
     * and insert the appropriate requests in the request queue
     */ 
    
    if (expired_) {
	newptr = build_new_request_(SC_RESUME, currtid,
				    currtptr->t_time, 1, OK);

#ifdef DEBUG
	printf("#%d %s QUANTUM EXPIRED\n", currtid);
#endif
	
	
	/* If any interrupts are pending, we call the processor's ihandler
	 * to service them. The current thread is temporarily marked 
	 * INTERRUPTED. As a result of the interrupts, the current thread
	 * may be killed or suspended. In those cases, request newptr
	 * is discarded and the processor is rescheduled. Otherwise, newptr
	 * is normally enqueued. Ihandler updates the timestamp of newptr
	 * to account for the interrupt service overhead.
	 */
	if ( ((currosptr->t_intrflag && intr_pending[processor_] > 0) ||
	      (currosptr->t_preempt  && ReadTimer(processor_)<=0))
	    /* && !currpptr->p_ihandleractive */) {
	    currosptr->t_state |= T_INTERRUPTED; /* set intr. flag */
	    currpptr->p_interrupted = currtid;
	    MAKE_RESUME_REQUEST(currpptr->p_ihandler, currpptr->p_time,
				newptr);
	} else {
	    enqueue_request_(newptr);
	}
    }
}


/* Context switch from user thread to simulator engine */

GLOBAL void TrapToSimulator() 
{
    Word saveresult;
    
    if (mode_ != USER)
      fatal("Will context switch (ctxsw) to ENGINE while mode is not USER.\n");
    
    ctxsw(currtptr->t_regs, sim_regs_);
    
    saveresult = result_;
    CheckForInterrupts();
    result_ = saveresult;
}


GLOBAL void PassControl()
{
    MAKE_RESUME_REQUEST(currtid, base_time_ - cycles_, 0);
    TrapToSimulator();
}



GLOBAL void PassControl_no_interrupts()
{
    MAKE_RESUME_REQUEST(currtid, base_time_ - cycles_, 0);

    if (mode_ != USER)
      fatal("Will context switch (ctxsw) to ENGINE while mode is not USER.\n");
    
    ctxsw(currtptr->t_regs, sim_regs_);
}
