
/****************************************************************************
 *                                                                          *
 *                     Parallel Architecture Simulator                      *
 *                Eric A. Brewer  and  Chris N. Dellarocas                  *
 *                     Laboratory for Computer Science                      *
 *                  Massachusetts Institute of Technology                   *
 *                                                                          *
 * Module: Memory management of simulator threads                           *
 *                                                                          *
 * Description:                                                             *
 *     This module contains routines that allocate and free memory to be    *
 *     used for stacks of dynamically created threads. Memory is allocated  *
 *     from a contiguous heap created by routine 'makepool'.                *
 *                                                                          *
 * Last Modified: 5-11-90 (cnd)                                             *
 *                                                                          *
 * Global Functions:                                                        *
 *     int allocstk(unsigned nblks)                                         *
 *         Allocates heap memory for a stack of size nblks*STK_BLK_SIZE     *
 *         Returns index of first block allocated in internal page table    *
 *                                                                          *
 *     int freestk(unsigned n)                                              *
 *         Frees previously allocated stack whose first block has index n   *
 *                                                                          *
 *     int extendstk(unsigned n)                                            *
 *         Try to extend previously allocated stack starting at n.          *
 *         New stack is between one block larger to double the original     *
 *         size. Index of new first block is returned.                      *
 *                                                                          *
 *    char *stktop(unsigned n)                                              *
 *    char *stkbase(unsigned n)                                             *
 *         Return top and base addresses of allocated stack whose first     *
 *         block index is n.                                                *
 *                                                                          *
 *     int makepool(unsigned maxblks)                                       *
 *         Create and initialize a heap of maxblks stack blocks.            *
 *                                                                          *
 * Global Variables: See file mem.h                                         *
 *                                                                          *
 ****************************************************************************
 * $Header: /psg/proteus/RCS/mem.c,v 1.3 92/11/19 13:42:50 brewer Exp $
 * $Log:	mem.c,v $
 * Revision 1.3  92/11/19  13:42:50  brewer
 * Eliminated gcc -Wall -O bogus uninitialized variable warning.
 * 
 * Revision 1.2  92/04/01  17:01:37  brewer
 * cleaned up gcc warnings, added ANSI prototypes
 * 
 * Revision 1.1  92/02/11  13:55:59  brewer
 * Initial revision
 * 
 ****************************************************************************/

#include "sim.h"
#include "sim.param"
#include "mem.h"

static StkBlk *pagetab;                /* Page table address */
static int freelist;                   /* Head of free block list */
static int maxpage;                    /* Maximum legal stack block no */
static char *poolbase;                 /* Start address of stack block pool */

static int allocstkbetween(unsigned nmin, unsigned nmax);
static void copystk(unsigned from, unsigned to, unsigned nblks);

/* allocstk -- Allocate nblks contiguous stack blocks. 
 * Returns index of first block in pagetable or ERROR if nblks < 1
 * or no contiguous area of adequate size exists in free list.
 */

int allocstk(unsigned nblks) 
{
    int p, q, leftover;

#ifdef MEMDEBUG
    printf("allocstk(%d) called.\n", nblks);
#endif

    /* If argument is invalid or free list is empty,return ERROR immediately */
    if (nblks < 1 || freelist == FL_NULL)
      return(ERROR);

    /* Sequentialy scan free list for first free block of adequate size.
     * p always points to the current block being scanned and
     * q points to the previous block in free list.
     */
    for (q = FL_NULL, p = freelist; p != FL_NULL;
	   q = p, p = pagetab[p].next)
      /* If a block of exact size is found, it is simply removed from fl */
      if (pagetab[p].size == nblks) {
	  if (q != FL_NULL) 
	    pagetab[q].next = pagetab[p].next;
	  else
	    freelist = pagetab[p].next;
	  pagetab[p].status = OCCUPIED;
	  return(p);
      /* If a larger block is found, nblks are claimed from it and the
       * rest is put back in the free list
       */
      } else if (pagetab[p].size > nblks) {
	  leftover = p + nblks;
	  if (q != FL_NULL) 
	    pagetab[q].next = leftover;
	  else
	    freelist = leftover;
	  pagetab[leftover].status = FREE;
	  pagetab[leftover].size   = pagetab[p].size - nblks;
	  pagetab[leftover].next   = pagetab[p].next;
	  pagetab[p].status = OCCUPIED;
	  pagetab[p].size   = nblks;
	  return(p);
      }
    return(ERROR);
}



int freestk(unsigned n) 
{
    int p, q, top;

#ifdef STKDEBUG
    printf("Called freestk(%d)\n",n);
#endif

    /* Check if n is a valid occupied stack block */
    if (n > maxpage || pagetab[n].status != OCCUPIED)
      return(ERROR);
    
    /* Sequentially scan free list to locate two free areas q , p
     * between which n must lie
     */
    for (q = FL_NULL, p = freelist; p != FL_NULL && p < n;
	   q = p, p = pagetab[p].next );

    /* If the block starting at n cannot fit between q and p
     * (i.e. there is some overlapping with either of them)
     * then some error has occurred.
     */
    top = (q != FL_NULL) ? q + pagetab[q].size : 0;
    if ((top > n) ||
	(p != FL_NULL &&  n + pagetab[n].size > p))
      return(ERROR);

    /* If the block starting at n is adjacent to any (or both) blocks
     * starting at q, p, coalesce them to come up with a large contiguous
     * free block.
     */
    if (q != FL_NULL && top == n )           /* Check if n adjacent to q */
      pagetab[q].size += pagetab[n].size;
    else {
	pagetab[n].status = FREE;
	pagetab[n].next = p;
	if (q != FL_NULL)
	  pagetab[q].next = n;
	else
	  freelist = n;
	q = n;
    }
    if (q + pagetab[q].size == p ) {        /* Check if n adjacent to p */
	pagetab[q].size += pagetab[p].size;
	pagetab[q].next =  pagetab[p].next;
    }
    return(OK);
}


char *stkbase(unsigned n) 
{
    if (n > maxpage)
      return((char *)ERROR);
    else
      return(poolbase + (maxpage - n - pagetab[n].size + 1) * STK_BLK_SIZE );
}

char *stktop(unsigned n) 
{
    if (n > maxpage)
      return((char *)ERROR);
    else
      return(poolbase + (maxpage - n + 1) * STK_BLK_SIZE);
}


int makepool(unsigned maxblks) 
{
    char *addr;
    ulong poolsize;
    char *sbrk();

    /* If makepool has been successfully executed before or if
     * maxblks is too small, return ERROR
     */
    if (maxpage != 0 || maxblks < MIN_STK_POOL)
      fatal("makepool: wrong arguments.\n");

    /* Calculate memory required for stack blocks plus assoc. pagetable */
    poolsize = (maxblks + 1) * STK_BLK_SIZE + maxblks * sizeof(StkBlk);

    /* Ask the OS for memory. If we cannot get it, return ERROR */
    if ((addr = sbrk(poolsize)) == (char *)-1 )
      fatal("makepool(%d): OS won't give me as much memory.\n", maxblks);

    /* Poolbase is set to the base of the first page that lies entirely
     * within the allocated space. That way all stack blocks contained
     * in the pool are page-aligned.
     */
    poolbase = roundpage(addr);
    maxpage  = maxblks - 1;

    /* Initialize pagetable and freelist */
    pagetab  = (StkBlk *)(poolbase + maxblks * STK_BLK_SIZE);
    pagetab[0].status = FREE;
    pagetab[0].size   = maxblks;
    pagetab[0].next   = FL_NULL;
    freelist = 0;

#ifdef STKDEBUG
    printf("Poolsize = %ld\n",poolsize);
    printf("Addr     = %08lx\n",addr);
    printf("Poolbase = %08lx\n",poolbase);
    printf("Pagetab  = %08lx\n",pagetab);
#endif

    return(OK);
}
    

int extendstk(unsigned n) 
{
    int mysize, p, q, leftover, newn;

    if (n > maxpage || pagetab[n].status != OCCUPIED)
      return(ERROR);

    mysize = pagetab[n].size;

#ifdef STKDEBUG
    printf("Entered extendstk(%d). mysize= %d\n",n,mysize);
#endif

    if (pagetab[n+mysize].status == FREE) {
#ifdef STKDEBUG
	printf("Found adjacent free stack area.\n");
#endif
	for (q = FL_NULL, p = freelist; p != n+mysize;
	       q = p, p = pagetab[p].next);
	if (pagetab[p].size <= mysize)  {
	    if (q != FL_NULL)
	      pagetab[q].next = pagetab[p].next;
	    else
	      freelist = pagetab[p].next;
	    pagetab[n].size += pagetab[p].size;
	  } else {
	    leftover = p + mysize;
	    if (q != FL_NULL )
	      pagetab[q].next = leftover;
	    else
	      freelist = leftover;
	    pagetab[leftover].status = FREE;
	    pagetab[leftover].size   = pagetab[p].size - mysize;
	    pagetab[leftover].next   = pagetab[p].next;
	    pagetab[n].size += mysize;
	  }
	return(n);
    } else {
#ifdef STKDEBUG
	printf("Did not find adjacent free stack area.\n");
#endif
	freestk(n);
	if ((newn = allocstkbetween(mysize+1,2*mysize)) == ERROR)
	  return(ERROR);
	else {
	    copystk(n,newn,mysize);
	    return(newn);
	}
    }
}



/* allocstkbetween -- Allocate a free block of size at least nmin and
 * at most nmax blocks.
 */

static int allocstkbetween(unsigned nmin, unsigned nmax) 
{
    int p, q, size, leftover;

#ifdef STKDEBUG
    printf("Called allocstkbetween(%d,%d)\n",nmin,nmax);
#endif

    /* If arguments invalid or free list is empty,return ERROR immediately */
    if (nmin < 1 || freelist == FL_NULL)
      return(ERROR);

    /* Sequentialy scan free list for first free block of adequate size.
     * p always points to the current block being scanned and
     * q points to the previous block in free list.
     */
    for (q = FL_NULL, p = freelist; p != FL_NULL;
	   q = p, p = pagetab[p].next)
      /* If a block in exact size range is found, 
       * it is simply removed from fl
       */
      if ((size = pagetab[p].size) >= nmin && size <= nmax) {
	  if (q != FL_NULL) 
	    pagetab[q].next = pagetab[p].next;
	  else
	    freelist = pagetab[p].next;
	  pagetab[p].status = OCCUPIED;
	  return(p);
      /* If a larger block is found, nmax blocks are claimed from it and the
       * rest is put back in the free list
       */
      } else if (size > nmax) {
	  leftover = p + nmax;
	  if (q != FL_NULL) 
	    pagetab[q].next = leftover;
	  else
	    freelist = leftover;
	  pagetab[leftover].status = FREE;
	  pagetab[leftover].size   = size - nmax;
	  pagetab[leftover].next   = pagetab[p].next;
	  pagetab[p].status = OCCUPIED;
	  pagetab[p].size = nmax;
	  return(p);
      }
    return(ERROR);
}



static void copystk(unsigned from, unsigned to, unsigned nblks) 
{
    ulong *fromptr, *toptr;
    int i, j;

    fromptr = (ulong *)stktop(from);
    toptr   = (ulong *)stktop(to);  

#ifdef STKDEBUG
    printf("Entered copystk(%d,%d,%d)\n",from,to,nblks);
    printf("fromptr = %08lx\n",fromptr);
    printf("toptr   = %08lx\n",toptr);
#endif

    for(i=0; i<nblks; i++)
      for(j=0; j< STK_BLK_SIZE/sizeof(ulong); j++)
	*toptr-- = *fromptr--;
}
