 /**************************************************************************\
 *
 *                 Proteus Parallel-Architecture Simulator
 *                Eric A. Brewer  and  Chris N. Dellarocas
 *                     Laboratory for Computer Science
 *                  Massachusetts Institute of Technology
 *
 * Module: OStraps.c
 *
 * Description: Defines default trap handlers and routines to
 *     change handlers
 *
 * Last Modified:  8-2-91  (eab)
 *
 * Global Functions:
 *     int _mem_retry(address, value, val, info)
 *     int _mem_switch(address, value, val, info)
 *     int _mem_mismatch(address, value, val, info)
 *     int _mem_error(address, value, val, info)
 *     FuncPtr define_timer_handler(p, handler)
 *     FuncPtr define_switch_handler(p, handler)
 *     FuncPtr define_mismatch_handler(p, handler) 
 *     FuncPtr define_error_handler(p, handler)
 *
 * Global Variables: none
 *    
 * Referenced parameters: NO_OF_PROCESSORS
 *
 ***************************************************************************
 *   Copyright 1991
 *   Eric A. Brewer  and  Chris N. Dellarocas
 *   Massachusetts Institute of Technology
 *
 *   Permission to use, copy, modify, and distribute this program
 *   for any purpose and without fee is hereby granted, provided
 *   that this copyright and permission notice appear on all copies
 *   and supporting documentation, the name of M.I.T. not be used
 *   in advertising or publicity pertaining to distribution of the
 *   program without specific prior permission, and notice be given
 *   in supporting documentation that copying and distribution is
 *   by permission of M.I.T.  M.I.T. makes no representations about
 *   the suitability of this software for any purpose.  It is pro-
 *   vided "as is" without express or implied warranty.		
 ****************************************************************************
 * $Header: /psg/proteus/RCS/OStraps.c,v 1.2 92/09/22 11:32:03 brewer Exp $
 * $Log:	OStraps.c,v $
 * Revision 1.2  92/09/22  11:32:03  brewer
 * Added some documentation
 * 
 * Revision 1.1  92/02/11  13:54:38  brewer
 * Initial revision
 * 
 \**************************************************************************/

#include "mem.param"
#include "sim.h"
#include "conf.h"
#include "rt_thread_def.h"
#include "simreq.h"
#include "processor.h"
#include "thread.h"
#include "cache.h"
#include "shmem.h"
  
int _mem_switch(), _mem_mismatch(), _mem_error(), _mem_retry();

#define DEFAULT_TIMER_HANDLER           _timer_handler
#define DEFAULT_SWITCH_HANDLER          _mem_switch
#define DEFAULT_MISMATCH_HANDLER        _mem_switch
#define DEFAULT_ERROR_HANDLER           _mem_error

GLOBAL void null_handler_() {}


GLOBAL int _mem_retry(address, value, val, info) 
Word address;
Word value;
Word *val;
Word info;
{
    cycles_ -= 1;
    return(MEM_RETRY);
}


GLOBAL int _mem_switch(address, value, val, info) 
Word address;
Word value;
Word *val;
Word info;
{
    cycles_ -= 1;

    /* printf("MEM_SWITCH: P%d #%d a=%d\n", processor_, currtid, address); */
    
#ifdef 0 /* should simulate multiple contexts for Alewife */
    cycles_ -= 11; /* add 11 cycles for fast context switch */
    
    _thread_yieldToScheduler();
    
    while ( intr_pending[processor_] > 0 ) {
	ServiceInterrupt(processor_);
    }
#endif
    
    return(MEM_RETRY);
}


GLOBAL int _mem_mismatch(address, value, val, info) 
Word address;
Word value; /* current value of address */
Word *val;  /* value that trapped access should return */
Word info;  /* operation type: MR_READ, MR_WRITE, etc. (see shmem.h) */
{
    fatal("Tag Mismatch at address %d (Processor %d Tid %d Time %d)",
	  address, processor_, currtid, CURRENTTIME);

    /* unreachable, return included to prevent warnings */
    return(ERROR);
}


GLOBAL int _mem_error(address, value, val, info) 
Word address;
Word value;
Word *val;
Word info;
{
    fatal("Cache Protocol Error at address %d (Processor %d Tid %d Time %d)",
	  address, processor_, currtid, CURRENTTIME);

    /* unreachable, return included to prevent warnings */
    return(ERROR);
}


GLOBAL FuncPtr define_timer_handler(p, handler) 
int p;
FuncPtr handler;
{
    ProcBlk *pptr;
    FuncPtr previous;
    int i;
    
    if (p == -1) {
	for(i=0; i<NO_OF_PROCESSORS; i++)
	  define_timer_handler(i, handler);
	return((FuncPtr)OK);
    }
    
    if (InvalidProc(p))
      return((FuncPtr)ERROR);
    
    previous = (pptr = &proc_table_[p])->p_timer_ihandler;
    
    if (handler == (FuncPtr)I_DEFAULT)
      pptr->p_timer_ihandler = (FuncPtr)DEFAULT_TIMER_HANDLER;
    else if (handler != NULL)
      pptr->p_timer_ihandler = handler;
    
    return(previous);
}


GLOBAL FuncPtr define_switch_handler(p, handler) 
int p;
FuncPtr handler;
{
    ProcBlk *pptr;
    FuncPtr previous;
    int i;
    
    if (p == -1) {
	for(i=0; i<NO_OF_PROCESSORS; i++)
	  define_switch_handler(i, handler);
	return((FuncPtr)OK);
    }
    
    if (InvalidProc(p))
      return((FuncPtr)ERROR);
    
    previous = (pptr = &proc_table_[p])->p_switch_ihandler;
    
    if (handler == (FuncPtr)I_DEFAULT)
      pptr->p_switch_ihandler = (FuncPtr)DEFAULT_SWITCH_HANDLER;
    else if (handler != NULL)
      pptr->p_switch_ihandler = handler;
    
    return(previous);
}


GLOBAL FuncPtr define_mismatch_handler(p, handler) 
int p;
FuncPtr handler;
{
    ProcBlk *pptr;
    FuncPtr previous;
    int i;
    
    if (p == -1) {
	for(i=0; i<NO_OF_PROCESSORS; i++)
	  define_mismatch_handler(i, handler);
	return((FuncPtr)OK);
    }
    
    if (InvalidProc(p))
      return((FuncPtr)ERROR);
    
    previous = (pptr = &proc_table_[p])->p_mismatch_ihandler;
    
    if (handler == (FuncPtr)I_DEFAULT)
      pptr->p_mismatch_ihandler = (FuncPtr)DEFAULT_MISMATCH_HANDLER;
    else if (handler != NULL)
      pptr->p_mismatch_ihandler = handler;
    
    return(previous);
}


GLOBAL FuncPtr define_error_handler(p, handler) 
int p;
FuncPtr handler;
{
    ProcBlk *pptr;
    FuncPtr previous;
    int i;
    
    if (p == -1) {
	for(i=0; i<NO_OF_PROCESSORS; i++)
	  define_error_handler(i, handler);
	return((FuncPtr)OK);
    }
    
    if (InvalidProc(p))
      return((FuncPtr)ERROR);
    
    previous = (pptr = &proc_table_[p])->p_error_ihandler;
    
    if (handler == (FuncPtr)I_DEFAULT)
      pptr->p_error_ihandler = (FuncPtr)DEFAULT_ERROR_HANDLER;
    else if (handler != NULL)
      pptr->p_error_ihandler = handler;
    
    return(previous);
}
