#define SAFE_MODE

 /**************************************************************************\
 *
 *                 Proteus Parallel-Architecture Simulator
 *       Eric A. Brewer, Chris N. Dellarocas, and Anthony D. Joseph
 *                     Laboratory for Computer Science
 *                  Massachusetts Institute of Technology
 *
 * Module: OSmem.h
 *
 * Description: Structures, definitions and constants for shared memory
 *
 * Last Modified: $Date: 92/12/09 15:18:47 $ ($Author: brewer $)
 *
 * Originally written by Anthony D. Joseph 11-92
 *
 * Data Structures: Shared memory allocation structures
 *
 * Constants: SPLITSIZE 
 *
 * Referenced parameters: MODULE_SIZE
 *
 * Macros: BUCKET(x)
 *
 ****************************************************************************
 *   Copyright 1992                                                      
 *   Eric A. Brewer, Chris N. Dellarocas, and Anthony D. Joseph
 *   Massachusetts Institute of Technology
 *
 *   Permission to use, copy, modify, and distribute this program
 *   for any purpose and without fee is hereby granted, provided
 *   that this copyright and permission notice appear on all copies
 *   and supporting documentation, the name of M.I.T. not be used
 *   in advertising or publicity pertaining to distribution of the
 *   program without specific prior permission, and notice be given
 *   in supporting documentation that copying and distribution is
 *   by permission of M.I.T.  M.I.T. makes no representations about
 *   the suitability of this software for any purpose.  It is pro-
 *   vided "as is" without express or implied warranty.		
 ***************************************************************************
 * $Header: /psg/proteus/RCS/OSmem.h,v 1.2 92/12/09 15:18:47 brewer Exp $
 * $Log:	OSmem.h,v $
 * Revision 1.2  92/12/09  15:18:47  brewer
 * changed dump_module to take FILE *
 * 
 * Revision 1.1  92/11/25  17:45:56  brewer
 * Initial revision
 * 
 * 
 \**************************************************************************/
/*
 * This storage allocator is a heavily modified version of the Unix memory
 * allocator:
 * 		malloc.c (Caltech) 2/21/82
 * 		Chris Kingsley, kingsley@cit-20.
 */

/* This is a very fast storage allocator.  It allocates blocks of a small 
 * number of different sizes, and keeps free lists of each size.  Blocks that
 * don't exactly fit are passed up to the next larger size.  In this 
 * implementation, the available sizes are 2^n-4 (or 2^n-10) bytes long.
 * 
 * This was orignally intended as an allocator for a virtual memory
 * uniprocessor (Unix), but has been modified for multiprocessor use. Each
 * processor maintains a bucket list of different size blocks. Access to
 * the bucket list on each processor is controlled by a per processor mutex.
 * The virtual memory version refilled empty buckets using VM, for the
 * multiprocessor version, however, we use a bucket splitting/coalescing
 * algorithm.
 */

/* The overhead on a block is at least 4 bytes.  When free, this space
 * contains a pointer to the next free block, and the bottom two bits must
 * be zero.  When in use, the first byte is set to MAGIC, and the second
 * byte is the size index.  The remaining bytes are for alignment.
 * If range checking is enabled then a second word holds the size of the
 * requested block, less 1, rounded up to a multiple of sizeof(RMAGIC).
 * The order of elements is critical: ov_magic must overlay the low order
 * bits of ov_next, and ov_magic can not be a valid ov_next bit pattern.
 */

/* Each bucket contains blocks of useable size 2^(i+x)-OVERHEAD,
 * where x is 3 and OVERHEAD is 4 when range checking is disabled. When
 * range checking is enabled, x is 4 and OVERHEAD is 10.
 *
 * The allocator uses a best-fit allocation scheme. A block from the
 * smallest bucket with useable space larger than space request is
 * returned for a request. If the bucket is empty, it checks larger
 * buckets. If a larger non-empty bucket is found, the block is removed
 * from the bucket and split into as follows: Starting with the next
 * smaller bucket and continuing until the original bucket is encountered,
 * the block is split in half and half is inserted into the bucket
 * (which was empty). This allocates half the block to the next smaller
 * bucket, one-quarter to the next smaller, one-eigth, etc. When the
 * original bucket is reached, the block is split in two and BOTH
 * blocks are inserted into the bucket (one is used to satisfy the memory
 * request).
 *
 * If there are no larger non-empty buckets, the allocator for OS_getmem
 * tries to allocate the memory from another module. It uses the same
 * algorithm as for the original module: check the original bucket; then try
 * to split a larger bucket.
 *
 * If this is unsucessful or OS_getmemfrommodule is being called, the
 * allocator tries to find a block by merging the blocks in smaller buckets.
 * Starting with the next smaller non-empty block, the allocator first sorts
 * the linked list by address. Next, it tries to find chains of adjacent
 * blocks that are long enough to fit the original bucket. The allocator
 * works downward looking for matching chains.
 * 
 * If this fails and OS_getmem is being called, the process is repeated for
 * all memory modules.
 *
 * Finally, if no matching blocks can be found, the allocator either
 * signals a fatal error or returns a NULL pointer (depending upon the
 * FATAL define).
 */


#include "user.h"
#include "cache.h"
#include "shmem.h"
#include "event.h"
#include <sys/types.h>

#ifdef SAFE_MODE		/* when in safe mode, do range checking */
#define RCHECK			/*  there is no performance penalty to using */
#else 				/*  range checking (cycle-counting is */
#undef RCHECK			/*  disabled during checks) */
#endif

/* Define the memory allocation structure */
union overhead {
  union	overhead *ov_next;	/* when free */
  struct {
    u_char ovu_magic;		/* magic number */
    u_char ovu_index;		/* bucket # */
    u_char ovu_module;		/* Memory module */
#ifdef RCHECK
    u_char ovu_rmagic;		/* range magic number */
    u_int ovu_size;		/* actual block size */
#endif
  } ovu;
};
typedef union overhead ShmemBlk;

/* Some macros to ease access to the elements */
#define	ov_magic	ovu.ovu_magic
#define	ov_index	ovu.ovu_index
#define	ov_module	ovu.ovu_module
#define	ov_rmagic	ovu.ovu_rmagic
#define	ov_size		ovu.ovu_size


#define	MAGIC		0xef		/* magic # on accounting info */
#define FRMAGIC		0x55		/* magic # on front range info */
#define RMAGIC		0x5555		/* magic # on end range info */

#ifdef RCHECK
#define	RSLOP		sizeof (u_short) /* space for the end marker */
#define BUCKET(x)	(1<<(x+4))	/* smallest bucket is 16 bytes */
#else
#define	RSLOP		0		/* no range checking, no extra space */
#define BUCKET(x)	(1<<(x+3))	/* smallest bucket is 8 bytes */
#endif

#define OVERHEAD	(sizeof(ShmemBlk) + RSLOP)	/* total overhead 
							   on mem requests */
/* max space in a single module */
#define MAXSPACE	(MODULE_SIZE - (NBUCKETS * sizeof(ShmemBlk *)))
extern int MaxSharedMemory;	      	/* Max useable space in module */

/*
 * MemoryTable[module][i] is the pointer to the next free block of
 * size 2^(i+3) on processor "module". The smallest allocatable block
 * is 8 bytes w/o range checking; 16 bytes if range checking is enabled.
 * The overhead information precedes the data area returned to the user.
 */
#define	NBUCKETS 28	/* this yields a max of 2147483648 bytes in the largest
			   bucket */

extern ShmemBlk **nextf;	/* local processor ptr to local bucket list */
extern ShmemBlk *shmem_pool;	/* base of shared memory (after semaphores) */

/* array of ptrs to bucket lists (treated as a list local to each proc) */
extern ShmemBlk **MemoryTable[NO_OF_MODULES];	
extern int MemoryCount[NO_OF_MODULES];	/* space used in module */
extern Word memlist_mutex[NO_OF_MODULES];	/* access ctrl semaphores */

extern void dump_module(FILE *fp, int module);

#define SPLITSIZE 32	/* This is the number of blocks that we try to find
			   when we refill an empty bucket using splitting */
