
#include <string.h>
#include "cprep.h"
#include "symtable.h"



void make_binary_expression(int opcode, Lex *e1,
			    Lex *op, Lex *e2, Bool freeflag )
{
    Expr *op1 = (Expr *)e1->tinfo;
    Expr *op2 = (Expr *)e2->tinfo;
    
    e1->text = op1->val = link_chain3(op1->val, op->text, op2->val);
    
    update_numval(opcode, op1, op2);
    
    /* For now, final expression keeps the type of its first operand */
    
    if (freeflag) {
	op1->attributes &= (~RID_SHARED);
	free_chain(op1->addr);
	op1->addr = (StrTab *)NULL;
    }
    free_chain(op2->addr);
    free_expr(op2);
}




void make_conditional_expression(Lex *e1, Lex *thn, Lex *e2,
				 Lex *els, Lex *e3 )
{
    Expr *op1 = (Expr *)e1->tinfo;
    Expr *op2 = (Expr *)e2->tinfo;
    Expr *op3 = (Expr *)e3->tinfo;
    
    e1->text = op1->val = link_chain5(op1->val, 
				      thn->text, 
				      op2->val,
				      els->text,
				      op3->val);
    
    if ( op1->numval != NONUMVAL )
      op1->numval = op1->numval ? op2->numval : op3->numval;
    
    /* For now, final expression keeps the type of its first operand */
    
    op1->attributes &= (~RID_SHARED);
    
    free_chain(op1->addr);
    op1->addr = (StrTab *)NULL;
    free_chain(op2->addr);
    free_chain(op3->addr);
    free_expr(op2);
    free_expr(op3);
}




/* Final expression keeps type of its left argument */

void make_assignment( Lex *e1, Lex *op, Lex *e2, char prepost )
{
    
    Expr *op1 = (Expr *)e1->tinfo;
    Expr *op2 = (Expr *)e2->tinfo;
    char access;
    
    /* If lval is not shared , just append text */
    if ( !(op1->attributes & RID_SHARED) ) { 
	
	e1->text = op1->val =
	  link_chain3(op1->val, op->text, op2->val);
	
	op1->numval = op2->numval;
	
	free_chain(op1->addr);
	op1->addr = (StrTab *)NULL;
	if ( op1 != op2 ) {
	    free_chain(op2->addr);
	    free_expr(op2);
	}
	
	return;
    }
    
    /* Cases when lval is shared */
    
    free_chain(op->text);
    
    if ( op1->addr == (StrTab *)NULL ) {
	error("invalid lval in assignment expression");
      error_return:
	free_chain(op1->addr); op1->addr = (StrTab *)NULL;
	free_chain(op1->val ); op1->val  = (StrTab *)NULL;
	free_chain(op2->addr); 
	free_chain(op2->val ); 
	free_expr(op2);
	return;
    }
    
    if ( op1 != op2 )
      free_chain(op1->val);
    
    /* Lval may be a TAG expression, signalling tag access */
    if ( op1->attributes & RID_TAGACCESS ) {
	
	e1->text = op1->val =
	  link_chain7( allocate_string(WRITETAG_TEXT),
		      op1->addr,
		      allocate_string(","),
		      op2->val,
		      allocate_string(","),
		      make_access_mode(op1->attributes, BYTE, 0, prepost),
		      allocate_string(")") );
    }
    else {
	/* Lval is a non-tag shared expression */
	
	switch( SIZE(op1->type) )
	  {
	    case 1:
	      access = BYTE; break;
	    case 2:
	      access = HALFWORD; break;
	    case 8:
	      access = DOUBLEWORD; break;
	    default:
	      access = WORD;
	  }
	
	switch( TID(op1->type) )
	  {
	    case RID_STRUCT:
	    case RID_UNION:
	      error("assignment to structures and unions not yet implemented");
	      goto error_return;
	    case RID_FLOAT:
	    case RID_DOUBLE:
	      
	      e1->text = op1->val = 
		link_chain9(allocate_string("("),
			    make_type_text(op1->type),
			    allocate_string(WRITEMEM_FL_TEXT),
			    op1->addr,
			    allocate_string(", (double)"),
			    parenthesize(op2->val),
			    allocate_string(","),
			    make_access_mode(op1->attributes, access, 0, prepost),
			    allocate_string("))") );
	      break;
	      
	    default:
	      
	      e1->text = op1->val = 
		link_chain9(allocate_string("("),
			    make_type_text(op1->type),
			    allocate_string(WRITEMEM_TEXT),
			    op1->addr,
			    allocate_string(", (unsigned long)"),
			    parenthesize(op2->val),
			    allocate_string(","),
			    make_access_mode(op1->attributes,
					     access, 0, prepost),
			    allocate_string("))") );
	  }
    }
    
    op1->addr = (StrTab *)NULL;
    
    op1->numval = op2->numval;
    op1->attributes &= (~RID_SHARED);
    
    if ( op1 != op2 ) {
	free_chain(op2->addr);
	free_expr(op2);
    }
}


/* Final expression keeps type of its left argument */

void make_abbreviated_assignment( Lex *e1, Lex *op, Lex *e2, char prepost )
{
    char *s;
    
    Expr *op1 = (Expr *)e1->tinfo;
    Expr *op2 = (Expr *)e2->tinfo;
    Lex empty;
    
    /* If lval is not shared , just append text */
    if ( !(op1->attributes & RID_SHARED) ) { 
	
	e1->text = op1->val =
	  link_chain3(op1->val, op->text, op2->val);
	
	update_numval(op->tinfo, op1, op2);
	
	free_chain(op1->addr);
	op1->addr = (StrTab *)NULL;
	free_chain(op2->addr);
	free_expr(op2);
	
	return;
    }
    
    
    /* Wipe equal sign from end of op= token */
    for(s = TEXT(op->text); *s != '='; s++);
    *s = 0;
    
    make_binary_expression(op->tinfo, e1, op, e2, FREENO );
    
    empty = empty_lex();
    make_assignment(e1, &empty, e1, prepost);
}




void make_increment_decrement(Lex *e, Lex *op, char prepost)
{
    Expr *op1 = (Expr *)e->tinfo;
    char access;
    
    Word incdec = (op->tinfo == PLUSPLUS_UNARY) ? INCREMENT : DECREMENT;
    
    /* If lval is not shared , just append text */
    if ( !(op1->attributes & RID_SHARED) ) { 
	
	e->text = op1->val =
	  (prepost == PRE) ?
	    link_chain2( op->text, op1->val ) :
	      link_chain2(op1->val, op->text);
	
	op1->numval = NONUMVAL;
	
	return;
    }
    
    /* Cases when lval is shared */
    
    free_chain(op->text);
    
    if ( op1->addr == (StrTab *)NULL ) {
	error("invalid lval in assignment expression");
	return;
    }
    
    
    /* Lval may be a TAG expression, signalling tag access */
    if ( op1->attributes & RID_TAGACCESS ) {
	error("incrementing/decrementing a tag is not allowed");
    }
    else {
	/* Lval is a non-tag shared expression */
	
	switch( SIZE(op1->type) )
	  {
	    case 1:
	      access = BYTE; break;
	    case 2:
	      access = HALFWORD; break;
	    case 8:
	      access = DOUBLEWORD; break;
	    default:
	      access = WORD;
	  }
	
	switch( TID(op1->type) )
	  {
	      
	    case RID_CHAR:
	    case RID_SHORT:
	    case RID_INT:
	    case RID_LONG:
	      
	      e->text = op1->val = 
		link_chain7(allocate_string("("),
			    make_type_text(op1->type),
			    allocate_string(WRITEMEM_TEXT),
			    op1->addr,
			    allocate_string(", 0 ,"),
			    make_access_mode(op1->attributes, access, 
					     incdec, prepost),
			    allocate_string("))") );
	      break;
	      
	    case RID_FLOAT:
	    case RID_DOUBLE:
	      
	      e->text = op1->val = 
		link_chain7(allocate_string("("),
			    make_type_text(op1->type),
			    allocate_string(WRITEMEM_FL_TEXT),
			    op1->addr,
			    allocate_string(", (double)0 ,"),
			    make_access_mode(op1->attributes, access, 
					     incdec, prepost),
			    allocate_string("))") );
	      break;
	      
	    case RID_PTR:
	      {
		  Expr *one = newexpr();
		  Lex  newop, onelex;
		  one->val = allocate_string("1");
		  onelex.tinfo = (Word)one;
		  newop.text = allocate_string( (incdec == INCREMENT) ? "+=" : "-=" );
		  newop.tinfo = (incdec == INCREMENT) ? PLUS_EXPR : MINUS_EXPR;
		  
		  make_abbreviated_assignment( e, &newop, &onelex, prepost );
		  return;
	      }
	      
	    default:
	      error("invalid type in increment/decrement operation");
	      return;
	  }
	
	op1->addr = (StrTab *)NULL;
	
	op1->numval = NONUMVAL;
	op1->attributes &= (~RID_SHARED);
	
    }
}



void update_numval(int opcode, Expr *op1, Expr *op2)
{
    
    if ( op1->numval != NONUMVAL && op2->numval != NONUMVAL ) {
	switch( opcode ) {
	  case PLUS_EXPR:
	    op1->numval += op2->numval; break;
	  case MINUS_EXPR:
	    op1->numval -= op2->numval; break;
	  case MULT_EXPR:
	    op1->numval *= op2->numval; break;
	  case TRUNC_DIV_EXPR:
	    op1->numval /= op2->numval; break;
	  case TRUNC_MOD_EXPR:
	    op1->numval %= op2->numval; break;
	  case LSHIFT_EXPR:
	    op1->numval <<= op2->numval; break;
	  case RSHIFT_EXPR:
	    op1->numval >>= op2->numval; break;	 
	  case EQ_EXPR:
	    op1->numval = (op1->numval == op2->numval); break;	  
	  case NEQ_EXPR:
	    op1->numval = (op1->numval != op2->numval); break;	  
	  case LESS_EXPR:
	    op1->numval = (op1->numval < op2->numval); break;	  
	  case GRT_EXPR:
	    op1->numval = (op1->numval > op2->numval); break;	  
	  case LEQ_EXPR:
	    op1->numval = (op1->numval <= op2->numval); break;	  
	  case GEQ_EXPR:
	    op1->numval = (op1->numval >= op2->numval); break;	  
	  case BIT_AND_EXPR:
	    op1->numval &= op2->numval; break;
	  case BIT_IOR_EXPR:
	    op1->numval |= op2->numval; break;
	  case BIT_XOR_EXPR:
	    op1->numval ^= op2->numval; break;
	  case AND_EXPR:
	    op1->numval = (op1->numval && op2->numval); break;	  
	  case OR_EXPR:
	    op1->numval = (op1->numval || op2->numval); break;	  
	}
    } else {
	op1->numval = NONUMVAL;
    }
}



long eval( Expr *e )
{
    char buffer[1024];
    int  totlen = 0;
    StrTab *s = e->val;
    char *status;
    long val;
    
    *buffer = 0;
    
    for( ; s != (StrTab *)NULL; s = s->next) {
	int len = strlen(TEXT(s));
	if (totlen + len >= 1020) {
	    warning("expression too large -- cannot determine numval");
	    return(NONUMVAL);
	}
	strcpy(&buffer[totlen], TEXT(s) );
	totlen += len;
    }
    
    val = strtol(buffer, &status, 0);
    
    if ( val == 0 && status == buffer )
      return(NONUMVAL);
    else
      return(val);
}
