/****************************************************************************
*****************************************************************************

Parallel Architecture Simulator
Eric A. Brewer  and  Chris N. Dellarocas
Laboratory for Computer Science
Massachusetts Institute of Technology

Module:  Augment-- Symbol Table

Description:
    This is the symbol table module used by 'augment'.  It uses a chained
    hash table to store the symbols.

Last Modified:  4-26-90  (eab)

Global Functions:
    void InitTable();
        Initialize the symbol table.

    char *LocateSymbol(char *name, int styp);
        Find symbol 'name' of type 'styp' in the table.  Return NULL
        if it is not there, else return the (unique) string for that
        symbol.

    char *LocateSymbol(char *name, int styp, nodeType **value);
        Locates the symbol as above, but also sets 'value'.

    char *InsertSymbol(char *name, int styp);
        Insert the symbol 'name' of type 'styp' into the table.  Define
        and return a unique string for the symbol.  If the symbol is
        already in the table, the existing unique string is returned.

    void SetSymbolNode(char *name, nodeType *node);
        Associate the node 'node' with the (label) symbol 'name'.

    char *InsertSymbolValue(char *name, int styp, nodeType *node);
        Equivalent to InsertSymbol followed by SetSymbolNode.

    nodeType *GetSymbolNode(char *name);
        Retrieve the node associated with the (label) symbol 'name'.

    int ClearType(char *sym, int typ);
        Clears type 'typ' of symbol 'sym'.  Returns true iff symbol
        wasn't found.

    void SymbolStats();
        Print statistics on the symbol table.
    
Global Variables:
    none

****************************************************************************
*   Copyright 1991                                                       
*   Eric A. Brewer  and  Chris N. Dellarocas                             
*   Massachusetts Institute of Technology                                
*                                                                        
*   Permission to use, copy, modify, and distribute this program         
*   for any purpose and without fee is hereby granted, provided          
*   that this copyright and permission notice appear on all copies       
*   and supporting documentation, the name of M.I.T. not be used         
*   in advertising or publicity pertaining to distribution of the        
*   program without specific prior permission, and notice be given       
*   in supporting documentation that copying and distribution is         
*   by permission of M.I.T.  M.I.T. makes no representations about       
*   the suitability of this software for any purpose.  It is pro-        
*   vided "as is" without express or implied warranty.		          
*****************************************************************************
****************************************************************************/


#include <stdio.h>
#include <malloc.h>
#include <string.h>
#include "lex.h"

extern void abort();

#define TABLE_SIZE  1811

typedef struct eT {
    char *id;
    short typ;
    nodeType *node;
    struct eT *next;
} entryType;

static entryType *hash_table[TABLE_SIZE];
static num_symbols=0;

/** INTERNAL ROUTINES **/

/* Store a symbol.
    Copy string to new location, and return the new location.
*/
static char *StoreSymbol(char *sym)
{
    char *new_loc;

    new_loc = malloc(strlen(sym) + 1);
    strcpy(new_loc, sym);
    return new_loc;
}


/* hash table function */
static short HashFunction(char *sym)
{
    unsigned short i, k;
    unsigned long val;

    assert(sym != NULL);
    
    val = (short)sym[0];
    for(i=1; i<8; i++) {
	if (sym[i]==0) break;
	k = sym[i] & 0x1f;
	val *= k;
    }
    return((short)(val % TABLE_SIZE));
}


static entryType *Locate(char *sym)
{
    entryType *e;
    
    e = hash_table[HashFunction(sym)];
    while (e != NULL) {
	if (strcmp(sym, e->id)==0) return e;
	e = e->next;
    }
    return NULL;
}

/**  EXTERNAL ROUTINES **/

void InitTable()
{
    int i;
    
    for (i=0; i<TABLE_SIZE; i++)  hash_table[i] = (entryType *) NULL;
}


/* Locate a symbol with a given name and entry type.
   Returns NULL if there is no such symbol.
*/  
char *LocateSymbol(char *sym, int styp)
{
    entryType *e;
    
    e = Locate(sym);
    if (e==NULL) return NULL;
    if (!(e->typ & styp)) return NULL;
    /* printf("Located \"%s\".\n", e->id); */
    return e->id;
}

char *LocateSymbolValue(char *sym, int styp, nodeType **value)
{
    entryType *e;
    
    e = Locate(sym);
    if (e==NULL) return NULL;
    if (!(e->typ & styp)) return NULL;
    /* printf("Located \"%s\".\n", e->id); */
    *value = e->node;
    return e->id;
}


int ClearType(char *sym, int styp)
{
    entryType *e;
    
    e = Locate(sym);
    if (e==NULL) return 1;
    e->typ &= ~styp;
    return 0;
}

char *InsertSymbol(char *sym, int styp)
{
    entryType *e;
    short bucket;

    e = Locate(sym);
    if (e==NULL) {  /* new symbol */
	bucket = HashFunction(sym);
	e = (entryType *) malloc(sizeof(entryType));
	e->id = StoreSymbol(sym);
	e->node = NULL;
	e->typ = styp;
	e->next = hash_table[bucket];
	hash_table[bucket] = e;
	num_symbols++;
        /* printf("Adding \"%s\" to table.\n", sym); */
    } else {  /* symbol already present */
	e->typ |= styp;
    }
    return e->id;
}

char *InsertSymbolValue(char *sym, int styp, nodeType *node)
{
    entryType *e;
    short bucket;

    e = Locate(sym);
    if (e==NULL) {  /* new symbol */
	bucket = HashFunction(sym);
	e = (entryType *) malloc(sizeof(entryType));
	e->id = StoreSymbol(sym);
	e->node = node;
	e->typ = styp;
	e->next = hash_table[bucket];
	hash_table[bucket] = e;
	num_symbols++;
        /* printf("Adding \"%s\" to table.\n", sym); */
    } else {  /* symbol already present */
	e->typ |= styp;
    }
    return e->id;
}

void SetSymbolNode(char *sym, nodeType *node)
{
    entryType *e;

    assert(node != NULL);
    e = Locate(sym);
    if (e==NULL) {
	fprintf(stderr,
	   "augment: INTERNAL SetSymbolNode: Missing symbol \"%s\".\n", sym);
    } else {
	/* printf("SetSymbolNode: %s\n", sym); */
	e->node = node;
    }
}

nodeType *GetSymbolNode(char *sym)
{
    entryType *e;

    e = Locate(sym);
    if (e==NULL)
      fprintf(stderr, 
	 "augment: INTERNAL GetSymbolNode: Missing symbol \"%s\".\n", sym);
    else {
	assert(e->node != NULL);
	return e->node;
    }
    return NULL;
}

void SymbolStats()
{
    printf("Total number of symbols: %d\n", num_symbols);
}
