package main

import (
	"encoding/json"
	"fmt"
	"io"
	"io/ioutil"
	"log"
	"net/http"
	"strings"
	"time"
)

type Lang struct {
	Name string
	Year int
	URL  string
}

var jsonLines = `
	{"Name": "Python", "Year": 1991, "URL": "http://python.org/"}
	{"Name": "Ruby", "Year": 1995, "URL": "http://www.ruby-lang.org/en/"}
	{"Name": "Scala", "Year": 2003, "URL": "http://www.scala-lang.org/"}
	{"Name": "Go", "Year": 2009, "URL": "http://golang.org/"}
`

func do(jsonLines string, f func(Lang)) {
	dec := json.NewDecoder(strings.NewReader(jsonLines))
	for {
		var lang Lang
		err := dec.Decode(&lang)
		if err != nil {
			if err == io.EOF {
				break
			}
			log.Fatal(err)
		}
		f(lang)
	}
}

func countWebpageContent(name, url string, c chan<- string) { //----- added a channel
	start := time.Now()
	r, err := http.Get(url)
	if err != nil {
		c <- fmt.Sprintf("%s: %s\n", name, err) //----- error message goes to the channel now
		return
	}
	// copy the content to a black hole, counting the bytes copied
	n, _ := io.Copy(ioutil.Discard, r.Body)
	r.Body.Close()
	c <- fmt.Sprintf("%s %d [%.2fs]\n", name, n, time.Since(start).Seconds()) //----- result to the channel too
}

func main() {
	start := time.Now()
	c := make(chan string)  //----- the channel needs constructing
	n := 0
	do(jsonLines, func(lang Lang) {
		n++
		go countWebpageContent(lang.Name, lang.URL, c) //----- pass in the channel
	})

	for i := 0; i < n; i++ {
		fmt.Printf(<-c) //----- consume results from the channel - easy!
	}

	fmt.Printf("%.2fs total\n", time.Since(start).Seconds())
	//----- no need for any rubbish time delays
}
