f is a simple general purpose text editor which was developed to aid the production
of program source.  The editor is based on the concept of folding, which is a
mechanism which allows text to be given a hierarchical structure so that detail may
be revealed or obscured as desired.  Folding can be used in any programming language
to construct well structured and easy to read code.

Folding works by marking blocks of text so that they may be displayed or hidden as
desired.  A block is marked by inserting a marker line above and below the block of
text.  The block of text and the two marker lines are called a fold.  Folds may
contain nested folds. Folds can be open, in which case the contents are displayed
between the two marker lines, or closed, in which case the contents are replaced by a
single fold marker.

The rest of this introduction is contained in a fold labelled "rest of introduction".
The fold may be OPENed by placing the cursor on the fold and using the Keypad 'End'
key or Double clicking the left mouse button with the Control key down.
The fold may be ENTERed by placing the cursor on the fold and using the Keypad 'Home'
key or Double clicking the left mouse button.

/*{{{  rest of introduction*/
Although f has been developed specifically as an editor for program source code, the
ability to impose structure and hierarchy on a document make it very suitable for
many other areas particularly documentation.  The structure of any document can be
developed and refined without all of the contents being completed just through the
hierarchical fold structure.

The editor is line based, rather than character based, and most functions use the
line as the element of operation.  A line, however, isnt necessarily a single line
of text and can be a very significant object.

In program source code folds provide a similar, but complementary, function to
procedures. Procedures and functions enable detailed portions of an algorithm to be
hidden from view to simplify and clarify the algorithm.  In turn, these functions may
call other functions so that the hierarchical structure of the program reflects the
hierarchical structure of the algorithm.  Folds provide a similar mechanism which can
be as finely grained as desired and which is located at the point of instance so the
contents are immediately available whenever needed.  Folds do not replace functions
and procedures but add an extra facility for assisting with the clarity of program
text.  Folded documents are, as a consequence, hierarchically structured in a manner
which provides a three dimensional viewing capability.  As a side effect of the
manner in which folds are created, folded source files tend to have comprehensive
and locally useful comments.

A fold has an indentation associated with it; no text may be inserted within the fold
to the left of this indentation. Folds may be entered or opened. When a fold is
entered the fold markers become the bounds of the editors view of the document; the
editor cannot `see' or display regions of text outside the fold. When a fold is
opened the contents of the fold are displayed within the context of the surrounding
folds. The comparison is analogous to the difference between opening a door to a
room and looking in compared with opening the door entering the room and closing the
door behind.

A closed fold is displayed and treated as a single line by the editor so that editor
functions which operate on lines, such as COPY, PICK and DELETE operate in an
identical manner on folds.  A line operation applied to a fold applies to all the
lines contained within that fold as well and as a consequence some functions may
result in a significant change to the file being edited.  For example COPY applied
to a fold containing 1000 lines would add 1000 lines to the file.  The nested fold
structure of a copied fold is duplicated identically.

To EXIT this fold type Keypad PgUp, or double click the right mouse button.
To CLOSE this fold type Keypad PgDn, or double click the right mouse button with
the Control key held down.
/*}}}  */

The following is a sample removed from a windows main program. It does not nest folds
very deeply but is intended to give a feeling for how folding can help in keeping
track of file contents.

/*{{{  example source code*/
/*{{{  includes*/
#include <windows.h>
#include <commdlg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include "data.h"
#include "winmain.h"
#include "winterm.h"
#include "fterm.h"
#include "editor.h"
#include "window.h"
#include "filemgr.h"
/*}}}  */
/*{{{  constant strings*/
const char Version[]  = "Version 2.00.00 for MS Windows, 12 March 1994";
PRIVATE const char ZERO_STRING[]  =  "\0";
PRIVATE const char STARTUP_FILE[] =  "c:\\f\\f.stp";
PRIVATE const char Copyright[]    = "Copyright (c) TLM Software, 1990-94.  All Rights Reserved.\n";
PRIVATE const char ProgramName[]  = "fedit" ;
/*}}}  */
/*{{{  static data*/
PRIVATE HWND    CurrentWindowHandle;
PRIVATE HWND    ClientWindowHandle, FrameWindowHandle;
PRIVATE HANDLE  Instance;
PRIVATE char    EditClass[]  = "EditChild";
PRIVATE char    FrameClass[]  = "MainFrame";
PRIVATE HMENU   FrameMenu, EditMenu;
PRIVATE HMENU   FrameMenuWindow, EditMenuWindow;
PRIVATE HDC     DeviceContext;
/*}}}  */
/*{{{  functions*/
/*{{{  tracing functions*/
/*{{{  void xprintf (HWND WindowHandle, const char* String, ... )*/
void xprintf (HWND WindowHandle, const char* String, ... )
{
  char Line[MAX_LINE_SIZE];
  va_list  ArgList;

  va_start (ArgList, String);
  vsprintf (Line, String, ArgList);
  va_end   (ArgList);

  MessageBox (WindowHandle, Line, ProgramName, MB_OK | MB_ICONEXCLAMATION);
}
/*}}}  */
/*{{{  void tprintf (const char* String, ... )*/
void tprintf (const char* String, ... )
{
  char Line[MAX_LINE_SIZE];
  va_list  ArgList;

  va_start (ArgList, String);
  vsprintf (Line, String, ArgList);
  va_end   (ArgList);

  MessageBox (ClientWindowHandle, Line, ProgramName, MB_OK | MB_ICONEXCLAMATION);
}
/*}}}  */
/*}}}  */
/*{{{  int PASCAL WinMain (HANDLE hInstance, HANDLE hPrevInstance, ...*/
int PASCAL WinMain (HANDLE hInstance, HANDLE hPrevInstance,
                    LPSTR CommandLine, int nCmdShow)
{
  /*{{{  locals*/
  MSG      WindowMessage;
  HWND     FrameWindowHandle, ClientWindowHandle;
  HANDLE   hAccel;
  WNDCLASS WindowClass;
  /*}}}  */
  /*{{{  copy instance to static data*/
  Instance = hInstance;
  /*}}}  */
  /*{{{  check for instance and register classes*/
  if (!hPrevInstance) 
    {
      /*{{{  initialise main window class*/
      WindowClass.style         = CS_HREDRAW | CS_VREDRAW | CS_DBLCLKS;
      WindowClass.lpfnWndProc   = FrameMessageProc;
      WindowClass.cbClsExtra    = 0;
      WindowClass.cbWndExtra    = 0;
      WindowClass.hInstance     = hInstance;
      WindowClass.hIcon         = LoadIcon (hInstance, ProgramName);
      WindowClass.hCursor       = LoadCursor (NULL, IDC_ARROW);
      WindowClass.hbrBackground = COLOR_APPWORKSPACE + 1;
      WindowClass.lpszMenuName  = NULL;
      WindowClass.lpszClassName = ProgramName;
      /*}}}  */
      RegisterClass (&WindowClass);
      /*{{{  initialse edit window class*/
      WindowClass.lpfnWndProc   = EditMessageProc;
      WindowClass.cbWndExtra    = sizeof (WINDOW*); // pointer to edit
      WindowClass.hbrBackground = GetStockObject (WHITE_BRUSH);
      WindowClass.lpszClassName = EditClass;
      /*}}}  */
      RegisterClass (&WindowClass);
    }

  /*}}}  */
  /*{{{  load menus and accelerators*/
  FrameMenu = LoadMenu (hInstance, "FrameMenu");
  EditMenu  = LoadMenu (hInstance, "EditMenu");

  FrameMenuWindow = GetSubMenu (FrameMenu, FRAME_MENU_POSITION);
  EditMenuWindow  = GetSubMenu (EditMenu, EDIT_MENU_POSITION);

  hAccel = LoadAccelerators (hInstance, "MdiAccelerators");
  /*}}}  */
  /*{{{  create and display window*/
  FrameWindowHandle = CreateWindow (ProgramName, "F for Windows",
                           WS_OVERLAPPEDWINDOW | WS_CLIPCHILDREN | WS_MAXIMIZE,
                           CW_USEDEFAULT, CW_USEDEFAULT,
                           CW_USEDEFAULT, CW_USEDEFAULT,
                           NULL, FrameMenu, hInstance, NULL);

  ClientWindowHandle = GetWindow (FrameWindowHandle, GW_CHILD);
  ShowWindow   (FrameWindowHandle, nCmdShow);
  UpdateWindow (FrameWindowHandle);
  /*}}}  */

  while (GetMessage (&WindowMessage, NULL, 0, 0))
    /*{{{  main message loop*/
    {
      if ((DialogWindowHandle == NULL) || (!IsDialogMessage (DialogWindowHandle, &WindowMessage)))
        {
          if ((!TranslateMDISysAccel (ClientWindowHandle, &WindowMessage)) &&
              (!TranslateAccelerator (FrameWindowHandle, hAccel, &WindowMessage)))
            /*{{{  handle message*/
            {
              TranslateMessage (&WindowMessage);
              DispatchMessage (&WindowMessage);
            }
            /*}}}  */
        }
    }
    /*}}}  */
  DestroyMenu (EditMenu);
  return WindowMessage.wParam;
}

/*}}}  */
/*}}}  */
/*}}}  */
