/*{{{  File banner*/
/*@(#)=====================================================*\
||@(#)  Project : GPMIMD ESPRIT P5404
||@(#)  Authors : Mark Debbage and Mark Hill
||@(#)            University of Southampton
||  
||@(#)    Title : Virtual processor interface module
||@(#)   System : VPI
||@(#) Filename : vpmain.c
||@(#)  Version : 1.30
||@(#)     Date : 10/15/92
\*@(#)====================================================*/
/*}}}*/

/*{{{  includes*/
#include <stdio.h>
#include <stdlib.h>

#include "vcr.h"
#include "vchan.h"
#include "vparse.h"
#include "vpi.h"

#include "vpmain.h"
#include "uchan.h"
/*}}}*/

/*{{{  debugging definitions*/
/*#define VPMAIN_DEBUG*/
#ifdef  VPMAIN_DEBUG
#define DEBUG(x)
#else
#define DEBUG(x)
#endif

#define TIME_INFO(x)
/*}}}*/

/*{{{  heap management*/
char   *ENV_SYSTEM_HEAP_BYTES_ROOT = "VPI_ROOT_HEAP" ;

#define DEF_SYSTEM_HEAP_BYTES_ROOT  100000
#define DEF_SYSTEM_HEAP_BYTES_NODE  50000
extern int *_IMS_heap_front ;
/*}}}*/
/*{{{  statics*/
PRIVATE char *_FILE_ = __FILE__ ;

PUBLIC VPI_GLOBALS vpi_globals ;

PRIVATE VCB  from_filer, to_filer ;
PRIVATE PORT from_filer_port, to_filer_port ;
PRIVATE BYTE from_filer_pkt[UPR_MAX_PKT_SIZE] ;

#ifndef UPR_RD
PRIVATE VCB *filer_to_users, *users_to_filer ;
#endif
/*}}}*/
/*{{{  topology structures*/
typedef struct
{
  int id ;         /* Corresponding processor id  */
  int size ;       /* Sub-tree size               */
  int son ;        /* Index of son                */
  int sibling ;    /* Index of next sibling       */
} BOOTNODE ;
/*}}}*/

/*{{{  PRIVATE void CallDynamicOccam(int *wptr, char *iptr)*/
/*{{{  constants for CallDynamicOccam*/
#define iptr_state 0
#define wptr_state 1
#define state_size 2

#define stub_size  4


PRIVATE int stub_code [stub_size] = {
   0xFB214460,   /* LDC -12   Offset to iptr state  */
                 /* LDPI                            */
   0xFB214460,   /* LDC -12   Offset to wptr state  */
                 /* LDPI                            */
   0xF0FC2330,   /* LDNL 0    Get wptr              */
                 /* GAJW                            */
                 /* REV                             */
   0x0000F630    /* LDNL 0    Get iptr              */
                 /* GCALL                           */
} ;
/*}}}*/
PRIVATE void CallDynamicOccam(int *wptr, char *iptr)
{
  int stub[state_size+stub_size] ;

  AsmMove (stub_code, &stub[state_size], sizeof(int)*stub_size) ;

  __asm {
          /*{{{  retain caller iptr*/
          ldlabeldiff done-here ;
          ldpi ;
          here: st stub[iptr_state] ;
          /*}}}*/
          /*{{{  retain caller wptr */
          ldlp 0 ;
          st stub[wptr_state] ;
          /*}}}*/
          /*{{{  force return to stub*/
          ld &stub[state_size] ;
          st *wptr ;
          /*}}}*/
          /*{{{  call code (at current priority)*/
          ld iptr ;
          ld wptr ;
          gajw ;
          rev ;
          gcall ;
          done: ;
          /*}}}*/
        } ;   
}
/*}}}*/
/*{{{  PRIVATE int  DynamicC(Channel *fs, Channel *ts, SCDB *scdb, int wsp_in_ints)*/
PRIVATE int  DynamicC(Channel *fs, Channel *ts, SCDB *scdb, int wsp_in_ints)
{
  int *stack_base, *stack_ptr;
  int stack_size;
  int *user_ws = (int*) malloc(wsp_in_ints*sizeof(int));

  if (user_ws==NULL)
    Exception(UPR_error,_FILE_,__LINE__,"Malloc of dynamic workspace failed\n");
  
  /* TYPE1 PROC MAIN.ENTRY has 6 parameters + vecspace + iptr */
  stack_size = scdb->wsp_size + (6+2);
  if ((stack_base = (int*) malloc(stack_size*sizeof(int)))==NULL)
    Exception(UPR_error,_FILE_,__LINE__,"Malloc of calling stack frame failed\n");
  stack_ptr  = stack_base + stack_size;

  {
    int *vs = (int*) malloc(scdb->vsp_size*sizeof(int));
    int dummy_ws ;

    /* STACK = stack ^ iptr | parameters | vec ptr */
    stack_ptr--;
    *(stack_ptr--) = (int) vs;         /* VEC SPACE ptr */
    *(stack_ptr--) = 0 ;
    *(stack_ptr--) = (int) &dummy_ws;  /* []INT ws2     */
    *(stack_ptr--) = wsp_in_ints;
    *(stack_ptr--) = (int) user_ws;    /* []INT ws1     */
    *(stack_ptr--) = (int) ts;         /* CHAN OF SP ts */
    *(stack_ptr--) = (int) fs;         /* CHAN OF SP fs */

    CallDynamicOccam(stack_ptr,scdb->code+scdb->ep_offset);
  }

  return(stack_ptr[1]) ;
}
/*}}}*/
/*{{{  PRIVATE void SP_ConvertBuffer(Process *p, Channel *sfs, Channel *sts, VCB *vfs, VCB *vts)*/
#define SP_MAX_PACKET_DATA_SIZE 510
#define SP_ConvertBufferWsp  400
PRIVATE void SP_ConvertBuffer(Process *p, Channel *sfs, Channel *sts, VCB *vfs, VCB *vts)
{
  int length = 0;
  char *buffer ;
  
  p=p;
  if ((buffer=malloc(SP_MAX_PACKET_DATA_SIZE)) == NULL)
    Exception(UPR_error,_FILE_,__LINE__,"Out of heap space whilst allocating hosthook buffer") ;  

  while (1)
  {
    ChanIn(sts,&length,2);
    ChanIn(sts,buffer,length);
    VirtualOut(vts,&length,2);
    VirtualOut(vts,buffer,length);
    VirtualIn(vfs,&length,2);
    VirtualIn(vfs,buffer,length);
    ChanOut(sfs,&length,2);
    ChanOut(sfs,buffer,length);
  }
}    
/*}}}*/

/*{{{  PRIVATE void  adjust_scdb  (SCDB *oldscdb, SCDB *scdb)*/
PRIVATE void  adjust_scdb  (SCDB *oldscdb, SCDB *scdb)
{
  int offset = ((char *) scdb) - ((char *) oldscdb) ;
  PDB **patch ;

  scdb->filename += offset ;
  scdb->code     += offset ;
  scdb->symbol   += offset ;

  patch=&scdb->patch ;

  while (*patch != NULL)
  {
    *((char **) patch) += offset ;

    (*patch)->symbol += offset ;
    (*patch)->filename += offset ;
    patch = &((*patch)->next) ;
  }
}
/*}}}*/
/*{{{  PRIVATE SCDB *ReadBootCode ()*/
PRIVATE SCDB *ReadBootCode ()
{
  SCDB *scdb ;
  int len;
  char *base ;
  
  ServerIn (&len, sizeof(int)) ;   /* length of code block */
  
  if ((scdb = (SCDB*) malloc(len)) == NULL)
    Exception(UPR_error,_FILE_,__LINE__,"Malloc for dynamically loaded code failed") ;
  
  ServerIn (scdb, len) ;            /* code block */
  
  ServerIn (&base, sizeof(int)) ;   /* relocation base */

  adjust_scdb((SCDB*) base, scdb);

  return(scdb) ;
}
/*}}}*/
/*{{{  PRIVATE void  SendBootCode (int target, SCDB *scdb)*/
PRIVATE void  SendBootCode (int target, SCDB *scdb)
{
  int len = scdb->size ;

  ClientOut (target, &len, sizeof(int)) ;  /* length of code block */
  ClientOut (target, scdb, len) ;          /* code block */
  ClientOut (target, &scdb, sizeof(int)) ; /* relocation base */
}
/*}}}*/

/*{{{  PRIVATE BOOTNODE *ReadBootTree ()*/
PRIVATE BOOTNODE *ReadBootTree ()
{
  BOOTNODE *tree ;
  int size ;

  ServerIn (&size, sizeof(int)) ;

  if ((tree = (BOOTNODE*) malloc(size)) == NULL)
    Exception(UPR_error,_FILE_,__LINE__,"Malloc for boot tree failed") ;
  
  ServerIn (tree, size) ;

  return(tree) ;
}
/*}}}*/
/*{{{  PRIVATE void SendBootTree (int target, BOOTNODE *tree)*/
PRIVATE void SendBootTree (int target, BOOTNODE *tree)
{
  int size = tree->size * sizeof(BOOTNODE) ;
  ClientOut (target, &size, sizeof(int)) ;
  ClientOut (target, tree, size) ;
}
/*}}}*/
/*{{{  PRIVATE void BootASon     (Process *p, int target, SCDB *scdb, BOOTNODE *tree)*/
PRIVATE void BootASon     (Process *p, int target, SCDB *scdb, BOOTNODE *tree)
{
  ClaimChannel (target) ;
  SendBootCode (target, scdb) ;
  SendBootTree (target, tree) ;
  ClientIn (target, NULL, 0) ;
  ReleaseChannel(target) ;
}
/*}}}*/
/*{{{  PRIVATE void BootSons     (SCDB *scdb, BOOTNODE *tree)*/
PRIVATE void BootSons     (SCDB *scdb, BOOTNODE *tree)
{
  int *proc = &tree->son ;
  Process *p[UPR_num_links] ;
  Process **q = p ;

  if (*proc != NOT_PROC)
  {
    do
    {
      int target = tree[*proc].id ;
      #ifdef VPMAIN_DEBUG
      {
        char s[80] ;
        sprintf(s,"Loading physical processor %d\n",tree[*proc].id) ;
        Exception (UPR_warning,_FILE_,__LINE__,s) ;
      }
      #endif

      if ((*q++ = ProcAlloc (BootASon, 0, 3, target, scdb, &tree[*proc])) == NULL)
        Exception (UPR_error,_FILE_,__LINE__,"Out of memory") ;
      
      proc = &tree[*proc].sibling ;
    } while (*proc != NOT_PROC) ;

    *q=NULL ;

    ProcParList(p) ;
  }
}
/*}}}*/
/*{{{  PUBLIC  int  *ComputeChildren  (BOOTNODE *tree)*/
PUBLIC  int  *ComputeChildren  (BOOTNODE *tree)
{
  int *children = malloc ((UPR_num_links+1) * sizeof(int)) ;
  int *c = children ;

  if (children==NULL)
    Error(_FILE_,__LINE__,"Out of heap") ;
  else
    {
      int proc = tree->son ;
      if (proc != NOT_PROC)
      {
        do
        {
          *c   = tree[proc].id ;
          proc = tree[proc].sibling ;
          c++ ;
        } while (proc != NOT_PROC) ;
      }
    }
  *c = NOT_PROC ;
    
  return(children) ;
}
/*}}}*/

/*{{{  PUBLIC int  NewVCE     (VCE *vce) */
PUBLIC int  NewVCE     (VCE *vce) 
{
  VPI_GLOBALS *vpig = (VPI_GLOBALS *) (* ((GLOBALS **) GLOBAL_PTR))->vpi_gp ;
  VCPAIR *vcp = (VCPAIR *) malloc (sizeof(VCPAIR)) ;

  if (vcp == NULL)
    return(-1) ;
  else
    {
      vce->proc = vpig->proc_id ;
      vce->vc   = (VC) vcp ;
      if (((vce->port_in  = PortGrab (&vcp->in_port)) < 0) ||
          ((vce->port_out = PortGrab (&vcp->out_port)) < 0))
        /*{{{  failed*/
        {
          free   (vcp) ;
          return (-1) ;
        }
        /*}}}*/
      else
        /*{{{  pre-initialise UCBs*/
        {
          PortInit (&vcp->in_port, VPI_InPreAction, TRUE, VPI_InPostAction, FALSE, (void *) &vcp->in_ucb) ;
          PortInit (&vcp->out_port, VPI_OutPreAction, TRUE, VPI_OutPostAction, FALSE, (void *) &vcp->out_ucb) ;
        
          UCB_Init (&vcp->in_ucb,  &vcp->in_port, vcp->packet, -1, -1) ;
          UCB_Init (&vcp->out_ucb, &vcp->out_port, NULL, -1, -1) ;
        
          vcp->in_ucb.next  = NeverQueue_p ;
          vcp->out_ucb.next = NeverQueue_p ;
        }
        /*}}}*/
    }
  return(0) ;
}
/*}}}*/
/*{{{  PUBLIC void ConnectVCE (VCE *source, VCE *destn) */
PUBLIC void ConnectVCE (VCE *source, VCE *destn) 
{
  VPI_GLOBALS *vpig = (VPI_GLOBALS *) (* ((GLOBALS **) GLOBAL_PTR))->vpi_gp ;
  VCPAIR *vcp = (VCPAIR *) source->vc ;
  
  if (source->proc != vpig->proc_id)
    Exception(UPR_error,_FILE_,__LINE__,"Cannot connect a remote channel end") ;

  HdrBuild (destn->proc, destn->port_out, 0, &vcp->in_ucb.orb.header) ;
  HdrBuild (destn->proc, destn->port_in, 0, &vcp->out_ucb.orb.header) ;
}
/*}}}*/
/*{{{  PUBLIC void FreeVC     (VC vc) */
PUBLIC void FreeVC     (VC vc) 
{
  VCPAIR *vcp = (VCPAIR *) vc ;
  PortRelease (&vcp->in_port) ;
  PortRelease (&vcp->out_port) ;
  free(vcp) ;
}
/*}}}*/

#ifndef UPR_RD
/*{{{  PRIVATE int TransformTree (BOOTNODE *tree, int proc, BOOTNODE *temp, int *pos)*/
PRIVATE int TransformTree (BOOTNODE *tree, int proc, VERTEX *vert, int *pos)
{
  int index = (*pos)++ ;
  VERTEX   *vertnode = &vert[proc] ;
  BOOTNODE *thisnode = &tree[index] ;

  thisnode->id = proc ;
  thisnode->sibling = NOT_PROC ;

  if (vertnode->son == NOT_PROC)
    thisnode->son = NOT_PROC ;
  else
    {
      int *vertsib = &vertnode->son ;
      int *thissib = &thisnode->son ;
      do
      {
        *thissib = TransformTree (tree, *vertsib, vert, pos) - index ;
        vertsib = &vert[*vertsib].sibling ;
        thissib = &tree[*thissib + index].sibling ;
      } while (*vertsib != NOT_PROC) ;
    }

  thisnode->size = (*pos) - index ;

  return(index) ;
}

/* Recursively tree-walks vert in order to build a tree such that
   the nodes comprising each sub-tree are contiguously stored */
/*}}}*/
/*{{{  PRIVATE void CallHostHook(Process *p, VCB *fs, VCB *ts)*/
#define HostHookWsp 4000  /* in bytes */
PRIVATE void CallHostHook(Process *p, VCB *fs, VCB *ts)
{
  Channel stopper;
  ChanInit(&stopper);
  p=p;

  /* This call is fixed for an inconsistency in VCR 2's internal structure */
  HostHook((VCB *) (((int) fs) ^ virtual_bit),
           (VCB *) (((int) ts) ^ virtual_bit),
           (VCB *) (&stopper)) ;
}
/*}}}*/
#endif

/*{{{  PUBLIC void InitVirtualProcessorInterface(int proc_id, int num_procs)*/
PUBLIC void InitVirtualProcessorInterface(int proc_id, int num_procs)
{
  GLOBALS *g = * ((GLOBALS **) GLOBAL_PTR) ;
  
  /*{{{  install VPI global pointer*/
  if (g->vpi_gp != NULL)
    Exception (UPR_fatal, _FILE_, __LINE__, "VCR globals pointer already occupied") ;
  g->vpi_gp = &vpi_globals ;
  /*}}}*/
  /*{{{  initialise VPI globals*/
  vpi_globals.proc_id   = proc_id ;
  vpi_globals.num_procs = num_procs ;
  vpi_globals.gsb       = (void *) *(&(proc_id)-1) ;
  
  vpi_globals.vpfunc.VarLenIn   = VarLenIn ;
  vpi_globals.vpfunc.VarLenOut  = VarLenOut ;
  vpi_globals.vpfunc.NewVCE     = NewVCE ;
  vpi_globals.vpfunc.ConnectVCE = ConnectVCE ;
  vpi_globals.vpfunc.FreeVC     = FreeVC ;
  /*}}}*/
  /*{{{  initialise client-servers*/
  {
    int server_in_base, server_out_base ;
    int client_in_base, client_out_base ;
    PORT *server_ports, *client_ports ;
  
    /*{{{  allocate and divide ports*/
    {
      PORT *ports ;
      int base ;
    
      if ((ports = (PORT *) malloc (sizeof(PORT)*4*num_procs)) == NULL)
        Exception(UPR_fatal,_FILE_,__LINE__,"Malloc for client/server ports failed") ;
    
      if ((base = MultiplePortGrab (ports, 4*num_procs)) < 0)
        Exception (UPR_fatal, _FILE_, __LINE__, "Out of ports") ;
    
      server_ports = ports ;
      vpi_globals.server_port_base = base ;
      server_in_base  = base ;
      server_out_base = base + num_procs ;
      
      client_ports = ports + 2*num_procs ;
      vpi_globals.client_port_base = base ;
      client_in_base  = base + 2*num_procs ;
      client_out_base = base + 3*num_procs ;
    }
    /*}}}*/
    /*{{{  initialise server channels*/
    {
      BYTE (*packets)[UPR_MAX_PKT_SIZE] ;
      int i ;
    
      vpi_globals.server.wdesc = NULL ;
      vpi_globals.server.head  = NULL ;
      vpi_globals.server.tail  = NULL ;
      
      if (((packets = (BYTE (*)[UPR_MAX_PKT_SIZE]) malloc (UPR_MAX_PKT_SIZE*num_procs)) == NULL) ||
          ((vpi_globals.server.in  = (UCB *) malloc (sizeof(UCB)*num_procs)) == NULL) ||
          ((vpi_globals.server.out = (UCB *) malloc (sizeof(UCB)*num_procs)) == NULL))
        Exception(UPR_fatal,_FILE_,__LINE__,"Malloc for server channels failed") ;
      
      for (i=0;i<num_procs;i++)
      {
        PortInit (&server_ports[i], VPI_InPreAction, TRUE,
                  VPI_InPostAction, FALSE, (void *) &vpi_globals.server.in[i]) ;
                  
        PortInit (&server_ports[num_procs+i], VPI_OutPreAction, TRUE,
                  VPI_OutPostAction, FALSE, (void *) &vpi_globals.server.out[i]) ;
      
        UCB_Init (&vpi_globals.server.in[i],  &server_ports[i], packets[i], i, client_out_base+proc_id) ;
        UCB_Init (&vpi_globals.server.out[i], &server_ports[num_procs+i], NULL, i, client_in_base+proc_id) ;
      }    
    }
    /*}}}*/
    /*{{{  initialise client channels*/
    {
      BYTE (*packets)[UPR_MAX_PKT_SIZE] ;
      int i ;
    
      if (((vpi_globals.client = (CLIENT *) malloc (sizeof(CLIENT)*num_procs)) == NULL) ||
          ((packets = (BYTE (*)[UPR_MAX_PKT_SIZE]) malloc (UPR_MAX_PKT_SIZE*num_procs)) == NULL))
        Exception(UPR_fatal,_FILE_,__LINE__,"Malloc for client channels failed") ;
    
      for (i=0;i<num_procs;i++)
      {
        CLIENT *client=&vpi_globals.client[i] ;
    
        sema_init (&client->sema,1) ;
        
        PortInit (&client_ports[i], VPI_InPreAction, TRUE,
                  VPI_InPostAction, FALSE, (void *) &client->in) ;
                  
        PortInit (&client_ports[num_procs+i], VPI_OutPreAction, TRUE,
                  VPI_OutPostAction, FALSE, (void *) &client->out) ;
      
        UCB_Init (&client->in,  &client_ports[i], packets[i], i, server_out_base+proc_id) ;
        UCB_Init (&client->out, &client_ports[num_procs+i], NULL, i, server_in_base+proc_id) ;
      }    
    }
    /*}}}*/
  }
  /*}}}*/
  /*{{{  initialise filer channels*/
  {
    int my_in_id, my_out_id ;
    int filer_in_id, filer_out_id ;
    
    /*{{{  initialise filer ports and channels on all nodes*/
    {
      my_in_id  = PortGrab (&from_filer_port) ;
      my_out_id = PortGrab (&to_filer_port) ;
      filer_in_id  = (my_in_id+2) + proc_id ;
      filer_out_id = (my_in_id+2) + num_procs + proc_id ;
      
      PortInit (&from_filer_port, VCR_InPreAction, TRUE,
                VCR_InPostAction, FALSE, (void *) &from_filer) ;
                
      PortInit (&to_filer_port, VCR_OutPreAction, TRUE,
                VCR_OutPostAction, FALSE, (void *) &to_filer) ;
    
      VCB_Init (&from_filer, &from_filer_port, from_filer_pkt, NULL, 0, filer_out_id) ;
      VCB_Init (&to_filer, &to_filer_port, NULL, NULL, 0, filer_in_id) ;
    }  
    /*}}}*/
    /*{{{  initiliase filer-users port and channels on root*/
    #ifndef UPR_RD
    {
      PORT *ports ;
      BYTE (*packets)[UPR_MAX_PKT_SIZE] ;
      int base, i ;
    
      if (((ports = (PORT *) malloc (sizeof(PORT)*2*num_procs)) == NULL) ||
          ((packets = (BYTE (*)[UPR_MAX_PKT_SIZE]) malloc (UPR_MAX_PKT_SIZE*num_procs)) == NULL) ||
          ((users_to_filer = (VCB *) malloc (sizeof(VCB)*num_procs)) == NULL) ||
          ((filer_to_users = (VCB *) malloc (sizeof(VCB)*num_procs)) == NULL))
        Exception(UPR_fatal,_FILE_,__LINE__,"Malloc for filer-users channels failed") ;
      
      if ((base = MultiplePortGrab (ports, 2*num_procs)) < 0)
        Exception (UPR_fatal, _FILE_, __LINE__, "Out of ports") ;
    
      if (base != filer_in_id)
        Exception (UPR_fatal, _FILE_, __LINE__, "Serious programming inconsistency") ;
      
      for (i=0;i<num_procs;i++)
      {
        PortInit (&ports[i], VCR_InPreAction, TRUE,
                  VCR_InPostAction, FALSE, (void *) &users_to_filer[i]) ;
                  
        PortInit (&ports[num_procs+i], VCR_OutPreAction, TRUE,
                  VCR_OutPostAction, FALSE, (void *) &filer_to_users[i]) ;
    
        VCB_Init (&users_to_filer[i], &ports[i], packets[i], NULL, i, my_out_id) ;
        VCB_Init (&filer_to_users[i], &ports[num_procs+i], NULL, NULL, i, my_in_id) ;
      }    
    }
    #endif
    /*}}}*/
  }
  /*}}}*/
}
/*}}}*/
/*{{{  PUBLIC int VirtualProcessorInterface(int argc, char *argv[])*/
PUBLIC int VirtualProcessorInterface(int argc, char *argv[])
{
  SCDB *scdb;

  init_scheduler() ;

  #ifndef UPR_RD
  /*{{{  root*/
  {
    FILE *fp, *ncf_fp ;
    char *program = argv[1];
    int start, finish ;
    NETWORK *net = UPR_GetNetworkInfo() ;
    BOOTNODE *tree ;
    int pos = 0 ;
  
    if (net == NULL)
      Exception (UPR_error,_FILE_,__LINE__,"Cannot get network information") ;
  
    if ((tree = (BOOTNODE *) malloc(vpi_globals.num_procs*sizeof(BOOTNODE))) == NULL)
      Exception (UPR_error,_FILE_,__LINE__,"Out of heap") ;
  
    TransformTree (tree, 0, net->vertex, &pos) ;
      
    vpi_globals.children = ComputeChildren (tree) ;
    vpi_globals.parent   = NOT_PROC ;
    
    /*{{{  debug*/
    #ifdef VPMAIN_DEBUG
    {
      int i ;
    
      for (i=0;i<net->num_edges;i++)
        printf("Processor %d to processor %d\n",net->edge[i][0],net->edge[i][1]) ;
    
      for (i=0;i<vpi_globals.num_procs;i++)
      {
        int *proc = &tree[i].son ;
    
        printf("Processor %d (sub-tree size %d)",
               tree[i].id, tree[i].size) ;
        
        if (*proc != NOT_PROC)
        {
          printf(", boots up ") ;
    
          do
          {
            printf("%d (%d) ", tree[*proc+i].id, *proc+i) ;
            proc = &tree[*proc+i].sibling ;
          } while (*proc != NOT_PROC) ;
        }
        printf("\n") ;
      }
    }
    #endif
    /*}}}*/
        
    if ((argc >= 2) && ((fp=fopen(program,"r")) != NULL))
    {
      fclose(fp) ;    
      TIME_INFO(start = ProcTime()) ;
      scdb = RPC_LoadCode(program);
      TIME_INFO(finish = ProcTime()) ;
      TIME_INFO(printf("Read SCDB of %d bytes in %d microseconds\n",scdb->size,ProcTimeMinus(finish,start)*64)) ;
      {
        TIME_INFO(start = ProcTime()) ;
  
        BootSons (scdb, tree) ;
  
        /*{{{  spawn hosthooks*/
        {
          int proc;
          
          for (proc=0; proc<vpi_globals.num_procs; proc++)
          {
            Process *p;
            VCB *fs = &filer_to_users[proc];
            VCB *ts = &users_to_filer[proc];
            if ((p=ProcAlloc(CallHostHook,HostHookWsp,2,fs,ts)) == NULL)
              Exception(UPR_error,_FILE_,__LINE__,"Out of heap") ;
            ProcRun(p);
          }
        }
        /*}}}*/
  
        TIME_INFO(finish = ProcTime()) ;
        TIME_INFO(printf("%d nodes, %d microseconds\n",vpi_globals.num_procs,ProcTimeMinus(finish,start)*64)) ;
  
        DEBUG(printf("Network loaded\n")) ;
     }
    }
    else
    {
      printf("VPI 2.0k : Virtual processor interface, %s %s\n",__TIME__,__DATE__) ;
      printf("M. Debbage, M. Hill, University Of Southampton, ESPRIT GP-MIMD P5404\n\n") ;
      printf("VPI requires user code filename\n") ;
      return(0) ;
    }
  }
  /*}}}*/
  #else  
  /*{{{  node*/
  /* For 2-phase HPR, nodes must not pre-acknowledge first communication */
  
  {
    BOOTNODE *tree ;
    
    vpi_globals.parent = GrantService() ;
    scdb = ReadBootCode() ;
    RPC_StackCode(scdb);
    tree = ReadBootTree() ;
    BootSons (scdb, tree) ;
    ServerOut(NULL, 0) ;
    EndService() ;
  
    vpi_globals.children = ComputeChildren (tree) ;
  }
  /*}}}*/
  #endif

  {
    Process *p;
    Channel sfs,sts ;
    int user_heap_in_ints ;
    #ifdef UPR_RD
    int system_heap_bytes = DEF_SYSTEM_HEAP_BYTES_NODE;
    #else
    char *env = getenv(ENV_SYSTEM_HEAP_BYTES_ROOT) ;
    int system_heap_bytes = (env==NULL) ? DEF_SYSTEM_HEAP_BYTES_ROOT : atoi(env) ;
    #endif

    ChanInit(&sfs);
    ChanInit(&sts);
    if ((p=ProcAlloc(SP_ConvertBuffer,SP_ConvertBufferWsp, 4, &sfs,&sts,&from_filer,&to_filer)) == NULL)
      Exception(UPR_error,_FILE_,__LINE__,"Out of heap") ;
    ProcRun(p) ;

    user_heap_in_ints = (proc_memory - (((int) _IMS_heap_front)^((int) NotProcess_p)) - system_heap_bytes) / sizeof(int);
    
    if (user_heap_in_ints <= 0)
      Exception(UPR_error,_FILE_,__LINE__,"Insufficient heap to run user heap") ;

    return(DynamicC(&sfs,&sts, scdb, user_heap_in_ints)) ;
  }
}
/*}}}*/

