/*{{{  File banner*/
/*@(#)=====================================================*\
||@(#)  Project : GPMIMD ESPRIT P5404
||@(#)  Authors : Mark Debbage and Mark Hill
||@(#)            University of Southampton
||  
||@(#)    Title : VPI client-server interface
||@(#)   System : VPI
||@(#) Filename : vpi.c
||@(#)  Version : 1.18
||@(#)     Date : 7/6/92
\*@(#)====================================================*/
/*}}}*/

#include <stdio.h>
#include <stdlib.h>

#include "vcr.h"
#include "vpi.h"
#include "vpmain.h"
#include "schan.h"
#include "uchan.h"

/* These routines may be called at either priority, but will be more
   efficient from HI. */

PRIVATE char *_FILE_ = __FILE__ ;

#define DEBUG(s)

/*{{{  Macros for entering and leaving VCR system*/
/* These macros are used when switching between VPI software
   scheduling and VCR microcoded scheduling. They are only
   needed if a VCR descheduling point is anticipated (such as
   a channel communication). A better, though more dramatic,
   approach would be to change the whole of VCR to use the
   threads library. */

#if ((defined MICROCODED_SCHEDULER) || (!defined LIBRAFY))
#define ENTER_VCR()
#define LEAVE_VCR()
#else
#define ENTER_VCR() deactivate() ;
#define LEAVE_VCR() reactivate() ;
#endif
/*}}}*/

/*{{{  PUBLIC int  GrantService()*/
PUBLIC int  GrantService()
{
  int priority ;
  int target;

  DEBUG(printf("Server granting\n");)
  
  ENTER_VCR() ;

  ProcGetPRI(&priority) ;
  if (priority==PROC_LOW) ProcToHI() ;
        
  /*{{{  critical code section*/
  {
    VPI_GLOBALS *vpig = (VPI_GLOBALS *) (* ((GLOBALS **) GLOBAL_PTR))->vpi_gp ;
    SERVER *server = &vpig->server;
    UCB *head = server->head;
    
    if (head==NULL)
    {
      ProcDesc(&server->wdesc);
      ProcSleep();
      head = server->head;
    }
  
    target = head->port->id - vpig->server_port_base ;
  }
  /*}}}*/

  if (priority==PROC_LOW) ProcToLO() ;

  LEAVE_VCR() ;

  DEBUG(printf("Server granted\n");)
  
  return(target);
}
/*}}}*/
/*{{{  PUBLIC int  ServerIn (void *message, int length)*/
PUBLIC int  ServerIn (void *message, int length)
{
  VPI_GLOBALS *vpig = (VPI_GLOBALS *) (* ((GLOBALS **) GLOBAL_PTR))->vpi_gp ;
  SERVER *server = &vpig->server ;
  UCB *head  = server->head ;
  int target = head->port->id - vpig->server_port_base ;
  int len ;

  DEBUG(printf("Server reading\n");)
  
  ENTER_VCR() ;

  if (target == vpig->proc_id)
    /*{{{  local transaction*/
    {
      len = SoftVarLenIn (&head->id, message, length) ;
    }
    /*}}}*/
  else
    /*{{{  non-local transaction*/
    {
      void **gsb = (void **) (&message-1) ;
      void *oldgsb ;
      oldgsb = *gsb ;
      *gsb = vpig->gsb ;
      len = vpig->vpfunc.VarLenIn (head, message, length) ;
      *gsb = oldgsb ;
    }
    /*}}}*/

  LEAVE_VCR() ;

  DEBUG(printf("Server read\n");)
  
  return(len) ;
}
/*}}}*/
/*{{{  PUBLIC void ServerOut(void *message, int length)*/
PUBLIC void ServerOut(void *message, int length)
{
  VPI_GLOBALS *vpig = (VPI_GLOBALS *) (* ((GLOBALS **) GLOBAL_PTR))->vpi_gp ;
  SERVER *server = &vpig->server;
  int target = server->head->port->id - vpig->server_port_base ;

  DEBUG(printf("Server sending\n");)
  
  ENTER_VCR() ;
  
  if (target == vpig->proc_id)
    /*{{{  local transaction*/
    {
      SoftVarLenOut (&vpig->client[target].in.id, message, length) ;
    }
    /*}}}*/
  else
    /*{{{  non-local transaction*/
    {
      void **gsb = (void **) (&message-1) ;
      void *oldgsb ;
      oldgsb = *gsb ;
      *gsb = vpig->gsb ;
      vpig->vpfunc.VarLenOut (&vpig->server.out[target], message, length) ;
      *gsb = oldgsb ;
    }
    /*}}}*/

  LEAVE_VCR() ;

  DEBUG(printf("Server sent\n");)
}
/*}}}*/
/*{{{  PUBLIC void EndService()*/
PUBLIC void EndService()
{
  int priority ;

  DEBUG(printf("Server ending\n");)
  
  ProcGetPRI(&priority) ;
  if (priority==PROC_LOW) ProcToHI() ;
  
  /*{{{  critical code section*/
  {        
    VPI_GLOBALS *vpig = (VPI_GLOBALS *) (* ((GLOBALS **) GLOBAL_PTR))->vpi_gp ;
    SERVER *server = &vpig->server;
    UCB  *head = server->head;
    PORT *port = head->port ;
    int target = head->port->id - vpig->server_port_base ;
  
    server->head = head->next ;
    head->next = NULL ;
  
    if (target == vpig->proc_id)
      /*{{{  local transaction*/
      {
        if (head->port->buffer != NotPointer_p)
        {
          head->port->buffer = NotPointer_p ;
          /*{{{  enqueue next transaction on server*/
          {
            if (server->head == NULL)
              /*{{{  make list*/
              {
                server->head = head ;
                server->tail = head ;
              }
              /*}}}*/
            else
              /*{{{  append to pending queue*/
              {
                server->tail->next = head ;
                server->tail = head ;
              }
              /*}}}*/
          }    
          /*}}}*/
        }
      }
      /*}}}*/
    else  
      /*{{{  non-local transaction*/
      if (head->id != NotProcess_p)
        /*{{{  re-enqueue since packet body has already arrived*/
        {
          if (server->head == NULL)
            /*{{{  make queue*/
            {
              server->head = head ;
              server->tail = head ;
            }
            /*}}}*/
          else
            /*{{{  append to queue*/
            {
              server->tail->next = head ;
              server->tail = head ;
            }
            /*}}}*/
        }
        /*}}}*/
      /*}}}*/
  }
  /*}}}*/

  if (priority==PROC_LOW) ProcToLO() ;

  DEBUG(printf("Server ended\n");)
}
/*}}}*/

/*{{{  PUBLIC void ClaimChannel  (int target)*/
PUBLIC void ClaimChannel  (int target)
{
  VPI_GLOBALS *vpig = (VPI_GLOBALS *) (* ((GLOBALS **) GLOBAL_PTR))->vpi_gp ;

  DEBUG(printf("Client claiming channel\n");)

  if ((target >= 0) && (target < vpig->num_procs))
    {
      CLIENT *client = &vpig->client[target] ;
      sema_wait (&client->sema) ;
      client->bound = FALSE ;
    }
  else
    Error(_FILE_,__LINE__,"Bad processor number") ;

  DEBUG(printf("Client got channel\n");)
}
/*}}}*/
/*{{{  PUBLIC void ClientOut     (int target, void *message, int length)*/
PUBLIC void ClientOut     (int target, void *message, int length)
{
  VPI_GLOBALS *vpig = (VPI_GLOBALS *) (* ((GLOBALS **) GLOBAL_PTR))->vpi_gp ;

  DEBUG(printf("Client sending\n");)
  
  ENTER_VCR() ;
  
  if ((target >= 0) && (target < vpig->num_procs))
    {
      CLIENT *client = &vpig->client[target] ;
      
      if (target == vpig->proc_id)
        /*{{{  local transaction*/
        {
          SERVER *server = &vpig->server ;
          UCB *ucb = &server->in[target] ;
          int priority ;
        
          ProcGetPRI(&priority) ;
          if (priority==PROC_LOW) ProcToHI() ;
        
          /*{{{  critical code section*/
          if (!client->bound)
          {
            if (server->head != ucb)
              /*{{{  only enqueue if not still top of queue (server handles other case)*/
              {
                if (server->head == NULL)
                  /*{{{  make list*/
                  {
                    server->head = ucb;
                    server->tail = ucb;
                  
                    if (server->wdesc != NULL)
                    {
                      ProcAwaken(server->wdesc);
                      server->wdesc = NULL;
                    }
                  }
                  /*}}}*/
                else
                  /*{{{  append to pending queue*/
                  {
                    server->tail->next = ucb ;
                    server->tail = ucb ;
                  }
                  /*}}}*/
              }
              /*}}}*/
            else
              /*{{{  still on queue*/
              {
                if (server->wdesc != NULL)
                  Error(_FILE_,__LINE__,"Invalid server state") ;
              
                ucb->port->buffer = message ;   /* Flags server of this condition */
              }
              /*}}}*/
          }
          /*}}}*/
          
          if (priority==PROC_LOW) ProcToLO() ;
          
          SoftVarLenOut (&ucb->id, message, length) ;
        }
        /*}}}*/
      else
        /*{{{  non-local transaction*/
        {
          void **gsb = (void **) (&target-1) ;
          void *oldgsb ;
          oldgsb = *gsb ;
          *gsb = vpig->gsb ;
          vpig->vpfunc.VarLenOut (&client->out, message, length) ;
          *gsb = oldgsb ;
        }
        /*}}}*/

      client->bound = TRUE ;
    }
  else
    Error(_FILE_,__LINE__,"Bad processor number") ;

  LEAVE_VCR() ;

  DEBUG(printf("Client sent\n");)
}
/*}}}*/
/*{{{  PUBLIC int  ClientIn      (int target, void *message, int length)*/
PUBLIC int  ClientIn      (int target, void *message, int length)
{
  VPI_GLOBALS *vpig = (VPI_GLOBALS *) (* ((GLOBALS **) GLOBAL_PTR))->vpi_gp ;
  int len ;

  DEBUG(printf("Client reading\n");)

  ENTER_VCR() ;

  if ((target >= 0) && (target < vpig->num_procs))
    {
      CLIENT *client = &vpig->client[target] ;
      
      if (client->bound)
      {
        if (target == vpig->proc_id)
          /*{{{  local transaction*/
          {
            len = SoftVarLenIn (&client->in.id, message, length) ;
          }
          /*}}}*/
        else
          /*{{{  non-local transaction*/
          { 
            void **gsb = (void **) (&target-1) ;
            void *oldgsb ;
            oldgsb = *gsb ;
            *gsb = vpig->gsb ;
            len = vpig->vpfunc.VarLenIn (&client->in, message, length) ;
            *gsb = oldgsb ;
          }
          /*}}}*/
      }
      else
        Error(_FILE_,__LINE__,"Client must send first") ;
    }
  else
    Error(_FILE_,__LINE__,"Bad processor number") ;

  LEAVE_VCR() ;

  DEBUG(printf("Client read\n");)
  
  return(len) ;
}
/*}}}*/
/*{{{  PUBLIC void ReleaseChannel(int target)*/
PUBLIC void ReleaseChannel(int target)
{
  VPI_GLOBALS *vpig = (VPI_GLOBALS *) (* ((GLOBALS **) GLOBAL_PTR))->vpi_gp ;

  DEBUG(printf("Client releasing\n");)
  
  if ((target >= 0) && (target < vpig->num_procs))
    sema_signal (&vpig->client[target].sema) ;
  else
    Error(_FILE_,__LINE__,"Bad processor number") ;

  DEBUG(printf("Client released\n");)
}
/*}}}*/

/*{{{  PUBLIC void Error (char file[], int line, char msg[])*/
PUBLIC void Error (char file[], int line, char msg[])
{
  VCR_GLOBALS *g = (VCR_GLOBALS *) (* ((GLOBALS **) GLOBAL_PTR))->vcr_gp ;  
  void **gsb = (void **) (&file-1) ;
  
  ENTER_VCR() ;
  
  *gsb = g->gsb ;
  g->vfunc.Exception (UPR_error, file, line, msg) ;
}
/*}}}*/

#ifdef LIBRAFY
/*{{{  PUBLIC int  NumProcs ()*/
PUBLIC int  NumProcs ()
{
  VPI_GLOBALS *vpig = (VPI_GLOBALS *) (* ((GLOBALS **) GLOBAL_PTR))->vpi_gp ;
  return(vpig->num_procs) ;
}
/*}}}*/
/*{{{  PUBLIC int  ProcId ()*/
PUBLIC int  ProcId ()
{
  VPI_GLOBALS *vpig = (VPI_GLOBALS *) (* ((GLOBALS **) GLOBAL_PTR))->vpi_gp ;
  return(vpig->proc_id) ;
}
/*}}}*/

/*{{{  PUBLIC int  NewVCE     (VCE *vce) */
PUBLIC int  NewVCE     (VCE *vce) 
{
  VPI_GLOBALS *vpig = (VPI_GLOBALS *) (* ((GLOBALS **) GLOBAL_PTR))->vpi_gp ;
  void **gsb = (void **) (&vce-1) ;
  *gsb = vpig->gsb ;
  return(vpig->vpfunc.NewVCE(vce)) ;
}
/*}}}*/
/*{{{  PUBLIC void ConnectVCE (VCE *source, VCE *destn) */
PUBLIC void ConnectVCE (VCE *source, VCE *destn) 
{
  VPI_GLOBALS *vpig = (VPI_GLOBALS *) (* ((GLOBALS **) GLOBAL_PTR))->vpi_gp ;
  void **gsb = (void **) (&source-1) ;
  *gsb = vpig->gsb ;
  vpig->vpfunc.ConnectVCE(source,destn) ;
}
/*}}}*/
/*{{{  PUBLIC VC   GetVC  (VCE *vce) */
PUBLIC VC   GetVC  (VCE *vce) 
{
  return (vce->vc) ;
}
/*}}}*/
/*{{{  PUBLIC void FreeVC (VC vc)*/
PUBLIC void FreeVC (VC vc)
{
  VPI_GLOBALS *vpig = (VPI_GLOBALS *) (* ((GLOBALS **) GLOBAL_PTR))->vpi_gp ;
  void **gsb = (void **) (&vc-1) ;
  *gsb = vpig->gsb ;
  vpig->vpfunc.FreeVC(vc) ;
}
/*}}}*/

/*{{{  PUBLIC void Send (VC vc, void *message, int length) */
PUBLIC void Send (VC vc, void *message, int length) 
{
  VPI_GLOBALS *vpig = (VPI_GLOBALS *) (* ((GLOBALS **) GLOBAL_PTR))->vpi_gp ;

  ENTER_VCR() ;
  
  /*{{{  do communication*/
  {
    void **gsb = (void **) (&vc-1) ;
    void *oldgsb ;
    oldgsb = *gsb ;
    *gsb = vpig->gsb ;
    vpig->vpfunc.VarLenOut (&((VCPAIR *) vc)->out_ucb, message, length) ;
    *gsb = oldgsb ;
  }
  /*}}}*/
  
  LEAVE_VCR() ;
}
/*}}}*/
/*{{{  PUBLIC int  Recv (VC vc, void *message, int length)*/
PUBLIC int  Recv (VC vc, void *message, int length) 
{
  VPI_GLOBALS *vpig = (VPI_GLOBALS *) (* ((GLOBALS **) GLOBAL_PTR))->vpi_gp ;
  int len ;

  ENTER_VCR() ;

  /*{{{  do communication*/
  {
    void **gsb = (void **) (&vc-1) ;
    void *oldgsb ;
    oldgsb = *gsb ;
    *gsb = vpig->gsb ;
    len = vpig->vpfunc.VarLenIn (&((VCPAIR *) vc)->in_ucb, message, length) ;
    *gsb = oldgsb ;
  }
  /*}}}*/
  
  LEAVE_VCR() ;
  
  return(len) ;
}
/*}}}*/

/*{{{  PUBLIC int *Children()*/
PUBLIC int *Children()
{
  VPI_GLOBALS *vpig = (VPI_GLOBALS *) (* ((GLOBALS **) GLOBAL_PTR))->vpi_gp ;
  return(vpig->children) ;
}
/*}}}*/
/*{{{  PUBLIC int  Parent()*/
PUBLIC int  Parent()
{
  VPI_GLOBALS *vpig = (VPI_GLOBALS *) (* ((GLOBALS **) GLOBAL_PTR))->vpi_gp ;
  return(vpig->parent) ;
}
/*}}}*/
#endif
