/*{{{  File banner*/
/*@(#)=====================================================*\
||@(#)  Project : GPMIMD ESPRIT P5404
||@(#)  Authors : Mark Debbage and Mark Hill
||@(#)            University of Southampton
||  
||@(#)    Title : Variable length channel i/o (based on vchan.c V2.4)
||@(#)   System : VPI
||@(#) Filename : uchan.c
||@(#)  Version : 1.10
||@(#)     Date : 10/15/92
\*@(#)====================================================*/
/*}}}*/

#include <stdio.h>
#include <stdlib.h>

#include "vcr.h"
#include "vpi.h"
#include "vpmain.h"
#include "uchan.h"
#include "schan.h"

PRIVATE char *_FILE_ = __FILE__ ;

#define USE_MULTI_PACKET_UPRI

/*{{{  PUBLIC void UCB_Init (UCB *ucb, PORT *port, BYTE *buffer, int destn_proc, int destn_port)*/
PUBLIC void UCB_Init (UCB *ucb, PORT *port, BYTE *buffer, int destn_proc, int destn_port)
{
  ucb->id   = NotProcess_p ;
  ucb->next = NULL ;
  ucb->port = port ;
  ucb->orb.list = InputServiced_p ;
  ucb->orb.bffr = NULL ;
  ucb->buffer = buffer ;
  ucb->readlen = NotLength_p ;

  HdrBuild (destn_proc, destn_port, 0, &(ucb->orb.header)) ;
}
/*}}}*/

/*{{{  PUBLIC void VPI_InPreAction (PORT *port, int length)*/
PUBLIC void VPI_InPreAction (PORT *port, int length)
{
  UCB *ucb = (UCB *) port->state ;

  switch ((int) ucb->id)
  {
    case (int) NotProcess_p :
      /*{{{  empty channel end*/
      port->buffer  = (BYTE *) ucb->buffer ;
      port->space   = UPR_MAX_PKT_SIZE ;
      port->do_post = TRUE ;
      break ;
      /*}}}*/

    case (int) InputWaiting_p :
      Exception(UPR_error, _FILE_, __LINE__, "Two early packets on virtual channel") ;
      break ;
  
    default :
      /*{{{  buffer overflow*/
      ucb->readlen = OverLength_p ;
      port->buffer  = (BYTE *) ucb->buffer ;
      port->space   = UPR_MAX_PKT_SIZE ;
      /*}}}*/
  }
}
/*}}}*/
/*{{{  PUBLIC void VPI_InPostAction (PORT *port, int length, BYTE *buffer)*/
PUBLIC void VPI_InPostAction (PORT *port, int length, BYTE *buffer)
{
  UCB *ucb = (UCB *) port->state ;

  switch ((int) ucb->id)
  {
    case (int) NotProcess_p :
      /*{{{  channel end still empty*/
      {
        ucb->id = InputWaiting_p ;
        port->buffer  = NotPointer_p ;
        port->do_pre  = FALSE ;
      
        if ((port->id >= vpi_globals.server_port_base) &&
            (port->id <  vpi_globals.server_port_base + (vpi_globals.num_procs)))
          if (vpi_globals.server.head != ucb)
            /*{{{  first packet on a server transaction*/
            {
              if (vpi_globals.server.head == NULL)
                /*{{{  make queue*/
                {
                  vpi_globals.server.head = ucb ;
                  vpi_globals.server.tail = ucb ;
                
                  if (vpi_globals.server.wdesc != NULL)
                  {
                    ProcAwaken(vpi_globals.server.wdesc);
                    vpi_globals.server.wdesc = NULL;
                  }            
                }
                /*}}}*/
              else
                /*{{{  append to queue*/
                {
                  vpi_globals.server.tail->next = ucb ;
                  vpi_globals.server.tail = ucb ;
                }
                /*}}}*/
            }
            /*}}}*/
      }
      
      break ;
      /*}}}*/
  
    case (int) InputWaiting_p :
      Exception(UPR_error, _FILE_, __LINE__, "Two early packets on virtual channel") ;
      break ;
  
    default :
      /*{{{  virtual in arrived*/
      {
        if (port->do_pre == TRUE)
          /*{{{  header-vin-body*/
          {
            if (length > port->space)
              ucb->readlen = OverLength_p ;
            else
              {
                if (length != 0)
                {
                  AsmMove(ucb->buffer, port->buffer, length) ;
                  port->buffer += length ;
                  port->space -= length ;
                }
                ucb->readlen = length ;
              }
          
            if (length == UPR_MAX_PKT_SIZE)
              {
                if (ucb->readlen == OverLength_p)
                  {
                    port->buffer = ucb->buffer ;
                    port->space  = UPR_MAX_PKT_SIZE ;
                  }
                port->do_pre = FALSE ;
              }
            else
              {
                ProcAwaken(ucb->id) ;
                ucb->id = NotProcess_p ;
                port->buffer  = NotPointer_p ;
                port->space   = 0 ;
                port->do_pre  = TRUE ;
                port->do_post = FALSE ;
              }    
          }
          /*}}}*/
        else
          /*{{{  all remaining packets */
          if (length == UPR_MAX_PKT_SIZE)
            {
              if (ucb->readlen == OverLength_p)
              {
                port->buffer  = (BYTE *) ucb->buffer ;
                port->space   = UPR_MAX_PKT_SIZE ;
              }
              else
                ucb->readlen += length ;
            }
          else
            {
              if (ucb->readlen != OverLength_p)
                ucb->readlen += length ;
            
              ProcAwaken(ucb->id) ;
              ucb->id = NotProcess_p ;
              port->buffer  = NotPointer_p ;
              port->space   = 0 ;
              port->do_pre  = TRUE ;
              port->do_post = FALSE ;
            }
          /*}}}*/
      }
      /*}}}*/
  }
}               
/*}}}*/
/*{{{  PUBLIC int  VarLenIn (UCB *ucb, void *message, int length)*/
PUBLIC int  VarLenIn (UCB *ucb, void *message, int length)
{
  BYTE *msg = (BYTE *) message ;

  if (length < 0)
    Exception(UPR_error, _FILE_, __LINE__, "Message input of negative length!") ;

  if (ucb->buffer == NULL)
    Exception(UPR_error, _FILE_, __LINE__, "Virtual input on output channel") ;

  if (HdrDestn(&ucb->orb.header) == vpi_globals.proc_id)
    /*{{{  internal virtual communication*/
    {
      return(SoftVarLenIn (&ucb->id, msg, length)) ;
    }
    /*}}}*/
  else
    /*{{{  external virtual communication*/
    {  
      int priority ;
    
      ProcGetPRI(&priority) ;
      if (priority==PROC_LOW) ProcToHI() ;
    
      /*{{{  critical code section*/
      {
        PORT *port = ucb->port ;
        
        /*{{{  enqueue acknowledgement*/
        UPR_Enqueue_NBK (&ucb->orb) ;
        /*}}}*/
        /*{{{  perform transaction*/
        switch ((int) ucb->id)
        {
          case (int) NotProcess_p :
            /*{{{  early virtual input*/
            {
              ucb->readlen  = 0 ;
            
              /* Note if header (but not body) has been read */
              if (port->buffer == NotPointer_p)
              {
                port->do_pre = FALSE ;
                port->do_post = TRUE ;
              }
            
              port->space  = length ;
              port->buffer = msg ;
              ProcDesc(&ucb->id) ;
              ProcSleep() ;
            }
            
            break ;
            /*}}}*/
        
          case (int) InputWaiting_p :
            /*{{{  late virtual input*/
            {
              int sentlen = UPR_MAX_PKT_SIZE - port->space ;
            
              if (sentlen > length)
                ucb->readlen = OverLength_p ;
              else
                {
                  if (sentlen != 0)
                  {
                    AsmMove(ucb->buffer, msg, sentlen) ;
                    msg += sentlen ;
                    length -= sentlen ;
                  }
                  ucb->readlen = sentlen ;
                }
            
              if (sentlen == UPR_MAX_PKT_SIZE)
                {
                  if (ucb->readlen == OverLength_p)
                    {
                      port->buffer = ucb->buffer ;
                      port->space  = UPR_MAX_PKT_SIZE ;
                    }
                  else
                    {
                      port->buffer = msg ;
                      port->space  = length ;
                    }        
                  ProcDesc(&ucb->id) ;
                  ProcSleep() ;
                }      
              else
                {
                  ucb->id     = NotProcess_p ;
                  port->do_pre  = TRUE ;
                  port->do_post = FALSE ;
                  port->space = 0 ;
                }
            }      
            
            break ;
            /*}}}*/
        
          default :
            Exception(UPR_error, _FILE_, __LINE__, "Invalid id on virtual in entry") ;
        }
        /*}}}*/
        /*{{{  check channel state*/
        if ((ucb->id != NotProcess_p) && (ucb->id != InputWaiting_p))
          Exception(UPR_error, _FILE_, __LINE__, "Invalid input id on virtual in exit") ;
        /*}}}*/
        /*{{{  block if acknowledge still here*/
        UPR_ORB_BK (&ucb->orb) ;
        /*}}}*/
      }
      
      /*}}}*/
    
      if (priority==PROC_LOW) ProcToLO() ;
    
      return(ucb->readlen) ;
    }
    /*}}}*/
}
/*}}}*/

/*{{{  PUBLIC void VPI_OutPreAction (PORT *port, int length)*/
PUBLIC void VPI_OutPreAction (PORT *port, int length)
{
  UCB *ucb = (UCB *) port->state ;

  if (length !=0)
    Exception(UPR_error, _FILE_, __LINE__, "Unsolicited packet on acknowledge port") ;

  switch ((int) ucb->id)
  {
    case (int) NotProcess_p :
      ucb->id = AckWaiting_p ;
      break ;

    case (int) AckWaiting_p :
      Exception(UPR_error, _FILE_, __LINE__, "Out of protocol acknowledgement received") ;
      break ;

    default :
      ProcAwaken(ucb->id) ;
      ucb->id = NotProcess_p ;
  }  
}               
/*}}}*/
/*{{{  PUBLIC void VPI_OutPostAction (PORT *port, int length, BYTE *buffer)*/
PUBLIC void VPI_OutPostAction (PORT *port, int length, BYTE *buffer)
{
  ;
}
/*}}}*/
/*{{{  PUBLIC void VarLenOut (UCB *ucb, void *message, int length)*/
PUBLIC void VarLenOut (UCB *ucb, void *message, int length)
{
  BYTE *msg = (BYTE *) message ;

  if (length < 0)
    Exception(UPR_error, _FILE_, __LINE__, "Message output of negative length!") ;

  if (ucb->buffer != NULL)
    Exception(UPR_error, _FILE_, __LINE__, "Virtual output on input channel") ;
  
  if (HdrDestn(&ucb->orb.header) == vpi_globals.proc_id)
    /*{{{  internal virtual communication*/
    {
      UCB *ucb_in = (UCB *) (PortPtr(HdrPort(&ucb->orb.header))->state) ;
      SoftVarLenOut (&ucb_in->id, msg, length) ;
    }
    /*}}}*/
  else
    /*{{{  external virtual communication*/
    {  
      int priority ;
      
      ProcGetPRI(&priority) ;
      if (priority==PROC_LOW) ProcToHI() ;
    
      /*{{{  critical code section*/
      {
        int size ;
        
        /*{{{  enqueue first packet*/
        size = min (UPR_MAX_PKT_SIZE, length) ;
        HdrModLength(size, &ucb->orb.header) ;
        ucb->orb.bffr = msg ;
        
        UPR_Enqueue_NBK (&ucb->orb) ;
        msg += size ;
        length -= size ;
        /*}}}*/
        /*{{{  block if acknowledgement not here*/
        switch ((int) ucb->id)
        {
          case (int) NotProcess_p :
            /*{{{  block until ack arrives*/
            ProcDesc(&ucb->id) ;
            ProcSleep() ;
            break ;
            /*}}}*/
        
          case (int) AckWaiting_p :
            ucb->id = NotProcess_p ;
            break ;
        
          default :
            Exception(UPR_error, _FILE_, __LINE__, "Invalid id on virtual out entry") ;
        }
        /*}}}*/
        /*{{{  block if first packet still here*/
        UPR_ORB_BK (&ucb->orb) ;
        /*}}}*/
        
        #ifdef USE_MULTI_PACKET_UPRI
          /*{{{  enqueue remaining packets with one blocking call*/
          {
            if (length != 0)
            {
              ucb->orb.bffr = msg ;
              UPR_MultipleEnqueue_BK(&ucb->orb,length) ;
          
              /* Empty packet to terminate message if last packet is full */
              if (length%UPR_MAX_PKT_SIZE == 0)
                UPR_MultipleEnqueue_BK(&ucb->orb,0) ;
            }
            else
              /* Empty packet to terminate message if only packet is full */
              if (size == UPR_MAX_PKT_SIZE)
                UPR_MultipleEnqueue_BK(&ucb->orb,0) ;
          }
          
          /*}}}*/
        #else
          /*{{{  enqueue remaining packets with repeated blocking calls*/
          while (size == UPR_MAX_PKT_SIZE)
          {
            size = min (UPR_MAX_PKT_SIZE, length) ;
            HdrModLength(size, &ucb->orb.header) ;
            ucb->orb.bffr = msg ;
            
            UPR_Enqueue_BK (&ucb->orb) ;
              
            msg += size ;
            length -= size ;
          }
          /*}}}*/
        #endif
      
        /*{{{  check channel state*/
        if ((ucb->id != NotProcess_p) && (ucb->id != AckWaiting_p))
          Exception(UPR_error, _FILE_, __LINE__, "Invalid id on virtual out exit") ;
        /*}}}*/
      }
      /*}}}*/
    
      if (priority==PROC_LOW) ProcToLO() ;
    }
    /*}}}*/
}
/*}}}*/

