/*{{{  File banner*/
/*@(#)=====================================================*\
||@(#)  Project : GPMIMD ESPRIT P5404
||@(#)  Authors : Mark Debbage and Mark Hill
||@(#)            University of Southampton
||  
||@(#)    Title : Machine dependent process library
||@(#)   System : VPI
||@(#) Filename : threads.c
||@(#)  Version : 1.13
||@(#)     Date : 7/8/92
\*@(#)====================================================*/
/*}}}*/

/*{{{  includes*/
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>

#include <process.h>
#include "upri.h"
#include "threads.h"
/*}}}*/

#ifdef TRANSPUTER
  #ifndef MICROCODED_SCHEDULER
    /*{{{  co-routined scheduling*/
    typedef struct
    {
      process *head ;
      process *tail ;
    } pqueue ;
    
    PRIVATE pqueue staticpq = { NULL, active_p } ;
    
    /*{{{  PUBLIC void init_scheduler(void)*/
    PUBLIC void init_scheduler(void)
    {
      (*((GLOBALS **)GLOBAL_PTR))->pqueue = &staticpq ;
    
      #ifndef USER_CODE_AT_LOW_PRI
      ProcToHI() ;
      #endif
    }
    /*}}}*/
    
    /*{{{  PUBLIC void deschedule (process *me)*/
    PUBLIC void deschedule (process *current)
    {
      pqueue *pq = (pqueue *) (* ((GLOBALS **)GLOBAL_PTR))->pqueue ;
      process *next = pq->head ;
    
      /*{{{  restart next process*/
      if (next == NULL)
        pq->tail = NULL ;
      else
        {
          pq->head = pq->head->next ;
          if (pq->head == NULL)
            pq->tail = active_p ;
          ProcAwaken(next->context) ;
        }
      /*}}}*/
      
      if (current != NULL)
        /*{{{  save and sleep current process state*/
        {
          ProcDesc(&current->context) ;
          ProcSleep() ;
        }
        /*}}}*/
    }
    /*}}}*/
    
    /*{{{  PUBLIC void reschedule (process *you)*/
    PUBLIC void reschedule (process *you)
    {
      pqueue *pq = (pqueue *) (* ((GLOBALS **)GLOBAL_PTR))->pqueue ;
    
      if (pq->head == NULL)
      {
        if (pq->tail == active_p)
          /*{{{  only item on queue*/
          {
            you->next = NULL ;
            pq->head = you ;
            pq->tail = you ;
          }
          /*}}}*/
        else
          /*{{{  no current process, only occurs when leaving VCR via action*/
          {
            pq->tail  = active_p ;
            ProcAwaken(you->context) ;
          }
          /*}}}*/
      }
      else
        /*{{{  add to queue*/
        {
          you->next = NULL ;
          pq->tail->next = you ;
          pq->tail = you ;
        }
        /*}}}*/
    }
    /*}}}*/
    
    /*{{{  PUBLIC void backofqueue (void)*/
    PUBLIC void backofqueue (void)
    {
      process me ;
      reschedule (&me) ;
      deschedule (&me) ;
    }
    /*}}}*/
    
    
    /* These two calls are peculiar to VCR and not really part of the interface */
    
    /*{{{  PUBLIC void deactivate (void)*/
    PUBLIC void deactivate (void)
    {
      pqueue *pq = (pqueue *) (* ((GLOBALS **)GLOBAL_PTR))->pqueue ;
      process *next = pq->head ;
    
      /*{{{  restart next process*/
      if (next == NULL)
        pq->tail = NULL ;
      else
        {
          pq->head = pq->head->next ;
          if (pq->head == NULL)
            pq->tail = active_p ;
          ProcAwaken(next->context) ;
        }
      /*}}}*/
    }
    /*}}}*/
     
    /*{{{  PUBLIC void reactivate (void)*/
    PUBLIC void reactivate (void)
    {
      pqueue *pq = (pqueue *) (* ((GLOBALS **)GLOBAL_PTR))->pqueue ;
      process me ;
    
      if (pq->head == NULL)
      {
        if (pq->tail == active_p)
          /*{{{  only item on queue*/
          {
            me.next  = NULL ;
            pq->head = &me ;
            pq->tail = &me ;
          }
          /*}}}*/
        else
          /*{{{  no current process, only occurs when leaving VCR via VPI*/
          {
            pq->tail  = active_p ;
            /*printf("Re-activating myself\n") ;*/
            return ;
          }
          /*}}}*/
      }
      else
        /*{{{  add to queue*/
        {
          me.next = NULL ;
          pq->tail->next = &me ;
          pq->tail = &me ;
        }
        /*}}}*/
    
      /*{{{  sleep unless top of queue*/
      {
        ProcDesc(&me.context) ;
        ProcSleep() ;
      }   
      /*}}}*/
    }
    /*}}}*/
    /*}}}*/
    /*{{{  binary syncs and channels*/
    PUBLIC channel idle_channel = {NULL, 0, NULL} ;
    
    /*{{{  PUBLIC void synchronize (sync *s)*/
    PUBLIC void synchronize (sync *s)
    {
      if (*s == NULL)
      {
        process me ;
        *s = &me ;
        deschedule(&me) ;
      }
      else
      {
        process *you = *s ;
        *s = NULL ;
        reschedule(you) ;
      }
    }
    /*}}}*/
    
    /*{{{  PUBLIC void output (channel *c, void *message, int length)*/
    PUBLIC void output (channel *c, void *message, int length)
    {
      if (c->id == NULL)
      {
        process me ;
        c->id = &me ;
        c->message = message ;
        c->length = length ;
        deschedule (&me) ;
      }
      else
      {
        process *you = c->id ;      
        c->id = NULL ;
        memcpy (c->message, message, length) ;
        reschedule(you) ;
      }
    }
    /*}}}*/
    
    /*{{{  PUBLIC void input  (channel *c, void *message, int length)*/
    PUBLIC void input  (channel *c, void *message, int length)
    {
      if (c->id == NULL)
      {
        process me ;
        c->id = &me ;
        c->message = message ;
        c->length = length ;
        deschedule (&me) ;
      }
      else
      {
        process *you = c->id ;
        c->id = NULL ;
        memcpy (message, c->message, length) ;
        reschedule(you) ;
      }
    }
    /*}}}*/
    /*}}}*/
    /*{{{  semaphores*/
    /*{{{  PUBLIC void sema_init (semaphore *sema, int value)*/
    PUBLIC void sema_init (semaphore *sema, int value)
    {
      sema->head = NULL ;
      sema->tail = NULL ;
      sema->count = value ;
    }
    /*}}}*/
    
    /*{{{  PUBLIC void sema_signal (semaphore *sema)*/
    PUBLIC void sema_signal (semaphore *sema)
    {
      if (sema->head != NULL)
        /*{{{  signal blocked process*/
        {
          process *next = sema->head->next ;
          reschedule (sema->head) ;
          sema->head = next ;
        }
        /*}}}*/
      else
        sema->count++ ;
    }
    /*}}}*/
    
    /*{{{  PUBLIC void sema_wait (semaphore *sema)*/
    PUBLIC void sema_wait (semaphore *sema)
    {
      if (sema->count == 0)
        /*{{{  block until signalled*/
        {
          process me ;
        
          if (sema->head == NULL)
          {
            sema->head = &me ;
            sema->tail = &me ;
          }
          else
          {
            sema->tail->next = &me ;
            sema->tail = &me ;
          }
          deschedule (&me) ;
        }
        /*}}}*/
      else
        sema->count-- ;
    }
    /*}}}*/
    /*}}}*/
    /*{{{  process masking*/
    /*{{{  PUBLIC void BEGIN_MASK (mstate *ms, mask_processes mp)*/
    PUBLIC void BEGIN_MASK (mstate *ms, mask_processes mp)
    {
      #ifdef USER_CODE_AT_LOW_PRI
      if (mp == comm_processes || mp == all_processes)
      {
        ProcGetPRI(&ms->priority) ;
    
        if (ms->priority==PROC_LOW)
          ProcToHI() ;
      }
      #endif
    }
    /*}}}*/
    
    /*{{{  PUBLIC void END_MASK (mstate *ms, mask_processes mp)*/
    PUBLIC void END_MASK (mstate *ms, mask_processes mp)
    {
      #ifdef USER_CODE_AT_LOW_PRI
      if (mp == comm_processes || mp == all_processes)
        if (ms->priority==PROC_LOW)
          ProcToLO() ;
      #endif
    }
    /*}}}*/
     
    /*}}}*/
  #endif
#else
  #error Architecture not implemented yet
#endif
