/*{{{  File banner*/
/*@(#)=====================================================*\
||@(#)  Project : PUMA ESPRIT P2701
||@(#)  Authors : Mark Debbage and Mark Hill
||@(#)            University of Southampton
||
||@(#)    Title : VCR server functions
||@(#)   System : VCR
||@(#) Filename : vserver.c
||@(#)  Version : 2.6
\*@(#)====================================================*/

/*}}}*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "vcr.h"
#include "vchan.h"
#include "vcrctrl.h"
#include "vparse.h"
#include "vmove.h"
#include "vrpc.h"

static char *_FILE_ = __FILE__ ;

/*{{{  local definitions*/
#define call_occam_safety 10
/*}}}*/

/*{{{  patch resolution definitions*/
/* Patch size in bytes, compiler must allow at least this much patch */
#define patch_size  6
#define nibble_size 4
#define nibble_mask ((1 << nibble_size) - 1)
#define patch_range (1 << (patch_size*nibble_size-1))

static int pos_patch_j [patch_size] = { 0x20, 0x20, 0x20, 0x20, 0x20, 0x00 } ;
static int neg_patch_j [patch_size] = { 0x6F, 0x20, 0x20, 0x20, 0x20, 0x00 } ;
/*}}}*/

/*{{{  PRIVATE void PatchJump (char *hole, int jump)*/
PRIVATE void PatchJump (char *hole, int jump)
{
  int  nibble, i;

  if (jump >= patch_range || jump <= -patch_range)
    Exception (UPR_error,_FILE_,__LINE__,"Jump too large for 6 byte patch") ;

  /* Patch is right justified to make offset calculations easy */

  if (jump >= 0)
    /*{{{  positive jump*/
    {  
      for (i=patch_size-1;i>=0;i--)
      {
        nibble  = jump & nibble_mask ;
    
        hole[i] = pos_patch_j[i] ^ nibble ;
    
        #ifdef UPR_DEBUG
        {
          char s[80] ;
          sprintf(s, "Jump %x : loc %d, patch %x",jump,i,hole[i]) ;
          Exception(UPR_warning,_FILE_,__LINE__,s) ;
        }
        #endif
    
        jump    = jump >> nibble_size ;
      }
    }
    /*}}}*/
  else
    /*{{{  negative jump*/
    {  
      for (i=patch_size-1;i>=0;i--)
      {
        nibble  = jump & nibble_mask ;
    
        hole[i] = neg_patch_j[i] ^ nibble ;
    
        #ifdef UPR_DEBUG
        {
          char s[80] ;
          sprintf(s, "Jump %x : loc %d, patch %x",jump,i,hole[i]) ;
          Exception(UPR_warning,_FILE_,__LINE__,s) ;
        }
        #endif
    
        jump    = jump >> nibble_size ;
      }
    }
    /*}}}*/
}
/*}}}*/
/*{{{  PRIVATE void CallStaticOccam*/
PRIVATE void CallStaticOccam (Process *p, void (*occam)() , Channel **chan_ins, Channel **chan_outs, int *vecspace)
{
  if (vecspace == NULL)
    occam (chan_ins, chan_outs) ;
  else
    occam (chan_ins, chan_outs, vecspace) ;
} 
/*}}}*/
/*{{{  PRIVATE void ConditionErrMode (ERR_MODE new)*/
PRIVATE void ConditionErrMode (ERR_MODE new)
{
  ERR_MODE *current = &vcr_globals.err_mode ;
  
  if (*current != new)
  {
    if (*current == universal)
    {
      if (new == halt)
      {
        *current = halt ;
        SetHaltOnErr() ;
      }
      else
      {
        *current = stop ;
        ClrHaltOnErr() ;
      }
    }
    else
      if (new != universal)
        Exception(UPR_warning,_FILE_,__LINE__,"User code error modes mismatch") ;
  }
}
/*}}}*/

/*{{{  PRIVATE void Server (Process *p, VCB *vcb_in, ... )*/
PRIVATE void Server (Process *p, VCB *vcb_in, VCB *vcb_out, SVR_MODE svr_mode)
{
  p=p ;
  
  switch (svr_mode)
  {
    case svr_pproc :
      /*{{{  physical processor load server*/
      {
        int num_ports, max_ports_per_vproc, base, num_vprocs, i, j, result, sync ;
        PORT  *ports ;
        PORT_ADDR *vca ;
        VTHREAD *vthreads ;
        VPROC vproc ;
        int info_size;
        char *info_base, *info_src_base;
      
        if (vcr_globals.proc_id!=0)
        {
          VirtualIn (vcb_in, &vcr_globals.num_devices, sizeof(int)) ;
          VirtualIn (vcb_in, &vcr_globals.num_devices_wow, sizeof(int)) ;
          if (vcr_globals.num_devices!=0)
            /*{{{  obtain device information*/
            {
              if ((vcr_globals.devices=(DEVICE*) malloc(sizeof(DEVICE)*vcr_globals.num_devices))==NULL)
                Exception(UPR_error,_FILE_,__LINE__,"Malloc of device struct array failed") ;
              VirtualIn (vcb_in, vcr_globals.devices, sizeof(DEVICE)*vcr_globals.num_devices) ;
              VirtualIn (vcb_in, &info_size,sizeof(int));
              if ((info_base=malloc(info_size))==NULL)
                Exception(UPR_error,_FILE_,__LINE__,"Malloc of Device info buffer failed");
              VirtualIn (vcb_in, info_base, info_size);
              VirtualIn (vcb_in, &info_src_base, sizeof(int));
              {
                int offset = info_base-info_src_base;
                int i;
                for (i=0;i<vcr_globals.num_devices;i++)
                  vcr_globals.devices[i].info += offset ;
              }
            } 
            /*}}}*/
        }
        
        VirtualIn (vcb_in, &num_vprocs, sizeof(int)) ;
        if (num_vprocs>0)
        {  
          /*{{{  startup transaction*/
          VirtualIn (vcb_in, &num_ports,  sizeof(int)) ;
          VirtualIn (vcb_in, &max_ports_per_vproc, sizeof(int)) ;
          
          if ((vthreads = (VTHREAD *) malloc (num_vprocs*sizeof(VTHREAD))) == NULL)
            Exception (UPR_error, _FILE_, __LINE__, "Out of heap space whilst allocating virtual processors") ;
          
          if (num_ports == 0)
            ports = NULL ;
          else
          {
            if ((ports = (PORT *) malloc (num_ports*sizeof(PORT))) == NULL)
              Exception (UPR_error, _FILE_, __LINE__, "Out of heap space whilst allocating ports") ;
          
            if ((base = MultiplePortGrab (ports, num_ports)) < 0)
              Exception (UPR_error, _FILE_, __LINE__, "Out of ports") ;
          }
          
          if (max_ports_per_vproc == 0)
            vca = NULL ;
          else
            if ((vca = (PORT_ADDR *) malloc (max_ports_per_vproc*sizeof(PORT_ADDR))) == NULL)
              Exception (UPR_error, _FILE_, __LINE__, "Out of heap space whilst allocating vca buffer") ;
          
          VirtualOut (vcb_out, &base, sizeof(int)) ;
          /*}}}*/
          /*{{{  receive virtual processors*/
          for (i=0;i<num_vprocs;i++)
          {
            VTHREAD *vthread = &vthreads[i] ;
            VCB *vcb_ins ;
            VCB *vcb_outs ;
            BYTE (*buffer)[UPR_MAX_PKT_SIZE] ; /* vcb_in packet buffering */
            
            /*{{{  receive code*/
            vthread->scdb = RPC_ReadCode(vcb_in) ;
            ConditionErrMode (vthread->scdb->err_mode) ;
            /*}}}*/
            /*{{{  resolve any patches*/
            {
              SCDB *scdb = vthread->scdb ;
              PDB *patch = scdb->patch ;
              int minus_one = -1 ;
            
              while (patch != NULL)
              {
                int len = strlen(patch->filename)+1 ;
                SCDB *called_scdb = RPC_FindCode(patch->filename) ;
                char *hole ;
                int  jump ;
            
                if (called_scdb == NULL)
                {
                  if (vcr_globals.proc_id == UPR_root)
                    called_scdb = RPC_LoadCode (patch->filename) ;
                  else
                    /*{{{  request code from root*/
                    {
                      VirtualOut (vcb_out, &len, sizeof(int)) ;
                      VirtualOut (vcb_out, patch->filename, len) ;
                      
                      called_scdb = RPC_ReadCode (vcb_in) ;
                      
                      RPC_StackCode (called_scdb) ;
                    }
                    /*}}}*/
                }
                ConditionErrMode (called_scdb->err_mode) ;
            
                if (called_scdb->patch != NULL)
                  Exception (UPR_error,_FILE_,__LINE__,"Cannot call an unresolved unit") ;
            
                hole = &scdb->code[patch->offset] ;
                jump = (called_scdb->code + called_scdb->ep_offset) -  (hole + patch_size) ;
            
                /*{{{  debug*/
                #ifdef UPR_DEBUG
                {
                  char s[80] ;
                  sprintf(s, "Patch at %x, scdb at %x : Jump of %x",(int) hole, (int) (called_scdb->code + called_scdb->ep_offset), jump) ;
                  Exception(UPR_warning,_FILE_,__LINE__,s) ;
                }
                #endif
                /*}}}*/
                
                PatchJump (hole, jump) ;
                
                patch = patch->next ;
              }
            
              VirtualOut (vcb_out, &minus_one, sizeof(int)) ;  /* request terminator */
            }
            /*}}}*/
            /*{{{  receive virtual processor structure*/
            VirtualIn (vcb_in, &vproc, sizeof(VPROC)) ;
            /*}}}*/
            /*{{{  allocate channels*/
            if (vproc.num_in_chans == 0)
            {
              vcb_ins = NULL ;
              vthread->chan_ins = NULL ;
            }
            else
            {
              if (((vcb_ins = (VCB *) malloc (vproc.num_in_chans*(sizeof(VCB)+UPR_MAX_PKT_SIZE))) == NULL) ||
                  ((vthread->chan_ins = (Channel **) malloc (vproc.num_in_chans*sizeof(Channel *))) == NULL))
                Exception (UPR_error, _FILE_, __LINE__, "Out of heap space whilst allocating virtual channels") ;
            
              buffer = (BYTE (*)[UPR_MAX_PKT_SIZE]) (vcb_ins + vproc.num_in_chans) ;
            }
            
            if (vproc.num_out_chans == 0)
            {
              vcb_outs = NULL ;
              vthread->chan_outs = NULL ;
            }
            else
            {
              if (((vcb_outs = (VCB *) malloc (vproc.num_out_chans*sizeof(VCB))) == NULL) ||
                  ((vthread->chan_outs = (Channel **) malloc (vproc.num_out_chans*sizeof(Channel *))) == NULL))
              Exception (UPR_error, _FILE_, __LINE__, "Out of heap space whilst allocating virtual channels") ;
            }
              
            if ((vproc.num_in_chans + vproc.num_out_chans) > max_ports_per_vproc)
              Exception (UPR_error, _FILE_, __LINE__, "Maximum ports bound exceeded") ;
            /*}}}*/
            /*{{{  receive virtual channel allocation information*/
            VirtualIn (vcb_in, vca, (vproc.num_in_chans + vproc.num_out_chans) * sizeof(PORT_ADDR)) ;
            /*}}}*/
            /*{{{  initialise in virtual channels*/
            for (j=0;j<vproc.num_in_chans;j++)  /* input channel end number */
            {
              int ap = vproc.base + j ;         /* absolute port number */
              int rp = ap - base ;              /* relative port number to pproc base */
              
              vthread->chan_ins[j] = (Channel *) (((int) &vcb_ins[j]) ^ virtual_bit) ;
            
              PortInit (&ports[rp], VCR_InPreAction, TRUE, VCR_InPostAction, FALSE, (void *) &vcb_ins[j]) ;
              VCB_Init (&vcb_ins[j], &ports[rp], buffer[j], NULL, vca[j].proc, vca[j].port) ;
            }
            /*}}}*/
            /*{{{  initialise out virtual channels*/
            for (j=0;j<vproc.num_out_chans;j++) /* output channel end number */
            {
              int k  = vproc.num_in_chans + j ; /* absolute channel end number */
              int ap = vproc.base + k ;         /* absolute port number */
              int rp = ap - base ;              /* relative port number to pproc base */
            
              vthread->chan_outs[j] = (Channel *) (((int) &vcb_outs[j]) ^ virtual_bit) ;
              
              PortInit (&ports[rp], VCR_OutPreAction, TRUE, VCR_OutPostAction, FALSE, (void *) &vcb_outs[j]) ;                  
              VCB_Init (&vcb_outs[j], &ports[rp], NULL, NULL, vca[k].proc, vca[k].port) ;
            }  
            /*}}}*/
            /*{{{  allocate any vecspace*/
            if (vthread->scdb->vsp_size == 0)
              vthread->vecspace = NULL ;
            else
              if ((vthread->vecspace = (int *) malloc (vthread->scdb->vsp_size * sizeof(int))) == NULL)
                Exception (UPR_error, _FILE_, __LINE__, "Out of heap space whilst allocating vecspace") ;
            /*}}}*/
            /*{{{  set up virtual processor calling thread*/
            if ((vthread->p=ProcAlloc (CallStaticOccam,
                 (vthread->scdb->wsp_size+call_occam_safety)*sizeof(int), 4,
                 (void (*)()) (vthread->scdb->code + vthread->scdb->ep_offset),
                 vthread->chan_ins, vthread->chan_outs, vthread->vecspace)) == NULL)
              Exception(UPR_error,_FILE_,__LINE__,"Out of heap space whilst allocating virtual processor thread") ;
            /*}}}*/
            /*{{{  synchronize with loader*/
            VirtualOut (vcb_out, &sync, sizeof(int)) ;
            /*}}}*/
          }    
          /*}}}*/
          /*{{{  synchronize*/
          VirtualIn (vcb_in, &sync, sizeof(int)) ;
          /*}}}*/
          /*{{{  launch virtual processors*/
          {
            Process **vps ;
          
            if ((vps = (Process **) malloc ((num_vprocs+1)*sizeof(Process *))) == NULL)
              Exception (UPR_error, _FILE_, __LINE__, "Out of heap space whilst allocating virtual processors") ;
          
            for (i=0;i<num_vprocs;i++)
              vps[i] = vthreads[i].p ;
          
            vps[num_vprocs] = NULL ;
          
            ProcParList (vps) ;
          
            free(vps) ;
          }
          /*}}}*/
          /*{{{  clean up*/
          MultiplePortRelease (ports, num_ports) ;
          
          for (i=0;i<num_vprocs;i++)
          {
            VTHREAD *vthread = &vthreads[i] ;
          
            ProcAllocClean (vthread->p) ;
            free (vthread->scdb) ;
          
            if (vthread->chan_ins != NULL)
            {
              free ((void *) (((int) vthread->chan_ins[0]) ^ virtual_bit)) ;
              free (vthread->chan_ins) ;
            }
          
            if (vthread->chan_outs != NULL)
            {
              free ((void *) (((int) vthread->chan_outs[0]) ^ virtual_bit)) ;
              free (vthread->chan_outs) ;
            }
          
            if (vthread->vecspace != NULL)
              free (vthread->vecspace) ;
          }
            
          free (vca) ;
          free (vthreads) ;
          free (ports) ;
          
          result = 0 ;
          VirtualOut (vcb_out, &result, sizeof(int)) ;
          /*}}}*/
        }
        break ;
      }
      /*}}}*/
      
    case svr_rpc :
      /*{{{  remote procedure call server*/
      {
        int sizes[2] ;
        char *filename ;
        PBLOCK *tpblock, *pblock ;
        SCDB *scdb ;
        int  total_size ; /* in bytes */
        int  *ws, *pb, *vs ;
      
        /*{{{  receive sizes and filename*/
        VirtualIn (vcb_in, sizes, 2*sizeof(int)) ;
        
        if ((filename = malloc(sizes[0])) == NULL)
          Exception(UPR_error,_FILE_,__LINE__,"Out of heap whilst allocating RPC filename") ;
          
        VirtualIn (vcb_in, filename, sizes[0]) ;
        /*}}}*/
        /*{{{  get code*/
        scdb = RPC_LoadCode (filename) ;
        
        if (scdb->patch != NULL)
          Exception(UPR_error,_FILE_,__LINE__,"Cannot RPC unresolved code") ;
        
        ConditionErrMode (scdb->err_mode) ;
        /*}}}*/
        /*{{{  allocate and segregate thread's workspace*/
        {
          int ws_bytes = max (sizeof(PBLOCK),(scdb->wsp_size+iptr_size)*sizeof(int)) ;
        
          total_size = ws_bytes + sizes[1] + sizeof(PBLOCK) + scdb->vsp_size*sizeof(int) ;
        
          if ((ws = (int *) malloc (total_size)) == NULL)
            Exception(UPR_error,_FILE_,__LINE__,"Out of heap whilst allocating RPC workspace") ;
          
          pb = (int *) ((char *) ws + ws_bytes) ;
          tpblock = ((PBLOCK *) pb) - 1 ;
          pblock  = (PBLOCK *) (((char *) pb) + sizes[1]) ;
          vs = (int *) (pblock + 1) ;
        }
        /*}}}*/
        /*{{{  receive parameter block*/
        VirtualIn (vcb_in, tpblock, sizeof(PBLOCK)+sizes[1]) ;
        AsmMove (tpblock,pblock,sizeof(PBLOCK)) ;
        pblock->pb = pb ;
        /*}}}*/
        
        DynamicOccam (pblock, vs, scdb) ;
        
        /*{{{  return channels*/
        {
          int i ;
        
          if (pblock->move == explicit_move)
            for (i=pblock->field[chans_f].start;i<pblock->field[chans_f].current;i++)
              DVC_ReturnChannel ((VCB *) (pb[i] & virtual_mask)) ;
          else
            if (pblock->move == implicit_move)
              for (i=pblock->field[chans_f].start;i<pblock->field[chans_f].current;i+=couple)
                DVC_ReturnChannel ((VCB *) (pb[pb[i+1]] & virtual_mask)) ;    
        }
        /*}}}*/
        /*{{{  return var data field*/
        VirtualOut (vcb_out, &pb[pblock->field[vardata_f].start],
         (pblock->field[vardata_f].current - pblock->field[vardata_f].start) *sizeof(int)) ;
        /*}}}*/
        /*{{{  clean up*/
        free (filename) ;
        free (ws) ;
        /*}}}*/
      
        break ;
      }
      /*}}}*/
  
    case load_code :
      /*{{{  serve load code request*/
      {
        char *file ;
        SCDB *scdb ;
        int  len ;
      
        VirtualIn (vcb_in, &len, sizeof(int)) ;   /* length of filename */
      
        if ((file=malloc(len))==NULL)
          Exception(UPR_error,_FILE_,__LINE__,"Malloc for filename failed") ;
          
        VirtualIn (vcb_in, (void *) file, len) ;  /* filename */
      
        scdb=RPC_CodeRequest(file) ;
      
        RPC_SendCode(vcb_out, scdb) ;
        
        break ;
      }  
      /*}}}*/
  
    case svr_test :
      /*{{{  server test request*/
      #define max_msg_size  400
      #define loops         100
      
      {
        BYTE msg[max_msg_size] ;
        int len, count ;
      
        while (TRUE)
        {
          for (len=10;len<=max_msg_size;len+=10)
          {
            for (count=0;count<loops;count++)
            {
              VirtualIn (vcb_in, msg, len) ;
              VirtualOut(vcb_out, msg, len) ;
            }
          }
        }
      }
      /*}}}*/
  
    default :
      Exception(UPR_error,_FILE_,__LINE__,"Unknown server mode") ;
  }      

  VCB_Free (vcb_in) ;
  VCB_Free (vcb_out) ;
  VCR_ProcessDone(p) ;
}

/*}}}*/

/*{{{  PUBLIC void SVR_Server (Process *p, VCR_CTRL_PKT ctrl_pkt)*/
PUBLIC void SVR_Server (Process *p, VCR_CTRL_PKT ctrl_pkt)
{
  VCB *vcb_in ;
  VCB *vcb_out ;
  ORB orb ;
  SVR_MODE svr_mode ;

  /*{{{  allocate channel ends and reply*/
  {
    SVR_REQUEST *svr_request = &ctrl_pkt.p.svr_request ;
  
    vcb_in  = VCB_InAlloc  (svr_request->src_proc_id, svr_request->src.out) ;
    vcb_out = VCB_OutAlloc (svr_request->src_proc_id, svr_request->src.in) ;
    svr_mode = svr_request->svr_mode ;
  
    orb.bffr = (BYTE *) &ctrl_pkt ;
    HdrBuild (svr_request->src_proc_id, svr_request->src_port_id, sizeof(SVR_REPLY)+ctrl_pkt_ovrhd, &orb.header) ;
  }
  
  ctrl_pkt.tag = svr_reply ;
  {
    SVR_REPLY *svr_reply = &ctrl_pkt.p.svr_reply ;
    svr_reply->dst.in    = vcb_in->port->id ;
    svr_reply->dst.out   = vcb_out->port->id ;
  }
  
  UPR_Enqueue_BK (&orb) ;
  /*}}}*/

  Server (p, vcb_in, vcb_out, svr_mode) ;
}
/*}}}*/

/*{{{  PUBLIC void SVR_PreAction (PORT *port, int length)*/
PUBLIC void SVR_PreAction (PORT *port, int length)
{
  SVR_STATE *svr_state = (SVR_STATE *) port->state ;

  port->buffer = (BYTE *) svr_state->ctrl_pkt ;
  port->space  = sizeof (SVR_REPLY) + ctrl_pkt_ovrhd ;
}
/*}}}*/
/*{{{  PUBLIC void SVR_PostAction (PORT *port, int length, BYTE *buffer)*/
PUBLIC void SVR_PostAction (PORT *port, int length, BYTE *buffer)
{
  SVR_STATE *svr_state = (SVR_STATE *) port->state ;

  if (svr_state->ctrl_pkt->tag != svr_reply)
    Exception(UPR_error, _FILE_, __LINE__, "Illegal tag on VCR server port") ;

  if (svr_state->restart_id == NULL)
    Exception(UPR_error, _FILE_, __LINE__, "No client on server connect port") ;
  
  ProcAwaken(svr_state->restart_id) ;   
}
/*}}}*/

/*{{{  PUBLIC void SVR_Request (int destn_proc, DUPLEX_ID *src_id, ... )*/
PUBLIC void SVR_Request (int destn_proc, DUPLEX_ID *src_id, DUPLEX_ID *dst_id, SVR_MODE svr_mode)
{
  VCR_CTRL_PKT ctrl_pkt ;
  SVR_STATE svr_state ;
  PORT *port = PortAlloc (SVR_PreAction, TRUE, SVR_PostAction, TRUE, &svr_state) ;
  ORB orb ;
  int priority ;

  ProcGetPRI(&priority) ;
  if (priority==PROC_LOW) ProcToHI() ;

  /*{{{  critical code section*/
  ctrl_pkt.tag = svr_request ;
  {
    SVR_REQUEST *svr_request = &ctrl_pkt.p.svr_request ;
    svr_request->src_proc_id = vcr_globals.proc_id ;
    svr_request->src_port_id = port->id ;
    svr_request->src.in      = src_id->in ;
    svr_request->src.out     = src_id->out ;
    svr_request->svr_mode    = svr_mode ;
  }
  
  orb.bffr = (BYTE *) &ctrl_pkt ;
  HdrBuild (destn_proc, vcr_globals.vcr_ctrl_port_id, sizeof(SVR_REQUEST)+ctrl_pkt_ovrhd, &orb.header) ;
  
  UPR_Enqueue_NBK (&orb) ;
  
  svr_state.ctrl_pkt = &ctrl_pkt ;
  ProcDesc(&svr_state.restart_id) ;
  ProcSleep() ;
  
  PortFree(port) ;
  
  {
    SVR_REPLY *svr_reply = &ctrl_pkt.p.svr_reply ;
    dst_id->in  = svr_reply->dst.in ;
    dst_id->out = svr_reply->dst.out ;
  }
  /*}}}*/

  if (priority==PROC_LOW) ProcToLO() ;
} 
/*}}}*/
/*{{{  PUBLIC void VCR_LaunchServer (DUPLEX_VC *vc, int destn_proc, ... )*/
PUBLIC void VCR_LaunchServer (DUPLEX_VC *vc, int destn_proc, SVR_MODE svr_mode)
{
  if (vcr_globals.proc_id == destn_proc)
    /*{{{  launch local server*/
    {
      PORT *ports[4] ;
      DUPLEX_VC vc_svr ;
      Process *svr ;
      int i ;
    
      for (i=0;i<4;i++)
      {
        if ((ports[i] = (PORT *) malloc (sizeof(PORT))) == NULL)
          Exception (UPR_error, _FILE_, __LINE__, "Out of heap space whilst allocating ports") ;
          
        if ((ports[i]->id = PortGrab(ports[i])) < 0)
          Exception (UPR_error, _FILE_, __LINE__, "Out of ports") ;   
      }
    
      if (((vc->in     = (VCB *) malloc (sizeof(VCB)+UPR_MAX_PKT_SIZE)) == NULL) ||
          ((vc_svr.in  = (VCB *) malloc (sizeof(VCB)+UPR_MAX_PKT_SIZE)) == NULL) ||
          ((vc->out    = (VCB *) malloc (sizeof(VCB))) == NULL) ||
          ((vc_svr.out = (VCB *) malloc (sizeof(VCB))) == NULL))
        Exception (UPR_error, _FILE_, __LINE__, "Out of heap space whilst allocating virtual channels") ;
    
      PortInit (ports[0], VCR_InPreAction,  TRUE, VCR_InPostAction,  FALSE, (void *) vc->in) ;
      PortInit (ports[1], VCR_OutPreAction, TRUE, VCR_OutPostAction, FALSE, (void *) vc->out) ;
      PortInit (ports[2], VCR_InPreAction,  TRUE, VCR_InPostAction,  FALSE, (void *) vc_svr.in) ;
      PortInit (ports[3], VCR_OutPreAction, TRUE, VCR_OutPostAction, FALSE, (void *) vc_svr.out) ;
                      
      VCB_Init (vc->in,  ports[0], (BYTE *) (vc->in+1), NULL, destn_proc, ports[3]->id) ;
      VCB_Init (vc->out, ports[1], NULL, NULL, destn_proc, ports[2]->id) ;
      VCB_Init (vc_svr.in,  ports[2], (BYTE *) (vc_svr.in+1), NULL, destn_proc, ports[1]->id) ;
      VCB_Init (vc_svr.out, ports[3], NULL, NULL, destn_proc, ports[0]->id) ;
    
      if ((svr=ProcAlloc(Server,SVR_wspace_size,3,vc_svr.in, vc_svr.out, svr_mode)) == NULL)
        Exception(UPR_error,_FILE_,__LINE__,"Server allocation failed") ;
    
      ProcRunLow(svr) ;
    }
    /*}}}*/
  else
    /*{{{  launch remote server*/
    {  
      PORT *port_in = PortAlloc (VCR_InPreAction, TRUE, VCR_InPostAction, FALSE, NULL);
      PORT *port_out = PortAlloc (VCR_OutPreAction, TRUE, VCR_OutPostAction, FALSE, NULL) ;
    
      DUPLEX_ID src_id ;
      DUPLEX_ID dst_id ;
    
      src_id.in  = port_in->id ;
      src_id.out = port_out->id ;
    
      if ((vc->in = (VCB *) malloc (sizeof(VCB)+UPR_MAX_PKT_SIZE)) == NULL)
        Exception (UPR_error, _FILE_, __LINE__, "Out of heap space whilst allocating vcb") ;
    
      if ((vc->out = (VCB *) malloc (sizeof(VCB))) == NULL)
        Exception (UPR_error, _FILE_, __LINE__, "Out of heap space whilst allocating vcb") ;
      
      port_in->state = (void *) vc->in ;
      VCB_Init (vc->in, port_in, (BYTE *) (vc->in+1), NULL, destn_proc, 0) ;
    
      port_out->state = (void *) vc->out ;    
      VCB_Init (vc->out, port_out, NULL, NULL, destn_proc, 0) ;
    
      SVR_Request (destn_proc, &src_id, &dst_id, svr_mode) ;
    
      HdrBuild (destn_proc, dst_id.in, 0, &vc->out->orb.header) ;
      HdrBuild (destn_proc, dst_id.out, 0, &vc->in->orb.header) ;
    }
    /*}}}*/
}
/*}}}*/
