/*{{{  File banner*/
/*@(#)=====================================================*\
||@(#)  Project : PUMA ESPRIT P2701
||@(#)  Authors : Mark Debbage and Mark Hill
||@(#)            University of Southampton
||  
||@(#)    Title : Code parsing functions
||@(#)   System : VCR
||@(#) Filename : vparse.c
||@(#)  Version : 2.7
\*@(#)====================================================*/
/*}}}*/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "vcr.h"
#include "vchan.h"
#include "tcoff.h"
#include "vshow.h"

static char *_FILE_ = __FILE__ ;

/*{{{  compile-time switches*/
#define TRANSPUTER 
/*#define VIEW_PARSE*/
/*#define CODE_CHECK*/
/*}}}*/

/*{{{  PRIVATE  void adjust_scdb(SCDB *oldscdb, SCDB *scdb)*/
PRIVATE  void adjust_scdb(SCDB *oldscdb, SCDB *scdb)
{
  int offset = ((char *) scdb) - ((char *) oldscdb) ;
  PDB **patch ;

  scdb->filename += offset ;
  scdb->code     += offset ;
  scdb->symbol   += offset ;

  patch=&scdb->patch ;

  while (*patch != NULL)
  {
    *((char **) patch) += offset ;

    (*patch)->symbol += offset ;
    (*patch)->filename += offset ;
    patch = &((*patch)->next) ;
  }
}
/*}}}*/

#ifndef UPR_RD
/*{{{  PRIVATE int read_number(FILE *tcoff_in)*/
PRIVATE int read_number(FILE *tcoff_in)
{
  int length;
  int sign ;
  unsigned char prefix ;
  unsigned char res[sizeof(int)];
  int *result = (int*) res;
  int byte ;

  if (feof(tcoff_in))
    Exception(UPR_error,_FILE_,__LINE__,"Unexpected end of TCOFF file");
    
  prefix = fgetc(tcoff_in);
  /*{{{  check sign;*/
  if (prefix==SIGN_INDICATOR)
  {
    sign = -1;
    prefix = (int) fgetc(tcoff_in);
  }
  else
    sign = 1;
  /*}}}*/

  if (!feof(tcoff_in))
  {
    if (prefix>=PFX_1_NUMBER)
      /*{{{  read numbers larger than 250*/
      {
        if (feof(tcoff_in))
          Exception(UPR_error,_FILE_,__LINE__,"Unexpected end of TCOFF file");
        
        /*{{{  select prefix operator*/
        switch ((int) prefix)
        {
          case PFX_1_NUMBER :
            length = 1 ;
            break ;
          case PFX_2_NUMBER :
            length = 2;
            break ;
          case PFX_4_NUMBER :
            length = 4;
            break ;
          case PFX_8_NUMBER :
            length = 8;
            Exception(UPR_warning,_FILE_,__LINE__,"INT64 number in TCOFF file");
            break;
        
          case SIGN_INDICATOR :
            Exception(UPR_warning,_FILE_,__LINE__,"Double SIGN in TCOFF file");
            return (0);
        }
        /*}}}*/
      
        *result = 0;
        for(byte=0;byte<length;byte++)
        {
          unsigned char next = fgetc(tcoff_in) ;
          #ifdef TRANSPUTER
            res[byte] = next ;
          #else
            *result = *result + (((unsigned int) next)<<(8*byte)) ;
          #endif
        }
      }
      /*}}}*/
    else
      *result = (int) prefix ;
  }    
  return( ((sign==1)?(*result):~(*result)) );
}
/*}}}*/
/*{{{  PRIVATE void skip_block(FILE *tcoff_in,int size)*/
PRIVATE void skip_block(FILE *tcoff_in,int size)
{
  for (;size>0;size--)
  {
    if (feof(tcoff_in))
      Exception(UPR_error,_FILE_,__LINE__,"Unexpected end of TCOFF file");
    fgetc(tcoff_in);
  }
}
/*}}}*/
/*{{{  PRIVATE void read_block(FILE *tcoff_in, char* addr, int size)*/
PRIVATE void read_block(FILE *tcoff_in, char* addr, int size)
{
  int read = fread (addr, sizeof(char), size, tcoff_in) ;
  if (read != size)
    Exception(UPR_error,_FILE_,__LINE__,"Unexpected end of TCOFF file");
}
/*}}}*/

/*{{{  PRIVATE char* balloc(CBB *cbb, int amount)*/
PRIVATE char* balloc(CBB *cbb, int amount)
{
  char* block;

  if ((amount + cbb->index)>=cbb->size)
  {
    SCDB *oldscdb= (SCDB*) cbb->base;

    do
    {
      cbb->size *= 2;
    }
    while ((amount + cbb->index)>=cbb->size);

    if ((cbb->base=realloc(cbb->base,cbb->size))==NULL)
      Exception(UPR_error,_FILE_,__LINE__,"Code buffer realloc failed");
    adjust_scdb(oldscdb,(SCDB*)cbb->base);
  }
  
  block = &(cbb->base[cbb->index]);
  cbb->index += amount;
  return(block);
}  
/*}}}*/
/*{{{  PRIVATE char* balloc_on_int(CBB *cbb, int amount)*/
PRIVATE char* balloc_on_int(CBB *cbb, int amount)
{
  char* block;
  int   overlap = (cbb->index)%sizeof(int);
  
  if (overlap != 0)
    cbb->index += (sizeof(int)-overlap);

  if ((amount + cbb->index)>=cbb->size)
  {
    SCDB *oldscdb= (SCDB*) cbb->base;

    do
    {
      cbb->size *= 2;
    }
    while ((amount + cbb->index)>=cbb->size);

    if ((cbb->base=realloc(cbb->base,cbb->size))==NULL)
      Exception(UPR_error,_FILE_,__LINE__,"Code buffer realloc failed");
    adjust_scdb(oldscdb,(SCDB*)cbb->base);
  }
  
  block = &(cbb->base[cbb->index]);
  cbb->index += amount;

  return(block);
}  
/*}}}*/
/*{{{  PRIVATE SCDB* TCOFF_FindEntrySymbol(char *file)*/
PRIVATE SCDB* TCOFF_FindEntrySymbol(char *symbol)
{
  SCDB* scdb ;

  scdb = vcr_globals.scdb_stack;

  while (scdb != ((SCDB*) NULL))
  {
    if (strcmp(scdb->symbol,symbol)==0)
      return (scdb) ;
      
    scdb = scdb->next;
  }
  
  return( scdb );
}
/*}}}*/

/*{{{  PRIVATE char* balloc_string(FILE *tcoff_in, CBB *cbb)*/
PRIVATE char* balloc_string(FILE *tcoff_in, CBB *cbb)
{
  char* addr ;
  int size;
  
  size = read_number(tcoff_in);
  addr = balloc(cbb, size+1);
  read_block(tcoff_in,addr, size);
  addr[size] = 0 ;
  return(addr);
}
/*}}}*/
/*{{{  PRIVATE BOOL read_nstring(FILE *tcoff_in, char* addr, int limit)*/
/*
PRIVATE BOOL read_nstring(FILE *tcoff_in, char* addr, int limit)
{
  int size = read_number(tcoff_in);
      
  if (size>=limit)
  {
    read_block(tcoff_in,addr,limit);
    skip_block(tcoff_in,size-limit);
    addr[limit-1] = 0;
    return(FALSE);
  }
  else
  {
    read_block(tcoff_in,addr, size);
    addr[size]=0 ;
    return(TRUE);
  }
}
*/
/*}}}*/
/*{{{  PRIVATE int TCOFF_LoadSCDB(FILE *tcoff_in,CBB *cbb, char *file, SCDB** scdbout, BOOL save)*/
PRIVATE int TCOFF_LoadSCDB(FILE *tcoff_in,CBB *cbb, char *file, SCDB** scdbout, BOOL save)
{
  int tag;
  BOOL got_text    = FALSE;
  BOOL got_symbol  = FALSE;
  BOOL got_offset  = FALSE;
  BOOL got_wsandvs = FALSE;
  BOOL got_start   = FALSE;
  BOOL doing_patch = FALSE;
  BOOL reading_static = FALSE ;
  int patch_offset= -1;
  int tcoff_state = PARSE_PROCEEDING ;
  SCDB **scdb;

  scdb = (SCDB**) &cbb->base;
  cbb->index = 0;  

  (*scdb) = (SCDB*) balloc_on_int(cbb,sizeof(SCDB));
  (*scdb)->next   = (SCDB*) NULL ;
  (*scdb)->patch  = (PDB*) NULL;
  (*scdb)->symbol = NULL;
  (*scdb)->code_size = 0 ;
  (*scdb)->static_size = -1 ;
  
  if (file != NULL)
  {
    (*scdb)->filename = balloc(cbb,strlen(file)+1);
    strcpy((*scdb)->filename,file);
  }
  else
    (*scdb)->filename = NULL;

  do
  {
    int length;
  
    #ifdef VIEW_PARSE
    printf("%08x ",(int) ftell(tcoff_in));
    #endif
    
    tag = read_number(tcoff_in);
    length = read_number(tcoff_in);
    
    /*{{{  parse directives*/
    switch( tag )
    {
      case START_MODULE_TAG :
        /*{{{  record filename*/
        #ifdef VIEW_PARSE
          printf("START_MODULE_TAG\n");
        #endif
        {
          int size ;
          int attrib ;
        
          read_number(tcoff_in) ;
          attrib = read_number(tcoff_in) ;
          read_number(tcoff_in) ;
        
          if (attrib & ATTRIB_HALT)
            (*scdb)->err_mode = halt ;
          else
            if (attrib & ATTRIB_STOP)
              (*scdb)->err_mode = stop ;
            else
              (*scdb)->err_mode = universal ;
        
          size = read_number(tcoff_in) ;
          if (size!=0)
          {
            char* tcoff_filename ;
            char *dot_ptr = strchr(file,'.') ;
            int match ;
        
            if ((tcoff_filename=malloc( size+1 ))==NULL)
              /*{{{  malloc failed*/
              {
                Exception(UPR_warning,_FILE_,__LINE__,"Malloc for filename failed") ;
                tcoff_state=PARSE_FAILED;
              }
              /*}}}*/
            else
             /*{{{  read and check filename*/
             {    
               read_block(tcoff_in,tcoff_filename,size);
               tcoff_filename[size] = 0;
               
               if (dot_ptr == NULL)
                 match = strcmp (file, tcoff_filename) ;
               else
                 match = strncmp(file, tcoff_filename, dot_ptr-file) ;
               
               if (match != 0)
               {
                 Exception(UPR_warning,_FILE_,__LINE__,"Filename mismatch in TCOFF file") ;
                 tcoff_state=PARSE_FAILED;
               }
               free(tcoff_filename);
             }
             /*}}}*/
          }
        }
        break;
        /*}}}*/
    
      case ADJUST_POINT_TAG:
        /*{{{  adjust point*/
        {
          int patch_size;
          
          if ((read_number(tcoff_in)!=CO_VALUE_TAG)||(patch_offset>=0))
          {
            Exception(UPR_warning,_FILE_,__LINE__,"Adjust Point Failed");
            tcoff_state=PARSE_FAILED;
          }
          patch_size = read_number(tcoff_in);
          patch_offset = (*scdb)->code_size + patch_size;
        
        #ifdef VIEW_PARSE
        printf("ADJUST_POINT_TAG - code:%d patch:%d size:%d\n",(*scdb)->code_size,patch_offset,patch_size);
        #endif
        }
        break;
        /*}}}*/
    
      case LOAD_PREFIX_TAG:
        /*{{{  adjust patch offset*/
        #ifdef VIEW_PARSE
        printf("LOAD_PREFIX_TAG\n");
        #endif
        {
          if (doing_patch)
          {
            int skip;
        
            skip = (int) ftell(tcoff_in);
            (*scdb)->patch->offset = patch_offset;
            patch_offset += read_number(tcoff_in);
            skip = ((int) ftell(tcoff_in)) - skip;
            skip_block(tcoff_in,length-skip);
            doing_patch = FALSE;
          }
          else
          {
            Exception(UPR_warning,_FILE_,__LINE__,"LOAD PREFIX while not patching");
            tcoff_state = PARSE_FAILED;
          }
        }
        break;
        /*}}}*/
    
      case LOAD_TEXT_TAG :
        /*{{{  malloc (*scdb) and load code*/
        {
          int code_length = read_number(tcoff_in) ;
        
          #ifdef VIEW_PARSE
            printf("LOAD_TEXT_TAG\n");
          #endif
        
          
          /*{{{  check only one code block*/
           if (got_text)
          {
            Exception(UPR_warning,_FILE_,__LINE__,"More TEXT after TEXT");
            skip_block(tcoff_in,code_length);
            tcoff_state = PARSE_FAILED;
            break;
          }      
          /*}}}*/
         
          (*scdb)->code_size = code_length ;
          (*scdb)->code = balloc_on_int(cbb,code_length);
          got_text = TRUE;
          read_block(tcoff_in,(*scdb)->code,code_length) ;
          break;
        }  
        /*}}}*/
    
      case SPECIFIC_SYMBOL_TAG :
        /*{{{  as per symbol tag*/
        #ifdef VIEW_PARSE
        printf("SPECIFIC_");
        #endif
        /*}}}*/
    
      case SYMBOL_TAG :
        /*{{{  read entry point name if a symbol*/
        #ifdef VIEW_PARSE
        printf("SYMBOL_TAG - Length %d\n",length);
        #endif
        if ((!got_symbol)&&(got_text))
        {
          int set;
          set = read_number(tcoff_in);
          if (set & EXPORT_USAGE)
            /*{{{  entry point name*/
            {
              (*scdb)->symbol=balloc_string(tcoff_in,cbb);
              got_symbol=TRUE;
            }
            /*}}}*/
          else if (set & ORIGIN_USAGE)
          {
            if ((doing_patch)||(patch_offset<0))
              /*{{{  error*/
              {
                Exception(UPR_warning,_FILE_,__LINE__,"SYMBOL arrived while Patching");
                tcoff_state = PARSE_FAILED;      
              }    
              /*}}}*/
            else
              /*{{{  create pdb*/
              {
                PDB* pdb;
                char *colon ;
              
                doing_patch = TRUE;
                pdb = (PDB*) balloc_on_int(cbb,sizeof(PDB));
                pdb->next = (*scdb)->patch;
                (*scdb)->patch = pdb;
                (*scdb)->patch->offset = 0;
                (*scdb)->patch->filename = balloc_string(tcoff_in,cbb);
              
                if ((colon = strchr((*scdb)->patch->filename,':')) != NULL)
                  *colon = 0;
              }
              /*}}}*/
          }
          else if (set & IMPORT_USAGE)
            /*{{{  add symbol to pdb*/
            if (patch_offset<0)
              /*{{{  error*/
              {
                Exception(UPR_warning,_FILE_,__LINE__,"SYMBOL Import received while not Patching");
                tcoff_state = PARSE_FAILED;      
              }    
              /*}}}*/
            else if (doing_patch)
              (*scdb)->patch->symbol = balloc_string(tcoff_in,cbb);
            else
              /*{{{  patch when SC already imported*/
              {
                SCDB *patch_scdb ;
                
                doing_patch = TRUE;
                {
                  PDB* pdb;
                  pdb = (PDB*) balloc_on_int(cbb,sizeof(PDB));
                  pdb->next = (*scdb)->patch;
                  (*scdb)->patch = pdb;
                }
                (*scdb)->patch->offset = 0;
                (*scdb)->patch->symbol = balloc_string(tcoff_in,cbb);
                patch_scdb = TCOFF_FindEntrySymbol((*scdb)->patch->symbol);
                if (patch_scdb==NULL)
                  /*{{{  Error- could not find SCDB*/
                  {
                    char s[100];
                    sprintf(s,"Could not find entry symbol %s in SCDBs",(*scdb)->patch->symbol);
                    Exception(UPR_warning,_FILE_,__LINE__,s);
                    tcoff_state=PARSE_FAILED;
                  }
                  /*}}}*/
                (*scdb)->patch->filename = balloc(cbb,strlen(patch_scdb->filename)+1);
                strcpy((*scdb)->patch->filename,patch_scdb->filename);
              }
              /*}}}*/
              
            /*}}}*/
          else
            /*{{{  skip*/
            {
              skip_block(tcoff_in,read_number(tcoff_in));
            }
            /*}}}*/
          
          if (tag==SPECIFIC_SYMBOL_TAG) read_number(tcoff_in);
        }
        else
        {
          int set;
          set = read_number(tcoff_in);
        
          if (set & LOCAL_USAGE)
          {
            int size   = read_number(tcoff_in);
            char *addr = malloc(size+1);
            if (addr==NULL)
              Exception(UPR_error,_FILE_,__LINE__,"Out of memory");
            read_block(tcoff_in, addr, size);
            addr[size] = 0 ;
        
            if (strcmp("static%base",addr)==0)
            {
              if ((*scdb)->static_size < 0)
                reading_static = TRUE ;
              else
              {
                Exception(UPR_error,_FILE_,__LINE__,"Expecting only one static size");
                tcoff_state = PARSE_FAILED;
              }
            }
            free(addr) ;
          }
          else
            /*{{{  skip*/
            {
              skip_block(tcoff_in,read_number(tcoff_in));
            }
            /*}}}*/
        
          if (tag==SPECIFIC_SYMBOL_TAG) read_number(tcoff_in);
        }  
        break;
        /*}}}*/
    
      case DEFINE_SYMBOL_TAG :
        /*{{{  read entry point offset into code*/
        #ifdef VIEW_PARSE
          printf("DEFINE_SYMBOL_TAG\n");
        #endif
        if ((!got_offset)&&(got_text))
        {
          int token;
          read_number(tcoff_in);
          if ((token=read_number(tcoff_in))==PLUS_OP)
            /*{{{  offset expression*/
            {
              /*{{{  check for symbol value   SV:2*/
              if (read_number(tcoff_in)!=SV_VALUE_TAG)
              {
                Exception(UPR_warning,_FILE_,__LINE__,"Expecting symbol value");
                tcoff_state=PARSE_FAILED;
                break;
              }
              else
                read_number(tcoff_in);
                
              /*}}}*/
              /*{{{  check for and read offset*/
              if (read_number(tcoff_in)!=CO_VALUE_TAG)
              {
                Exception(UPR_warning,_FILE_,__LINE__,"Expecting constant");
                tcoff_state=PARSE_FAILED;
                break;
              }
              else
              {
                (*scdb)->ep_offset = read_number(tcoff_in);
                got_offset = TRUE;
              }
              /*}}}*/
            }
            /*}}}*/
          else if (token==CO_VALUE_TAG)
            /*{{{  stated offset*/
            {
              (*scdb)->ep_offset = read_number(tcoff_in);
              got_offset = TRUE;
            }
            /*}}}*/
          else
            /*{{{  error*/
            {
              Exception(UPR_warning,_FILE_,__LINE__,"Expecting plus operand");
              tcoff_state=PARSE_FAILED;
              break;
            }
            /*}}}*/
        }
        else if (reading_static)
        {
          int token;
          read_number(tcoff_in);
        
          if ((token=read_number(tcoff_in))==CO_VALUE_TAG)
            /*{{{  stated offset*/
            {
              (*scdb)->static_size = read_number(tcoff_in);
              reading_static = FALSE ;
            }
            /*}}}*/
          else
            /*{{{  error*/
            {
              Exception(UPR_warning,_FILE_,__LINE__,"Expecting constant static size");
              tcoff_state=PARSE_FAILED;
              break;
            }
            /*}}}*/
        }
        else
          skip_block(tcoff_in,length);
        break;
        /*}}}*/
    
      case DESCRIPTOR_TAG :
        /*{{{  record wsp and vsp requirements*/
        #ifdef VIEW_PARSE
          printf("DESCRIPTOR_TAG\n");
        #endif
        if ((!got_wsandvs)&&(got_start)&&(got_text))
        {
          int mark;
          int size;
          
          read_number(tcoff_in);
          /*{{{  check for OCCAM*/
          {
            int lang;
            lang = read_number(tcoff_in);
            if ((lang!=LANG_OCCAM)&&(lang!=LANG_OCCAM_HARNESS))
            {
              Exception (UPR_warning,_FILE_,__LINE__,"Expecting OCCAM");
              tcoff_state=PARSE_FAILED;
              break;
            }
          }
          /*}}}*/
          size = read_number(tcoff_in);
          mark = (int) ftell(tcoff_in) ;
          (*scdb)->wsp_size = read_number(tcoff_in);
          (*scdb)->vsp_size = read_number(tcoff_in);
          skip_block(tcoff_in,(mark+size)-((int) ftell(tcoff_in)));
          got_wsandvs = TRUE;
        }  
        else
        {
          skip_block(tcoff_in,length);
        }
        break;
        /*}}}*/
        
      case SECTION_TAG :
        /*{{{  record endpoint*/
        #ifdef VIEW_PARSE
          printf("SECTION_TAG\n");
        #endif
        {
          int token = read_number(tcoff_in);
          read_number(tcoff_in);
          skip_block(tcoff_in,read_number(tcoff_in));
        
          if ((token & EXECUTE_SECTION)==0) break;
          
          got_start = TRUE;
        }
        break;
        /*}}}*/
        
      case END_MODULE_TAG :
        /*{{{  record endpoint*/
        skip_block(tcoff_in,length);
        #ifdef VIEW_PARSE
          printf("END_MODULE\n");
        #endif
        if ((got_text)&&(got_symbol)&&(got_offset)&&(got_wsandvs))
          tcoff_state=PARSE_COMPLETED ;
        else
          tcoff_state=PARSE_FAILED ;
        
        /*}}}*/
        
      default :
        /*{{{  skip unrecognised tag*/
        {
          #ifdef VIEW_PARSE
            printf("Tag %02x Length %d\n",tag,length);
          #endif
          skip_block(tcoff_in,length);
        }
        /*}}}*/
    }        
    /*}}}*/
    
    if (feof(tcoff_in)) return(PARSE_FAILED);
  }
  while (tcoff_state==PARSE_PROCEEDING);

  (*scdb)->size = cbb->index;

  if (save)
  {
    if ((*scdbout=(SCDB*)malloc(cbb->index))==NULL)  
      Exception(UPR_error,_FILE_,__LINE__,"SCDB malloc failed");
    memcpy(*scdbout,*scdb,(*scdb)->size);
    adjust_scdb(*scdb,*scdbout);
  }
  else
    *scdbout=*scdb;
    
  #ifdef VIEW_PARSE
  if (save)
    printf("Relocated SCDB to :\n");
  else
    printf("Leave SCDB at:\n");
  ViewSCDB(*scdbout);
  #endif

  return(tcoff_state);
}
/*}}}*/

/*{{{  PUBLIC FILE *fopenenv(char *file, char *opts, char *env, char *def)*/
PUBLIC FILE *fopenenv(char *file, char *opts, char *env, char *def)
{
  char *tryname;
  char *gotenv;
  FILE *fp;
  char *start, *end;

  if ((fp=fopen(file,opts))!=NULL) return fp;

  if ((gotenv=getenv(env))==NULL)
    {
      if (def != NULL)
        /*{{{  try default*/
        {
          char s[100];
          
          sprintf(s,"No environment variable %s using default %s",env,def);
          Exception(UPR_warning,_FILE_,__LINE__,s);
        
          tryname = (char *) malloc(strlen(def)+strlen(file)+1) ;
          strcpy(tryname,def) ;
          strcat(tryname,file);
        
          fp=fopen(tryname,opts);
          free(tryname);
          return(fp);
        }
        
        /*}}}*/
      else
        /*{{{  Error as no default exists*/
        {
          char s[100];
          sprintf(s,"Failed to find environment string for %s and no default\n",env) ;
          Exception(UPR_error,_FILE_,__LINE__,s);
        }
        /*}}}*/
    }
  else
    /*{{{  check through fullnames*/
    {
      start = gotenv;
      do
      {
        while ((*start==' ')||(*start==';')) start++;
        end=start;
        while ((*end!=' ')&&(*end!=';')&&(*end!=0)) end++;
        if ((end-start)>0)
        {
          /*{{{  malloc and build full filename*/
          if ((tryname = malloc((end-start)+strlen(file)+1)) == NULL)
            Exception(UPR_error,_FILE_,__LINE__,"Out of heap space") ;
            
          strncpy(tryname,start,(end-start));
          tryname[end-start]=0;
          strcat(tryname,file);
          /*}}}*/
          #ifdef DVC_DEBUG
            printf("Trying to open %s\n",tryname);
          #endif
          fp=fopen(tryname,opts);
          free(tryname);
          if (fp!=NULL) return fp;
        }
        start = end + 1;
      } while (*end != 0);
      return NULL;
    }
    /*}}}*/
}
/*}}}*/
/*{{{  PUBLIC int* TCOFF_read_init(FILE *tcoff_in)*/
PUBLIC int* TCOFF_read_init(FILE *tcoff_in)
{
  BOOL got_text = FALSE;
  BOOL got_end = FALSE;
  int* address;  

  while (!got_end)
  {
    int tag,length;
  
    #ifdef VIEW_PARSE
    printf("%08x ",(int) ftell(tcoff_in));
    #endif
    
    tag = read_number(tcoff_in);
    length = read_number(tcoff_in);
    switch (tag)
    {
      /*{{{  case LOAD_TEXT_TAG :*/
      case LOAD_TEXT_TAG :
      #ifdef VIEW_PARSE
      printf("LOAD_TEXT_TAG\n");
      #endif
      {
        int size = read_number(tcoff_in);
      
        address = (int*) malloc(size);
        got_text = TRUE;
        if (address==NULL) Exception(UPR_error,_FILE_,__LINE__,"Malloc failed during VCR configuration load");
        read_block(tcoff_in, (char*) address,size);
      }
      break;
      /*}}}*/
      /*{{{  case END_MODULE_TAG :*/
      case END_MODULE_TAG :
      #ifdef VIEW_PARSE
      printf("END_MODULE_TAG\n");
      #endif
      {
        got_end = TRUE;
        break;      
      }
      
      /*}}}*/
      /*{{{  default:*/
      default:
      #ifdef VIEW_PARSE
      printf("Tag %02x Length %d\n",tag,length);
      #endif
        skip_block(tcoff_in,length);
      /*}}}*/
    }    
    if (feof(tcoff_in))
      Exception(UPR_error,_FILE_,__LINE__,"Error during VCR configuration file parse");
  }
  return address;
}

/*}}}*/
/*{{{  PUBLIC int* TCOFF_read_chans(FILE *tcoff_in, int num_chans)*/
PUBLIC int* TCOFF_read_chans(FILE *tcoff_in, int num_chans)
{
  int length;
  int *address = NULL;

  #ifdef VIEW_PARSE
  printf("Reading Channel Table \n");
  #endif

  /*{{{  skip SYMBOL TAG*/
  if (read_number(tcoff_in)!=SYMBOL_TAG)
    Exception(UPR_error,_FILE_,__LINE__,"Expected SYMBOL_TAG");
  length = read_number(tcoff_in);
  skip_block(tcoff_in,length);
  /*}}}*/
  /*{{{  check for LOAD_TEXT*/
  if (num_chans == 0)
    address = NULL ;
  else  
    if (read_number(tcoff_in)==LOAD_TEXT_TAG)
      /*{{{  read text*/
      {
        length = read_number(tcoff_in); /* record length */
        length = read_number(tcoff_in); /* text length */
      
        if (length/(2*sizeof(int)) != num_chans)
          Exception(UPR_error,_FILE_,__LINE__,"Number of channels mismatch in vcr file") ;
        
        if ((address=malloc(length))==NULL)
          Exception(UPR_error,_FILE_,__LINE__,"Malloc Failed");
      
        read_block(tcoff_in,(char*) address,length);  
      }
      /*}}}*/
    else
      /*{{{  skip text*/
      {
        skip_block(tcoff_in,read_number(tcoff_in));
      }
      /*}}}*/
  /*}}}*/

  return address;
}  
/*}}}*/
/*{{{  PUBLIC SCDB* TCOFF_read_code(FILE *tcoff_in)*/
PUBLIC SCDB* TCOFF_read_code(FILE *tcoff_in)
{
  SCDB* scdb;
  CBB cbb;
  
  if ((cbb.base=malloc(STATIC_CODE_BUFFER))==NULL) 
    Exception(UPR_error,_FILE_,__LINE__,"Code buffer malloc failed");
  
  cbb.index = 0;
  cbb.size = STATIC_CODE_BUFFER;
  
  if (TCOFF_LoadSCDB(tcoff_in, &cbb,"stub",&scdb,FALSE)==PARSE_FAILED)
    Exception(UPR_error,_FILE_,__LINE__,"Failed during code stub parse");
  else
    return scdb;    
}  
/*}}}*/
#endif
#ifdef CODE_CHECK
/*{{{  PRIVATE int sum_byte_array (BYTE *a, int len)*/
PRIVATE int sum_byte_array (BYTE *a, int len)
{
  int i, sum=0 ;
  for (i=0;i<len;i++)
    sum += a[i] ;
  return(sum) ;
}
/*}}}*/
#endif

/*{{{  PUBLIC SCDB *RPC_ReadCode(VCB *vcb_in)*/
PUBLIC SCDB *RPC_ReadCode(VCB *vcb_in)
{
  SCDB *scdb ;
  int len;
  char *base ;
  
  /*{{{  fetch scdb and code from server*/
  VirtualIn  (vcb_in,  &len, sizeof(int)) ;   /* length of code block */
  
  if ((scdb = (SCDB*) malloc(len)) == NULL)
    Exception(UPR_error,_FILE_,__LINE__,"Malloc for dynamically loaded code failed") ;
  
  VirtualIn  (vcb_in, scdb, len) ;            /* code block */
  
  VirtualIn  (vcb_in, &base, sizeof(int)) ;   /* relocation base */
  #ifdef CODE_CHECK
  {
    int check;
  VirtualIn  (vcb_in, &check, sizeof(int)) ;
  
  if (check != sum_byte_array((BYTE *) scdb, len))
    Exception(UPR_error,_FILE_,__LINE__,"Code failed check-sum test") ;
  }
  #endif
  
  /*}}}*/
  adjust_scdb((SCDB*) base, scdb);

  return(scdb) ;
}
/*}}}*/
/*{{{  PUBLIC void  RPC_SendCode(VCB *vcb_out, SCDB *scdb)*/
PUBLIC void  RPC_SendCode(VCB *vcb_out, SCDB *scdb)
{
  int len = scdb->size ;
#ifdef CODE_CHECK
  int check = sum_byte_array((BYTE *) scdb, len) ;
#endif
  VirtualOut (vcb_out, &len, sizeof(int)) ;  /* length of code block */
  VirtualOut (vcb_out, scdb, len) ;          /* code block */
  VirtualOut (vcb_out, &scdb, sizeof(int)) ; /* relocation base */
#ifdef CODE_CHECK
  VirtualOut (vcb_out, &check, sizeof(int)); /* checksum */
#endif
  
}
/*}}}*/
/*{{{  PUBLIC void  RPC_StackCode(SCDB *scdb)*/
PUBLIC void  RPC_StackCode(SCDB *scdb)
{
  VCR_GLOBALS *g = &vcr_globals ;
  int priority ;
  
  ProcGetPRI(&priority);
  if (priority==PROC_LOW) ProcToHI();
  
  /*{{{  add to code stack*/
  scdb->next = g->scdb_stack ;
  g->scdb_stack  = scdb ;
  /*}}}*/

  if (priority==PROC_LOW) ProcToLO() ;
}
/*}}}*/

/*{{{  PUBLIC SCDB* RPC_FindCode(char *file)*/
PUBLIC SCDB* RPC_FindCode(char *file)
{
  SCDB* scdb ;

  scdb = vcr_globals.scdb_stack;

  while (scdb != ((SCDB*) NULL))
  {
    if (strcmp(scdb->filename,file)==0)
      return (scdb) ;
      
    scdb = scdb->next;
  }
  
  return( scdb );
}
/*}}}*/

/*{{{  PUBLIC SCDB* RPC_CodeRequest(char *file)*/
PUBLIC SCDB* RPC_CodeRequest(char *file)
{
  VCR_GLOBALS *g = &vcr_globals ;
  CQB *cqb = &g->cqb ;
  CRB crb;
  int priority;

  if (g->proc_id != UPR_root)
    Exception(UPR_error,_FILE_,__LINE__,"Code request called on reduced node") ;

  if ((crb.scdb = RPC_FindCode(file)) == NULL)
    /*{{{  queue request on code handler*/
    {
      #ifdef UPR_DEBUG
      printf("Request for %s : enqueuing on handler\n", file) ;
      #endif
    
      ProcDesc(&(crb.wdesc));
      ProcGetPRI(&priority);
    
      if (priority==PROC_LOW) ProcToHI();
        
      crb.next = (CRB*) NULL ;
      crb.name = file ;
      
      if (cqb->t == (CRB*) NULL)
      {
        ProcAwaken(cqb->wdesc);
        cqb->h = &crb;
        cqb->t = &crb;
      }
      else
      {  
        cqb->t->next = &crb;
        cqb->t = &crb;
      }
      
      ProcSleep();
    }
    /*}}}*/
  #ifdef UPR_DEBUG
  else
    printf("Request for %s : already loaded\n", file) ;
  #endif

  return(crb.scdb);
}
/*}}}*/
/*{{{  PUBLIC void  RPC_CodeHandler(Process *p,CQB *cqb)*/
PUBLIC void  RPC_CodeHandler(Process *p,CQB *cqb)
{
  CBB cbb;
  
  p=p ;

  #ifdef UPR_RD
    Exception (UPR_fatal,_FILE_,__LINE__ ,"VCR code handler launched on reduced node") ;
  #else
    /*{{{  server code requests*/
    {
      if ((cbb.base=malloc(INIT_CODE_BUFFER))==NULL)
        Exception(UPR_error,_FILE_,__LINE__,"Code buffer malloc failed");
      
      cbb.index = 0;
      cbb.size = INIT_CODE_BUFFER;
      
      ProcDesc(&(cqb->wdesc));
      cqb->h = (CRB*) NULL;
      
      while (TRUE)
      {
        cqb->t = (CRB*) NULL;
        ProcSleep();
      
        while (cqb->h!=(CRB*) NULL)
        {
          if ((cqb->h->scdb = RPC_FindCode(cqb->h->name)) == (SCDB*) NULL)
            /*{{{  load code*/
            {
              SCDB *scdb;
              FILE *tcoff_in ;
            
              #if ((defined UPR_DEBUG) || (defined VIEW_PARSE))
              printf("Loading code unit %s\n", cqb->h->name) ;
              #endif
            
              if ((tcoff_in = fopenenv(cqb->h->name,"rb","ISEARCH",NULL))== (FILE *) NULL)
              {
                char s[80] ;
                sprintf(s, "Could not find file %s", cqb->h->name) ;
                Exception(UPR_error,_FILE_,__LINE__,s) ;
              }
            
              if (TCOFF_LoadSCDB(tcoff_in, &cbb,cqb->h->name, &scdb,TRUE)==PARSE_FAILED)
                Exception(UPR_error,_FILE_,__LINE__,"Parse failed");
            
              fclose(tcoff_in) ;
            
              RPC_StackCode (scdb) ;  
            
              cqb->h->scdb = scdb ;
            }
            /*}}}*/
          
          if (cqb->h->wdesc != (WDESC) NotProcess_p)
            ProcAwaken(cqb->h->wdesc);
      
          cqb->h = cqb->h->next;
        }
      }
    }
    /*}}}*/
  #endif
}
/*}}}*/

/*{{{  PUBLIC SCDB* RPC_LoadCode(char *file)*/
PUBLIC SCDB* RPC_LoadCode(char *file)
{
  #ifdef UPR_RD
    /*{{{  load code on network node*/
    {
      SCDB *scdb ;
    
      if ((scdb = RPC_FindCode(file)) == NULL)
        /*{{{  fetch code from root*/
        {
          DUPLEX_VC vc ;
          int len = strlen(file)+1 ;
        
          VCR_LaunchServer (&vc, UPR_root, load_code) ;
        
          VirtualOut (vc.out, &len, sizeof(int)) ;   /* length of filename */
          VirtualOut (vc.out, file, len) ;           /* filename */
        
          scdb = RPC_ReadCode(vc.in) ;
        
          RPC_StackCode (scdb) ;
          
          VCB_Free (vc.in) ;
          VCB_Free (vc.out) ;
        }  
        /*}}}*/
    
      return(scdb) ;
    }
    /*}}}*/
  #else  
    /*{{{  enqueue load code request*/
    {
      return (RPC_CodeRequest(file)) ;
    }
    /*}}}*/
  #endif
}
/*}}}*/
