/*{{{  File banner*/
/*@(#)=====================================================*\
||@(#)  Project : PUMA ESPRIT P2701
||@(#)  Authors : Mark Debbage and Mark Hill
||@(#)            University of Southampton
||  
||@(#)    Title : VCR system main
||@(#)   System : VCR
||@(#) Filename : vmain.c
||@(#)  Version : 2.11
||@(#)     Date : 10/15/92
\*@(#)====================================================*/
/*}}}*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <misc.h>
#include <iocntrl.h>

#include "vcr.h"
#include "vchan.h"
#include "vparse.h"

PRIVATE char *_FILE_ = __FILE__ ;
const char EOLN = 10 ;

EXTERN  int   ProcGuardedAltList ( Channel **, BOOL * ) ;

#ifdef UPR_RD
  /*{{{  network nodes*/
  PUBLIC int VirtualChannelRouter(int argc, char *argv[])
  {
    return(0) ;
  }
  /*}}}*/
#else
  /*{{{  loader on root*/
  /*{{{  constants*/
  #define sys_ports_per_vproc 2
  #define sys_ports_per_local_vproc 2
  /*}}}*/
  /*{{{  statics*/
  PPROC *pprocs ;
  VPROC *vprocs = NULL ;
  
  int   num_pprocs ;
  int   max_ports = 0;       /* this is computed but no longer required */
  
  static char *cfg_ext = ".cfg" ;
  static char *vcr_ext = ".vcr" ;
  static char *dvc_ext = ".dvc" ;
  static char *net_env = "NETSEARCH" ;
  static char *net_def = "nets" ;
  static char *vcr_env = "VCR" ;
  /*}}}*/
  
  /*{{{  PRIVATE void read_init (FILE *cfg_code_fp)*/
  PRIVATE void read_init (FILE *cfg_code_fp)
  {
    int *data = TCOFF_read_init (cfg_code_fp) ;
    int num_vprocs = data[0] ;
    int i ;
    
    if (num_vprocs == 0)
      Exception(UPR_fatal,_FILE_,__LINE__,"VCR application contains no processors!");
      
    for (i=0;i<num_vprocs;i++)
    {
      VPROC *vproc ;
      int   pproc_id ;
  
      /*{{{  allocate and initialise vproc*/
      if ((vproc = (VPROC *) malloc (sizeof (VPROC))) == NULL)
        Exception(UPR_fatal,_FILE_,__LINE__,"Out of heap space whilst allocating virtual processor") ;
      
      /* decode data */
      vproc->vproc_id      = data[i*3+1] ;
      vproc->num_in_chans  = data[i*3+2] ;
      vproc->num_out_chans = data[i*3+3] ;
      /*}}}*/
      /*{{{  append to vprocs list*/
      if (vprocs == NULL)
        vprocs = vproc ;
      else
      {
        VPROC *next = vprocs ;
      
        while (next->list != NULL)
          next = next->list ;
      
        next->list = vproc ;
      }
      
      vproc->list = NULL ;
      
      /*}}}*/
      /*{{{  connect to associated pproc*/
      pproc_id = VCR_processor_mapping (vproc->vproc_id) ;
      vproc->pproc = &pprocs[pproc_id] ;
      /*}}}*/
      /*{{{  update pproc*/
      {
        int num_ports = vproc->num_in_chans+vproc->num_out_chans ;
      
        vproc->pproc->depth++ ;
        vproc->pproc->count += num_ports ;
        vproc->pproc->max_ports = max(num_ports, vproc->pproc->max_ports) ;  /* maximum this pproc */
      
        max_ports = max (vproc->pproc->count, max_ports) ;            /* overall maximum    */
      }
      /*}}}*/
    }
  
    free (data) ;
  }
  /*}}}*/
  
  /*{{{  PRIVATE VPROC *find_vproc(int vproc_id)*/
  PRIVATE VPROC *find_vproc(int vproc_id)
  {
    VPROC *this = vprocs ;
      
    while (this != NULL)
    {
      if (this->vproc_id == vproc_id)
        return(this) ;
  
      this = this->list ;
    }
    
    return(NULL) ;
  }
  /*}}}*/
  
  /*{{{  PUBLIC int VirtualChannelRouter (int argc, char *argv[])*/
  PUBLIC int VirtualChannelRouter (int argc, char *argv[])
  {
    int   silent = FALSE ;
    int   debug  = FALSE ;
    char  *program = NULL ;
    char  *cfg_code ;
    FILE  *cfg_code_fp ;
    int info_size=0;
    char  *info_base;
    NETWORK *net = UPR_GetNetworkInfo() ;
  
    num_pprocs = vcr_globals.num_nodes ;
  
    /*{{{  parse CLI parameters*/
    {
      int p ;
    
      for (p=1;p<argc;p++)
      {
        if (argv[p][0] == switch_char)
        {
          if ((argv[p][1] == 'd') || (argv[p][1] == 'D'))
            debug = TRUE ;
          else
            if ((argv[p][1] == 's') || (argv[p][1] == 'S'))
              silent = TRUE ;
            else
              Exception(UPR_warning,_FILE_,__LINE__,"Unknown command-line option") ;
        }
        else
          if (program == NULL)
            program = argv[p] ;
          else
            break ;
      }
    
      vcr_globals.user_argc = argc-p ;
      vcr_globals.user_argv = argv+p ;
    }
    /*}}}*/
    /*{{{  startup message*/
    if (!silent)
    {
      printf("vcr : Virtual Channel Router, Version 2.0k, %s %s\n",__TIME__,__DATE__) ;
      printf("M. Debbage, M. Hill, University Of Southampton, ESPRIT PUMA P2701\n\n") ;
    }
    
    if (program == NULL)
    {
      printf("Usage: vcr <topology> [%cld] [%ckd %ckf] [%cd %cs] <program>\n\n",
       switch_char,switch_char,switch_char,switch_char,switch_char) ;
      printf("Loader Options:\n");
      printf(" ld   Debug loader\n");
      printf("Kernel Options:\n");
      printf(" kd   Kernel debug\n");
      printf(" kf   Go faster kernel (default secure)\n");
      printf("VCR Options:\n");
      printf(" d    debug\n") ;
      printf(" s    silent\n") ;
      
      exit_terminate(EXIT_FAILURE) ;
    }
    /*}}}*/
    /*{{{  load device details*/
    if (net == NULL)
      Exception(UPR_warning,_FILE_,__LINE__,"Unable to retrieve network information so no devices") ;
    else
    {
      char *dvc_info ;
      char *def ;
      FILE *dvc_info_fp;
      
      /*{{{  find default pathname*/
      {
        char *vcr ;
        char *tmp ;
      
        vcr=getenv(vcr_env);
        if (vcr==NULL)
          Exception(UPR_fatal,_FILE_,__LINE__,"No VCR environment variable");
        if ((def = malloc(strlen(vcr)+strlen(net_def)+2)) == NULL)
          Exception(UPR_fatal,_FILE_,__LINE__,"Out of heap space whilst evaluating default path");
        strcpy( def, vcr );
        tmp = &def[strlen(vcr)];
        *tmp++ = slash_char ;
        strcpy( tmp, net_def);
      }
      /*}}}*/
      /*{{{  generate device filename*/
      if (net->name == NULL) 
        Exception(UPR_fatal,_FILE_,__LINE__,"Topology name not found");
      
      if ((dvc_info = malloc(strlen(net->name)+strlen(dvc_ext)+1)) == NULL)
        Exception(UPR_fatal,_FILE_,__LINE__,"Out of heap space whilst allocating filename");
      strcpy (dvc_info, net->name) ;
      strcat (dvc_info, dvc_ext) ;
      /*}}}*/
    
      if ((dvc_info_fp = fopenenv(dvc_info,"r",net_env,def)) != NULL)
        /*{{{  parse <topology>.dvc file*/
        {
          int i ;
          char *info; 
          int dummy;
        
          fscanf(dvc_info_fp,"%d",&vcr_globals.num_devices);
          if (vcr_globals.num_devices!=0)
          {
            if ((vcr_globals.devices = (DEVICE*) malloc(sizeof(DEVICE)*vcr_globals.num_devices))==NULL)
              Exception(UPR_fatal,_FILE_,__LINE__,"Device array malloc failed");
            
            for (i=0;i<vcr_globals.num_devices;i++)
              /*{{{  extract device record*/
              {
                DEVICE *device = &vcr_globals.devices[i];
                
                if ((ferror(dvc_info_fp)) ||
                    (fscanf(dvc_info_fp,"%d%d%d",  &device->user_id,&device->physical_id, &device->wrap) != 3))
                  Exception(UPR_fatal,_FILE_,__LINE__,"Failed during parse of <topology>.dvc file");
              
                info_size +=2;                                  
                while (fgetc(dvc_info_fp) != EOLN) info_size++;
              
                if (!device->wrap)
                {
                  int j;
                  
                  for (j=0;j<i;j++)
                    if (vcr_globals.devices[j].physical_id==device->physical_id)
                      break;
                  if (j==i) vcr_globals.num_devices_wow++;
                }
              }
              /*}}}*/
            
            if ((info_base = malloc(info_size))==NULL)
              Exception(UPR_fatal,_FILE_,__LINE__,"Device info malloc failed");
            if ((debug)||(!silent))
              printf("Found %d devices of which %d exclude wrapping\n",vcr_globals.num_devices,vcr_globals.num_devices_wow);
            
            rewind(dvc_info_fp);
            info=info_base;
            fscanf(dvc_info_fp,"%d",&dummy);
            for (i=0;i<vcr_globals.num_devices;i++)
              /*{{{  extract free format info string from input*/
              {
                DEVICE *device = &vcr_globals.devices[i];
                int space = info_size-(info-info_base);
              
                if (space<=0)
                  Exception(UPR_error,_FILE_,__LINE__,"Ran out of space in device info buffer");
                fscanf(dvc_info_fp,"%d%d%d",&dummy,&dummy,&dummy);
                device->info = fgets(info,space,dvc_info_fp);
                info = info + strlen(info) + 1;
              
                if (debug)
                  printf("%3d %3d %2d |%s\n",device->user_id, device->physical_id, device->wrap, device->info);           
              }      
              /*}}}*/
          }
          fclose(dvc_info_fp);
        }
        /*}}}*/
    
      if ((vcr_globals.num_devices==0)&&((debug)||(!silent)))
        printf("No devices for %s\n",net->name);
    
      free(dvc_info);
      free(def);
    }
    /*}}}*/
    /*{{{  open vcr library file*/
    if ((cfg_code = malloc(strlen(program)+strlen(vcr_ext)+1)) == NULL)
      Exception(UPR_fatal,_FILE_,__LINE__,"Out of heap space whilst allocating filename") ;
       
    strcpy (cfg_code, program) ;
    strcat (cfg_code, vcr_ext) ;
    
    if ((cfg_code_fp = fopen(cfg_code,"rb")) == NULL)
    {
      char s[40] ;
      sprintf(s, "Failed to open file %s", cfg_code) ;
      Exception(UPR_fatal,_FILE_,__LINE__,s) ;
    }
    /*}}}*/
    /*{{{  call application*/
    {
      int i,j, sync ;
    
      /*{{{  allocate and initialise physical processors*/
      if ((pprocs = (PPROC *) malloc (num_pprocs*sizeof (PPROC))) == NULL)
        Exception(UPR_fatal,_FILE_,__LINE__,"Out of heap space whilst allocating physical processors") ;
      
      for (i=0;i<num_pprocs;i++)
      {
        pprocs[i].pproc_id = i ;
        pprocs[i].depth = 0 ;
        pprocs[i].count = 0 ;
        pprocs[i].max_ports = 0 ;
      }
      /*}}}*/
      
      read_init (cfg_code_fp) ;
    
      /*{{{  launch physical processor servers*/
      for (i=0;i<num_pprocs;i++)
      {
        PPROC *pproc = &pprocs[i] ;
        
      
        if (debug)
          printf("Physical processor %d (%d ports), %d virtual processors (max has %d ports)\n", i, pproc->count, pproc->depth, pproc->max_ports) ;
      
        VCR_LaunchServer (&pproc->vc, i, svr_pproc) ;
      
        if (i != 0)
          /*{{{  send device details*/
          {
            VirtualOut(pproc->vc.out, &vcr_globals.num_devices, sizeof(int));
            VirtualOut(pproc->vc.out, &vcr_globals.num_devices_wow, sizeof(int));
            if (vcr_globals.num_devices!=0)
            {
              VirtualOut(pproc->vc.out, vcr_globals.devices,sizeof(DEVICE)*vcr_globals.num_devices);
              VirtualOut(pproc->vc.out, &info_size, sizeof(int));
              VirtualOut(pproc->vc.out, info_base, info_size);
              VirtualOut(pproc->vc.out, &info_base, sizeof(int));
            }
          }
          /*}}}*/
      
        VirtualOut(pproc->vc.out, &pproc->depth, sizeof(int)) ;
      
        if (pproc->depth>0)
        {
          VirtualOut(pproc->vc.out, &pproc->count, sizeof(int)) ;
          VirtualOut(pproc->vc.out, &pproc->max_ports, sizeof(int)) ;
          VirtualIn (pproc->vc.in,  &pproc->base,  sizeof(int)) ;
        }
        pproc->count = pproc->base ; /* used as a counter in ensuing update */
      }
      /*}}}*/
      /*{{{  adjust virtual processors' bases*/
      {
        VPROC *this = vprocs ;
        
        while (this != NULL)
        {
          this->base = this->pproc->count ;
          this->pproc->count += (this->num_in_chans + this->num_out_chans) ;
      
          if (debug)
            printf("Virtual processor %d has port base of %d\n", this->vproc_id, this->base) ;
      
          this = this->list ;
        }
      }
      /*}}}*/
      /*{{{  load virtual processors with vca and code*/
      {
        VPROC *this = vprocs ;
        
        while (this != NULL)
        {
          PORT_ADDR *vca = (PORT_ADDR *) TCOFF_read_chans (cfg_code_fp, this->num_in_chans+this->num_out_chans) ;
          SCDB *scdb = TCOFF_read_code(cfg_code_fp) ;
          PPROC *pproc = this->pproc ;
      
          /*{{{  information*/
          if (!silent)
            printf("Loading virtual processor %d", this->vproc_id) ;
          
          if (debug)
            printf(" (onto physical processor %d)", pproc->pproc_id) ;
          
          if (!silent)
            printf("\n") ;
          /*}}}*/
            
          /*{{{  send code*/
          RPC_SendCode (pproc->vc.out, scdb) ;
          /*}}}*/
          /*{{{  serve patch requests*/
          {
            int len ;
            int buflen = 0;
            char *name = NULL ;
            SCDB *scdb ;
          
            VirtualIn (pproc->vc.in, &len, sizeof(int)) ;
          
            while (len > 0)
            {
              if (len > buflen)
              {
                if (name != NULL) free(name) ;
                if ((name = malloc (len)) == NULL)
                  Exception(UPR_error,_FILE_,__LINE__,"Out of heap whilst allocating filename") ;
                buflen = len ;
              }
              
              VirtualIn (pproc->vc.in, name, len) ;
          
              scdb = RPC_LoadCode (name) ;
              RPC_SendCode (pproc->vc.out, scdb) ;
          
              VirtualIn (pproc->vc.in, &len, sizeof(int)) ;
            }
          
            if (name != NULL) free(name) ;
          }
          /*}}}*/
          /*{{{  send virtual processor data structure*/
          VirtualOut(pproc->vc.out, this, sizeof(VPROC)) ;
          /*}}}*/
          /*{{{  translate in connections*/
          for (j=0;j<this->num_in_chans;j++)
          {
            VPROC *other ;
            
            if ((other = find_vproc (vca[j].proc)) == NULL)
              Exception(UPR_fatal,_FILE_,__LINE__,"Inconsistency in virtual channel connections") ;
          
            vca[j].proc = other->pproc->pproc_id ;
            vca[j].port += other->base + other->num_in_chans ;
          
           if (debug)
             printf("%d, %d inputs from %d, %d\n", this->pproc->pproc_id,  this->base + j, vca[j].proc, vca[j].port) ;
          }
          /*}}}*/
          /*{{{  translate out connections*/
          for (j=0;j<this->num_out_chans;j++)
          {
            int k = j+this->num_in_chans ;
            VPROC *other ;
          
            if ((other = find_vproc (vca[k].proc)) == NULL)
              Exception(UPR_fatal,_FILE_,__LINE__,"Inconsistency in virtual channel connections") ;
           
            vca[k].proc = other->pproc->pproc_id ;
            vca[k].port += other->base ; 
          
            if (debug)
              printf("%d, %d outputs to %d, %d\n", this->pproc->pproc_id, this->base + k, vca[k].proc, vca[k].port) ;
          }
          /*}}}*/
          /*{{{  send virtual channel allocation information*/
          VirtualOut(pproc->vc.out, vca, (this->num_in_chans + this->num_out_chans) * sizeof(PORT_ADDR)) ;
          /*}}}*/
          /*{{{  synchronize with server*/
          VirtualIn (pproc->vc.in, &sync, sizeof(int)) ;
          /*}}}*/
          /*{{{  clean up*/
          free(scdb) ;
          free(vca) ;
          /*}}}*/
            
          this = this->list ;
        }
      }
      /*}}}*/
      /*{{{  launch application*/
      for (i=0;i<num_pprocs;i++)
      {
        PPROC *pproc = &pprocs[i] ;
        
        if (pproc->depth != 0)
        {
          if (debug)
            printf("Launching physical processor %d\n", i) ;
      
          VirtualOut(pproc->vc.out, &sync, sizeof(int)) ;
        }
      }
      
      if (!silent)
        printf("Network loaded successfully\n") ;
      /*}}}*/
      /*{{{  wait for termination*/
      {
        BOOL *guard ;
        VCB  **vcb_ins ;
        int  pcount=0;
      
        if (((guard = (BOOL *) malloc (num_pprocs*sizeof (BOOL))) == NULL) ||
            ((vcb_ins = (VCB **) malloc ((num_pprocs+1)*sizeof (VCB *))) == NULL))
          Exception(UPR_fatal,_FILE_,__LINE__,"Out of heap space whilst allocating termination alt") ;
      
        for (i=0;i<num_pprocs;i++)
        {
          PPROC *pproc = &pprocs[i] ;
      
          if (pproc->depth != 0)
            /*{{{  active pproc*/
            {
              guard[i] = TRUE ;
              vcb_ins[i] = pprocs[i].vc.in ;
              pcount++;
            }
            /*}}}*/
          else
            /*{{{  inactive pproc*/
            {
              guard[i] = FALSE ;
              vcb_ins[i] = (void *) (~(int) NULL) ;
            }
            /*}}}*/
        }
        vcb_ins[num_pprocs] = NULL ;
        
        while (pcount != 0)
        {
          int selected = ProcGuardedAltList ((Channel **) vcb_ins, guard) ;
          int result ;
      
          VirtualIn (vcb_ins[selected], &result, sizeof(int)) ;
          guard[selected] = FALSE ;
          pcount-- ;
      
          if (debug)
            printf("Physical processor %d has terminated\n", selected) ;
        }
      
        free(guard) ;
        free(vcb_ins) ;
      
        if (!silent)
          printf("Network terminated successfully\n") ;
      }
      /*}}}*/
      /*{{{  clean up and terminate*/
      {
        VPROC *this = vprocs ;
        
        while (this != NULL)
        {
          VPROC *next = this->list ;
          free (this) ;
          this = next ;
        }
      }
      
      for (i=0;i<num_pprocs;i++)
      {
        PPROC *pproc = &pprocs[i] ;
      
        if (pproc->depth != 0)
        {
          VCB_Free (pproc->vc.in) ;
          VCB_Free (pproc->vc.out) ;
        }
      }
      
      free(pprocs) ;
      /*}}}*/
    }
    /*}}}*/
    /*{{{  clean up*/
    fclose (cfg_code_fp) ;
    free (cfg_code) ;
    /*}}}*/
  
    return(0) ;
  }
  /*}}}*/
  
  /*}}}*/
#endif
