/*{{{  File banner*/
/*@(#)=====================================================*\
||@(#)  Project : PUMA ESPRIT P2701
||@(#)  Authors : Mark Debbage and Mark Hill
||@(#)            University of Southampton
||  
||@(#)    Title :  Virtual channel control port functions
||@(#)   System : VCR
||@(#) Filename : vcrctrl.c
||@(#)  Version : 2.4
\*@(#)====================================================*/
/*}}}*/

#include <stdio.h>
#include <stdlib.h>

#include "vcr.h"
#include "vserver.h"
#include "vchan.h"
#include "vmove.h"
#include "vpktio.h"

static char *_FILE_ = __FILE__ ;

/*#define DVC_DEBUG*/

/*{{{  PUBLIC void VCR_CtrlPreAction (PORT *port, int length)*/
PUBLIC void VCR_CtrlPreAction (PORT *port, int length)
{
  VCR_CTRL_PKT *ctrl_state = (VCR_CTRL_PKT *) port->state ;
  int i ;

  for (i=0;i<num_ctrl_pkts;i++)
    if (ctrl_state[i].tag == empty_ctrl) break ;

  if (i == num_ctrl_pkts)
    Exception(UPR_error, _FILE_, __LINE__, "Cannot service control packet - no state") ;

  ctrl_state[i].tag = ~empty_ctrl ;
  
  port->buffer = (BYTE *) &ctrl_state[i] ;
  port->space  = sizeof (VCR_CTRL_PKT) ;
}
/*}}}*/
/*{{{  PUBLIC void VCR_CtrlPostAction (PORT *port, int length, BYTE *buffer)*/
PUBLIC void VCR_CtrlPostAction (PORT *port, int length, BYTE *buffer)
{
  VCR_CTRL_PKT *ctrl_pkt = (VCR_CTRL_PKT *) buffer ;

  switch (ctrl_pkt->tag)
  {
    /*{{{  server tags*/
    case svr_request :
      /*{{{  launch server*/
      {
        Process *svr ;
      
        if ((svr=ProcAlloc(SVR_Server,SVR_wspace_size,sizeof(VCR_CTRL_PKT)/sizeof(int),*(ctrl_pkt))) == NULL)
          Exception(UPR_error,_FILE_,__LINE__,"Server allocation failed") ;
      
        ProcRunLow(svr);
        
        break ;
      }
      /*}}}*/
    /*}}}*/

    /*{{{  dynamic virtual channel tags*/
    case dvc_request :
      /*{{{  request for dynamic channel*/
      {
        DVC_REQUEST *dvcrq_in = &(ctrl_pkt->p.dvc_request) ;
        VCB *vc ;
        PORT *port;
        DCB *dcb;
        
        if (dvcrq_in->way == input_d)
          /*{{{  request for input*/
          {
            if ((vc = (VCB*) malloc(sizeof(VCB)+UPR_MAX_PKT_SIZE+sizeof(DCB)))==NULL)
              Exception(UPR_error,_FILE_,__LINE__,"DVCB malloc failed");
            port=PortAlloc (VCR_InPreAction,TRUE,VCR_InPostAction,FALSE,vc);
            dcb = (DCB*)( ((char*) vc)+UPR_MAX_PKT_SIZE+sizeof(VCB));
            VCB_Init(vc,port,(BYTE*) (vc+1), dcb, dvcrq_in->proc_id,dvcrq_in->port_id);
          }
          /*}}}*/
        else if (dvcrq_in->way ==output_d)
          /*{{{  request for output*/
          {
            if ((vc = (VCB*) malloc(sizeof(VCB)+sizeof(DCB)))==NULL)
              Exception(UPR_error,_FILE_,__LINE__,"DVCB malloc failed");
            port=PortAlloc (VCR_OutPreAction,TRUE,VCR_OutPostAction,FALSE,vc);
            dcb = (DCB*)( ((char*) vc)+sizeof(VCB));
            VCB_Init(vc,port,NULL,dcb,dvcrq_in->proc_id,dvcrq_in->port_id);
          }
          /*}}}*/
        else
          Exception(UPR_error,_FILE_,__LINE__,"Invalid direction during DVCB create");
      
        DVC_InitDCB(dcb, &vc->orb.header);
        
        /*{{{  send dvc_reply using dcb*/
        {
          DCP *dcp = &(vc->dcb->dcp) ;
          ORB *orb = &(vc->dcb->orb) ;
          
          dcp->tag = dvc_reply;
          dcp->p.dvc_reply.port_id = port->id ;
          dcp->p.dvc_reply.src_port_id = dvcrq_in->port_id ;
          dcp->p.dvc_reply.vc_ptr  = vc ;
        
          HdrBuild( dvcrq_in->proc_id, vcr_globals.vcr_ctrl_port_id, sizeof(DVC_REPLY)+ctrl_pkt_ovrhd,&(orb->header));
          
          VCR_Enqueue_NBK( orb );
        }
        /*}}}*/
      
        break;
      }
      /*}}}*/
    
    case dvc_reply :
      /*{{{  reply to dynamic channel request*/
      {
        DVC_REPLY *dvcrp_in = &(ctrl_pkt->p.dvc_reply) ;
        VCB *vcb = PortPtr(dvcrp_in->src_port_id)->state;
      
        AsmMove(buffer,&(vcb->dcb->dcp),length);
        ProcAwaken(vcb->dcb->restart);
        break;
      }
      /*}}}*/
    /*}}}*/
      
    /*{{{  move channel tags*/
    case dvc_offload :
      /*{{{  rx request to Offload channel at destination*/
      {
        DVC_OFFLOAD *dvc_offload = &ctrl_pkt->p.dvc_offload ;
        VCB *vcb;
        PORT *port;
        DCB *dcb ;
      
        /*{{{  debug Offload*/
        #ifdef DVC_DEBUG
        {
          char s[100];
          sprintf(s,"Offload [ %d:%d -> %d ] (%d:%d) ",dvc_offload->src_p,dvc_offload->src_c,vcr_globals.proc_id,dvc_offload->other_p,dvc_offload->other_c);
          Exception(UPR_warning,_FILE_,__LINE__,s);
        }
        #endif
        /*}}}*/
      
        if (dvc_offload->make_way != neuter_d)
          /*{{{  OF1: Offload [SRC->DST] (OTHER) actions*/
          {
            if (dvc_offload->make_way==input_d)
              /*{{{  request for input*/
              {
                if ((vcb = (VCB*) malloc(sizeof(VCB)+UPR_MAX_PKT_SIZE+sizeof(DCB)))==NULL)
                  Exception(UPR_error,_FILE_,__LINE__,"DVCB malloc failed");
                port=PortAlloc (VCR_InPreAction,TRUE,VCR_InPostAction,FALSE,vcb);
                dcb = (DCB*)( ((char*) vcb)+UPR_MAX_PKT_SIZE+sizeof(VCB));
                VCB_Init(vcb,port,(BYTE*) (vcb+1), dcb, dvc_offload->other_p,dvc_offload->other_c);
              }
              /*}}}*/
            else
              /*{{{  request for output*/
              {
                if ((vcb = (VCB*) malloc(sizeof(VCB)+sizeof(DCB)))==NULL)
                  Exception(UPR_error,_FILE_,__LINE__,"DVCB malloc failed");
                port=PortAlloc (VCR_OutPreAction,TRUE,VCR_OutPostAction,FALSE,vcb);
                dcb = (DCB*)( ((char*) vcb)+sizeof(VCB));
                VCB_Init(vcb,port,NULL,dcb,dvc_offload->other_p,dvc_offload->other_c);
              }
              
              /*}}}*/
            DVC_InitDCB(dcb, &vcb->orb.header );
            HdrBuild(dvc_offload->src_p,dvc_offload->src_c,VALID_HEAD_L,&dcb->parent);
            /*{{{  SEND DoneEnd ; src_c ; dst_p ; dst_c ; vcb*/
            {
              DCP *dcp = &(vcb->dcb->dcp) ;
              ORB *orb = &(vcb->dcb->orb) ;
              
              dcp->tag = dvc_doneend;
              dcp->p.dvc_doneend.src_c = dvc_offload->src_c ;
              dcp->p.dvc_doneend.dst_p = vcr_globals.proc_id ;
              dcp->p.dvc_doneend.dst_c = vcb->port->id ;
              dcp->p.dvc_doneend.vcb = vcb ;
            
              HdrBuild( dvc_offload->src_p, vcr_globals.vcr_ctrl_port_id, sizeof(DVC_DONEEND)+ctrl_pkt_ovrhd,&(orb->header));
            
              VCR_Enqueue_NBK( orb );
            }
            /*}}}*/
          }
          /*}}}*/
        else
        {
          vcb = PortPtr(dvc_offload->dst_c)->state;
          dcb = vcb->dcb ;
          
          if (dcb->state==dvc_gone)
            /*{{{  OF2: Offload actions (Return channel)*/
            {
              vcb = PortPtr(dvc_offload->dst_c)->state;
              dcb = vcb->dcb ;
            
              if ((dvc_offload->src_p != HdrDestn(&vcb->dcb->other))||
                  (dvc_offload->src_c != HdrPort(&vcb->dcb->other)))
                Exception(UPR_error,_FILE_,__LINE__,"Rxd Offload from invalid source");
              HdrBuild(dvc_offload->other_p,dvc_offload->other_c,0,&vcb->dcb->other) ;
              HdrBuild(dvc_offload->src_p,dvc_offload->src_c,VALID_HEAD_L,&dcb->child);
              dcb->state = dvc_ready ;
            
              /*{{{  SEND DoneEnd ; src_c ; dst_p ; dst_c ; vcb*/
              {
                DCP *dcp = &(vcb->dcb->dcp) ;
                ORB *orb = &(vcb->dcb->orb) ;
              
                if (orb->list != InputServiced_p)
                  Exception(UPR_error,_FILE_,__LINE__,"ORB in use upon OF2") ;
               
                dcp->tag = dvc_doneend;
                dcp->p.dvc_doneend.src_c = dvc_offload->src_c ;
                dcp->p.dvc_doneend.dst_p = vcr_globals.proc_id ;
                dcp->p.dvc_doneend.dst_c = vcb->port->id ;
                dcp->p.dvc_doneend.vcb = vcb ;
              
                HdrBuild( dvc_offload->src_p, vcr_globals.vcr_ctrl_port_id, sizeof(DVC_DONEEND)+ctrl_pkt_ovrhd,&(orb->header));
                
                VCR_Enqueue_NBK( orb );
              }
              /*}}}*/
            }
            /*}}}*/
          else 
            Exception(UPR_error,_FILE_,__LINE__,"Rxd invalid offload");
        }
        break;
      }
      /*}}}*/
    
    case dvc_doneend :
      /*{{{  rx doneend acknowledgment at source*/
      {
        DVC_DONEEND *dvc_doneend = &(ctrl_pkt->p.dvc_doneend) ;
        VCB *vcb = PortPtr(dvc_doneend->src_c)->state;
        DCB *dcb = vcb->dcb ;
      
        /*{{{  debug Donend*/
        #ifdef DVC_DEBUG
        {
          char s[100];
          sprintf(s,"DoneEnd [ %d:%d -> _:%d ] (%08x) ",dvc_doneend->dst_p,dvc_doneend->dst_c,dvc_doneend->src_c,(int)dvc_doneend->vcb);
          Exception(UPR_warning,_FILE_,__LINE__,s);
        }
        #endif
        /*}}}*/
          
        dcb->dcp.p.dvc_doneend.vcb = dvc_doneend->vcb ;
        
        if (dcb->state==dvc_finding)
          /*{{{  DE1: offload successful*/
          {
            dcb->state=dvc_gone;
            /*{{{  SEND Point ; other_c ; dst_p ; dst_c*/
            {
              DCP *dcp = &(vcb->dcb->dcp) ;
              ORB *orb = &(vcb->dcb->orb) ;
              
              if (orb->list != InputServiced_p)
                Exception(UPR_error,_FILE_,__LINE__,"ORB in use upon DE1") ;
            
              dcp->tag = dvc_point;
              dcp->p.dvc_point.other_c = HdrPort(&vcb->dcb->other);
              dcp->p.dvc_point.dst_p = dvc_doneend->dst_p ;
              dcp->p.dvc_point.dst_c = dvc_doneend->dst_c ;
            
              HdrBuild( HdrDestn(&vcb->dcb->other), vcr_globals.vcr_ctrl_port_id, sizeof(DVC_POINT)+ctrl_pkt_ovrhd,&orb->header);
              
              VCR_Enqueue_NBK( orb );
            }
            /*}}}*/
            HdrBuild(dvc_doneend->dst_p,dvc_doneend->dst_c,0,&vcb->dcb->other) ;
            HdrCopy (&vcb->dcb->other, &vcb->orb.header) ;
          }
          /*}}}*/
        else if (dcb->state==dvc_retry)
          /*{{{  DE2: retry offload*/
          {
            dcb->state=dvc_finding ;
            /*{{{  SEND RetryOffload ; dst_c ; newother_p ; newother_c */
            {
              DCP *dcp = &(vcb->dcb->dcp) ;
              ORB *orb = &(vcb->dcb->orb) ;
            
              if (orb->list != InputServiced_p)
                Exception(UPR_error,_FILE_,__LINE__,"ORB in use upon DE2") ;
              
              dcp->tag = dvc_retryoffload;
              dcp->p.dvc_retryoffload.dst_c = dvc_doneend->dst_c ;
              dcp->p.dvc_retryoffload.newother_p = HdrDestn(&vcb->dcb->other) ;
              dcp->p.dvc_retryoffload.newother_c = HdrPort(&vcb->dcb->other) ;
              HdrBuild(dvc_doneend->dst_p,vcr_globals.vcr_ctrl_port_id,sizeof(DVC_RETRYOFFLOAD)+ctrl_pkt_ovrhd,&orb->header);                                
            
              VCR_Enqueue_NBK(orb);
            }
            /*}}}*/
          }      
          /*}}}*/
        else
          Exception(UPR_error,_FILE_,__LINE__,"Done End rxed by channel in invalid state");
        break;  
      }  
      /*}}}*/
      
    case dvc_point :
      /*{{{  rx point at other end*/
      {
        DVC_POINT *dvc_point = &(ctrl_pkt->p.dvc_point) ;
        PORT *port = PortPtr(dvc_point->other_c);
        VCB *vcb = port->state ;
        DCB *dcb = vcb->dcb ;
      
        /*{{{  invalid port*/
        if (port==NULL)
        {
          char s[100];
          sprintf(s,"Point rxed on vacant port %d, repoint was to (%d:%d)",dvc_point->other_c,dvc_point->dst_p,dvc_point->dst_c);
          Exception(UPR_error,_FILE_,__LINE__,s);
        }
        /*}}}*/
        /*{{{  debug Point*/
        #ifdef DVC_DEBUG
        {
          char s[100];
          sprintf(s,"Point [ _:_ -> %d:%d ] (%d:%d) ",vcr_globals.proc_id,dvc_point->other_c,dvc_point->dst_p,dvc_point->dst_c);
          Exception(UPR_warning,_FILE_,__LINE__,s);
        }
        #endif
        /*}}}*/
      
        if (dcb->state==dvc_ready)
          /*{{{  PT1: Point actions*/
          {
            /*{{{  SEND  Moved ; src_p*/
            {
              DCP *dcp = &(vcb->dcb->dm_dcp) ;
              ORB *orb = &(vcb->dcb->dm_orb) ;
            
              if (orb->list != InputServiced_p)
                Exception(UPR_error,_FILE_,__LINE__,"ORB in use upon PT1") ;
              
              dcp->tag = dvc_moved;
              dcp->p.dvc_moved.src_c = HdrPort(&vcb->dcb->other);
            
              HdrBuild( HdrDestn(&vcb->dcb->other), vcr_globals.vcr_ctrl_port_id, sizeof(DVC_MOVED)+ctrl_pkt_ovrhd,&(orb->header));
            
              VCR_Enqueue_NBK( orb );
            }
            /*}}}*/
          
            #ifdef DVC_DEBUG
            if (vcb->orb.list != InputServiced_p)
              Exception(UPR_warning,_FILE_,__LINE__,"ORB in use upon header repoint") ;
            #endif
            
            HdrBuild(dvc_point->dst_p,dvc_point->dst_c,0,&vcb->dcb->other);
          }
          /*}}}*/
        else if (dcb->state==dvc_gone)
          /*{{{  PT2: Point action : SEND DualMove*/
          {
            DCP *dcp = &(vcb->dcb->dm_dcp) ;
            ORB *orb = &(vcb->dcb->dm_orb) ;
          
            if (orb->list != InputServiced_p)
              Exception(UPR_error,_FILE_,__LINE__,"ORB in use upon PT2") ;
            
            dcp->tag = dvc_dualmove;
            dcp->p.dvc_dualmove.dst_c = dvc_point->dst_c ;
            dcp->p.dvc_dualmove.newother_p = HdrDestn(&vcb->dcb->other);
            dcp->p.dvc_dualmove.newother_c = HdrPort(&vcb->dcb->other);
            
            HdrBuild( dvc_point->dst_p, vcr_globals.vcr_ctrl_port_id, sizeof(DVC_DUALMOVE)+ctrl_pkt_ovrhd,&(orb->header));
            
            VCR_Enqueue_NBK( orb );  
          }  
          /*}}}*/
        else if (dcb->state==dvc_finding)
          /*{{{  PT3: Point Action */
          {
            dcb->state=dvc_retry;
            /*{{{  SEND Moved ; src_c*/
            {
              DCP *dcp = &(vcb->dcb->dm_dcp) ;
              ORB *orb = &(vcb->dcb->dm_orb) ;
            
              if (orb->list != InputServiced_p)
                Exception(UPR_error,_FILE_,__LINE__,"ORB in use upon PT3") ;
              
              dcp->tag = dvc_moved;
              dcp->p.dvc_moved.src_c = HdrPort(&vcb->dcb->other);
            
              HdrBuild( HdrDestn(&vcb->dcb->other), vcr_globals.vcr_ctrl_port_id, sizeof(DVC_MOVED)+ctrl_pkt_ovrhd,&(orb->header));
              
              VCR_Enqueue_NBK( orb );
            }
            /*}}}*/
            HdrBuild(dvc_point->dst_p,dvc_point->dst_c,0,&vcb->dcb->other);
          }
          /*}}}*/
        else
          Exception(UPR_error,_FILE_,__LINE__,"Point rxed on DVC end in invalid state");
        break;    
      }      
      /*}}}*/
    
    case dvc_moved :
      /*{{{  dvc_moved rxed at source*/
      {
        DVC_MOVED *dvc_moved = &(ctrl_pkt->p.dvc_moved);
        VCB *vcb = PortPtr(dvc_moved->src_c)->state;
        
        /*{{{  debug Moved*/
        #ifdef DVC_DEBUG
        {
          char s[100];
          sprintf(s,"Moved [ _:_ -> %d:%d ] {%08x}",vcr_globals.proc_id,dvc_moved->src_c,(int) vcb->id);
          Exception(UPR_warning,_FILE_,__LINE__,s);
        }
        #endif
        /*}}}*/
                 
        if (vcb->dcb->state==dvc_gone)
          /*{{{  MD1: Moved actions*/
          {
            if (vcb->id==InputWaiting_p)
              /*{{{  Pass on packet*/
              {
                int size = UPR_MAX_PKT_SIZE - vcb->port->space;
              
                if (vcb->orb.list != InputServiced_p)
                  Exception(UPR_error,_FILE_,__LINE__,"ORB in use upon PKT forwarding") ;
                
                HdrModLength(size,&vcb->orb.header);
                vcb->orb.bffr = vcb->buffer ;
                VCR_Enqueue_NBK(&vcb->orb);
                
                vcb->id = NotProcess_p;
                vcb->port->space = 0;
                vcb->port->do_pre = TRUE;
                
                /*{{{  debug Packet forward*/
                #ifdef DVC_DEBUG
                {
                  ORB *orb = &vcb->orb ;
                  char s[100];
                  sprintf(s,"Forwarding packet D%d-P%d-L%d",HdrDestn(&orb->header),HdrPort(&orb->header),HdrLength(&orb->header));
                  Exception(UPR_warning,_FILE_,__LINE__,s);
                }
                #endif
                /*}}}*/
              }
              /*}}}*/
            else if (vcb->id==AckWaiting_p)
              /*{{{  Pass on Ack*/
              {
                if (vcb->orb.list != InputServiced_p)
                  Exception(UPR_error,_FILE_,__LINE__,"ORB in use upon ACK forwarding") ;
              
                HdrModLength(0,&vcb->orb.header);
                VCR_Enqueue_NBK(&vcb->orb);
              
                vcb->id = NotProcess_p;
                /*{{{  debug ack forward*/
                #ifdef DVC_DEBUG
                {
                  ORB *orb = &vcb->orb ;
                  char s[100];
                  sprintf(s,"Forwarding ack D%d-P%d-L%d",HdrDestn(&orb->header),HdrPort(&orb->header),HdrLength(&orb->header));
                  Exception(UPR_warning,_FILE_,__LINE__,s);
                }
                #endif
                /*}}}*/
              }
              /*}}}*/
              
            /*{{{  debug restart*/
            #ifdef DVC_DEBUG
            {
              char s[100];
              sprintf(s,"Restarting thread %08x",(int) vcb->dcb->restart);
              Exception(UPR_warning,_FILE_,__LINE__,s);
            }
            #endif
            /*}}}*/
            ProcAwaken(vcb->dcb->restart);  
          }
          /*}}}*/
        else
          Exception(UPR_error,_FILE_,__LINE__,"Not gone on Moved arrival");
      
        break;
      }  
      /*}}}*/
    
    case dvc_dualmove :
      /*{{{  dvc_dualmove rxed*/
      {
        DVC_DUALMOVE *dvc_dualmove = &(ctrl_pkt->p.dvc_dualmove);
        VCB *vcb = PortPtr(dvc_dualmove->dst_c)->state;
        DCB *dcb = vcb->dcb ;
      
        /*{{{  debug Dualmove*/
        #ifdef DVC_DEBUG
        {
          char s[100];
          sprintf(s,"DualMove [ _:_ -> %d:%d ] (%d:%d) ",vcr_globals.proc_id,dvc_dualmove->dst_c,dvc_dualmove->newother_p,dvc_dualmove->newother_c);
          Exception(UPR_warning,_FILE_,__LINE__,s);
        }
        #endif
        /*}}}*/
        
        if (dcb->state==dvc_ready)
        {
          if (HdrLength(&dcb->child)==VALID_HEAD_L)
            /*{{{  DM1: Dual Move actions*/
            {
              HdrBuild(dvc_dualmove->newother_p,dvc_dualmove->newother_c,0,&vcb->dcb->other);
              /*{{{  SEND Moved ; src_c*/
              {
                DCP *dcp = &(vcb->dcb->dcp) ;
                ORB *orb = &(vcb->dcb->orb) ;
              
                if (orb->list != InputServiced_p)
                  Exception(UPR_error,_FILE_,__LINE__,"ORB in use upon DM1") ;
                
                dcp->tag = dvc_moved;
                dcp->p.dvc_moved.src_c = HdrPort(&dcb->child);
              
                HdrBuild( HdrDestn(&dcb->child), vcr_globals.vcr_ctrl_port_id, sizeof(DVC_MOVED)+ctrl_pkt_ovrhd,&(orb->header));
                
                VCR_Enqueue_NBK( orb );
              }      
              /*}}}*/
            }
            /*}}}*/
          else if (HdrLength(&dcb->parent)==VALID_HEAD_L)
            /*{{{  DM2: Dual Move actions*/
            {
              HdrBuild(dvc_dualmove->newother_p,dvc_dualmove->newother_c,0,&vcb->dcb->other);
              /*{{{  SEND Moved ; src_c */
              {
                DCP *dcp = &(vcb->dcb->dcp) ;
                ORB *orb = &(vcb->dcb->orb) ;
              
                if (orb->list != InputServiced_p)
                  Exception(UPR_error,_FILE_,__LINE__,"ORB in use upon DM2") ;
                
                dcp->tag = dvc_moved;
                dcp->p.dvc_moved.src_c = HdrPort(&dcb->parent);
              
                HdrBuild( HdrDestn(&dcb->parent), vcr_globals.vcr_ctrl_port_id, sizeof(DVC_MOVED)+ctrl_pkt_ovrhd,&(orb->header));
                
                VCR_Enqueue_NBK( orb );
              }      
              /*}}}*/
            }
            /*}}}*/
          else
            Exception(UPR_error,_FILE_,__LINE__,"Neither child or parent valid on DM arrival");
        }
        else
          Exception(UPR_error,_FILE_,__LINE__,"Received dualmove on unready channel");
      
        break;
      }   
      /*}}}*/
    
    case dvc_retryoffload :
      /*{{{  RetryOffload received on destination*/
      {
        DVC_RETRYOFFLOAD *dvc_retryoffload = &(ctrl_pkt->p.dvc_retryoffload);
        VCB *vcb = PortPtr(dvc_retryoffload->dst_c)->state;
        DCB *dcb = vcb->dcb ;
      
        /*{{{  debug RetryOffload*/
        #ifdef DVC_DEBUG
        {
          char s[100];
          sprintf(s,"RetryOffload [ _:_ -> %d:%d ] (%d:%d) ",vcr_globals.proc_id,dvc_retryoffload->dst_c,dvc_retryoffload->newother_p,dvc_retryoffload->newother_c);
          Exception(UPR_warning,_FILE_,__LINE__,s);
        }
        #endif
        /*}}}*/
      
        if (dcb->state==dvc_ready)
        {
          if (HdrLength(&dcb->child)==VALID_HEAD_L)
            /*{{{  RO1: Retry Offload actions */
            {
              HdrBuild(dvc_retryoffload->newother_p,dvc_retryoffload->newother_c,0,&vcb->dcb->other);
              /*{{{  SEND DoneEnd ; src_c ; dst_p ; dst_c ; vcb*/
              {
                DCP *dcp = &(vcb->dcb->dcp) ;
                ORB *orb = &(vcb->dcb->orb) ;
                
                dcp->tag = dvc_doneend;
                dcp->p.dvc_doneend.src_c = HdrPort(&dcb->child);
                dcp->p.dvc_doneend.dst_p = vcr_globals.proc_id ;
                dcp->p.dvc_doneend.dst_c = vcb->port->id ;
                dcp->p.dvc_doneend.vcb = vcb ;
              
                HdrBuild( HdrDestn(&dcb->child), vcr_globals.vcr_ctrl_port_id, sizeof(DVC_DONEEND)+ctrl_pkt_ovrhd,&(orb->header));
                
                VCR_Enqueue_NBK( orb );
              }
              /*}}}*/
            }
            /*}}}*/
          else if (HdrLength(&dcb->parent)==VALID_HEAD_L)
            /*{{{  RO2: Retry Offload actions */
            {
              HdrBuild(dvc_retryoffload->newother_p,dvc_retryoffload->newother_c,0,&vcb->dcb->other);
              /*{{{  SEND DoneEnd ; src_c ; dst_p ; dst_c ; vcb*/
              {
                DCP *dcp = &(vcb->dcb->dcp) ;
                ORB *orb = &(vcb->dcb->orb) ;
                
                dcp->tag = dvc_doneend;
                dcp->p.dvc_doneend.src_c = HdrPort(&dcb->parent);
                dcp->p.dvc_doneend.dst_p = vcr_globals.proc_id ;
                dcp->p.dvc_doneend.dst_c = vcb->port->id ;
                dcp->p.dvc_doneend.vcb = vcb ;
              
                HdrBuild( HdrDestn(&dcb->parent), vcr_globals.vcr_ctrl_port_id, sizeof(DVC_DONEEND)+ctrl_pkt_ovrhd,&(orb->header));
                
                VCR_Enqueue_NBK( orb );
              }
              /*}}}*/
            }
            /*}}}*/
          else
            Exception(UPR_error,_FILE_,__LINE__,"Neither child or parent valid");
        }
        else
          Exception(UPR_error,_FILE_,__LINE__,"Received retryoffload on unready channel");
      
        break;
      }
      /*}}}*/
    /*}}}*/
         
    /*{{{  empty and unknown tags*/
    case empty_ctrl :
      Exception(UPR_error, _FILE_, __LINE__, "Cannot service empty control packet") ;
    
    default :
      Exception(UPR_error, _FILE_, __LINE__, "Unknown control packet type") ;
    /*}}}*/
  }   
      
  ctrl_pkt->tag = empty_ctrl ;
}     
/*}}}*/
/*{{{  PUBLIC void VCR_CtrlPortAlloc ()*/
PUBLIC void VCR_CtrlPortAlloc ()
{
  PORT *ctrl_port ;
  VCR_CTRL_PKT *ctrl_state ;
  int i ;

  if ((ctrl_port = (PORT *) malloc (sizeof(PORT))) == NULL)
    Exception (UPR_fatal, _FILE_, __LINE__, "Out of heap space whilst allocating port") ;

  if ((ctrl_state = (VCR_CTRL_PKT *) malloc (sizeof(VCR_CTRL_PKT)*num_ctrl_pkts)) == NULL)
    Exception (UPR_fatal, _FILE_, __LINE__, "Out of heap space whilst allocating control handler state") ;

  for (i=0;i<num_ctrl_pkts;i++)
    ctrl_state[i].tag = empty_ctrl ;    

  if (PortGrab(ctrl_port) < 0)
    Exception(UPR_fatal,_FILE_,__LINE__,"Cannot allocate VCR control port") ;

  PortInit (ctrl_port, VCR_CtrlPreAction, TRUE,
            VCR_CtrlPostAction, TRUE, (void *) ctrl_state) ;

  vcr_globals.vcr_ctrl_port_id = ctrl_port->id ;
}
/*}}}*/
  
/*{{{  PUBLIC void VCR_ProcessCollector (Process *p, PCQB *pcqb)*/
PUBLIC void VCR_ProcessCollector (Process *p, PCQB *pcqb)
{
  p=p ;
  pcqb->h = NULL ;
  ProcDesc (&pcqb->wdesc) ;

  while (TRUE)
  {
    pcqb->t = NULL ;
    ProcSleep() ;

    while (pcqb->h != NULL)
    {
      PCRB *next = pcqb->h->next ;

      ProcAllocClean (pcqb->h->p) ;
      pcqb->h = next ;
    }
  }
}
/*}}}*/
/*{{{  PUBLIC void VCR_ProcessDone (Process *p)*/
PUBLIC void VCR_ProcessDone (Process *p)
{
  VCR_GLOBALS *g = &vcr_globals ;
  PCRB pcrb ;
  int priority;

  ProcGetPRI(&priority);
  if (priority==PROC_LOW) ProcToHI();

  /*{{{  critical code*/
  pcrb.p = p ;
  pcrb.next = NULL ;
  
  if (g->pcqb.t == NULL)
  {
    ProcAwaken(g->pcqb.wdesc);
    g->pcqb.h = &pcrb;
    g->pcqb.t = &pcrb;
  }
  else
  {  
    g->pcqb.t->next = &pcrb;
    g->pcqb.t = &pcrb;
  }
  
  ProcSleep();
  /*}}}*/
}
/*}}}*/
