/*{{{  File banner*/
/*@(#)=====================================================*\
||@(#)  Project : PUMA ESPRIT P2701
||@(#)  Authors : Mark Debbage and Mark Hill
||@(#)            University of Southampton
||  
||@(#)    Title : Common routines between system library and SC interface
||@(#)   System : vcr
||@(#) Filename : vcommon.c
||@(#)  Version : 2.3
||@(#)     Date : 10/2/91
\*@(#)====================================================*/
/*}}}*/

#include <stdlib.h>
#include "vcr.h"

/* Common routines between the system and applications libraries.
   THESE MUST USE NO STATICS AND NO HEAP */

/*{{{  PRIVATE routines*/
/*{{{  PRIVATE void CheckRange(PBLOCK *pblock, int field, int offset)*/
PRIVATE void CheckRange(PBLOCK *pblock, int field, int offset)
{
  int field_dir = FIELD_DIR(field) ;
  int nbour_dir = FIELD_DIR(field+field_dir) ;
  FIELD *this   = &pblock->field[field] ;
  FIELD *nbour  = this + field_dir ;
  int bound     = ((field_dir==nbour_dir) ? nbour->start : nbour->current) ;

  if (((bound-offset)*field_dir) < 0)
    Exception(UPR_error,__FILE__,__LINE__,"RPC pblock field has overflowed") ;

  if (((offset-this->start)*field_dir) < 0)
    Exception(UPR_error,__FILE__,__LINE__,"RPC pblock field has underflowed") ;
}
/*}}}*/
/*{{{  PRIVATE int AddElement (PBLOCK *pblock, int field, int element)*/
PRIVATE int AddElement (PBLOCK *pblock, int field, int element)
{
  FIELD *this ;
  int result ;
  
  if (field >= num_fields-1)
    Exception(UPR_error,__FILE__,__LINE__,"Add element to illegal field") ;

  this = &pblock->field[field] ;

  result = this->current ;
  CheckRange (pblock, field, result) ;
  this->current++ ;
  pblock->pb[result] = element ;

  return(result) ;
}
/*}}}*/
/*{{{  PRIVATE int AddCouplet (PBLOCK *pblock, int field, void *ptr, int element)*/
PRIVATE int AddCouplet (PBLOCK *pblock, int field, void *ptr, int element)
{
  FIELD *this ;
  int result ;
  
  if (field >= num_fields-1)
    Exception(UPR_error,__FILE__,__LINE__,"Add element to illegal field") ;

  this = &pblock->field[field] ;

  result = this->current ;
  CheckRange (pblock, field, result+1) ;
  this->current += couple ;
  pblock->pb[result]   = (int) ptr ;
  pblock->pb[result+1] = element ;

  return(result) ;
}
/*}}}*/
/*{{{  PRIVATE int AddData    (PBLOCK *pblock, int field, int num_bytes)*/
PRIVATE int AddData    (PBLOCK *pblock, int field, int num_bytes)
{
  int field_dir = FIELD_DIR(field) ;
  FIELD *this = &pblock->field[field] ;
  int num_words = ((num_bytes-1)/sizeof(int)) + 1 ;
  int result ;

  if (field_dir>0)
  {
    result = this->current ;
    this->current += num_words ;
    CheckRange (pblock, field, this->current-1) ;
  }
  else
  {
    this->current -= num_words ;
    result = this->current ;
    CheckRange (pblock, field, result) ;
  }

  return(result) ;
}
/*}}}*/
/*{{{  PRIVATE int SubData    (PBLOCK *pblock, int num_bytes)*/
PRIVATE int SubData    (PBLOCK *pblock, int num_bytes)
{
  FIELD *this = &pblock->field[vardata_f] ;
  int num_words = ((num_bytes-1)/sizeof(int)) + 1 ;

  this->current -= num_words ;
  CheckRange (pblock, vardata_f, this->current) ;

  return (this->current) ;
}
/*}}}*/
/*}}}*/

/*{{{  PUBLIC int shift_register (int *seed)        feedback = 400431*/
PUBLIC int shift_register (int *seed)
{
  int newbit, priority, random ;

  ProcGetPRI(&priority) ;
  if (priority==PROC_LOW) ProcToHI() ;

  /*{{{  critical code section*/
  random = *seed ;
            
  newbit = (random & 0x00001) ;
  newbit = newbit ^ ((random & 0x00008) >> 3) ;
  newbit = newbit ^ ((random & 0x00010) >> 4) ;
  newbit = newbit ^ ((random & 0x00100) >> 8) ;
  newbit = newbit << 16 ;
  random = random >> 1 ;
  random = random | newbit ;
  *seed  = random ;
  /*}}}*/

  if (priority==PROC_LOW) ProcToLO() ;

  return(random) ;
}
/*}}}*/

/*{{{  RPC library calls*/
/*{{{  PUBLIC PBLOCK *RPC_InitCall (int target, int num_chans, int num_ptrs, int num_vals, int data_size)*/
PUBLIC PBLOCK *RPC_InitCall (int target, int num_chans, int num_ptrs, int num_vals, int data_size)
{
  /* Extra parameter is implicit vecspace pointer */
  int num_params = max (num_ptrs+num_vals+1, min_params) ;
  int pb_size    = num_params + data_size + num_ptrs*(1+couple) + num_chans*couple ;
  PBLOCK *pblock ;
  int *pb ;

  /*{{{  allocate pblock*/
  if ((pblock = (PBLOCK *) Malloc (sizeof(PBLOCK) + pb_size*sizeof(int))) == NULL)
    Exception(UPR_error,__FILE__,__LINE__,"Out of heap whilst allocating RPC pblock") ;
  
  pb = (int *) (pblock + 1) ;
  pblock->pb = pb ;
  /*}}}*/
  /*{{{  partition pblock*/
  {
    int index = 0 ;
  
    pblock->target = VCR_processor_mapping(target) ;
    pblock->move   = no_move ;
    pblock->ret    = new_ret ;
  
    pblock->field[params_f].start   = index ;
    pblock->field[params_f].current = index ;
    index += num_params ;
    
    pblock->field[vardata_f].start   = index ;
    pblock->field[vardata_f].current = index ;
    index += data_size ;
    pblock->field[valdata_f].start   = index ;
    pblock->field[valdata_f].current = index ;
  
    pblock->field[relocs_f].start   = index ;
    pblock->field[relocs_f].current = index ;
    index += num_ptrs ;
    
    pblock->field[chans_f].start   = index ;
    pblock->field[chans_f].current = index ;
    index += num_chans * couple ;
    
    pblock->field[varptrs_f].start   = index ;
    pblock->field[varptrs_f].current = index ;
    index += num_ptrs * couple ;
    
    pblock->field[end_f].start   = index ;
    pblock->field[end_f].current = -1 ;   /* mark end field */
  
    if (index != pb_size)
      Exception(UPR_error,__FILE__,__LINE__,"pblock init fault - total size mismatch") ;
  }
  
  /*}}}*/
  /*{{{  COMMENT CLEAR PB*/
  /*
  {
    int i ;
  
    for (i=0;i<pb_size;i++)
      pb[i] = 0 ;
  }
  */
  /*}}}*/

  return(pblock) ;
}
/*}}}*/
/*{{{  PUBLIC void RPC_DoneCall    (PBLOCK *pblock)*/
PUBLIC void RPC_DoneCall    (PBLOCK *pblock)
{
  if (pblock->ret == new_ret)
  {
    int i ;
    int base = pblock->field[varptrs_f].start ;

    for (i=pblock->field[varptrs_f].current-2; i>=base; i-=2)
    {
      int num_bytes = pblock->pb[i+1] ;
      int source = SubData (pblock, num_bytes) ;

      AsmMove (&pblock->pb[source], pblock->pb[i], num_bytes) ;
    }
  }

  Free(pblock) ;
}
/*}}}*/

/*{{{  PUBLIC void RPC_NewChannel (VCB **local_out, VCB **local_in)*/
PUBLIC void RPC_NewChannel (VCB **local_out, VCB **local_in)
{
  VCR_GLOBALS *g = (VCR_GLOBALS *) (* ((GLOBALS **) GLOBAL_PTR))->vcr_gp ;

  DVC_CreateChannelOut (local_out, local_in, g->proc_id) ;
}  
/*}}}*/
/*{{{  PUBLIC void RPC_DisposeChannel (VCB *local_out, VCB *local_in)*/
PUBLIC void RPC_DisposeChannel (VCB *local_out, VCB *local_in)
{
  DVC_DestroyChannelEnd (local_out) ;
  DVC_DestroyChannelEnd (local_in) ;
}
/*}}}*/

/*{{{  PUBLIC int RPC_ProcId ()*/
PUBLIC int RPC_ProcId ()
{
  VCR_GLOBALS *g = (VCR_GLOBALS *) (* ((GLOBALS **) GLOBAL_PTR))->vcr_gp ;
  return(g->proc_id) ;
}
/*}}}*/
/*{{{  PUBLIC int RPC_NumProcs ()*/
PUBLIC int RPC_NumProcs ()
{
  VCR_GLOBALS *g = (VCR_GLOBALS *) (* ((GLOBALS **) GLOBAL_PTR))->vcr_gp ;
  return(g->num_nodes - 1 - g->num_devices_wow) ;
}
/*}}}*/
/*{{{  PUBLIC int RPC_RandomProc ()*/
PUBLIC int RPC_RandomProc ()
{
  VCR_GLOBALS *g = (VCR_GLOBALS *) (* ((GLOBALS **) GLOBAL_PTR))->vcr_gp ;

  if (g->num_nodes == 1)
    return(0) ;
  else
  {
    int random = shift_register (&g->seed) ;
    
    return(((random >> 2) % (g->num_nodes - 1 - g->num_devices_wow)) + 1) ;
  }
}
/*}}}*/
/*{{{  PUBLIC void RPC_PassByValue (PBLOCK *pblock, int value)*/
PUBLIC void RPC_PassByValue (PBLOCK *pblock, int value)
{
  AddElement (pblock, params_f, value) ;
}
/*}}}*/
/*{{{  PUBLIC void RPC_PassVarByRef(PBLOCK *pblock, char *pointer, int num_bytes)*/
PUBLIC void RPC_PassVarByRef(PBLOCK *pblock, char *pointer, int num_bytes)
{
  int destn ;

  AddCouplet (pblock, varptrs_f, pointer, num_bytes) ;
  
  if (num_bytes == 0)
  {
    AddElement (pblock, relocs_f,
      AddElement (pblock, params_f, pblock->field[vardata_f].current)) ;
  }
  else
  {
    AddElement (pblock, relocs_f,
      AddElement (pblock, params_f,
        (destn=AddData (pblock, vardata_f, num_bytes)))) ;

    AsmMove (pointer, &pblock->pb[destn], num_bytes) ;
  }
}
/*}}}*/
/*{{{  PUBLIC void RPC_PassValByRef(PBLOCK *pblock, char *pointer, int num_bytes)*/
PUBLIC void RPC_PassValByRef(PBLOCK *pblock, char *pointer, int num_bytes)
{
  int destn ;

  if (num_bytes == 0)
  {
    AddElement (pblock, relocs_f,
      AddElement (pblock, params_f, pblock->field[valdata_f].current)) ;
  }
  else
  {
    AddElement (pblock, relocs_f,
      AddElement (pblock, params_f,
        (destn=AddData (pblock, valdata_f, num_bytes)))) ;

    AsmMove (pointer, &pblock->pb[destn], num_bytes) ;
  }
}
/*}}}*/
/*{{{  PUBLIC void RPC_PassChannel (PBLOCK *pblock, VCB *channel)*/
PUBLIC void RPC_PassChannel (PBLOCK *pblock, VCB *channel)
{
  /*{{{  check move mode*/
  switch (pblock->move)
  {
    case no_move :
      pblock->move = implicit_move ;
      break ;
  
    case explicit_move :
      Exception (UPR_error,__FILE__,__LINE__,"RPC uses both old and new style channel moves") ;
  }
  /*}}}*/

  if ((int) channel & virtual_bit == 0)
    Exception(UPR_error,__FILE__,__LINE__,"Can only pass virtual channel parameters") ;

  AddCouplet (pblock, chans_f, channel,
    AddElement (pblock, params_f, (int) channel)) ;
}
/*}}}*/
/*{{{  PUBLIC void RPC_PassChannelArray (PBLOCK *pblock, VCB **channel, int num_chans)*/
PUBLIC void RPC_PassChannelArray (PBLOCK *pblock, VCB **channel, int num_chans)
{
  int destn, i ;
  int num_bytes = num_chans * sizeof(int) ;

  /*{{{  check move mode*/
  switch (pblock->move)
  {
    case no_move :
      pblock->move = implicit_move ;
      break ;
  
    case explicit_move :
      Exception (UPR_error,__FILE__,__LINE__,"RPC uses both old and new style channel moves") ;
  }
  /*}}}*/
  
  AddElement (pblock, relocs_f,
    AddElement (pblock, params_f,
      (destn=AddData (pblock, valdata_f, num_bytes)))) ;

  AsmMove (channel, &pblock->pb[destn], num_bytes) ;
      
  for (i=0;i<num_chans;i++)
  {
    if ((int) channel[i] & virtual_bit == 0)
      Exception(UPR_error,__FILE__,__LINE__,"Can only pass virtual channel parameters") ;
      
    AddCouplet (pblock, chans_f, channel[i], destn+i) ;
  }
}
/*}}}*/
/*{{{  PUBLIC void RPC_Target (PBLOCK *pblock, int target)*/
PUBLIC void RPC_Target (PBLOCK *pblock, int target)
{
  pblock->target = VCR_processor_mapping(target) ;    
}
/*}}}*/
/*}}}*/

/*{{{  VCR 1.8 RPC interface calls retained for compatability*/
/*{{{  PUBLIC void RPC_MoveChannel (PBLOCK *pblock, VCB *local_chan, VCB **remote_chan)*/
PUBLIC void RPC_MoveChannel (PBLOCK *pblock, VCB *local_chan, VCB **remote_chan)
{
  /*{{{  check move mode*/
  switch (pblock->move)
  {
    case no_move :
      pblock->move = explicit_move ;
      break ;
  
    case implicit_move :
      Exception (UPR_error,__FILE__,__LINE__,"RPC uses both old and new style channel moves") ;
  }
  /*}}}*/

  AddElement (pblock, chans_f,
      (int) (*remote_chan = DVC_MoveChannel (pblock->target, local_chan))) ;
}
/*}}}*/
/*{{{  PUBLIC void RPC_ReturnData  (PBLOCK *pblock, char *pointer, int num_bytes)*/
PUBLIC void RPC_ReturnData  (PBLOCK *pblock, char *pointer, int num_bytes)
{
  int source = SubData (pblock, num_bytes) ;

  AsmMove (&pblock->pb[source], pointer, num_bytes) ;
  pblock->ret = old_ret ;
}
/*}}}*/
/*{{{  PUBLIC void RPC_SkipData    (PBLOCK *pblock, int num_bytes)*/
PUBLIC void RPC_SkipData    (PBLOCK *pblock, int num_bytes)
{
  SubData (pblock, num_bytes) ;
  pblock->ret = old_ret ;
}
/*}}}*/
/*}}}*/
