/*{{{  File banner*/
/*@(#)=====================================================*\
||@(#)  Project : PUMA ESPRIT P2701
||@(#)  Authors : Mark Debbage and Mark Hill
||@(#)            University of Southampton
||  
||@(#)    Title : Virtual channels functions
||@(#)   System : VCR
||@(#) Filename : vchan.c
||@(#)  Version : 2.7
\*@(#)====================================================*/
/*}}}*/

#include <stdio.h>
#include <stdlib.h>

#include "vcr.h"
#include "vpktio.h"

static char *_FILE_ = __FILE__ ;

#define USE_MULTI_PACKET_UPRI

/*{{{  common routines*/
/*{{{  PUBLIC void VCB_Init (VCB *vcb, PORT *port, BYTE *buffer, DCB *dcb, int destn_proc, int destn_port)*/
PUBLIC void VCB_Init (VCB *vcb, PORT *port, BYTE *buffer, DCB *dcb, int destn_proc, int destn_port)
{

  vcb->id   = NotProcess_p ;
  vcb->port = port ;
  vcb->dcb  = dcb ;
  vcb->orb.list = InputServiced_p ;
  vcb->orb.bffr = NULL ;
  vcb->buffer = buffer ;

  HdrBuild (destn_proc, destn_port, 0, &(vcb->orb.header)) ;
}
/*}}}*/
/*{{{  PUBLIC void VCB_Free (VCB *vcb)*/
PUBLIC void VCB_Free (VCB *vcb)
{
  if (vcb->id==NotProcess_p)
  {  
    PortFree(vcb->port) ;
    free(vcb) ;
  }
  else
    Exception(UPR_error,_FILE_,__LINE__,"Tried to destroy busy channel end");
}  
/*}}}*/
/*}}}*/

/*{{{  input  routines*/
/*{{{  PUBLIC void VCR_InPreAction (PORT *port, int length)*/
PUBLIC void VCR_InPreAction (PORT *port, int length)
{
  VCB *vcb = (VCB *) port->state ;

  switch ((int) vcb->id)
  {
    case (int) NotProcess_p :
      /*{{{  empty channel end*/
      port->buffer  = (BYTE *) vcb->buffer ;
      port->space   = UPR_MAX_PKT_SIZE ;
      port->do_post = TRUE ;
      break ;
      /*}}}*/

    case (int) InputWaiting_p :
      Exception(UPR_error, _FILE_, __LINE__, "Two early packets on virtual channel") ;
      break ;
  
    default :
      if (port->buffer == NotPointer_p)
        /*{{{  alt has occured */
        {
          port->buffer  = (BYTE *) vcb->buffer ;
          port->space   = UPR_MAX_PKT_SIZE ;
          port->do_post = TRUE ;
          break ;
        }
        /*}}}*/
      else
        /*{{{  commit to communication*/
        {
          if (length > port->space)
            Exception(UPR_error, _FILE_, __LINE__, "Virtual input length mismatch") ;
          
          port->do_pre = FALSE ;
        }
        /*}}}*/
  }  
}
/*}}}*/
/*{{{  PUBLIC void VCR_InPostAction (PORT *port, int length, BYTE *buffer)*/
PUBLIC void VCR_InPostAction (PORT *port, int length, BYTE *buffer)
{
  VCB *vcb = (VCB *) port->state ;

  if (port->do_post)
    /*{{{  check channel state*/
    switch ((int) vcb->id)
    {
      case (int) NotProcess_p :
        /*{{{  channel end still empty*/
        vcb->id = InputWaiting_p ;
        port->buffer  = NotPointer_p ;
        port->do_pre  = FALSE ;
        port->do_post = FALSE ;
        break ;
        /*}}}*/
    
      case (int) InputWaiting_p :
        Exception(UPR_error, _FILE_, __LINE__, "Two early packets on virtual channel") ;
        break ;
    
      default :
        if ((port->buffer >= vcb->buffer) &&
            (port->buffer <= (vcb->buffer + UPR_MAX_PKT_SIZE)))
          /*{{{  alt has occured */
          {
            Channel id = vcb->id ;
            vcb->id = InputWaiting_p ;
            port->buffer  = NotPointer_p ;
            port->do_pre  = FALSE ;
            port->do_post = FALSE ;
            ProcRestartAlt(&id) ;
            break ;
          }
          /*}}}*/
        else
          /*{{{  virtual in has become active*/
          {
            if (length == 0)
              /*{{{  null communication*/
              {
                if (port->space != length)
                  Exception(UPR_error, _FILE_, __LINE__, "Virtual input length mismatch") ;
              
                ProcAwaken(vcb->id) ;
                vcb->id = NotProcess_p ;
                port->buffer  = NotPointer_p ;
                port->do_pre  = TRUE ;
                port->do_post = FALSE ;
              }
              /*}}}*/
            else
              /*{{{  copy packet from wait buffer*/
              {
                if (port->space < length)
                  Exception(UPR_error, _FILE_, __LINE__, "Virtual input length mismatch") ;
                
                AsmMove(vcb->buffer, port->buffer, length) ;
                port->buffer += length ;
                port->space  -= length ;
               
                if (port->space == 0)
                  /*{{{  terminate communication*/
                  {
                    ProcAwaken(vcb->id) ;
                    vcb->id = NotProcess_p ;
                    port->buffer  = NotPointer_p ;
                    port->do_pre  = TRUE ;
                    port->do_post = FALSE ;
                  }
                  /*}}}*/
                else
                  /*{{{  commit to communication*/
                  {
                    port->do_pre  = FALSE ;
                    port->do_post = FALSE ;
                  }
                  /*}}}*/
              }
              /*}}}*/
          }
          /*}}}*/
    }  
    /*}}}*/
  else
    /*{{{  terminate communication*/
    {
      ProcAwaken(vcb->id) ;
      vcb->id = NotProcess_p ;
      port->buffer  = NotPointer_p ;
      port->do_pre  = TRUE ;
      port->do_post = FALSE ;
    }
    /*}}}*/
}               
/*}}}*/

/*{{{  PUBLIC void VirtualIn (VCB *vcb, void *message, int length)*/
PUBLIC void VirtualIn (VCB *vcb, void *message, int length)
{
  BYTE *msg = (BYTE *) message ;

  if (length < 0)
    Exception(UPR_error, _FILE_, __LINE__, "Message input of negative length!") ;

  if (vcb->buffer == NULL)
    Exception(UPR_error, _FILE_, __LINE__, "Virtual input on output channel") ;
  
  if ((vcb->dcb == NULL) && (HdrDestn(&vcb->orb.header) == vcr_globals.proc_id))
    /*{{{  internal virtual communication*/
    {
      #if (UPR_HISTORY != 0)
      UPR_Record (vin_thread,0,vin_internal,(int) &vcb->id) ;
      #endif
      
      SoftChanIn (&vcb->id, msg, length) ;
    }
    /*}}}*/
  else
    /*{{{  external virtual communication*/
    {  
      int priority ;
    
      ProcGetPRI(&priority) ;
      if (priority==PROC_LOW) ProcToHI() ;
    
      /*{{{  critical code section*/
      {
        int size ;
        
        size = min (UPR_MAX_PKT_SIZE, length) ;
      
        /*{{{  if dynamic copy other header into ORB*/
        if (vcb->dcb != NULL)
        {
          HdrCopy (&vcb->dcb->other, &vcb->orb.header) ;
        
          #ifdef DVC_DEBUG
          {
            char s[100];
            sprintf(s,"ACK [ %d:%d -> %d:%d ]",vcr_globals.proc_id, vcb->port->id, HdrDestn(&vcb->orb.header), HdrPort (&vcb->orb.header)) ;
            Exception(UPR_warning,_FILE_,__LINE__,s);
          }
          #endif
        }
        /*}}}*/
        /*{{{  enqueue acknowledgement*/
        #if (UPR_HISTORY != 0)
        UPR_Record (vin_thread,0,vin_send_ack,(int) &vcb->id) ;
        #endif
        
        VCR_Enqueue_NBK (&vcb->orb) ;
        /*}}}*/
        /*{{{  perform transaction*/
        switch ((int) vcb->id)
        {
          case (int) NotProcess_p :
            /*{{{  early virtual input*/
            ProcDesc(&vcb->id) ;
            vcb->port->space  = length ;
            vcb->port->buffer = msg ;
            
            #if (UPR_HISTORY != 0)
            UPR_Record (vin_thread,0,vin_early,(int) &vcb->id) ;
            #endif
            
            ProcSleep() ;
            break ;
            /*}}}*/
        
          case (int) InputWaiting_p :
            /*{{{  late virtual input*/
            #if (UPR_HISTORY != 0)
            UPR_Record (vin_thread,0,vin_late,(int) &vcb->id) ;
            #endif
            
            if (length == 0)
              /*{{{  null communication*/
              {
                if (vcb->port->space != UPR_MAX_PKT_SIZE)
                  Exception(UPR_error, _FILE_, __LINE__, "Virtual input length mismatch") ;
                  
                vcb->id     = NotProcess_p ;
                vcb->port->do_pre = TRUE ;
                vcb->port->space = 0 ;
              }
              /*}}}*/
            else
              /*{{{  copy packet from wait buffer*/
              {
                AsmMove(vcb->buffer, msg, size) ;
                msg += size ;
                length -= size ;
              
                if (vcb->port->space != UPR_MAX_PKT_SIZE-size)
                  Exception(UPR_error, _FILE_, __LINE__, "Virtual input length mismatch") ;
              
                if (length==0)
                {
                  vcb->id     = NotProcess_p ;
                  vcb->port->do_pre = TRUE ;
                  vcb->port->space = 0 ;
                }
                else
                {
                  ProcDesc(&vcb->id) ;
                  vcb->port->buffer = msg ;
                  vcb->port->space  = length ;
                  ProcSleep() ;
                }      
              }
              /*}}}*/
            
            break ;
            /*}}}*/
        
          default :
            Exception(UPR_error, _FILE_, __LINE__, "Invalid id on virtual in entry") ;
        }
        /*}}}*/
        /*{{{  check channel state*/
        if ((vcb->id != NotProcess_p) && (vcb->id != InputWaiting_p))
          Exception(UPR_error, _FILE_, __LINE__, "Invalid input id on virtual in exit") ;
        /*}}}*/
        /*{{{  block if acknowledge still here*/
        UPR_ORB_BK (&vcb->orb) ;
        /*}}}*/
      
        #if (UPR_HISTORY != 0)
        UPR_Record (vin_thread,0,vin_done,(int) &vcb->id) ;
        #endif
      }
      
      /*}}}*/
    
      if (priority==PROC_LOW) ProcToLO() ;
    }
    /*}}}*/
}
/*}}}*/
/*{{{  PUBLIC void VChanIn  (VCB *c, void *message, int length)*/
PUBLIC void VChanIn  (VCB *c, void *message, int length)
{
  if (((int) c & virtual_bit) != 0)
    VirtualIn ( (VCB *) ((int) c ^ virtual_bit), message, length) ;
  else
    SoftChanIn ( c, message, length) ;
}    
/*}}}*/

/*{{{  PUBLIC VCB *VCB_InAlloc (int destn_proc, int destn_port)*/
PUBLIC VCB *VCB_InAlloc (int destn_proc, int destn_port)
{
  VCB  *vcb ;
  PORT *port ;

  if ((vcb = (VCB *) malloc (sizeof(VCB)+UPR_MAX_PKT_SIZE)) == NULL)
    Exception (UPR_error, _FILE_, __LINE__, "Out of heap space whilst allocating vcb") ;

  port = PortAlloc (VCR_InPreAction, TRUE, VCR_InPostAction, FALSE, (void *) vcb);
    
  VCB_Init (vcb, port, (BYTE *) (vcb+1), NULL, destn_proc, destn_port) ;

  return(vcb) ;
}
/*}}}*/
/*}}}*/

/*{{{  output routines*/
/*{{{  PUBLIC void VCR_OutPreAction (PORT *port, int length)*/
PUBLIC void VCR_OutPreAction (PORT *port, int length)
{
  VCB *vcb = (VCB *) port->state ;

  if (length !=0)
    Exception(UPR_error, _FILE_, __LINE__, "Unsolicited packet on acknowledge port") ;

  switch ((int) vcb->id)
  {
    case (int) NotProcess_p :
      vcb->id = AckWaiting_p ;
      break ;

    case (int) AckWaiting_p :
      Exception(UPR_error, _FILE_, __LINE__, "Out of protocol acknowledgement received") ;
      break ;

    default :
      ProcAwaken(vcb->id) ;
      vcb->id = NotProcess_p ;
  }  
}               
/*}}}*/
/*{{{  PUBLIC void VCR_OutPostAction (PORT *port, int length, BYTE *buffer)*/
PUBLIC void VCR_OutPostAction (PORT *port, int length, BYTE *buffer)
{
  ;
}
/*}}}*/

/*{{{  PUBLIC void VirtualOut (VCB *vcb, void *message, int length)*/
PUBLIC void VirtualOut (VCB *vcb, void *message, int length)
{
  BYTE *msg = (BYTE *) message ;

  if (length < 0)
    Exception(UPR_error, _FILE_, __LINE__, "Message output of negative length!") ;

  if (vcb->buffer != NULL)
    Exception(UPR_error, _FILE_, __LINE__, "Virtual output on input channel") ;
  
  if ((vcb->dcb == NULL) && (HdrDestn(&vcb->orb.header) == vcr_globals.proc_id))
    /*{{{  internal virtual communication*/
    {
      VCB *vcb_in = (VCB *) (PortPtr(HdrPort(&vcb->orb.header))->state) ;
    
      #if (UPR_HISTORY != 0)
      UPR_Record (vout_thread,0,vout_internal,(int) &vcb_in->id) ;
      #endif
      
      SoftChanOut (&vcb_in->id, msg, length) ;
    }
    /*}}}*/
  else
    /*{{{  external virtual communication*/
    {  
      int priority ;
    
      ProcGetPRI(&priority) ;
      if (priority==PROC_LOW) ProcToHI() ;
    
      /*{{{  critical code section*/
      {
        int size ;
        
        /*{{{  if dynamic copy other header into ORB*/
        if (vcb->dcb != NULL)
        {
          HdrCopy (&vcb->dcb->other, &vcb->orb.header) ;
        
          #ifdef DVC_DEBUG
          {
            char s[100];
            sprintf(s,"PKT [ %d:%d -> %d:%d ]",vcr_globals.proc_id, vcb->port->id, HdrDestn(&vcb->orb.header), HdrPort (&vcb->orb.header)) ;
            Exception(UPR_warning,_FILE_,__LINE__,s);
          }
          #endif
        }
        /*}}}*/
        /*{{{  enqueue first packet*/
        size = min (UPR_MAX_PKT_SIZE, length) ;
        HdrModLength(size, &vcb->orb.header) ;
        vcb->orb.bffr = msg ;
        
        #if (UPR_HISTORY != 0)
        UPR_Record (vout_thread,0,vout_send_first,(int) &vcb->id) ;
        #endif
        
        VCR_Enqueue_NBK (&vcb->orb) ;
          
        msg += size ;
        length  -= size ;
        /*}}}*/
        /*{{{  block if acknowledgement not here*/
        switch ((int) vcb->id)
        {
          case (int) NotProcess_p :
            /*{{{  block until ack arrives*/
            ProcDesc(&vcb->id) ;
            
            #if (UPR_HISTORY != 0)
            UPR_Record (vout_thread,0,vout_early,(int) &vcb->id) ;
            #endif
            
            ProcSleep() ;
            break ;
            /*}}}*/
        
          case (int) AckWaiting_p :
            vcb->id = NotProcess_p ;
            break ;
        
          default :
            Exception(UPR_error, _FILE_, __LINE__, "Invalid id on virtual out entry") ;
        }
        /*}}}*/
        /*{{{  block if first packet still here*/
        UPR_ORB_BK (&vcb->orb) ;
        /*}}}*/
      
        #ifdef USE_MULTI_PACKET_UPRI
          /*{{{  enqueue remaining packets with one blocking call*/
          if (length != 0)
          {
            vcb->orb.bffr = msg ;
            VCR_MultipleEnqueue_BK(&vcb->orb,length) ;
          }
          /*}}}*/
        #else
          /*{{{  enqueue remaining packets with repeated blocking calls*/
          while (length != 0)
          {
            size = min (UPR_MAX_PKT_SIZE, length) ;
            HdrModLength(size, &vcb->orb.header) ;
            vcb->orb.bffr = msg ;
          
            #if (UPR_HISTORY != 0)
            UPR_Record (vout_thread,0,vout_send_next,(int) &vcb->id) ;
            #endif
          
            VCR_Enqueue_BK (&vcb->orb) ;
              
            msg += size ;
            length  -= size ;
          }
          /*}}}*/
        #endif
        
        /*{{{  check channel state*/
        if ((vcb->id != NotProcess_p) && (vcb->id != AckWaiting_p))
          Exception(UPR_error, _FILE_, __LINE__, "Invalid id on virtual out exit") ;
        /*}}}*/
      
        #if (UPR_HISTORY != 0)
        UPR_Record (vout_thread,0,vout_done,(int) &vcb->id) ;
        #endif
      }
      /*}}}*/
    
      if (priority==PROC_LOW) ProcToLO() ;
    }
    /*}}}*/
}
/*}}}*/
/*{{{  PUBLIC void VChanOut (VCB *c, void *message, int length)*/
PUBLIC void VChanOut (VCB *c, void *message, int length)
{
  if (((int) c & virtual_bit) != 0)
    VirtualOut ( (VCB *) ((int) c ^ virtual_bit), message, length) ;
  else
    SoftChanOut ( c, message, length) ;
}    
/*}}}*/

/*{{{  PUBLIC VCB *VCB_OutAlloc (int destn_proc, int destn_port)*/
PUBLIC VCB *VCB_OutAlloc (int destn_proc, int destn_port)
{
  VCB  *vcb ;
  PORT *port ;

  if ((vcb = (VCB *) malloc (sizeof(VCB))) == NULL)
    Exception (UPR_error, _FILE_, __LINE__, "Out of heap space whilst allocating vcb") ;
  
  port = PortAlloc (VCR_OutPreAction, TRUE, VCR_OutPostAction, FALSE, (void *) vcb) ;

  VCB_Init (vcb, port, NULL, NULL, destn_proc, destn_port) ;

  return(vcb) ;
}
/*}}}*/
/*}}}*/

