/*{{{  Title*/
/*  TITLE : NETWORK VIRTUALISATION PROGRAM
    --------------------------------------

    VERSION : 1.2   31/5/91   Uses stdin/stdout as input/output files
                              Messages use stderr with silent option
                              Single processor networks handled
                              Dynamic arrays now global with macros
                              Memory required messages added
                              ANSI C compatible
                              Reverse links included and double links handled
              1.1   28/2/91   Removal of bug and modified *.ncf file format
              1.0   21/1/91   Initial program derived from genjmx.c

    AIM     : Generate a deadlock-free network configuration file, by
              optimised addition of virtual links, from the output file
              generated by router.c.

    BASED   : Optimised virtual link insertion algorithm obtained from
              genjmx.c.

    AUTHORS : David Pritchard, Mark Debbage and Mark Hill

    USAGE   : virtual [-s] <infile >outfile
              -s      : Disable screen messages
              infile  : Routing tables input filename (*.brf)
              outfile : Network configuration output filename (*.ncf)
*/
/*}}}*/
/*{{{  Libraries and constants*/
#include <stdio.h>
#include <ctype.h>
#include <string.h>
#ifdef __ZTC__
#include <stdlib.h>
#endif
#ifdef _ICC
#include <stdlib.h>
#endif

#define not_used         -1
#define TRUE             1
#define FALSE            0
#define no_arc           -1
#define max_dependency   3
/*}}}*/
/*{{{  Global declarations*/
long int silent = FALSE, route_size, dist_size, mask [2] ;

/* Array declarations */
char *processor ;           /* Processor type of node n (char c) */
long int *mem_size ;        /* Memory size of node n */
long int *node_con ;        /* Node connected to link l from node n */
long int *reverse_link_no ; /* Reverse link of link l from node n */
long int *route ;           /* Link from node s to reach node d */
long int *routed_flag ;     /* Flag indicating node n is routed */
long int *boot_from ;       /* Node and link from which node n is booted from */
long int *branch_conv ;     /* Equivalent node to level a, leaf b */
long int *ldg_matrix ;      /* dth absolute link on absolute link al */
long int *distance ;        /* No of virtual links from node n to dest node */
/*}}}*/
/*{{{  Macro definitions*/
#define PROCESSOR(n,c)        (*(processor + (n) * 8 +(c)))
#define MEM_SIZE(n)           (*(mem_size + (n)))
#define NODE_CON(n,l)         (*(node_con + (n) * 4 + (l)))
#define REVERSE_LINK_NO(n,l)  (*(reverse_link_no + (n) * 4 + (l)))
#define ROUTE(s,d)            (*(route + (d) * route_size + (s) / 15))
#define ROUTED_FLAG(n)        (*(routed_flag + (n)))
#define BOOT_FROM(n)          (*(boot_from + (n)))
#define BRANCH_CONV(a,b)      (*(branch_conv + base [a] + (b) / 2))
#define LDG_MATRIX(al,d)      (*(ldg_matrix + (d) * no_abs_links + (al)))
#define DISTANCE(s,d)         (*(distance + (d) * dist_size + (s) / 7))
#define CLEAR_ABS_LINK(al)    (*(clear_abs_link + (al)))
#define CYCLE(p)              (*(cycle + (p)))
#define COST(al,l)            (*(cost + (al) * 4 + (l)))
#define INC_DIST_FLAG(n)      (*(inc_dist_flag + (n)))

#define ROUTE_write(s,d,l)           ROUTE(s,d) += (l) << (((s) % 15) * 2)
#define ROUTE_read(s,d)              (ROUTE(s,d) % (1 << ((((s) % 15) + 1) * 2))) / (1 << (((s) % 15) * 2))
#define BOOT_FROM_write(n,d,l)       BOOT_FROM(n) = (d) * 4 + (l)
#define BOOT_FROM_node(n)            (BOOT_FROM(n) / 4)
#define BOOT_FROM_link(n)            (BOOT_FROM(n) % 4)
#define BRANCH_CONV_write(a,b,n)     BRANCH_CONV(a,b) += (n) << (((b) % 2) * 16)
#define BRANCH_CONV_null_route(a,b)  BRANCH_CONV(a,b) = (BRANCH_CONV(a,b) & mask [((b) + 1) % 2]) + mask [(b) % 2]
#define BRANCH_CONV_read(a,b)        ((BRANCH_CONV(a,b) >> (((b) % 2) * 16)) & mask[0])
#define DISTANCE_inc(s,d)            DISTANCE(s,d) += 1 << (((s) % 7) * 4)
#define DISTANCE_read(s,d)           (DISTANCE(s,d) % (1 << ((((s) % 7) + 1) * 4))) / (1 << (((s) % 7) * 4))
/*}}}*/

/*{{{  Procedure to find cycles in LDG and break by virtualising links*/
void add_virtual_links (number_nodes, no_abs_links)
/*{{{  Declarations*/
long int number_nodes, no_abs_links ;
{    long int clear_flag, reduced_flag, ldg_cycle, node, dest_node, next_node ;
     long int abs_link, dependency, new_abs_link, count, new_cost ;
     long int cycle_size, cycle_start, current_node, best_cost, best_pos, link ;
     long int abs_link_a, abs_link_b, no_classes = 0, init_flag = FALSE ;
     long int *clear_abs_link ; /* Flag showing abs link al is removed from LDG */
     long int *cycle ;          /* Absolute link in position p of LDG cycle */
     long int *cost ;           /* Max class dependent on abs link al, next link l */
     long int *inc_dist_flag ;  /* Flag showing abs link al needs inc in distance */
/*}}}*/
/*{{{  Initialisation*/
     /* clear_abs_link indicates whether each link is removed from LDG */
     if ((clear_abs_link = (long int *) malloc (no_abs_links * 4)) == NULL)
     {    fprintf (stderr, "\nError-%s(%d)- Failed to allocate memory for clear_abs_link array\n",__FILE__,__LINE__) ;
          exit (-1) ;
     }
     /* cycle stores links within the current LDG cycle */
     if ((cycle = (long int *) malloc (no_abs_links * 4)) == NULL)
     {    fprintf (stderr, "\nError-%s(%d)- Failed to allocate memory for cycle array\n",__FILE__,__LINE__) ;
          exit (-1) ;
     }
     /* cost stores the maximum buffer class dependent on each link */
     if ((cost = (long int *) malloc (no_abs_links * 16)) == NULL)
     {    fprintf (stderr, "\nError-%s(%d)- Failed to allocate memory for cost array\n",__FILE__,__LINE__) ;
          exit (-1) ;
     }
     /* inc_dist_flag indicates which nodes require increase in distance */
     if ((inc_dist_flag = (long int *) malloc (number_nodes * 4)) == NULL)
     {    fprintf (stderr, "\nError-%s(%d)- Failed to allocate memory for inc_dist_flag array\n",__FILE__,__LINE__) ;
          exit (-1) ;
     }
     /* Clear clear_abs_link and cost arrays */
     for (abs_link = 0; abs_link < no_abs_links; abs_link++)
     {    CLEAR_ABS_LINK(abs_link) = FALSE ;
          for (link = 0; link < 4; link++)
              COST(abs_link,link) = 0 ;
     }
/*}}}*/
     do
     {
          /*{{{  Search LDG matrix for clear rows/links and reduce*/
          reduced_flag = TRUE ;
          /* Loop until no reductions are made to LDG matrix */
          while (reduced_flag)
          {    reduced_flag = FALSE ;
               /*{{{  Search for clear rows and remove corresponding links*/
               for (abs_link = 0; abs_link < no_abs_links; abs_link++)
               {    /* If first iteration after virtualising a link,
                       check the removed link only */
                    if (init_flag == TRUE) abs_link = abs_link_a ;
                    if (CLEAR_ABS_LINK(abs_link) == FALSE)
                    {    clear_flag = TRUE ;
                         for (dependency = 0; dependency < max_dependency; dependency++)
                              if (LDG_MATRIX(abs_link,dependency) != no_arc) clear_flag = FALSE ;
                         /* If new clear row found then set flags and remove
                            corresponding link from LDG */
                         if (clear_flag)
                         {    reduced_flag = TRUE ;
                              CLEAR_ABS_LINK(abs_link) = TRUE ;
                              for (new_abs_link = 0; new_abs_link < no_abs_links; new_abs_link++)
                                   for (dependency = 0; dependency < max_dependency; dependency++)
                                        if (LDG_MATRIX(new_abs_link,dependency) == abs_link)
                                             LDG_MATRIX(new_abs_link,dependency) = no_arc ;
                         }
                    }
                    if (init_flag == TRUE) break ;
               }
               /*}}}*/
               /*{{{  Search for clear links and remove corresponding rows*/
               for (abs_link = 0; abs_link < no_abs_links; abs_link++)
               {    /* If first iteration after virtualising a link,
                       check the removed link only */
                    if (init_flag == TRUE) abs_link = abs_link_b ;
                    if (CLEAR_ABS_LINK(abs_link) == FALSE)
                    {    clear_flag = TRUE ;
                         for (new_abs_link = 0; new_abs_link < no_abs_links; new_abs_link++)
                              for (dependency = 0; dependency < max_dependency; dependency++)
                                   if (LDG_MATRIX(new_abs_link,dependency) == abs_link) clear_flag = FALSE ;
                         /* If current link not present in LDG, set flags
                            and remove corresponding row */
                         if (clear_flag)
                         {    reduced_flag = TRUE ;
                              CLEAR_ABS_LINK(abs_link) = TRUE ;
                              for (dependency = 0; dependency < max_dependency; dependency++)
                                   LDG_MATRIX(abs_link,dependency) = no_arc ;
                         }
                    }
                    if (init_flag == TRUE) break ;
               }
               /*}}}*/
               init_flag = FALSE ;
          } ;
          /*}}}*/
          /*{{{  Check whether a cycle exists in LDG matrix*/
          ldg_cycle = FALSE ;
          for (abs_link = 0; abs_link < no_abs_links; abs_link++)
               if (CLEAR_ABS_LINK(abs_link) == FALSE)
               {    ldg_cycle = TRUE ;
                    break ;
               }
          /*}}}*/
          /* If a cycle is present then find best link and virtualise */
          if (ldg_cycle)
          {
               /*{{{  Find links which form the cycle*/
               /* Find a link not yet removed from LDG matrix */
               for (abs_link = 0; abs_link < no_abs_links; abs_link++)
                    if (CLEAR_ABS_LINK(abs_link) == FALSE) break ;
               cycle_size = 0 ;
               /* Find link which depends on current link and repeat
                  until a cycle is formed */
               do
               {    CYCLE(cycle_size) = abs_link ;
                    /* Find next link in dependency cycle */
                    for (dependency = 0; dependency < max_dependency; dependency++)
                         if (LDG_MATRIX(abs_link,dependency) != no_arc) break ;
                    abs_link = LDG_MATRIX(abs_link,dependency) ;
                    /* Check whether cycle present yet, by searching cycle
                       array for a match to the new link */
                    clear_flag = TRUE ;
                    for (count = 0; count < cycle_size; count++)
                         if (CYCLE(count) == abs_link)
                         {    clear_flag = FALSE ;
                              break ;
                         }
                    cycle_size++ ;
               }
               while (clear_flag) ;
               /* Calculate cycle start and size */
               cycle_start = count ;
               cycle_size = cycle_size - cycle_start ;
               /*}}}*/
               /*{{{  Find best link pair within the cycle*/
               best_cost = 16 ;
               /* Find link pair in cycle which have the lowest cost */
               for (count = cycle_start; count < cycle_start + cycle_size; count++)
               {    link = CYCLE((count - cycle_start + 1) % cycle_size + cycle_start) % 4 ;
                    if (COST(CYCLE(count),link) < best_cost)
                    {     best_cost = COST(CYCLE(count),link) ;
                          best_pos = count ;
                    }
               }
               abs_link_a = CYCLE(best_pos) ;
               abs_link_b = CYCLE((best_pos - cycle_start + 1) % cycle_size + cycle_start) ;
               if (!silent) fprintf (stderr, "%ld->%ld ", abs_link_a, abs_link_b) ;
               /*}}}*/
               /*{{{  Update virtual distance table and cost matrix*/
               for (dest_node = 0; dest_node < number_nodes; dest_node++)
               {
                    /*{{{  Find nodes which route via link pair to dest node*/
                    for (node = 0; node < number_nodes; node++)
                         INC_DIST_FLAG(node) = FALSE ;
                    for (node = 0; node < number_nodes; node++)
                         if (node != dest_node)
                         {    /* Trace route from node to dest node until
                                 reach dest node or node in centre of
                                 selected link pair */
                              next_node = node ;
                              do
                              {    current_node = next_node ;
                                   next_node = NODE_CON(next_node,ROUTE_read(next_node,dest_node)) ;
                              }
                              while ((next_node != abs_link_b / 4) && (next_node != dest_node)) ;
                              /* If route is via selected link pair then
                                 set inc_dist_flag */
                              if ((next_node != dest_node) && (current_node == abs_link_a / 4) && (ROUTE_read(next_node,dest_node) == abs_link_b % 4))
                                   INC_DIST_FLAG(node) = TRUE ;
                         }
                    /*}}}*/
                    /* Increment distance and update costs if route from
                       current node to dest node is via link pair */
                    for (node = 0; node < number_nodes; node++)
                         if (INC_DIST_FLAG(node) == TRUE)
                         {
                              /*{{{  Increment distance matrix*/
                              DISTANCE_inc(node,dest_node) ;
                              new_cost = DISTANCE_read(node,dest_node) ;
                              if (new_cost > no_classes) no_classes = new_cost ;
                              /*}}}*/
                              /*{{{  Update cost matrix*/
                              next_node = node ;
                              link = ROUTE_read(node,dest_node) ;
                              /* For each link on route from node to dest
                                 node check cost is at least new distance */
                              do
                              {    abs_link = next_node * 4 + link ;
                                   next_node = NODE_CON(next_node,link) ;
                                   link = ROUTE_read(next_node,dest_node) ;
                                   if (next_node != dest_node)
                                        if (COST(abs_link,link) < new_cost)
                                             COST(abs_link,link) = new_cost ;
                              }
                              while (next_node != dest_node) ;
                              /*}}}*/
                         }
               }
               /*}}}*/
               /*{{{  Remove selected link pair edge from LDG matrix*/
               count = 0 ;
               while (LDG_MATRIX(abs_link_a,count) != abs_link_b) count++ ;
               LDG_MATRIX(abs_link_a,count) = no_arc ;
               init_flag = TRUE ;
               /*}}}*/
          }
     }
     while (ldg_cycle) ;
     if (!silent) fprintf (stderr, "\n\nNumber of buffer classes = %ld", no_classes + 1) ;
     free (clear_abs_link) ;
     free (cycle) ;
     free (cost) ;
     free (inc_dist_flag) ;
}
/*}}}*/

int main (argc, argv)
/*{{{  Declarations*/
int argc ;
char *argv[] ;
{ long int node, start_node, last_leaf = 1, base [15], dest_node, no_routed ;
long int link, current_node, count, abs_link, next_abs_link, next_node, new_leaf ;
long int base_node, root_node, root_link, leaf, level, entry ;
long int number_nodes, no_abs_links, diameter ;
char buffer ;
/*}}}*/
/*{{{  Check for silent parameter*/
if (argc > 1)
     if ((argv [1] [0] == '-') || (argv [1] [0] == '/'))
          if ((argv [1] [1] == 's') || (argv [1] [1] == 'S'))
               silent = TRUE ;
/*}}}*/
/*{{{  Read in information from routing input file*/
if (!silent)
{    fprintf (stderr, "\n\n               NETWORK VIRTUALISATION PROGRAM") ;
     fprintf (stderr, "\n\n\nReading network routing file") ;
}
/*{{{  Read in header line of routing file*/
/* Read in number of nodes, diameter, root node and root link */
if (fscanf (stdin, "%ld %ld %ld %ld", &number_nodes, &diameter, &root_node, &root_link) != 4)
{    /* Try next line if header line incorrect */
     while (fgetc (stdin) != '\n') ;           /* Skip rest of line */
     if (fscanf (stdin, "%ld %ld %ld %ld", &number_nodes, &diameter, &root_node, &root_link) != 4)
     {    fprintf (stderr, "\nError-%s(%d)- Routing file contains error in header line data\n",__FILE__,__LINE__) ;
          return (-1) ;
     }
}
/*}}}*/
/*{{{  Initialisation of arrays used to extract routing file information*/
/* processor stores string holding processor imformation for each node */
if ((processor = (char *) malloc (number_nodes * 8)) == NULL)
{    fprintf (stderr, "\nError-%s(%d)- Failed to allocate memory for processor array\n",__FILE__,__LINE__) ;
     return (-1) ;
}
/* mem_size stores total memory size for each node */
if ((mem_size = (long int *) malloc (number_nodes * 4)) == NULL)
{    fprintf (stderr, "\nError-%s(%d)- Failed to allocate memory for mem_size array\n",__FILE__,__LINE__) ;
     return (-1) ;
}
/* node_con stores node number at the end of each link of each node */
if ((node_con = (long int *) malloc (number_nodes * 16)) == NULL)
{    fprintf (stderr, "\nError-%s(%d)- Failed to allocate memory for node_con array\n",__FILE__,__LINE__) ;
     return (-1) ;
}
/* reverse_link_no stores link number which is at the other end of each link */
if ((reverse_link_no = (long int *) malloc (number_nodes * 16)) == NULL)
{    fprintf (stderr, "\nError-%s(%d)- Failed to allocate memory for reverse_link_no array\n",__FILE__,__LINE__) ;
     return (-1) ;
}
route_size = number_nodes / 15 + 1 ;
/* route gives link to leave source node on in order to reach dest node */
if ((route = (long int *) malloc (number_nodes * route_size * 4)) == NULL)
{    fprintf (stderr, "\nError-%s(%d)- Failed to allocate memory for route array\n",__FILE__,__LINE__) ;
     return (-1) ;
}
/* Clear route array */
for (count = 0; count < route_size * number_nodes; count++)
     *(route + count) = 0 ;
/*}}}*/
/*{{{  Extract routing file information*/
for (entry = 0; entry < number_nodes; entry++)
{    while (fgetc (stdin) != '\n') ;           /* Skip rest of line */
     while (fgetc (stdin) != '\n') ;           /* Skip rest of line */
     if (fscanf (stdin, "%ld", &node) != 1)     /* Read node number */
     {    fprintf (stderr, "\nError-%s(%d)- Routing file contains insufficient data for declared number of nodes\n",__FILE__,__LINE__) ;
          return (-1) ;
     }
     if ((node < 0) || (node > number_nodes))
     {    fprintf (stderr, "\nError-%s(%d)- Routing file contains node greater than declared number of nodes\n",__FILE__,__LINE__) ;
          return (-1) ;
     }
     while (fgetc (stdin) != '\n') ;           /* Skip rest of line */
     /* Read processor type */
     count = 0 ;
     buffer = fgetc (stdin) ;
     while (buffer != '\n')
     {    if (count < 8) PROCESSOR(node,count) = buffer ;
          count++ ;
          buffer = fgetc (stdin) ;
     }
     if (fscanf (stdin, "%lu", mem_size + node) != 1)  /* memory size */
     {    fprintf (stderr, "\nError-%s(%d)- Routing file contains incorrect memory size for node %ld\n",__FILE__,__LINE__,node) ;
          return (-1) ;
     }
     while (fgetc (stdin) != '\n') ;           /* Skip rest of line */
     /*{{{  Read in node and reverse link number to each link of current node*/
     for (link = 0; link < 4; link++)
     {    if ((fscanf (stdin, "%ld", node_con + node * 4 + link) != 1) || (NODE_CON(node,link) < -1) || (abs (NODE_CON(node,link)) > number_nodes))
          {    fprintf (stderr,"\nError-%s(%d)- Routing file contains incorrect link connection data for node %ld\n",__FILE__,__LINE__,node) ;
               return (-1) ;
          }
          if (NODE_CON(node,link) != not_used)
              if ((fscanf (stdin, ":%ld", reverse_link_no + node * 4 + link) != 1) || (REVERSE_LINK_NO(node,link) < 0) || (REVERSE_LINK_NO(node,link) > 3))
              {    fprintf (stderr,"\nError-%s(%d)- Routing file contains incorrect link connection data for node %ld\n",__FILE__,__LINE__,node) ;
                   return (-1) ;
              }
     }
     while (fgetc (stdin) != '\n') ;           /* Skip rest of line */
     /*}}}*/
     /*{{{  Read in output link number to take in order to reach each dest node*/
     for (dest_node = 0; dest_node < number_nodes; dest_node++)
     {    if ((fscanf (stdin, "%ld", &link) != 1) || (link < -1) || (link > 3) || ((link == -1) && (node != dest_node)))
          {    fprintf (stderr, "\nError-%s(%d)- Routing file contains incorrect routing data for node %ld\n",__FILE__,__LINE__,node) ;
               return (-1) ;
          }
          if (dest_node != node) ROUTE_write(node,dest_node,link) ;
     }
     /*}}}*/
}
/*}}}*/
/*}}}*/
/*{{{  Construct tree using root node as root*/
/*{{{  Initialisation*/
mask [0] = 65535 ;
mask [1] = -65536 ;
/* Generate base array which is used to calculate overall leaf number
   from level and leaf number on that level */
base[0] = 0 ;
for (count = 1; count <= diameter; count++)
     base [count] = base [count - 1] + (1 << (count * 2 - 1)) ;
if (!silent) fprintf (stderr, "\nCreating boot tree - Dynamic memory requirement = %ld bytes",
        number_nodes * (52 + route_size * 4) + base[diameter] * 4) ;

/* routed_flag indicates whether each source node has been routed yet */
if ((routed_flag = (long int *) malloc (number_nodes * 4)) == NULL)
{    fprintf (stderr, "\nError-%s(%d)- Failed to allocate memory for routed_flag array\n",__FILE__,__LINE__) ;
     return (-1) ;
}
/* boot_from contains node number from which node is booted from */
if ((boot_from = (long int *) malloc (number_nodes * 4)) == NULL)
{    fprintf (stderr, "\nError-%s(%d)- Failed to allocate memory for boot_from array\n",__FILE__,__LINE__) ;
     return (-1) ;
}
/* branch_conv stores node which corresponds to level,leaf number */
if ((branch_conv = (long int *) malloc (base [diameter] * 4)) == NULL)
{    fprintf (stderr, "\nError-%s(%d)- Failed to allocate memory for branch_conv array\n",__FILE__,__LINE__) ;
     return (-1) ;
}
/* Clear branch_conv array */
for (count = 0; count < base [diameter]; count++)
    *(branch_conv + count) = 0 ;
/*}}}*/
for (node = 0; node < number_nodes; node++)
     ROUTED_FLAG(node) = FALSE ;
ROUTED_FLAG(root_node) = TRUE ;
no_routed = 1 ;
level = 0;
do
{    if ((level >= diameter) && (level > 0))
     {    fprintf (stderr, "\nError-%s(%d)- Actual diameter is greater than declared diameter of %ld\n",__FILE__,__LINE__,diameter) ;
          return (-1) ;
     }
     for (leaf = 0; leaf < last_leaf; leaf++)
     {    if (level == 0) base_node = root_node ;
          else base_node = BRANCH_CONV_read(level - 1,leaf) ;
          /* Find the 4 nodes connected to the current leaf node */
          for (link = 0; link < 4; link++)
          {    new_leaf = leaf * 4 + link ;
               if (abs (base_node) == mask [0]) node = not_used ;
               else node = NODE_CON(base_node,link) ;
               if (node == not_used) BRANCH_CONV_null_route(level,new_leaf) ;
               /* Store node which corresponds to current branch */
               else
               {    BRANCH_CONV_write(level,new_leaf,node) ;
                    if (ROUTED_FLAG(node) == FALSE)
                    {    ROUTED_FLAG(node) = TRUE ;
                         no_routed++ ;
                    }
               }
          }
     }
     last_leaf *= 4 ;
     level++ ;
}
while (no_routed < number_nodes) ;
/*}}}*/
/*{{{  Output information for each processor in distance order from root node*/
if (!silent) fprintf (stderr, "\nWriting out processor information in distance order") ;
/* Output header line */
fprintf (stdout, "%ld %ld\n", number_nodes, diameter) ;
/*{{{  Initialisation*/
for (node = 0; node < number_nodes; node++)
{    ROUTED_FLAG(node) = FALSE ;
     BOOT_FROM(node) = not_used ;
}
no_routed = 0 ;
leaf = -1 ;
/*}}}*/
do
{    /* Find next node on tree in distance order from root node */
     if (leaf == -1) node = root_node ;
     else node = BRANCH_CONV_read(0,leaf) ;
     /* If current node is valid and not yet been output, then output
        information on that node */
     if (abs (node) != mask [0])
          if (ROUTED_FLAG(node) == FALSE)
          {    /* Output processor number, type and memory values */
               fprintf (stdout,"\n%ld\n", node) ;
               for (count = 0; count < 8; count++)
                    fputc (PROCESSOR(node,count), stdout) ;
               fprintf (stdout, "\n%ldK\n", MEM_SIZE(node)) ;
               /*{{{  Output link information*/
               /* Output link type : x = not used, h = host connection,
                  i = input link on boot tree, o = output link on boot tree,
                  n = non-special link */
               for (link = 0; link < 4; link++)
               {    next_node = NODE_CON(node,link) ;
                    if ((node == root_node) && (link == root_link)) fprintf (stdout, "h ") ;
                    else if (next_node == not_used) fprintf (stdout, "x ") ;
                         else if ((BOOT_FROM_node(node) == next_node) && (BOOT_FROM_link(node) == link)) fprintf (stdout, "i%ld:%ld ", next_node, REVERSE_LINK_NO(node,link)) ;
                              else if ((BOOT_FROM(next_node) == not_used) && (next_node != root_node))
                                   {    fprintf (stdout, "o%ld:%ld ", next_node, REVERSE_LINK_NO(node,link)) ;
                                        BOOT_FROM_write(next_node,node,REVERSE_LINK_NO(node,link)) ;
                                   }
                                   else fprintf (stdout, "n%ld:%ld ", next_node, REVERSE_LINK_NO(node,link)) ;
               }
               fprintf (stdout, "\n") ;
               /*}}}*/
               ROUTED_FLAG(node) = TRUE ;
               no_routed++ ;
          }
     leaf++ ;
}
while (no_routed < number_nodes) ;
/*{{{  Deallocate arrays*/
free (branch_conv) ;
free (routed_flag) ;
free (mem_size) ;
free (boot_from) ;
free (processor) ;
free (reverse_link_no) ;
/*}}}*/
/*}}}*/
/*{{{  Search for cycles of dependency and virtualise links as necessary*/
/*{{{  Initialisation*/
no_abs_links = number_nodes * 4 ;
dist_size = number_nodes / 7 + 1 ;
if (!silent) fprintf (stderr, "\n\nMapping virtual links - Dynamic memory requirement = %ld bytes",
        no_abs_links * (29 + route_size + dist_size + max_dependency * 4)) ;
/* ldg_matrix stores link dependencies in order to eliminate loops */
if ((ldg_matrix = (long int *) malloc (max_dependency * no_abs_links * 4)) == NULL)
{    fprintf (stderr, "\nError-%s(%d)- Failed to allocate memory for ldg_matrix array\n",__FILE__,__LINE__) ;
     return (-1) ;
}
/* distance stores number of virtual links traversed on route to dest node */
if ((distance = (long int *) malloc (number_nodes * dist_size * 4)) == NULL)
{    fprintf (stderr, "\nError-%s(%d)- Failed to allocate memory for distance array\n",__FILE__,__LINE__) ;
     return (-1) ;
}

/* Clear LDG and distance matrices */
for (abs_link = 0; abs_link < no_abs_links; abs_link++)
     for (count = 0; count < max_dependency; count++)
          LDG_MATRIX(abs_link,count) = no_arc ;
for (count = 0; count < number_nodes * dist_size; count++)
     *(distance + count) = 0 ;
/*}}}*/
/*{{{  Insert arcs into link dependency graph (LDG) matrix*/
for (start_node = 0; start_node < number_nodes; start_node++)
{    for (dest_node = 0; dest_node < number_nodes; dest_node++)
     {    if (start_node != dest_node)
          {    next_node = start_node ;
               next_abs_link = start_node * 4 + ROUTE_read(start_node,dest_node) ;
               /* Find each link traversed on route from current source
                  to current dest and insert into LDG matrix if length > 1 */
               do
               {    current_node = next_node ;
                    abs_link = next_abs_link ;
                    next_node = NODE_CON(current_node,ROUTE_read(current_node,dest_node)) ;
                    if (next_node != dest_node)
                    {    next_abs_link = next_node * 4 + ROUTE_read(next_node,dest_node) ;
                         count = 0 ;
                         /* Insert next link on route into next spare
                            space along current link row of LDG matrix
                            unless already added */
                         while ((LDG_MATRIX(abs_link,count) != no_arc) && (LDG_MATRIX(abs_link,count) != next_abs_link))
                              count++ ;
                         LDG_MATRIX(abs_link,count) = next_abs_link ;
                    }
               }
               while (next_node != dest_node) ;
          }
     }
}
/*}}}*/
if (!silent) fprintf (stderr, "\nDividing Link Pairs : ") ;
/* Find cycles in LDG matrix, select best link pairs to virtualise which
   keep number of buffer classes to a minimum and update distance matrix */
add_virtual_links (number_nodes, no_abs_links) ;
free (ldg_matrix) ;
/*}}}*/
/*{{{  Output route and virtual link info in order of increasing node number*/
if (!silent) fprintf (stderr, "\n\nWriting out route and virtual link information\n") ;
for (node = 0; node < number_nodes; node++)
{    /* Output node number and size of route/virtual link list */
     fprintf (stdout,"\n%ld %ld\n", node, number_nodes * 2) ;
     count = 0 ;
     for (dest_node = 0; dest_node < number_nodes; dest_node++)
     {    /* Output link number from current node to reach dest node and
             virtual link class to be sent out on */
          if (dest_node == node) fprintf (stdout, "4 0 ") ;
          else fprintf (stdout, "%ld %ld ", ROUTE_read(node,dest_node), DISTANCE_read(node,dest_node)) ;
          if (++count == 10)
          {    fprintf (stdout, "\n") ;
               count = 0 ;
          }
     }
     if (count != 0) fprintf (stdout, "\n") ;
}
return (0) ;
}
/*}}}*/

