/*{{{  File banner*/
/*@(#)=====================================================*\
||@(#)  Project : PUMA ESPRIT P2701
||@(#)  Authors : Mark Debbage and Mark Hill
||@(#)            University of Southampton
||  
||@(#)    Title : VCR driver
||@(#)   System : VCR
||@(#) Filename : vdriver.c
||@(#)  Version : 2.12
||@(#)     Date : 11/16/92
\*@(#)====================================================*/
/*}}}*/

/*{{{  includes*/
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>

/*}}}*/
/*{{{  defines*/
#define TRUE  (1)
#define FALSE (0)

#define MAX_LINE 80
#define MAX_DOS_CL 124

/* command-line parameter offsets */
#define top  1
#define params 2

#define ndirection 4
/*}}}*/
/*{{{  strings*/
static char *ISERVER = "iserver" ;
static char *ISENV   = "ISERVER" ;

static char *NCF = ".ncf" ;
static char *BTL = ".btl" ;
static char *SPACE = " " ;

static char *NETROOT = "netroot%d.btl" ;
static char *NETNODE = "netnode.btl" ;
static char *NETS = "nets" ;

#ifdef UNIX
static char *SLASH = "/" ;
static char *ISOPTS = "-se -sb" ;
#else
static char *SLASH = "\\" ;
#ifdef NORESET
static char *ISOPTS = "/se/ss/si/sc" ;
#else
static char *ISOPTS = "/se/sb" ;
#endif
#endif

static char *NETSEARCH = "NETSEARCH" ;
static char *VCR = "VCR" ;

#ifdef STATIC
static char *_FILE_  = "svdriver" ;
#else
static char *_FILE_  = "vdriver" ;
#endif

/*}}}*/

#ifndef STATIC
int  linenum = 0 ;
/*{{{  string parsing routines*/
/*{{{  char *skip_non_blanks (char *s)*/
char *skip_non_blanks (char *s)
{
  while ((*s != 0) && (!isspace(*s)))
    s++ ;
  return(s) ;
}

/*}}}*/
/*{{{  char *skip_blanks (char *s)*/
char *skip_blanks (char *s)
{
  while ((*s != 0) && (isspace(*s)))
    s++ ;
  return(s) ;
}

/*}}}*/
/*{{{  char *skip_digits (char *s)*/
char *skip_digits (char *s)
{
  while ((*s != 0) && (isdigit(*s)))
    s++ ;
  return(s) ;
}

/*}}}*/
/*}}}*/
/*{{{  file reading primitives*/
#define buflen 160
char linebuf[buflen];
char *linepos = linebuf ;
char *strbuf = NULL ;

/*{{{  char *fgetstring (line, n, file)*/
/*
   This routine reads a line, strips out # comments and strips
   out lines of just white space. Usage much like fgets.
*/

char *fgetstring (line, n, file)
char *line ;
int n ;
FILE *file ;
{
  char *s ;

  do
  { 
    if ((s=fgets (line, n, file)) == NULL)
      return(NULL) ;
       
    if (!feof(file) && (s[0] != 0))
    {
      char *hash = strchr (s, '#') ;
      int i ;
       
      if ((strlen(s) == n-1) && (s[n-1] != '\n'))
        fprintf(stderr, "Warning-%s(%d)- Line %d too long so wrapped round\n",__FILE__,__LINE__,linenum) ;
      else
        linenum++ ;
       
      if (hash != NULL)
        *hash = 0 ;
       
      for (i=0;i<strlen(s);i++)
        if (!isspace(s[i]))
          break ;
       
      if (i==strlen(s))
        s[0] = 0 ;    
    }
  } while (*s == 0) ;

  return(s) ;
}
/*}}}*/
  
/*{{{  int   read_number (file)*/
int   read_number (file)
FILE *file ;
{
  int i ;

  linepos = skip_blanks(linepos) ;
  
  if (*linepos == 0)
  {
    if (fgetstring(linebuf, buflen, file) == NULL)
    {
      fprintf(stderr, "Error-%s(%d)- Parse failed reading number on line %d\n",__FILE__,__LINE__,linenum) ;
      exit(-1) ;
    }
    linepos = linebuf ;
  }

  if (feof(file) || (*linepos == 0))
  {
    fprintf(stderr, "Error-%s(%d)- Parse failed reading number on line %d\n",__FILE__,__LINE__,linenum) ;
    exit(-1) ;
  }

  if (sscanf(linepos,"%d",&i) != 1)
  {
    fprintf(stderr, "Error-%s(%d)- Parse failed reading number on line %d\n",__FILE__,__LINE__,linenum) ;
    exit(-1) ;
  }

  linepos = skip_digits(linepos) ;

  return(i) ;  
}
/*}}}*/
/*{{{  char *read_string (file)*/
char *read_string (file)
FILE *file ;
{
  linepos = skip_blanks(linepos) ;

  if (*linepos == 0)
  {
    if (fgetstring(linebuf, buflen, file) == NULL)
    {
      fprintf(stderr, "Error-%s(%d)- Parse failed reading string on line %d\n",__FILE__,__LINE__,linenum) ;
      exit(-1) ;
    }
    linepos = linebuf ;
  }

  if (feof(file) || (*linepos == 0))
  {
    fprintf(stderr, "Error-%s(%d)- Parse failed reading string on line %d\n",__FILE__,__LINE__,linenum) ;
    exit(-1) ;
  }

  if (strbuf != NULL)
    free(strbuf) ;
    
  strbuf = (char *) malloc(strlen(linepos)+1) ;

  if (sscanf(linepos,"%s",strbuf) != 1)
  {
    fprintf(stderr, "Error-%s(%d)- Parse failed reading string on line %d\n",__FILE__,__LINE__,linenum) ;
    exit(-1) ;
  }

  linepos = skip_non_blanks(linepos) ;

  return(strbuf) ;
}
/*}}}*/
/*{{{  char  read_char   (file)*/
char  read_char   (file)
FILE *file ;
{
  char c ;
  
  if (*linepos == 0)
  {
    if (fgetstring(linebuf, buflen, file) == NULL)
    {
      fprintf(stderr, "Error-%s(%d)- Parse failed reading character on line %d\n",__FILE__,__LINE__,linenum) ;
      exit(-1) ;
    }
    linepos = linebuf ;
  }

  if (feof(file) || (*linepos == 0))
  {
    fprintf(stderr, "Error-%s(%d)- Parse failed reading character on line %d\n",__FILE__,__LINE__,linenum) ;
    exit(-1) ;
  }

  if (sscanf(linepos,"%c",&c) != 1)
  {
    fprintf(stderr, "Error-%s(%d)- Parse failed reading character on line %d\n",__FILE__,__LINE__,linenum) ;
    exit(-1) ;
  }

  linepos++ ;
  
  return(c) ;  
}
/*}}}*/
/*{{{  void  next_field  (file)*/
void  next_field  (file)
FILE *file ;
{
  linepos = skip_blanks(linepos) ;
}
/*}}}*/
/*}}}*/
/*{{{  int GetBootLink (net_fp)*/
int GetBootLink (net_fp)
FILE *net_fp ;
{
  int node_cnt, diameter, index ;
  int BootLink = -1 ;

  /*{{{  read node_cnt diameter*/
  node_cnt = read_number(net_fp) ;
  diameter = read_number(net_fp) ;
  
  if (node_cnt <= 0 || diameter < 0)
  {
    fprintf(stderr, "Error-%s(%d)- Invalid NCF header\n",__FILE__,__LINE__);
    return(-1);
  }
  /*}}}*/

  for (index=0;index<node_cnt;index++)
  {
    int n;
    char line[MAX_LINE];
    
    /*{{{  read n*/
    n = read_number(net_fp) ;
    
    /*}}}*/
    /*{{{  read part*/
    {
      char *s = read_string (net_fp) ;
    }
    /*}}}*/
    /*{{{  read memory*/
    {
      int size  = read_number(net_fp) ;
      char unit = read_char (net_fp) ;
    }
    /*}}}*/
    /*{{{  read links*/
    {
      int type, dir ;
    
      for (dir=0;dir<ndirection;dir++)
      {
        next_field(net_fp) ;
        type = read_char (net_fp) ;
    
        if  (type == 'h')
          return(dir) ;
      }
    }            
    /*}}}*/
  }

  fprintf(stderr, "Error-%s(%d)- Failed to find boot link\n", _FILE_,__LINE__) ;
  exit(-1) ;
}
/*}}}*/
#endif

/*{{{  FILE *fopenenv(file,opts,env,fullname)*/
FILE *fopenenv(file,opts,env,def,fullname)
char *file ;
char *opts ;
char *env ;
char *def ;
char **fullname ;

/* Policy :
     If local file exists use that.
     If environment string is given, is set, and file exists then use that.
     If default string is given, and file exists then use that.
     Otherwise failed.
*/
     
{
  char *tryname;
  char *gotenv;
  FILE *fp;
  char *start, *end;

  if ((fp=fopen(file,opts))!=NULL) return fp;

  if (env == NULL)
    gotenv = NULL ;
  else
    gotenv = (char *) getenv(env) ;
    
  if (gotenv == NULL)
  {
    if (def != NULL)
      /*{{{  use default*/
      {
        /*{{{  malloc and build full filename*/
        tryname = (char *) malloc(strlen(def)+strlen(file)+1) ;
        strcpy(tryname,def) ;
        strcat(tryname,file);
        /*}}}*/
         
        fp=fopen(tryname,opts);
        
        if (fp!=NULL)
        {
          *fullname=tryname ;
          return fp;
        }
        else
        {
          free(tryname);
          return NULL;
        }
      }
      /*}}}*/
    else
      return NULL ;
  }
  else
    /*{{{  use environment string*/
    {
      start = gotenv;
      do
      {
        while ((*start==' ')||(*start==';')) start++;
        end=start;
        while ((*end!=' ')&&(*end!=';')&&(*end!=0)) end++;
        if ((end-start)>0)
        {
          /*{{{  malloc and build full filename*/
          tryname = (char *) malloc((end-start)+strlen(file)+1) ;
          strncpy(tryname,start,(end-start));
          tryname[end-start]=0;
          strcat(tryname,file);
          /*}}}*/
      
          /* printf("Trying to open %s\n",tryname); */
      
          fp=fopen(tryname,opts);
      
          if (fp!=NULL)
          {
            *fullname=tryname ;
            return fp;
          }
          else
            free(tryname);
        }
        start = end + 1;
      } while (*end != 0);
      return NULL;
    }
    /*}}}*/
}
/*}}}*/

int main(argc, argv)
int argc ;
char *argv[] ;
{
  int size ;
  char *net_fn, *call, *vcr_nets, *iserver ;
  FILE *net_fp ;
  
  if (argc<params)
  {
#ifdef __DATE__
    printf("VCR 2.0k, M. Debbage, M. Hill, %s\n",__DATE__) ;
#else
    printf("VCR 2.0k, M. Debbage, M. Hill\n") ;
#endif
    printf("Usage : %s <topology> {params}\n", _FILE_) ;
    return(-1) ;
  }

  /*{{{  attempt to get iserver from ISERVER environment variable*/
  if ((iserver = (char *) getenv(ISENV)) == NULL)
    iserver = ISERVER ;
  /*}}}*/
  
  /*{{{  find vcr_nets*/
  {
    char *vcr ;
  
    if ((vcr = (char *) getenv (VCR)) == NULL)
    {
      fprintf(stderr, "Warning-%s(%d)- Failed to find environment string for %s\n", _FILE_,__LINE__,VCR) ;
      vcr_nets = (char *) malloc(strlen(NETS)+strlen(SLASH)+1) ;
      strcpy(vcr_nets, NETS) ;
      strcat(vcr_nets, SLASH) ;
    }
    else
    {
      vcr_nets = (char *) malloc(strlen(vcr)+strlen(SLASH)+strlen(NETS)+strlen(SLASH)+1) ;
      strcpy(vcr_nets, vcr) ;
      strcat(vcr_nets, SLASH) ;
      strcat(vcr_nets, NETS) ;
      strcat(vcr_nets, SLASH) ;
    }
  }
  /*}}}*/
  /*{{{  build net_fn (ncf file)*/
  size = strlen(argv[top]) + strlen(NCF) + 1 ;
    
  if ((net_fn = (char *) malloc(size)) == NULL)
  {
    fprintf(stderr, "Error-%s(%d)- Out of heap\n", _FILE_,__LINE__) ;
    return(-1) ;
  }
  
  sprintf(net_fn, "%s%s", argv[top], NCF) ;
  /*}}}*/

#ifdef STATIC
  /*{{{  static loading*/
  {
    char *top_fn ;
    FILE *top_fp ;
  
    /*{{{  open net_fp*/
    if ((net_fp = fopenenv (net_fn, "r", NETSEARCH, vcr_nets, &net_fn)) == NULL)
      net_fn = argv[top] ;
    
    /* NCF does not exist for HPR so pass just topology name */
    /*}}}*/
    /*{{{  find topology.btl*/
    size = strlen(argv[top]) + strlen(BTL) + 1 ;
      
    if ((top_fn = (char *) malloc(size)) == NULL)
    {
      fprintf(stderr, "Error-%s(%d)- Out of heap\n", _FILE_,__LINE__) ;
      return(-1) ;
    }
    
    strcpy (top_fn, argv[top]) ;
    strcat (top_fn, BTL) ;
    
    if ((top_fp = fopenenv (top_fn, "rb", NETSEARCH, vcr_nets, &top_fn)) == NULL)
    {
      fprintf(stderr, "Error-%s(%d)- Cannot open static network bootable %s\n", _FILE_,__LINE__,top_fn) ;
      return(-1) ;
    }
    /*}}}*/
    /*{{{  form calling string*/
    {
      int p ;
    
      size = strlen(iserver) + strlen(ISOPTS) + strlen(top_fn) + strlen(net_fn) + 20 ;
    
      for (p=params-1;p<argc;p++)
        size += (strlen(SPACE) + strlen(argv[p])) ;
        
      if ((call = (char *) malloc(size)) == NULL)
      {
        fprintf(stderr, "Error-%s(%d)- Out of heap\n", _FILE_,__LINE__) ;
        return(-1) ;
      }
      
      sprintf(call, "%s %s %s %s", iserver, ISOPTS, top_fn, net_fn) ;
    
      for (p=params;p<argc;p++)
      {
        strcat(call, SPACE) ;
        strcat(call, argv[p]) ;
      }
    }    
    /*}}}*/
    /*{{{  close files*/
    fclose(top_fp) ;
    if (net_fp != NULL)
      fclose(net_fp) ;
    /*}}}*/
  }
  /*}}}*/
#else
  /*{{{  dynamic loading*/
  {
    long kernel_size=0 ;
    int  BootLink ;
    char *netnode_fn, *netroot_fn ;
    FILE *netnode_fp, *netroot_fp ;
  
    /*{{{  open net_fp*/
    if ((net_fp = fopenenv (net_fn, "r", NETSEARCH, vcr_nets, &net_fn)) == NULL)
    {
      fprintf(stderr, "Error-%s(%d)- Cannot open network file %s\n", _FILE_,__LINE__,net_fn) ;
      return(-1) ;
    }
    /*}}}*/
    /*{{{  open netnode.btl*/
    netnode_fn = NETNODE ;
    
    if ((netnode_fp = fopenenv (netnode_fn, "rb", NULL, vcr_nets, &netnode_fn)) == NULL)
    {
      fprintf(stderr, "Error-%s(%d)- Cannot open network node bootable %s\n", _FILE_,__LINE__,netnode_fn) ;
      return(-1) ;
    }
    /*}}}*/
    /*{{{  get system parameters*/
    while (!feof(netnode_fp))
    {
      getc(netnode_fp) ;
      kernel_size++ ;
    }
    
    BootLink = GetBootLink (net_fp) ;
    /*}}}*/
    /*{{{  open netroot%d.btl*/
    if ((netroot_fn = (char *) malloc(strlen(NETROOT))) == NULL)
    {
      fprintf(stderr, "Error-%s(%d)- Out of heap\n", _FILE_,__LINE__) ;
      return(-1) ;
    }
    
    sprintf(netroot_fn, NETROOT, BootLink) ;
    
    if ((netroot_fp = fopenenv (netroot_fn, "rb", NULL, vcr_nets, &netroot_fn)) == NULL)
    {
      fprintf(stderr, "Error-%s(%d)- Cannot open network root bootable %s\n", _FILE_,__LINE__,netroot_fn) ;
      return(-1) ;
    }
    /*}}}*/
    /*{{{  form calling string*/
    {
      int p ;
    
      size = strlen(iserver) + strlen(ISOPTS) + strlen(netroot_fn) 
             + strlen(net_fn) + strlen(netnode_fn) + 20 ;
    
      for (p=params;p<argc;p++)
        size += (strlen(SPACE) + strlen(argv[p])) ;
        
      if ((call = (char *) malloc(size)) == NULL)
      {
        fprintf(stderr, "Error-%s(%d)- Out of heap\n", _FILE_,__LINE__) ;
        return(-1) ;
      }
      
      sprintf(call, "%s %s %s %s %s %ld", iserver, ISOPTS, netroot_fn,
              net_fn, netnode_fn, kernel_size) ;
    
      for (p=params;p<argc;p++)
      {
        strcat(call, SPACE) ;
        strcat(call, argv[p]) ;
      }
    }    
    /*}}}*/
    /*{{{  close files*/
    fclose(net_fp) ;
    fclose(netnode_fp) ;
    fclose(netroot_fp) ;
    /*}}}*/
  }
  /*}}}*/
#endif

  fprintf(stderr, "%s\n", call) ;

#ifndef UNIX
  /*{{{  check CLI length if necessary*/
  if (strlen(call) > MAX_DOS_CL)
  {
    fprintf(stderr, "Error-%s(%d)- Command line too long for MSDOS (%d characters)\n", _FILE_,__LINE__, strlen(call)) ;
    return(-1) ;
  }
  /*}}}*/
#endif

  return(system(call)) ;
}
