/*{{{  Title*/
/*  TITLE : LOAD BALANCED ROUTING FILE GENERATOR
    --------------------------------------------

    VERSION : 1.2   31/5/91   Uses stdin/stdout as input/output files
                              Input file read without use of rewind
                              Wiring diagram format added incl processor params
                              Array macros and memory required message added
                              Messages use stderr with silent option
                              Single processor networks handled
                              Dynamic arrays now global and program in ANSI C
                              Outputs reverse link numbers
              1.1   28/1/91   Added error check for no host connection
              1.0   21/1/91   Initial program derived from gend4mx.c

    AIM     : Generate a routing file, which has even load balanced, from
              a network dump from inmos checkout software. Output file is
              used by virtual.c.

    BASED   : Network load balancing algorithm obtained from continuous
              development by gen*.c programs. Final version is gend4mx.c.

    AUTHORS : David Pritchard, Mark Debbage and Mark Hill

    USAGE   : router [-m {mode} -s -w {type} {mem} {link}] <infile >outfile
              -m {mode} : Defines number of re-runs to improve load balancing :
                          0 = No re-runs
                          1 = 1 re-run
                          2 = Re-run until link utilisation unchanged (default)
              -s        : Disable screen messages
              -w        : Wiring diagram input format (default is check|mtest)
              {type}    : Processor type - in form T???
              {mem}     : Processor memory size - either ???K or ???M
              {link}    : Link number from root node to host
              infile    : check|mtest (*.chk) or wiring diagram (*.map) filename
              outfile   : Balanced routing tables output filename (*.brf)
*/
/*}}}*/
/*{{{  Libraries and constants*/
#include <stdio.h>
#include <ctype.h>
#include <string.h>
#ifdef _ICC
#include <stdlib.h>
#endif
#ifdef __ZTC__
#include <stdlib.h>
#endif

#define not_used            -1
#define null_route          16383
#define TRUE                1
#define FALSE               0
#define maxint              999999
#define EOLN                10
#define max_nodes           8192
#define alloc_block         256
#define mem_size_default    256
#define root_link_default   0
/*}}}*/
/*{{{  Global declarations*/
long int silent = FALSE, wiring_diagram = FALSE, mtest = TRUE ;
long int mask [2], number_nodes = 0, route_size, usage_size ;
long int base [15] ;
static char *processor_default = "T414" ;
static char *wdg_connect = "Connect processor" ;
static char *wdg_link = "link" ;
static char *wdg_to = "to processor" ;

/* Array declarations */
long int *real_node ;        /* Actual node number of node n */
long int *node_con ;         /* Node connected to link l from node n */
long int *reverse_link_no ;  /* Reverse link of link l from node n */
long int *mem_size ;         /* Memory size of node n */
char *processor ;       /* Processor type of node n (char c) */
long int *route ;            /* Link from node s to reach node d */
long int *overall_usage ;    /* Total usage along link l from node n */
long int *old_route ;        /* Old link from node s to current dest */
long int *link_usage ;       /* Usage and link from node n to node d */
long int *new_usage ;        /* New usage and link from node n */
/*}}}*/
/*{{{  Macro definitions*/
#define NODE_DISTANCE(n)      (*(node_distance + (n)))
#define ROUTED_FLAG(n)        (*(routed_flag + (n)))
#define REAL_NODE(n)          (*(real_node + (n)))
#define NODE_CON(n,l)         (*(node_con + (n) * 4 + (l)))
#define REVERSE_LINK_NO(n,l)  (*(reverse_link_no + (n) * 4 + (l)))
#define MEM_SIZE(n)           (*(mem_size + (n)))
#define PROCESSOR(n,c)        (*(processor + (n) * 8 +(c)))
#define ROUTE(s,d)            (*(route + (d) * route_size + (s) / 15))
#define OVERALL_USAGE(n,l)    (*(overall_usage + (n) * 4 + (l)))
#define BRANCH_CONV(a,b)      (*(branch_conv + base [a] + (b) / 2))
#define OLD_ROUTE(n)          (*(old_route + (n) / 15))
#define LINK_USAGE(d,n)       (*(link_usage + (d) * usage_size + (n) / 2))
#define NEW_USAGE(n)          (*(new_usage + (n) / 2))

#define ROUTE_write(s,d,l)           ROUTE(s,d) += (l) << (((s) % 15) * 2)
#define ROUTE_read(s,d)              (ROUTE(s,d) % (1 << ((((s) % 15) + 1) * 2))) / (1 << (((s) % 15) * 2))
#define BRANCH_CONV_write(a,b,n,l)   BRANCH_CONV(a,b) += ((n) * 4 + (l)) << (((b) % 2) * 16)
#define BRANCH_CONV_null_route(a,b)  BRANCH_CONV(a,b) = (BRANCH_CONV(a,b) & mask [((b) + 1) % 2]) + mask [(b) % 2]
#define BRANCH_CONV_node(a,b)        ((BRANCH_CONV(a,b) >> (((b) % 2) * 16 + 2)) & 16383)
#define BRANCH_CONV_link(a,b)        ((BRANCH_CONV(a,b) >> (((b) % 2) * 16)) & 3)
#define LINK_USAGE_usage(d,n)        ((LINK_USAGE(d,n) >> (((n) % 2) * 16 + 2)) & 16383)
#define LINK_USAGE_link(d,n)         ((LINK_USAGE(d,n) >> (((n) % 2) * 16)) & 3)
#define NEW_USAGE_init(n,l)          NEW_USAGE(n) += (l) << (((n) % 2) * 16)
#define NEW_USAGE_inc(n)             NEW_USAGE(n) += 4 << (((n) % 2) * 16)
#define NEW_USAGE_usage(n)           ((NEW_USAGE(n) >> (((n) % 2) * 16 + 2)) & 16383)
#define NEW_USAGE_link(n)            ((NEW_USAGE(n) >> (((n) % 2) * 16)) & 3)

/*}}}*/

/*{{{  Procedure to get the next line of text from a file*/
long int get_line (ip, buffer, line)
FILE * ip ;
char * buffer ;
char line[] ;
{    long int letter ;
     letter = 0;

     while ((*buffer != (char) EOLN) && (*buffer != (char) EOF))
     {    line[letter] = *buffer ;
          letter++ ;
          *buffer = getc (ip);
     }

     *buffer = getc (ip);
     line [letter] = 0;
     return (letter) ;
}
/*}}}*/
/*{{{  Function to search for a specifed string in another string*/
long int search (line, match)
char line[], match[] ;
{    long int line_i, match_i ;
     line_i = 0 ;
     match_i = 0;

     while (line [line_i] != 0)
     {    if (line [line_i] == match [match_i])
          {    match_i++ ;
               /* If all characters matched then return position in string */
               if (match [match_i] == 0)
                    return (line_i - match_i + 1) ;
          }
          else
               match_i = 0;
          line_i++;
     }
     return (-1) ;
}
/*}}}*/
/*{{{  Procedure to skip along text file until specified character is reached*/
void skip_till (inp, buffer, stopper)
char *buffer, stopper ;
FILE * inp ;

{    while ((*buffer != stopper) && (*buffer != (char) EOF))
          *buffer = getc (inp) ;
}
/*}}}*/
/*{{{  Function defining position in line after specified sting*/
long int after_match (line, match)
char line [], match [] ;
{    return (search (line, match) + strlen (match)) ;
}
/*}}}*/
/*{{{  Procedure to reallocate memory to node connection input file arrays*/
void realloc_arrays (old_size, new_size)
long int old_size, new_size ;
{    long int node, link ;

     if ((real_node = (long int *) realloc (real_node, new_size * 4)) == NULL)
     {    fprintf (stderr, "\n\nError-%s(%d)- Realloc for real_node array failed\n", __FILE__, __LINE__) ;
          exit (-1) ;
     }
     if ((node_con = (long int *) realloc (node_con, new_size * 16)) == NULL)
     {    fprintf (stderr, "\n\nError-%s(%d)- Realloc for node_con array failed\n", __FILE__, __LINE__) ;
          exit (-1) ;
     }
     if ((reverse_link_no = (long int *) realloc (reverse_link_no, new_size * 16)) == NULL)
     {    fprintf (stderr, "\n\nError-%s(%d)- Realloc for reverse_link_no array failed\n", __FILE__, __LINE__) ;
          exit (-1) ;
     }
     if (wiring_diagram)
     {    /* Initialise node connection array */
          for (node = old_size; node < new_size; node++)
              for (link = 0; link < 4; link++)
                  NODE_CON(node,link) = not_used ;
     }
     else
     {    if ((mem_size = (long int *) realloc (mem_size, new_size * 4)) == NULL)
          {    fprintf (stderr, "\n\nError-%s(%d)- Realloc for mem_size array failed\n", __FILE__, __LINE__) ;
               exit (-1) ;
          }
          if ((processor = (char *) realloc ((void *)processor, new_size * 8)) == NULL)
          {    fprintf (stderr, "\n\nError-%s(%d)- Realloc for processor array failed\n", __FILE__, __LINE__) ;
               exit (-1) ;
          }
     }
}
/*}}}*/
/*{{{  Procedure to add a wiring diagram connection*/
void add_connection (node_a, link_a, node_b, link_b)
long int node_a, link_a, node_b, link_b ;
{    long int node, new_flag = TRUE ;

     /* Convert source node into contiguous node number */
     for (node = 0; node < number_nodes; node++)
          if (node_a == REAL_NODE(node))
          {    node_a = node ;
               new_flag = FALSE ;
               break ;
          }
     /* If node number is new then give it an equivalent contiguous number
        and increment node count */
     if (new_flag == TRUE)
     {    REAL_NODE(number_nodes) = node_a ;
          node_a = number_nodes ;
          number_nodes++ ;
          if (number_nodes > max_nodes)
          {    fprintf (stderr, "\n\nError-%s(%d)- Network exceed processor limit (%d)\n", __FILE__, __LINE__, max_nodes) ;
               exit (-1) ;
          }
     }
     if (node_b > max_nodes)
     {    fprintf (stderr, "\n\nError-%s(%d)- Node number exceeds processor limit (%d)\n", __FILE__, __LINE__, max_nodes) ;
          exit (-1) ;
     }
     /* Add connection to node_con and reverse_link_no arrays */
     if (NODE_CON(node_a,link_a) == not_used)
     {    NODE_CON(node_a,link_a) = node_b ;
          REVERSE_LINK_NO(node_a,link_a) = link_b ;
     }
     else
     {    fprintf (stderr, "\n\nError-%s(%d)- Processor %ld link %ld declared twice\n",__FILE__,__LINE__,REAL_NODE(node_a),link_a) ;
          exit (-1) ;
     }
}
/*}}}*/
/*{{{  Procedure to extract information from wiring diagram file*/
void read_wdg_file ()
{    char buffer, input[80] ;
     long int line_no = 0, pos, count ;
     long int source_node, source_link, dest_node, dest_link ;

     /* Initialise node connection array */
     for (source_node = 0; source_node < alloc_block; source_node++)
         for (source_link = 0; source_link < 4; source_link++)
             NODE_CON(source_node,source_link) = not_used ;

     /*{{{  Skip till start of processor connection information*/
     buffer = getc (stdin) ;
     if (buffer == (char) EOF)
     {    number_nodes = 1 ;
          return ;
     }
     else do
          {    get_line(stdin, &buffer, input) ;
               line_no++ ;
               if (buffer == (char) EOF)
               {    number_nodes = 1 ;
                    return ;
               }
          }
          while (search (input, wdg_connect) < 0) ;
     /*}}}*/
     /*{{{  Read node connection list*/
     do
     {    pos = 0 ;
          /*{{{  Read source processor / link*/
          if ((pos += after_match (&input [pos], wdg_connect)) < 0)
          {    fprintf(stderr, "\n\nError-%s(%d)- Bad text in wiring diagram, line %ld\n",__FILE__,__LINE__,line_no) ;
               exit (-1) ;
          }
          if (sscanf (&input [pos], "%ld", &source_node) != 1)
          {    fprintf(stderr, "\n\nError-%s(%d)- Bad source processor number in wiring diagram, line %ld\n",__FILE__,__LINE__,line_no) ;
               exit (-1) ;
          }
          if ((pos += after_match (&input [pos], wdg_link)) < 0)
          {    fprintf(stderr, "\n\nError-%s(%d)- Bad text in wiring diagram, line %ld\n",__FILE__,__LINE__,line_no) ;
               exit (-1) ;
          }
          if (sscanf (&input [pos], "%ld", &source_link) != 1)
          {    fprintf(stderr, "\n\nError-%s(%d)- Bad source link number in wiring diagram, line %ld\n",__FILE__,__LINE__,line_no) ;
               exit (-1) ;
          }
          /*}}}*/
          /*{{{  Read destination processor / link*/
          if ((pos += after_match (&input [pos], wdg_to)) < 0)
          {    fprintf(stderr, "\n\nError-%s(%d)- Bad text in wiring diagram, line %ld\n",__FILE__,__LINE__,line_no) ;
               exit (-1) ;
          }
          if (sscanf (&input [pos], "%ld", &dest_node) != 1)
          {    fprintf(stderr, "\n\nError-%s(%d)- Bad destination processor link number in wiring diagram, line %ld\n",__FILE__,__LINE__,line_no) ;
               exit (-1) ;
          }
          if ((pos += after_match (&input[pos], wdg_link)) < 0)
          {    fprintf(stderr, "\n\nError-%s(%d)- Bad text in wiring diagram, line %ld\n",__FILE__,__LINE__,line_no) ;
               exit (-1) ;
          }
          if (sscanf (&input [pos], "%ld", &dest_link) != 1)
          {    fprintf(stderr, "\n\nError-%s(%d)- Bad destination link number in wiring diagram, line %ld\n",__FILE__,__LINE__,line_no) ;
               exit (-1) ;
          }
          /*}}}*/
          /* Write connection pair to node_con and reverse_link_no arrays.
             If new processor number then add to real_node array and increment
             node count. Reallocate memory if arrays are full */
          add_connection (source_node, source_link, dest_node, dest_link) ;
          if (number_nodes % alloc_block == 0) realloc_arrays (number_nodes, number_nodes + alloc_block) ;
          add_connection (dest_node, dest_link, source_node, source_link) ;
          if (number_nodes % alloc_block == 0) realloc_arrays (number_nodes, number_nodes + alloc_block) ;
          /* Read next node connection line */
          get_line(stdin, &buffer, input) ;
          line_no++ ;
     }
     while (strlen (input) != 0) ;
     /*}}}*/
     /*{{{  Replace actual node numbers in node_con array with contiguous numbers*/
     for (source_node = 0; source_node < number_nodes; source_node++)
          for (source_link = 0; source_link < 4; source_link++)
               for (count = 0; count < number_nodes; count++)
                    if (NODE_CON(source_node,source_link) == REAL_NODE(count))
                    {    NODE_CON(source_node,source_link) = count ;
                         break ;
                    }
     /*}}}*/
}
/*}}}*/
/*{{{  Procedure to extract information from chk file*/
void read_chk_file (root_node, root_link)
long int *root_node, *root_link ;
{    long int node, link, memory, cycle, count ;
     char buffer, input [80] ;

     buffer = getc (stdin) ;
     get_line (stdin, &buffer, input) ;       /* Program title */
     if (search (input, "check") < 0)         /* Ensure input is check file */
     {    fprintf (stderr, "\n\nError-%s(%d)- Incorrect file type - check | mtest output file required\n", __FILE__, __LINE__) ;
          exit (-1) ;
     }
     /* Check that file has memory info - if not use default/parameter value */
     if (search (input, "mtest") < 0) mtest = FALSE ;
     get_line (stdin, &buffer, input) ;               /* Header for table */

     /* Line format is : */
     /* 1 T414b-20 0.49 0 [    0:2    3:1    4:3    5:2 ] 2K,1+256K,5; */
     while (buffer != (char) EOF)
     {
          /*{{{  Extract information from current line*/
          /* Read actual node number unless EOF */
          if (fscanf (stdin, "%ld", real_node + number_nodes) != 1) break ;
          buffer = getc (stdin) ;                     /* Skip space */
          fgets (processor + 8 * number_nodes, 9, stdin) ;  /* Processor type */
          fscanf (stdin, "%s", input) ;               /* Link speed */
          fscanf (stdin, "%s", input) ;               /* Boot link */
          buffer = getc (stdin) ;
          skip_till (stdin, &buffer, '[') ;           /* Match bracket */
          
          for (link = 0; link < 4; link++)
          {    fscanf (stdin, "%s", input) ;          /* Link connection */
               if (isdigit (input[0]))
                    sscanf (input, "%ld %*c %ld", node_con + number_nodes * 4 + link, reverse_link_no + number_nodes * 4 + link) ;
               else
               {    if (search (input, "HOST") >= 0)
                    {    *root_node = number_nodes ;
                         *root_link = link ;
                    }
                    NODE_CON(number_nodes,link) = not_used ;
               }
          }
          
          buffer = getc (stdin) ;
          skip_till (stdin, &buffer, ']') ;           /* Match bracket */
          
          /* If memory information then add up total memory at current node */
          if (mtest)
          {    MEM_SIZE(number_nodes) = 0 ;
               buffer = getc (stdin) ;                /* Skip space */
               do
               {    fscanf (stdin, "%ld %*c %*c %ld", &memory, &cycle) ;    /* Memory size, cycle */
                    MEM_SIZE(number_nodes) += memory ;
                    buffer = getc (stdin) ;
               }
               while (buffer == '+') ;
               if ((buffer == '|') || (buffer == '?'))
               {    fprintf (stderr, "\n\nError-%s(%d)- Node connections map file contains an error\n", __FILE__, __LINE__) ;
                    exit (-1) ;
               }
          }
          /*}}}*/
          number_nodes++ ;
          if (number_nodes > max_nodes)
          {    fprintf (stderr, "\n\nError-%s(%d)- Network exceed processor limit (%d)\n", __FILE__, __LINE__, max_nodes) ;
               exit (-1) ;
          }
          /* Reallocate memory if arrays are full */
          if (number_nodes % alloc_block == 0) realloc_arrays (number_nodes, number_nodes + alloc_block) ;
     }
     if (number_nodes == 0)
     {    fprintf (stderr, "\n\nError-%s(%d)- Network contains no processors\n", __FILE__, __LINE__) ;
          exit (-1) ;
     }
     if (*root_node == -1)
     {    fprintf (stderr, "\n\nError-%s(%d)- Node connections map file contains no HOST\n", __FILE__, __LINE__) ;
          exit (-1) ;
     }
     /*{{{  Replace actual node numbers in node_con array with contiguous numbers*/
     for (node = 0; node < number_nodes; node++)
          for (link = 0; link < 4; link++)
               for (count = 0; count < number_nodes; count++)
                    if (NODE_CON(node,link) == REAL_NODE(count))
                    {    NODE_CON(node,link) = count ;
                         break ;
                    }
     /*}}}*/
}
/*}}}*/
/*{{{  Function to find diameter of the network*/
long int find_diameter ()
{    long int *node_distance ;           /* Distance to node n */
     long int diameter = 0, source_node, level, node, link, no_found ;
     /* node_distance stores min distance to each node from source node */
     if ((node_distance = (long int *) malloc (number_nodes * 4)) == NULL)
     {    fprintf (stderr, "\n\nError-%s(%d)- Malloc for node_distance array failed\n", __FILE__, __LINE__) ;
          exit (-1) ;
     }
     /* For each source node find distance to each node */
     for (source_node = 0; source_node < (number_nodes - 1); source_node++)
     {    /* Initialise node_distance array */
          for (node = 0; node < number_nodes; node++)
               NODE_DISTANCE(node) = not_used ;
          NODE_DISTANCE(source_node) = 0 ;
          level = 0 ;
          do
          {    level++ ;
               /* For each node at (level - 1) away, find adjacent nodes and
                  if not already found, store level in node_distance array */
               for (node = 0; node < number_nodes; node++)
                    if (NODE_DISTANCE(node) == (level - 1))
                         for (link = 0; link < 4; link++)
                              if (NODE_CON(node,link) >= 0)
                                   if (NODE_DISTANCE(NODE_CON(node,link)) == not_used)
                                        NODE_DISTANCE(NODE_CON(node,link)) = level ;
               /* Find number of nodes > source node, found so far */
               no_found = 0 ;
               for (node = source_node + 1; node < number_nodes; node++)
                    if (NODE_DISTANCE(node) > not_used) no_found++ ;
          }
          /* Repeat until all nodes > source node found */
          while (no_found < (number_nodes - source_node - 1)) ;
          /* Update diameter if necessary */
          if (level > diameter) diameter = level ;
     }
     free (node_distance) ;
     return (diameter) ;
}
/*}}}*/
/*{{{  Function to find link with maximum usage from source leaf to root node*/
long int find_max_usage (branch_conv, level, source_leaf)
long int *branch_conv, level, source_leaf ;
{    long int max_usage = 0, current_level, node, link ;
     /* Find usage of each link on route by finding node/leaf at each level
        of the tree for the branch leading to source leaf */
     for (current_level = 0; current_level <= level; current_level++)
     {    /* Find node and link corresponding to current leaf */
          node = BRANCH_CONV_node(current_level,source_leaf >> ((level - current_level) * 2)) ;
          link = BRANCH_CONV_link(current_level,source_leaf >> ((level - current_level) * 2)) ;
          /* Check whether link usage of current leaf exceeds previous max */
          if (max_usage < OVERALL_USAGE(node,link)) max_usage = OVERALL_USAGE(node,link) ;
     }
     return (max_usage) ;
}
/*}}}*/
/*{{{  Procedure to find all nodes leading to given dest (root) node*/
void find_leaves (root_node, diameter)
/*{{{  Declarations*/
long int root_node ;
long int diameter ;
{    long int *branch_conv ;  /* Equivalent node, link to level a, leaf b */
     long int *routed_flag ;  /* Flag indicating node n is routed */
     long int link, leaf, best_usage, best_leaf, current_node ;
     long int source_node, new_max_usage, new_leaf, new_level ;
     long int count, node, last_leaf = 1, no_routed = 0, level = 0 ;
/*}}}*/
     /*{{{  Initialisation*/
     /* branch_conv stores node, link which corresponds to level, leaf */
     if ((branch_conv = (long int *) malloc (base [diameter] * 4)) == NULL)
     {    fprintf (stderr, "\n\nError-%s(%d)- Malloc for branch_conv array failed\n", __FILE__, __LINE__) ;
          exit (-1) ;
     }
     /* routed_flag indicates whether each source node has been routed yet */
     if ((routed_flag = (long int *) malloc (number_nodes * 4)) == NULL)
     {    fprintf (stderr, "\n\nError-%s(%d)- Malloc for routed_flag array failed\n", __FILE__, __LINE__) ;
          exit (-1) ;
     }
     for (node = 0; node < number_nodes; node++)
          ROUTED_FLAG(node) = FALSE ;
     for (count = 0; count < route_size; count++)
          *(route + root_node * route_size + count) = 0 ;
     if (new_usage != 0)
          for (count = 0; count < usage_size; count++)
               *(new_usage + count) = 0 ;
     /*}}}*/
     /* Generate route array which lists which link to take to arrive at
        current destination node from each source node (with balanced link
        usage) which are distance level + 1 away unless already routed */
     do
     {
          /*{{{  Find next level of nodes and link usages*/
          /* Clear leaf information for current level */
          for (leaf = 0; leaf < last_leaf * 4; leaf += 2)
               BRANCH_CONV(level,leaf) = 0 ;
          /* Create next level of tree by finding node number and link usage
             corresponding to the four links emerging from each leaf node */
          for (leaf = 0; leaf < last_leaf; leaf++)
          {    /* Find node number corresponding to current leaf */
               if (level == 0) current_node = root_node ;
               else current_node = BRANCH_CONV_node(level - 1,leaf) ;
               /* Find the 4 nodes connected to the current leaf node */
               for (link = 0; link < 4; link++)
               {    new_leaf = leaf * 4 + link ;
                    if (abs (current_node) != null_route) source_node = NODE_CON(current_node,link) ;
                    /* Check whether current node is part of a rejected
                       route, link is unconnected or new leaf node is same
                       as the root node - if so indicate null route */
                    if ((abs (current_node) == null_route) || (source_node == -1) || (source_node == root_node))
                        BRANCH_CONV_null_route(level,new_leaf) ;
                    /* Store node/link which corresponds to current branch */
                    else BRANCH_CONV_write(level,new_leaf,source_node,REVERSE_LINK_NO(current_node,link)) ;
               }
          }
          /*}}}*/
          /*{{{  Select the best leaf nodes which have yet to be routed*/
          /* Select the leaves (source nodes) which,
             a) route from a node not already routed at a higher level
             b) do not route via an already rejected route
             c) has the lowest maximum link usage on route compared with
                other leaves on this level having the same source node */
          for (leaf = 0; leaf < (last_leaf * 4); leaf++)
          {    /* Find node number corresponding to current leaf */
               node = BRANCH_CONV_node(level,leaf) ;
               if (abs (node) != null_route)
               if (ROUTED_FLAG(node) == FALSE)
               {    /* If node not routed and not part of a rejected route
                       then find maximum link usage on route between current
                       node and destination node - init as best leaf */
                    best_usage = find_max_usage (branch_conv, level, leaf) ;
                    best_leaf = leaf ;
                    /* Look for other leaves with same source node */
                    for (new_leaf = (leaf + 1); new_leaf < (last_leaf * 4); new_leaf++)
                    {    if (node == BRANCH_CONV_node(level,new_leaf))
                         {    /* Find maximum link usage of new leaf node */
                              new_max_usage = find_max_usage (branch_conv, level, new_leaf) ;
                              /* If new leaf node has lower max link usage
                                 then make it the new best leaf and make
                                 previous best leaf a null route */
                              if (new_max_usage < best_usage)
                              {    BRANCH_CONV_null_route(level,best_leaf) ;
                                   best_usage = new_max_usage ;
                                   best_leaf = new_leaf ;
                              }
                              /* otherwise make new leaf a null route */
                              else BRANCH_CONV_null_route(level,new_leaf) ;
                         }
                    }
                    /* Store link number to leave source node in order to
                       reach dest node via the best route */
                    if (level == 0)
                         ROUTE_write(node,root_node,REVERSE_LINK_NO(root_node,best_leaf)) ;
                    else ROUTE_write(node,root_node,REVERSE_LINK_NO(BRANCH_CONV_node(level - 1,best_leaf >> 2), best_leaf & 3)) ;
                    /* Increment link usage of each link used on route */
                    for (new_level = 0; new_level <= level; new_level++)
                    {    node = BRANCH_CONV_node(new_level,best_leaf >> ((level - new_level) * 2)) ;
                         link = BRANCH_CONV_link(new_level,best_leaf >> ((level - new_level) * 2)) ;
                         OVERALL_USAGE(node,link) ++ ;
                         /* If mode > 0 then also increment new_usage and
                            store corresponding link number if necessary
                            (note only 1 link per node is used if routing
                            to a single dest node) */
                         if (new_usage != 0)
                         {    if (NEW_USAGE_usage(node) == 0) NEW_USAGE_init(node,link) ;
                              NEW_USAGE_inc(node) ;
                         }
                    }
                    /* Set routing flag of current source node */
                    ROUTED_FLAG(node) = TRUE ;
                    /* Increment number of nodes routed counter */
                    no_routed ++ ;
               }
          }
          /* Increment level and number of leaves on current level */
          level++ ;
          last_leaf *= 4 ;
          /*}}}*/
     }
     /* Look at next level of nodes unless all nodes have been routed */
     while (no_routed < (number_nodes - 1)) ;
     free (branch_conv) ;
     free (routed_flag) ;
}
/*}}}*/
/*{{{  Procedure to calculate and display overall link usage parameters*/
void calc_usage_params (max_usage, sum_usage)
long int *max_usage, *sum_usage ;
{    long int node, link ;
     *max_usage = 0 ;
     *sum_usage = 0 ;
     /* Check each node/link to find maximum and sum of link usages */
     for (node = 0; node < number_nodes; node++)
          for (link = 0; link < 4; link++)
          {    if (OVERALL_USAGE(node,link) > *max_usage) *max_usage = OVERALL_USAGE(node,link) ;
               (*sum_usage) += OVERALL_USAGE(node,link) ;
          }
     /* Display overall link utilisation */
     if (!silent) fprintf (stderr, "Link Utilisation = %2.1f%%", (((((float) *sum_usage) * 100.0) / ((float) (number_nodes * 4))) / ((float) *max_usage))) ;
}
/*}}}*/

int main (argc, argv)
/*{{{  Declarations*/
int argc ;
char *argv[] ;
{
long int node = 0, diameter = 0, dest_node, link = -1, max_usage, sum_usage ;
long int best_max_usage = maxint, run_no = 0, prev_best_max, mode = 2 ;
long int root_node = -1, root_link = root_link_default, count ;
long int wdg_mem_size = mem_size_default ;
char *wdg_processor = processor_default ;
/*}}}*/
/*{{{  Read program parameters*/
for (count = 1; count < argc; count++)
{
     /*{{{  Check for s, w or m parameter*/
     if ((argv [count] [0] == '-') || (argv [count] [0] == '/'))
     {    char c = argv [count] [1] ;
          if (isalpha (c))
          {    char option = (isupper(c) ? tolower(c) : c) ;
               if (option == 's')
                    silent = TRUE ;
               if (option == 'w')
                    wiring_diagram = TRUE ;
               if (option == 'm')
               {    count++ ;
                    if (sscanf (argv [count], "%ld", &mode) != 1)
                    {    fprintf(stderr, "\n\nError-%s(%d)- Mode parameter value undefined\n", __FILE__,__LINE__) ;
                         return (-1) ;
                    }
                    if ((mode < 0) || (mode > 2))
                    {    fprintf(stderr, "\n\nError-%s(%d)- Invalid mode parameter %ld (must be between 0 and 2)\n", __FILE__,__LINE__, mode) ;
                         return (-1) ;
                    }
               }
          }
     }
     /*}}}*/
     /*{{{  Check and read processor details*/
     else if ((argv [count] [0] == 'T') || (argv [count] [0] == 't'))
          {    wdg_processor = argv [count] ;
               wdg_processor [0] = 'T' ;
               if ((atoi(&wdg_processor[1])<100)||(atoi(&wdg_processor[1])>999))
               {
                 fprintf(stderr,"\nError-%s(%d)- Invalid processor name given on command line\n",__FILE__,__LINE__);
                 return(-1);
               }  
          }
          else
          {    long int value ;
               if (sscanf (argv [count], "%ld", &value) == 1)
               {    if ((strchr (argv [count], 'M') != NULL) || (strchr (argv [count], 'm') != NULL))
                         wdg_mem_size = value * 1024 ;
                    else if ((strchr (argv [count], 'K') != NULL) || (strchr (argv [count], 'k') != NULL))
                         wdg_mem_size = value ;
                    else
                    {    root_link = value ;
                         if ((root_link < 0) || (root_link > 3))
                         {    fprintf(stderr, "\n\nError-%s(%d)- Invalid link to host parameter %ld\n", __FILE__,__LINE__, root_link) ;
                              return (-1) ;
                         }
                    }
               }
          }
     /*}}}*/
}
/*}}}*/
if (!silent) fprintf (stderr, "\n\n               LOAD BALANCED ROUTING FILE GENERATOR\n\n\n");
/*{{{  Read in node connections from input file*/
if (!silent) fprintf (stderr, "Reading in node connections map") ;
/*{{{  Initialisation of arrays used to extract map file information*/
/* real_node stores actual node number of each node */
if ((real_node = (long int *) malloc (alloc_block * 4)) == NULL)
{    fprintf (stderr, "\n\nError-%s(%d)- Malloc for real_node array failed\n", __FILE__, __LINE__) ;
     return (-1) ;
}
/* node_con stores node number at the end of each link of each node */
if ((node_con = (long int *) malloc (alloc_block * 16)) == NULL)
{    fprintf (stderr, "\n\nError-%s(%d)- Malloc for node_con array failed\n", __FILE__, __LINE__) ;
     return (-1) ;
}
/* reverse_link_no stores link number which is at other end of each link */
if ((reverse_link_no = (long int *) malloc (alloc_block * 16)) == NULL)
{    fprintf (stderr, "\n\nError-%s(%d)- Malloc for reverse_link_no array failed\n", __FILE__, __LINE__) ;
     return (-1) ;
}
/* Following arrays are not needed if wiring diagram as input file */
if (!wiring_diagram)
{    /* mem_size stores total memory size for each node */
     if ((mem_size = (long int *) malloc (alloc_block * 4)) == NULL)
     {    fprintf (stderr, "\n\nError-%s(%d)- Malloc for mem_size array failed\n", __FILE__, __LINE__) ;
          return (-1) ;
     }
     /* processor stores string holding processor imformation for each node */
     if ((processor = (char *) malloc (alloc_block * 8)) == NULL)
     {    fprintf (stderr, "\n\nError-%s(%d)- Malloc for processor array failed\n", __FILE__, __LINE__) ;
          return (-1) ;
     }
}
/*}}}*/
/* Extract information from input file */
if (wiring_diagram)
{    root_node = 0 ;     /* Host is first processor stated in wiring diagram */
     read_wdg_file () ;
}
else read_chk_file (&root_node, &root_link) ;
/* Reallocate arrays to remove unused sections */
realloc_arrays (number_nodes, number_nodes) ;
free (real_node) ;
/*}}}*/
/* Route network unless single processor */
if (number_nodes == 1) diameter = 0 ;
else
{
     /*{{{  Initialisation*/
     diameter = find_diameter () ;
     if (diameter > 14)
     {    fprintf (stderr, "\n\nError-%s(%d)- Exceeded diameter limit (14)\n", __FILE__, __LINE__) ;
          return (-1) ;
     }
     /* Generate base array which is used to calculate overall leaf number
        from level and leaf number on that level */
     base[0] = 0 ;
     for (count = 1; count <= diameter; count++)
          base [count] = base [count - 1] + (1 << (count * 2 - 1)) ;
     
     route_size = number_nodes / 15 + 1 ;
     usage_size = number_nodes / 2 + 1 ;
     /*{{{  Display total dynamic memory requirements*/
     if (mode == 0)
     {    if (!silent) fprintf (stderr, "\nTotal dynamic memory requirement = %ld bytes (%ld bytes if mode = 1 or 2)",
                  number_nodes * (64 - 12 * wiring_diagram) + number_nodes *
                  route_size * 4 + 4 * base [diameter], number_nodes * (64 -
                  12 * wiring_diagram) + (route_size + usage_size) * 4 +
                  number_nodes * (route_size + usage_size) * 4 + 4 *
                  base [diameter]) ;
     }
     else
     {    if (!silent) fprintf (stderr, "\nTotal dynamic memory requirement = %ld bytes (%ld bytes if mode = 0)",
                  number_nodes * (64 - 12 * wiring_diagram) + (route_size +
                  usage_size) * 4 + number_nodes * (route_size + usage_size) *
                  4 + 4 * base [diameter], number_nodes * (64 - 12 *
                  wiring_diagram) + number_nodes * route_size * 4 + 4 *
                  base [diameter]) ;
     }
     /*}}}*/
     /* Initialise masks to split a 32-bit number into 2 16-bit numbers */
     mask [0] = 65535 ;
     mask [1] = -65536 ;
     /*{{{  Malloc arrays*/
     /* route gives link to leave source node on in order to reach dest node */
     if ((route = (long int *) malloc (number_nodes * route_size * 4)) == NULL)
     {    fprintf (stderr, "\n\nError-%s(%d)- Malloc for route array failed\n", __FILE__, __LINE__) ;
          return (-1) ;
     }
     /* overall_usage stores usage of each link for routing to all dest nodes */
     if ((overall_usage = (long int *) malloc (number_nodes * 16)) == NULL)
     {    fprintf (stderr, "\n\nError-%s(%d)- Malloc for overall_usage array failed\n", __FILE__, __LINE__) ;
          return (-1) ;
     }
     
     if (mode == 0) new_usage = 0 ;
     /* Following arrays are not needed if initialisation run only mode */
     else
     {    /* old_route stores old route info to dest node being re-routed */
          if ((old_route = (long int *) malloc (route_size * 4)) == NULL)
          {    fprintf (stderr, "\n\nError-%s(%d)- Malloc for old_route array failed\n", __FILE__, __LINE__) ;
               return (-1) ;
          }
          /* link_usage stores usage of each link to route to each dest node */
          if ((link_usage = (long int *) malloc (number_nodes * usage_size * 4)) == NULL)
          {    fprintf (stderr, "\n\nError-%s(%d)- Malloc for link_usage array failed\n", __FILE__, __LINE__) ;
               return (-1) ;
          }
          /* new_usage stores usage of each link for re-routed dest node */
          if ((new_usage = (long int *) malloc (usage_size * 4)) == NULL)
          {    fprintf (stderr, "\n\nError-%s(%d)- Malloc for new_usage array failed\n", __FILE__, __LINE__) ;
               return (-1) ;
          }
     }
     /*}}}*/
     /* Clear overall_usage array */
     for (node = 0; node < number_nodes; node++)
          for (link = 0; link < 4; link++)
               OVERALL_USAGE(node,link) = 0 ;
     /*}}}*/
     /*{{{  Initial run to find shortest paths between all nodes*/
     if (!silent) fprintf (stderr, "\n\nInitialising Run : ") ;
     
     for (dest_node = 0; dest_node < number_nodes; dest_node++)
     {
          if (!silent) fprintf (stderr, "%ld ", dest_node) ;
          /* Find link to take for each source node to reach current dest node
             in order that the network remains load balanced */
          find_leaves (dest_node, diameter) ;
          /* Record link usage for routing to current dest node unless mode = 0 */
          if (mode > 0)
               for (node = 0; node < number_nodes; node += 2)
                    LINK_USAGE(dest_node,node) = NEW_USAGE(node) ;
     }
     if (!silent) fprintf (stderr, "\n") ;
     /* Calculate and display link utilisation figures for initial run */
     calc_usage_params (&best_max_usage, &sum_usage) ;
     /*}}}*/
     /*{{{  Re-run to improve link usages unless mode = 0*/
     if (mode > 0)
          do
          {    /* Record current best link utilisations */
               prev_best_max = best_max_usage ;
               for (dest_node = 0; dest_node < number_nodes; dest_node++)
               {    if (!silent) fprintf (stderr, "\nRun Number = %ld,%ld : ", run_no, dest_node) ;
                    /*{{{  Remove usage for routing to dest node from overall usage*/
                    for (node = 0; node < number_nodes; node++)
                    {    if (node % 15 == 0) OLD_ROUTE(node) = ROUTE(node,dest_node) ;
                         for (link = 0; link < 4; link++)
                              if (link == LINK_USAGE_link(dest_node,node))
                                   OVERALL_USAGE(node,link) -= LINK_USAGE_usage(dest_node,node) ;
                    }
                    /*}}}*/
                    /* Re-route to current dest node using updated overall link
                       usage as basis for load balancing network */
                    find_leaves (dest_node, diameter);
                    /*{{{  Display link utilisations and check whether improved run*/
                    /* Calculate and display new link utilisation figures */
                    calc_usage_params (&max_usage, &sum_usage) ;
                    /* If link utilisation is improved or equalled then record
                       new values */
                    if (max_usage <= best_max_usage)
                    {    best_max_usage = max_usage ;
                         if (!silent) fprintf (stderr, " *") ;
                         /* Record new link usage for current dest node */
                         for (node = 0; node < number_nodes; node++)
                              LINK_USAGE(dest_node,node) = NEW_USAGE(node) ;
                    }
                    /* else return overall_usage and route to original values */
                    else
                    {    for (node = 0; node < number_nodes; node++)
                         {    if (node % 15 == 0) ROUTE(node,dest_node) = OLD_ROUTE(node) ;
                              for (link = 0; link < 4; link++)
                              {    if (link == NEW_USAGE_link(node))
                                        OVERALL_USAGE(node,link) -= NEW_USAGE_usage(node) ;
                                   if (link == LINK_USAGE_link(dest_node,node))
                                        OVERALL_USAGE(node,link) += LINK_USAGE(dest_node,node) ;
                              }
                         }
                    }
                    /*}}}*/
               }
               run_no++ ;
          }
          /* Re-run again if mode = 2 and average link utilisation improved */
          while ((mode > 1) && (best_max_usage != prev_best_max)) ;
     /*}}}*/
}
/*{{{  Output routing file*/
if (!silent) fprintf (stderr, "\n\nWriting out routing file\n") ;
/* Output header line */
fprintf (stdout, "%ld %ld %ld %ld\n", number_nodes, diameter, root_node, root_link) ;
/*{{{  Output information for each processor*/
for (node = 0; node < number_nodes; node++)
{
     /*{{{  Output processor number, type and memory size*/
     fprintf (stdout, "\n%ld\n", node) ;
     if ((wiring_diagram) || (!mtest))
          fprintf (stdout, "%s\n%ldK\n", wdg_processor, wdg_mem_size) ;
     else
     {    for (count = 0; count < 8; count++)
               fputc (PROCESSOR(node,count), stdout) ;
          fprintf (stdout, "\n%ldK\n", MEM_SIZE(node)) ;
     }
     /*}}}*/
     /*{{{  Output link connections*/
     for (link = 0; link < 4; link++)
     {    fprintf (stdout, "%ld", NODE_CON(node,link)) ;
          /* output reverse link number unless link not used */
          if (NODE_CON(node,link) == not_used) fprintf (stdout, " ") ;
          else fprintf (stdout, ":%d ", REVERSE_LINK_NO(node,link)) ;
     }     
     fprintf (stdout, "\n") ;
     /*}}}*/
     /*{{{  Output route directions table*/
     count = 0 ;
     for (dest_node = 0; dest_node < number_nodes; dest_node++)
     {    if (dest_node == node) fprintf (stdout, "-1 ") ;
          else fprintf (stdout, "%ld ", ROUTE_read(node,dest_node)) ;
          if (++count == 20)
          {    fprintf (stdout, "\n") ;
               count = 0 ;
          }
     }
     if (count != 0) fprintf (stdout, "\n") ;
     /*}}}*/
}
/*}}}*/
/*}}}*/
return (0) ;
}

