/*{{{  File banner*/
/*@(#)=====================================================*\
||@(#)  Project : PUMA ESPRIT P2701
||@(#)  Authors : Mark Debbage and Mark Hill
||@(#)            University of Southampton
||  
||@(#)    Title : Route Generating Software (from chk or map files)
||@(#)   System : ROUTEGEN
||@(#) Filename : routegen.c
||@(#)  Version : 2.33
||@(#)     Date : 10/15/92
\*@(#)====================================================*/
/*}}}*/

/*{{{  includes*/
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <limits.h>
#include <stdlib.h>
/*}}}*/
/*{{{  definitions*/
typedef unsigned char BYTE ;
typedef int BOOL;

#define false                (0)
#define true                 (1)

#define MAX_BYTE     ((BYTE) 255)
#define MAX_VALUE    1000000
#define MAX_LINE     100
#define MAX_PART     10

#define EOLN         10

#define maxnodes    1000
#define ndirection  4             /* no of transputer links */

#define kilobyte 1024

#define not_used ((int) (-1))    /* marker for unused connections */

#define path_end ((BYTE) (-1))
#define not_done ((BYTE) (-2))

#define ZERO     ((BYTE) 0)

#define TRUE  (1)
#define FALSE (0)

#define SQR(x) ((x)*(x))

#define memory_default 256
#define boot_link_default 0

#define MAX(a,b) ((a)>(b) ? (a) : (b))
#define MIN(a,b) ((a)<(b) ? (a) : (b))

static char *ROUTEGEN_ARG = "ROUTEGEN_ARG" ;
/*}}}*/
/*{{{  struct NET*/
typedef struct
{
  int unmapped_id ;          /* Unmapped node id */
  int con_proc[ndirection] ; /* Neighbouring processor */
  int con_link[ndirection] ; /* Neighbouring processor's link */
  int boot ;                 /* Boot link */
  char part[MAX_PART];       /* Part Name */
  int memory;                /* Memory in K */
  int edge[ndirection];      /* Maps compass directions to links */
}
NET ;
/*}}}*/
/*{{{  manipulators*/
/*{{{  int get_line( FILE * ip, char * buffer, char line[])*/
int get_line( FILE * ip, char * buffer, char line[])
{
  int letter = 0;

  while ((*buffer!= (char) EOLN) && (*buffer!= (char) EOF))
  {
    line[letter++] = *buffer ;
    *buffer = getc(ip);
  }

  *buffer = getc(ip);
  line[letter] = 0;
  return(letter) ;
}
/*}}}*/
/*{{{  int search( char line[], char match[])*/
int search( char line[], char match[])
{
  int line_i = 0;
  int match_i = 0;

  while (line[line_i]!=0)
  {
    if (line[line_i] == match[match_i])
    {
      match_i++ ;
      if (match[match_i] == 0)
        return( line_i - match_i + 1) ;
    }
    else
      match_i = 0;
    line_i++;
  }
  return(-1) ;
}

/*}}}*/
/*{{{  int after_match(char line[], char match[])*/
int after_match(char line[], char match[])
{
  return(search(line, match) + strlen(match)) ;
}
/*}}}*/
/*{{{  void skip_till( FILE * inp, char *buffer, char stopper )*/
void skip_till( FILE * inp, char *buffer, char stopper )
{
  while ((*buffer != stopper)&&(*buffer != (char) EOF))
    *buffer = getc(inp) ;
}
/*}}}*/

/*}}}*/
/*{{{  network description statics*/
static NET **node ;
static int node_cnt    = 0;
static int free_links  = 0;
static int root        = not_used;
static int diameter ;
static current_maxnodes = maxnodes ;

static int debug        = FALSE ;
static int silent       = FALSE ;
static int wiring_dgm   = FALSE ;
static int show_map     = FALSE ;
static int net_match    = FALSE ;
static int cl_boot_link = -1 ;
static int cl_memory    = -1 ;
static char *cl_part    = NULL ;
static char *part_default = "T414" ;

static char *wdg_connect = "Connect processor" ;
static char *wdg_link    = "link" ;
static char *wdg_to      = "to processor" ;

static int *old_to_new_ids ;
static int *new_to_old_ids ;

/*}}}*/
/*{{{  debug*/
#ifdef HEAP_DEBUG
static void *malloc_tmp ;
#define malloc(b) (fprintf(stderr,"malloc %d bytes gives %x\n",b,(int)(malloc_tmp=malloc(b))),malloc_tmp)
#define free(p)   (fprintf(stderr,"free pointer %x\n",(int)p),free(p))
#endif
/*}}}*/
/*{{{  access arrays by pointer expansions*/
#define DISTANCE(s,d)    (*(distance + (s)*node_cnt + (d)))
#define DIRECTION(s,d,a) (*(direction + (s)*node_cnt*ndirection + (d)*ndirection + (a)))
#define ROUTE_CTR(s,d)   (*(route_ctr + (s)*node_cnt + (d)))
#define CHOSEN_DIR(s,d) (*(chosen_dir + (s)*node_cnt + (d)))
#define CHOSEN_VIR(s,d) (*(chosen_vir + (s)*node_cnt + (d)))
/*}}}*/

/*{{{  int LoadCheckerFile(FILE *input_fptr)*/
int LoadCheckerFile(FILE *input_fptr)
{
  char buffer ;
  char input[MAX_LINE];
  int max_node_id = 0;  /*  Largest id used */
  BOOL mtest ;
 
  if (!silent)
    fprintf(stderr, "\nReading in connections map :\n") ;
  
  /*{{{  clear NET pointers*/
  {
    int i ;
    for (i=0;i<current_maxnodes;i++)
      node[i] = NULL ;
  }
  /*}}}*/
  /*{{{  setup read*/
  buffer = getc(input_fptr) ;
  get_line(input_fptr, &buffer, input) ;   /* Identifies check file */
  /*{{{  check for check*/
  if (search(input, "check ") < 0)
  {
    fprintf(stderr, "Error-%s(%d)- Not a checkout output file\n",__FILE__,__LINE__) ;
    return(-1) ;
  }
  /*}}}*/
  /*{{{  check for mtest*/
  if (search(input, "mtest ") < 0)
  {
    if (cl_memory < 0)
    {
      fprintf(stderr, "Error-%s(%d)- No Memory Sizing and no command-line value\n",__FILE__,__LINE__) ;
      return(-1) ;
    }
    else
      mtest = FALSE ;
  }
  else
    mtest = TRUE ;
  /*}}}*/
  get_line(input_fptr, &buffer, input) ;   /* Header for table */
  
  /*}}}*/

  /* Line format is :   1 T414b-20 0.49 0 [    0:2    3:1    4:3    5:2 ]     */
  /* Record ids of connections then update to get contiguous node address set */

  /*{{{  parse processor records*/
  while (buffer != (char) EOF) 
  {
    int id;
    
    /*{{{  read node id*/
    if (fscanf(input_fptr, "%d", &id) == EOF) break ;
    
    if (id>max_node_id) max_node_id = id ;
    
    if (id>=current_maxnodes)
      /*{{{  out of nodes*/
      {
        int old_maxnodes = current_maxnodes ;
        current_maxnodes = 2*current_maxnodes ;
      
        if (!silent)
          fprintf(stderr, "\nReallocating nodes bound to %d\n", current_maxnodes) ;
      
        if ((node = (NET **) realloc (node, current_maxnodes * sizeof(NET *))) == NULL)
        {
          fprintf(stderr, "Error-%s(%d)- Realloc for net pointers array failed\n", __FILE__,__LINE__) ;
          return(-1) ;
        }
      
        {
          int i ;
          for (i=old_maxnodes;i<current_maxnodes;i++)
            node[i] = NULL ;
        }
      }  
      /*}}}*/
    
    /*{{{  allocate node*/
    if (node[id] == NULL)
    {
      if ((node[id] = (NET *) malloc (current_maxnodes * sizeof(NET *))) == NULL)
      {
        fprintf(stderr, "Error-%s(%d)- Malloc for net instance failed\n", __FILE__,__LINE__) ;
        return(-1) ;
      }
    }  
    else
    {
      fprintf(stderr, "Error-%s(%d)- Processor %d declared twice\n",__FILE__,__LINE__,id);
      return(-1);
    }
    
    node[id]->unmapped_id = id ;
    node[id]->boot = not_used ;
    /*}}}*/
      
    if (!silent)
    {
      fprintf(stderr, "<%04d>  ", id) ;
      if ((node_cnt % 5) == 4) fprintf (stderr, "\n") ;
    }
    
    /*}}}*/
    /*{{{  read type, speed + boot link*/
    fscanf(input_fptr, "%s", node[id]->part) ;    /* Part type */
    fscanf(input_fptr, "%s", input) ;    /* Link speed */
    fscanf(input_fptr, "%s", input) ;    /* Boot link */
    /*}}}*/
    /*{{{  match open square bracket*/
    buffer = getc(input_fptr) ;
    skip_till(input_fptr, &buffer, '[') ;    /* Match bracket */
    /*}}}*/
    /*{{{  read connections*/
    {
      int link ;
      for (link=0;link<ndirection;link++)
      {
        fscanf(input_fptr, "%s", input) ; /* Link connection */
        if (isdigit(input[0]))
        {
          if ((sscanf(input,"%d:%d",&node[id]->con_proc[link],&node[id]->con_link[link])) != 2)
          {
            fprintf(stderr, "\nError-%s(%d)- Link connection data not colon separated\n",__FILE__,__LINE__) ;
            return(-1) ;
          }
        }
        else
        {
          if (search(input,"HOST")>=0)
          {
            root = id ;
            node[id]->boot = link ;
          }
          else
            free_links++ ;
          node[id]->con_proc[link] = not_used ;
          node[id]->con_link[link] = not_used ;
        }
      }
    }
    /*}}}*/
    /*{{{  match and skip close square bracket*/
    buffer = getc(input_fptr) ;
    skip_till(input_fptr, &buffer, ']') ;    /* Match bracket */
    buffer = getc(input_fptr) ;              /* Skip bracket */
    /*}}}*/
    /*{{{  get memory size*/
    if (mtest)
    {
      int index;
      int end;
      fscanf(input_fptr, "%s", input) ;
      node[id]->memory=atoi(input);
      index = (end = search(input,"+"));
      while (end>=0)
      {
        node[id]->memory += atoi(&(input[++index]));
        index += (end = search(&input[index],"+"));
      }
    }
    else
      node[id]->memory = cl_memory ;
  /*}}}*/
    node_cnt++ ;
  }
  /*}}}*/

  /*{{{  if (!silent)*/
  if (!silent)
   fprintf(stderr, "\n\n") ;
  /*}}}*/
  /*{{{  check for contiguous address set*/
  {
    if ((node_cnt-1)!=max_node_id)
    {
      fprintf(stderr, "Error-%s(%d)- Non-contiguous processor labelling\n",__FILE__,__LINE__);
      return(-1) ;
    }    
  }
  /*}}}*/
  /*{{{  if (debug) */
  if (debug) 
  {
    int start, link;
    for (start=0;start<node_cnt;start++)
    {
      fprintf(stderr, "Node %d : ", start) ;
      for (link=0;link<ndirection;link++)
        fprintf(stderr, "%3d:%-3d ", node[start]->con_proc[link], node[start]->con_link[link]) ;
      fprintf(stderr, "\n") ;
    }
  }
  /*}}}*/
}
/*}}}*/
/*{{{  int WDG_FindNode(int unmapped_id)*/
int WDG_FindNode(int unmapped_id)
{
  int id ;

  for (id=0;id<node_cnt;id++)
    if (node[id]->unmapped_id == unmapped_id)
      break ;

  if (id == node_cnt)
    /*{{{  new node*/
    {
      if (id>=current_maxnodes)
        /*{{{  out of nodes*/
        {
          int old_maxnodes = current_maxnodes ;
          current_maxnodes = 2*current_maxnodes ;
        
          if (!silent)
            fprintf(stderr, "\nReallocating nodes bound to %d\n", current_maxnodes) ;
        
          if ((node = (NET **) realloc (node, current_maxnodes * sizeof(NET *))) == NULL)
          {
            fprintf(stderr, "Error-%s(%d)- Realloc for net pointers array failed\n", __FILE__,__LINE__) ;
            return(-1) ;
          }
        
          {
            int i ;
            for (i=old_maxnodes;i<current_maxnodes;i++)
              node[i] = NULL ;
          }
        }  
        /*}}}*/
    
      /*{{{  allocate node*/
      if (node[id] == NULL)
      {
        if ((node[id] = (NET *) malloc (sizeof(NET))) == NULL)
        {
          fprintf(stderr, "Error-%s(%d)- Malloc for net instance failed\n", __FILE__,__LINE__) ;
          return(-1) ;
        }
      }  
      else
      {
        fprintf(stderr, "Error-%s(%d)- Processor %d declared twice\n",__FILE__,__LINE__,id);
        return(-1);
      }
      
      if (debug)
        fprintf(stderr, "Mapped wdg proc %d to ncf proc %d\n", unmapped_id, id) ;
        
      /*}}}*/
      /*{{{  initialise node*/
      node[id]->boot = not_used ;
      node[id]->unmapped_id = unmapped_id ;
      node[id]->memory = cl_memory ;
      strcpy(node[id]->part, cl_part) ;
      
      {
        int i ;
      
        for(i=0;i<ndirection;i++)
        {
          node[id]->con_proc[i] = not_used ;
          node[id]->con_link[i] = not_used ;
        }
      }
      /*}}}*/
    
      node_cnt++ ;
    }
    /*}}}*/

  if (root == not_used)
  {
    root = id ;
    node[id]->boot = cl_boot_link ;

    if (!silent)
      fprintf(stderr, "Host processor is wdg proc %d (ncf proc %d)\n", unmapped_id, id) ;
  }

  return(id) ;
}

  
/*}}}*/
/*{{{  void WDG_Connect(int sp, int sl, int dp, int dl)*/
void WDG_Connect(int sp, int sl, int dp, int dl)
{
  int sp_id = WDG_FindNode(sp) ;
  int dp_id = WDG_FindNode(dp) ;

  if (node[sp_id]->con_proc[sl] != not_used)
  {
    fprintf(stderr,"Badly formed wiring diagram, processor %d link %d reused\n",
            node[sp_id]->unmapped_id, sl) ;
    exit(-1) ;
  }

  if (node[dp_id]->con_proc[dl] != not_used)
  {
    fprintf(stderr,"Badly formed wiring diagram, processor %d link %d reused\n",
            node[dp_id]->unmapped_id, dl) ;
    exit(-1) ;
  }
  
  node[sp_id]->con_proc[sl] = dp_id ;
  node[sp_id]->con_link[sl] = dl ;
  node[dp_id]->con_proc[dl] = sp_id ;
  node[dp_id]->con_link[dl] = sl ;
}
/*}}}*/
/*{{{  int LoadWiringDiagram(FILE *input_fptr)*/
int LoadWiringDiagram(FILE *input_fptr)
{
  char buffer ;
  char input[MAX_LINE] ;
  int  line=0 ;
  BOOL single=FALSE ;
 
  if (!silent)
    fprintf(stderr, "\nReading in connections map :\n") ;
  
  /*{{{  clear NET pointers*/
  {
    int i ;
    for (i=0;i<current_maxnodes;i++)
      node[i] = NULL ;
  }
  /*}}}*/

  /*{{{  setup read*/
  buffer = getc(input_fptr) ;
  
  get_line(input_fptr, &buffer, input) ;
  line++ ;
  /*}}}*/
  /*{{{  search for first connect*/
  while ((search(input, wdg_connect)) < 0)
  {
    if (buffer == (char) EOF)
    {
      single=TRUE ;
      break ;
    }
    
    get_line(input_fptr, &buffer, input) ;
    line++ ;
  }
  /*}}}*/
  
  if (!single)
    /*{{{  read connects*/
    do
    {
      int pos=0 ;
    
      /*{{{  read wiring diagram connects*/
      {
        int sp, sl, dp, dl ;
      
        /*{{{  get source processor*/
        if ((pos+=after_match(&input[pos], wdg_connect)) < 0)
        {
          fprintf(stderr, "Error-%s(%d)- Bad text in wiring diagram, line %d\n",__FILE__,__LINE__,line) ;
          return(-1) ;
        }
        
        if (sscanf(&input[pos], "%d", &sp) != 1)
        {
          fprintf(stderr, "Error-%s(%d)- Bad source processor number in wiring diagram, line %d\n",__FILE__,__LINE__,line) ;
          return(-1) ;
        }
        /*}}}*/
        /*{{{  get source link*/
        if ((pos+=after_match(&input[pos], wdg_link)) < 0)
        {
          fprintf(stderr, "Error-%s(%d)- Bad text in wiring diagram, line %d\n",__FILE__,__LINE__,line) ;
          return(-1) ;
        }
        
        if (sscanf(&input[pos], "%d", &sl) != 1)
        {
          fprintf(stderr, "Error-%s(%d)- Bad source link number in wiring diagram, line %d\n",__FILE__,__LINE__,line) ;
          return(-1) ;
        }
        /*}}}*/
        /*{{{  get destination processor*/
        if ((pos+=after_match(&input[pos], wdg_to)) < 0)
        {
          fprintf(stderr, "Error-%s(%d)- Bad text in wiring diagram, line %d\n",__FILE__,__LINE__,line) ;
          return(-1) ;
        }
        
        if (sscanf(&input[pos], "%d", &dp) != 1)
        {
          fprintf(stderr, "Error-%s(%d)- Bad destination processor link number in wiring diagram, line %d\n",__FILE__,__LINE__,line) ;
          return(-1) ;
        }
        /*}}}*/
        /*{{{  get destination link*/
        if ((pos+=after_match(&input[pos], wdg_link)) < 0)
        {
          fprintf(stderr, "Error-%s(%d)- Bad text in wiring diagram, line %d\n",__FILE__,__LINE__,line) ;
          return(-1) ;
        }
        
        if (sscanf(&input[pos], "%d", &dl) != 1)
        {
          fprintf(stderr, "Error-%s(%d)- Bad destination link number in wiring diagram, line %d\n",__FILE__,__LINE__,line) ;
          return(-1) ;
        }
        /*}}}*/
      
        if (debug)
          fprintf(stderr,"%s %d %s %d %s %d %s %d\n",
                         wdg_connect, sp, wdg_link, sl,
                         wdg_to, dp, wdg_link, dl) ;
                         
        WDG_Connect(sp, sl, dp, dl) ;
      }
      /*}}}*/
     
      get_line(input_fptr, &buffer, input) ;
      line++ ;
    } while (strlen(input) != 0) ;
    /*}}}*/
  else
    /*{{{  no connects to read*/
    {
      if (!silent)
        fprintf(stderr, "Single processor network\n") ;
    
      WDG_FindNode(0) ;
    }
    /*}}}*/

  /*{{{  count up free links*/
  {
    int id, i ;
  
    for (id=0;id<node_cnt;id++)
      for (i=0;i<ndirection;i++)
        if (node[id]->con_proc[i] == not_used)
          free_links++ ;
  }        
  /*}}}*/
  
  /*{{{  if (debug) */
  if (debug) 
  {
    int start, link;
    for (start=0;start<node_cnt;start++)
    {
      fprintf(stderr, "Node %d : ", start) ;
      for (link=0;link<ndirection;link++)
        fprintf(stderr, "%3d:%-3d ", node[start]->con_proc[link], node[start]->con_link[link]) ;
      fprintf(stderr, "\n") ;
    }
  }
  /*}}}*/

  if (node_cnt ==0)
  {
    fprintf(stderr, "Error-%s(%d)- No nodes found in wiring diagram\n",__FILE__,__LINE__) ;
    return(-1) ;
  }
}
/*}}}*/

#ifdef CUBE_MATCH
/*{{{  BOOL match_to_cube(BYTE *chosen_dir, BYTE *chosen_vir)*/
BOOL match_to_cube(BYTE *chosen_dir, BYTE *chosen_vir)
{
  int *cube_ids = (int*) malloc(node_cnt*sizeof(int));

  if (cube_ids==NULL)
  { fprintf(stderr,"Error- Malloc Failed\n"); exit(-1); }
  
  /*{{{  clear cube labels*/
  {
    int i;
    for (i=0;i<node_cnt;i++) cube_ids[i] = not_used; 
  }
  /*}}}*/
  {
    int start;
    /*{{{  find the node with the most*/
    {
      int i=0;
      int most=0;
      for (i=0;i<node_cnt;i++)
      {
        int j;
        int no_links = 0;
        for (j=0;j<ndirection;j++)
          if (node[i]->con_proc[j]!=not_used) no_links++;
        if (most<no_links)
        {
          start=i;
          most = no_links;
        }
      }
    }
      
    /*}}}*/
    /*{{{  label nodes adjacent to most*/
    {
      int i;
      int adj = 1;
      cube_ids[start] = 0;
      for (i=0;i<ndirection;i++)
      {
        if (node[start]->con_proc[i]!=not_used)
        {
          cube_ids[node[start]->con_proc[i]] = adj;
          adj<<=1;
        }
      }
    }
    /*}}}*/
  }
  /*{{{  label nodes with two adjacent labels*/
  {
    int i  ;
    for (i=0;i<ndirection;i++)
    {
      int j;
      for (j=0;j<node_cnt;j++)
        if (cube_ids[j]==not_used)
          {
            int k;
            int no_adj = 0;
            int orall = 0;
            for (k=0;k<ndirection;k++)
              if (node[j]->con_proc[k]!=not_used)
              {
                int adj_cube_id = cube_ids[node[j]->con_proc[k]];
                if (adj_cube_id!=not_used)
                {
                  no_adj++;
                  switch (no_adj)
                  {
                    case 1:
                      orall = adj_cube_id;
                      break;
                    case 2:
                      orall |= adj_cube_id;
                      cube_ids[j] = orall; 
                      break;
                    default:
                      if ((no_adj>2)&&(orall != (orall | adj_cube_id)))
                      {
                        free(cube_ids) ;
                        return(FALSE);
                      }
                  }
                }
              }
          }
    }
  }
                
      
  /*}}}*/
  /*{{{  join in single connections*/
  {
    int i;
    for (i=0;i<node_cnt;i++)
      if (cube_ids[i]==not_used)
      {
        int no_adj=0;
        int j;
        for (j=0;j<ndirection;j++)
          if (node[i]->con_proc[j]!=not_used)
          {
            int adj_id;
            int k;
            int orall = 0;
            int dir = 1;
            
            adj_id = node[i]->con_proc[j];
            if (no_adj++==1)
            {
              free(cube_ids) ;
              return(FALSE);
            }
            for (k=0;k<ndirection;k++)
              if (node[adj_id]->con_proc[k]!=not_used)
                orall |= cube_ids[node[adj_id]->con_proc[k]];
            while ((orall&dir)!=0) dir <<= 1;
            cube_ids[i] = cube_ids[adj_id] | dir;
            if (cube_ids[i]>=(ndirection<<1))
            {
              free(cube_ids) ;
              return(FALSE);
            }
         }
        
      }
  } 
                    
  /*}}}*/
  /*{{{  generate chosen directions and virtual links*/
  {
    int start,finish;
    for (start=0;start<node_cnt;start++)
    {
      int dims[ndirection];
      int i, dir, diff;
  
      for (i=0;i<ndirection;i++)
        dims[i] = not_used;
      /*{{{  record link associated with each dimension*/
      for (i=0;i<ndirection;i++)
      {
        if (node[start]->con_proc[i]!=not_used)
        {
          diff = cube_ids[start] ^ cube_ids[node[start]->con_proc[i]];
          dir =0;
          while ((diff>>dir)>1) dir++;
          dims[dir] = i;
        }
      }
      /*}}}*/
      /*{{{  route to finish on highest dimension first*/
      for (finish=0;finish<node_cnt;finish++)
      {
        if (start==finish)
          CHOSEN_DIR(start,finish) = path_end;
        else
        {
          diff = cube_ids[start] ^ cube_ids[finish];
          dir = 0;
          while ((diff>>dir)>1) dir++;
          while (dims[dir]==not_used)
            dir = (ndirection + dir -1) % ndirection;
          CHOSEN_DIR(start,finish) = dims[dir];
        }
        CHOSEN_VIR(start,finish) = 0 ;
      }
      /*}}}*/
    }      
  }
  /*}}}*/
  free(cube_ids) ;
  return(TRUE);
}
/*}}}*/
#endif

/*{{{  BOOL match_to_grid(BYTE *chosen_dir, BYTE *chosen_vir)*/
/*#define GRID_DEBUG*/
#define GRID_MATCH_FAILED(s) { fprintf(stderr,"Grid match failed (line %d): %s\n",__LINE__,s) ; return(FALSE) ; }

/*{{{  definitions*/
/* This code is heavily tailored to two-dimensional grids/torii */

#define north 0
#define east  1
#define west  2
#define south 3

int orthogonal[ndirection][2] = {{east,west},{north,south},{north,south},{east,west}} ;
int opposite[ndirection] = {south,west,east,north} ;
int corners[ndirection][2] = {{north,east},{south,east},{south,west},{north,west}} ;
/* corners progressions must only toggle one direction each time */

BOOL wrapx = TRUE, wrapy = TRUE ;

int side_limit[ndirection] ;
int centre, count ;
BOOL progress ;
typedef enum {torus_check,grid_check,other_check} CHECK_MODE ;
CHECK_MODE check_mode ;

#define unknown (-1)
#define new_id  (-2)
#define old_id  (-3)

/* value must only be evaluated once */
#ifdef GRID_DEBUG
#define EDGE(d1,d2,value) {int v=value; fprintf(stderr,"Set node[%d]->edge[%d] to %d\n",d1,d2,v) ; node[d1]->edge[d2]=v ; progress=TRUE ;}
#else
#define EDGE(d1,d2,value) {node[d1]->edge[d2]=value ; progress=TRUE ;}
#endif
/*}}}*/

/*{{{  static data for extras, routing*/
typedef struct
{
  int num ;
  int pnode ;
  int plink ;
} one_node ;

typedef struct
{
  int num ;
  int pnode[2] ;
  int plink[2] ;
} two_node ;

int extra_one=0 ;
int extra_two=0 ;
one_node *one ;
two_node *two ;

int nx, ny, hnx, hny ;
int offset = 0 ;
enum {none,spliced_in_x,spliced_in_y} sp = none ;
int first_id ;

#define XY_TO_ID(x,y) (first_id+(y)*nx+(x))
#define ID_TO_X(id)   ((id-first_id+nx)%nx)
#define ID_TO_Y(id)   ((id-first_id)/nx)
  
/*}}}*/

/*{{{  int  valency (int n)*/
int  valency (int n)
{
  int num=0, i ;
  for(i=0;i<ndirection;i++)
    if (node[n]->con_proc[i] != not_used)
      num++ ;
  return(num) ;
}  
/*}}}*/

/*{{{  void print_grid()*/
void print_grid()
{
  #ifdef GRID_DEBUG
  int origin = centre ;
  
  while (node[origin]->edge[north] != unknown)
    origin = node[origin]->con_proc[node[origin]->edge[north]] ;
  while (node[origin]->edge[west] != unknown)
    origin = node[origin]->con_proc[node[origin]->edge[west]] ;

  {
    int col = origin ;
  
    do
    {
      int row = col ;
      fprintf(stderr,"%.2d",row) ;
      while (node[row]->edge[east] != unknown)
      {
        row = node[row]->con_proc[node[row]->edge[east]] ;
        if (check_mode==torus_check && valency(row) == 2)
        {
          fprintf(stderr,"-%.2d-",row) ;
          if (node[row]->edge[east] == unknown)
            break ;
          else
            row = node[row]->con_proc[node[row]->edge[east]] ;
        }
        else
          fprintf(stderr,"----") ;
        fprintf(stderr,"%.2d",row) ;
      }
      row=col ;
      fprintf(stderr,"\n") ;
      do
      {
        if (check_mode==torus_check && valency(row) == 2)
        {
          if (node[row]->edge[east] == unknown)
            break ;
          else
            row = node[row]->con_proc[node[row]->edge[east]] ;
        }

        if (node[row]->edge[south] == unknown)
          fprintf(stderr,"      ") ;
        else
          fprintf(stderr," |    ") ;

        if (node[row]->edge[east] == unknown)
          break ;
        else
          row = node[row]->con_proc[node[row]->edge[east]] ;
      } while(1) ;
      fprintf(stderr,"\n") ;
  
      if (node[col]->edge[south] == unknown)
        break ;
      else
        col = node[col]->con_proc[node[col]->edge[south]] ;
    } while(1) ;
  }
  #endif
}  
/*}}}*/

/*{{{  int  get_node (int here, int pdir)*/
int  get_node (int here, int pdir)
{
  if (pdir == unknown)
    return(not_used) ;
  else
  {  
    int prev ;
    int next = node[here]->con_proc[pdir] ;
    int first = next ;
      
    if (next == not_used || check_mode==other_check)
      return(next) ;

    #ifdef GRID_DEBUG
    fprintf(stderr,"get_node with mode %d\n",check_mode) ;
    #endif

    while (1)
    {
      int i, num=0 ;
    
      prev = here ;
      here = next ;

      #ifdef GRID_DEBUG
      fprintf(stderr,"Moved from %d to %d\n",prev,here) ;
      #endif
     
      for (i=0;i<ndirection;i++)
        if (node[here]->con_proc[i] != not_used)
        {
          if (node[here]->con_proc[i] != prev)
            next = node[here]->con_proc[i] ;
          num++ ;
        }
      
      if (num == 1)
        return(not_used) ;
        
      if (num != 2)
      {
        if (check_mode==torus_check)
          return(here) ;
        else if (check_mode==grid_check)
          return(first) ;
      }
      else if (next == first)     
        /*{{{  catch 1-d rings*/
        return(first) ;      
        /*}}}*/
    }
  }
}

/* If torus_check, skips over 2-valent nodes
      grid_check,  ignores chains ending in 1-valent nodes
      otherwise,   includes all nodes */
/*}}}*/

/*{{{  int  add_node (int here, int ldir)*/
int  add_node (int here, int ldir)
{
  int next = node[here]->con_proc[node[here]->edge[ldir]] ;
  int prev ;

  if (next == not_used)
  { fprintf(stderr,"Error- Illegal use of add_node") ; exit(-1) ; }

  if (check_mode==torus_check)
    /*{{{  fill in details for 2-valent nodes*/
    while (1)
    {
      int i, num=0, pdir ;
    
      EDGE(next,opposite[ldir],node[here]->con_link[node[here]->edge[ldir]]) ;
      old_to_new_ids[next] = old_id ;
      
      prev = here ;
      here = next ;
    
      for (i=0;i<ndirection;i++)
        if (node[here]->con_proc[i] != not_used)
        {
          if (node[here]->con_proc[i] != prev)
          {
            pdir = i ;
            next = node[here]->con_proc[i] ;
          }
          num++ ;
        }
      if (num != 2)
        return(here) ;
    
      EDGE(here,ldir,pdir) ;
    }
    /*}}}*/
  else
    {
      old_to_new_ids[next] = old_id ;
      EDGE(next,opposite[ldir],node[here]->con_link[node[here]->edge[ldir]]) ;
      return(next) ;
    }
}
/*}}}*/

/*{{{  int  test_square (int start, int n1, int n2, int *d1, int *d2)*/
int  test_square (int start, int n1, int n2, int *d1, int *d2)
{
  if (start < 0 || n1 < 0 || n2 < 0)
  { fprintf(stderr,"Error- Illegal parameters to test_square : %d, %d, %d\n",
    start, n1, n2) ; exit(-1); }

  #ifdef GRID_DEBUG
  fprintf(stderr,"Test square given nodes %d %d %d\n",start,n1,n2) ;
  #endif
  if (start!=n1 && n1!=n2 && n2!=start)
    for (*d1=0;*d1<ndirection;(*d1)++)
    {
      int n3 = get_node(n1,*d1) ;
      if ((n3 != not_used) && (n3 != start))
      {
        #ifdef GRID_DEBUG
        fprintf(stderr,"Test square formed by nodes %d %d %d %d\n",start,n1,n2,n3) ;
        #endif
        for (*d2=0;*d2<ndirection;(*d2)++)
          if (get_node(n2,*d2) == n3)
          {
            #ifdef GRID_DEBUG
            fprintf(stderr,"Square formed by nodes %d %d %d %d, ",start,n1,n2,n3) ;
            fprintf(stderr,"procs %d %d %d %d\n",
              node[start]->unmapped_id, node[n1]->unmapped_id,
              node[n2]->unmapped_id,     node[n3]->unmapped_id) ;
            #endif
            return(n3) ;
          }
      }
    }
  return(-1) ;
}
/*}}}*/

/*{{{  BOOL push_out (int dir, int base, int *new)*/
BOOL push_out (int dir, int base, int *new)
{ 
  int m,c=not_used ;
  BOOL available[ndirection] ;

  if (node[base]->edge[dir] != unknown)
  { fprintf(stderr,"Error- Illegal use of push_out, from %d in dir %d\n",base,dir); print_grid() ; exit(-1); }
  
  for (m=0;m<ndirection;m++)
    available[m] = (node[base]->con_proc[m] != not_used) &&
                   (old_to_new_ids[node[base]->con_proc[m]] == new_id) ;
  for (m=0;m<ndirection;m++)
    if (node[base]->edge[m] != unknown)
        available[node[base]->edge[m]] = FALSE ;
        
  for (m=0;m<ndirection;m++)
    if (available[m])
    {
      if (c==not_used)
        c=m ;
      else
      { fprintf(stderr,"Error- Illegal use of push_out, from %d in dir %d\n",base,dir); print_grid() ; exit(-1); }
    }

  if (c != not_used)
    *new = get_node(base,c) ;
    
  if (c == not_used || *new == not_used)
    if (side_limit[dir] < count)
    { fprintf(stderr,"Error- Illegal push_out condition\n"); print_grid() ; exit(-1); }
    else
    {
      #ifdef GRID_DEBUG
      fprintf(stderr,"Hit limit on side %d, count %d\n",dir,count) ;
      #endif
      side_limit[dir] = count ;
    }
  else
  {
    old_to_new_ids[*new] = old_id ;
    EDGE(base,dir,c) ;
    add_node (base,dir) ;
  }

  return(TRUE) ;
}
/*}}}*/

/*{{{  BOOL expand_line (int main_dir, int orth_dir, int mid, int *end)*/
BOOL expand_line (int main_dir, int orth_dir, int mid, int *end)
{
  int l ;
  int prev = get_node(mid,node[mid]->edge[opposite[main_dir]]) ;

  #ifdef GRID_DEBUG
  fprintf(stderr,"EXPAND LINE : count %d, main %d, orth %d, mid %d\n",count,main_dir,orth_dir,mid) ;
  #endif

  if (prev == not_used)
  { fprintf(stderr,"Error- Internal expand_line error\n"); print_grid() ; exit(-1); }
  
  for (l=0;l<count-2;l++)
    if (l < side_limit[orth_dir])
      /*{{{  walk along orth_dir*/
      {
        int new, m ;
        
        if ((prev=get_node(prev,node[prev]->edge[orth_dir])) == not_used)
        { fprintf(stderr,"Error- Internal expand_line error\n"); print_grid() ; exit(-1); }
      
        if ((new=get_node(prev,node[prev]->edge[main_dir])) == not_used)
        { fprintf(stderr,"Error- Internal expand_line error\n"); print_grid() ; exit(-1); }
      
        for (m=0;m<ndirection;m++)
          if (get_node(mid,m) == new)
            break ;
      
        if (m==ndirection)
        { fprintf(stderr,"Error- Internal expand_line error\n"); print_grid() ; exit(-1); }
            
        EDGE(mid,orth_dir,m) ;
        add_node(mid,orth_dir) ;
        mid=new ;
      }
      /*}}}*/
    
  if ((count > 1) && (count-1 <= side_limit[orth_dir]))
    /*{{{  fill in near the corner*/
    {
      int next, new, next_dir, mid_dir ;
    
      if ((next = get_node(prev,node[prev]->edge[orth_dir])) == not_used)
      { fprintf(stderr,"Error- Internal expand_line error\n"); print_grid() ; exit(-1); }
    
      if ((new = test_square (prev,next,mid,&next_dir,&mid_dir)) < 0)
        GRID_MATCH_FAILED("Test_square failed")
      else
        if (old_to_new_ids[new] == old_id)
          GRID_MATCH_FAILED("Old id in square")
        else
          old_to_new_ids[new] = old_id ;
    
      EDGE(mid,orth_dir,mid_dir) ;
      if (add_node (mid,orth_dir) != new)
      { fprintf(stderr,"Error- Internal inconsistency on corner") ; exit(-1) ; }
    
      EDGE(next,main_dir,next_dir) ;
      if (add_node (next,main_dir) != new)
      { fprintf(stderr,"Error- Internal inconsistency on corner") ; exit(-1) ; }
    
      mid=new ;
    }
    /*}}}*/
      
  *end = mid ;
  
  #ifdef GRID_DEBUG
  fprintf(stderr,"EXPAND LINE : end-point %d\n",*end) ;
  #endif
  return(TRUE) ;
}

/*}}}*/

/*{{{  BOOL expand_grid ()*/
BOOL expand_grid ()
{
  int i ;
  int mid[ndirection] ;
  count = 1 ;

  /*{{{  expand from centre in compass directions*/
  for (i=0;i<ndirection;i++)
    if (node[centre]->edge[i] != unknown)
      mid[i] = add_node(centre,i) ;
    else
      side_limit[i] = 0 ;
  /*}}}*/
  
  do
  {
    progress = FALSE ;
    #ifdef GRID_DEBUG
    fprintf(stderr,"MAIN LOOP COUNT %d\n",count) ;
    #endif

    /*{{{  push up to the corners*/
    for (i=0;i<ndirection;i++)
    {
      int cor1, cor2, d1, d2, corner ;
    
      if (count <= side_limit[corners[i][0]])
        if (!expand_line (corners[i][0],corners[i][1],mid[corners[i][0]],&cor1))
          GRID_MATCH_FAILED("Failed to expand line in direction")
    
      if (count <= side_limit[corners[i][1]])
        if (!expand_line (corners[i][1],corners[i][0],mid[corners[i][1]],&cor2))
          GRID_MATCH_FAILED("Failed to expand line in direction")
    
      if ((count <= side_limit[corners[i][0]]) && (count <= side_limit[corners[i][1]]))
      {
        if ((corner=test_square(node[cor1]->con_proc[node[cor1]->edge[opposite[corners[i][0]]]],cor1,cor2,&d1,&d2)) < 0)
          GRID_MATCH_FAILED("Failed to insert corner")
        else
        {
          if (old_to_new_ids[corner] == old_id)
            GRID_MATCH_FAILED("Old id in square")
          else
            /*{{{  fill in corner details*/
            {
              EDGE(cor1,corners[i][1],d1) ;
              if (add_node (cor1,corners[i][1]) != corner)
              { fprintf(stderr,"Error- Internal inconsistency on corner") ; exit(-1) ; }
            
              EDGE(cor2,corners[i][0],d2) ;
              if (add_node (cor2,corners[i][0]) != corner)
              { fprintf(stderr,"Error- Internal inconsistency on corner") ; exit(-1) ; }
            }
            /*}}}*/
        }
      }
    }
    /*}}}*/

    #ifdef GRID_DEBUG
    for (i=0;i<node_cnt;i++)
      fprintf(stderr,"Node %2d : %2d %2d %2d %2d\n",i,node[i]->edge[0],node[i]->edge[1],node[i]->edge[2],node[i]->edge[3]) ;
    fprintf(stderr,"##################################\n") ;
    #endif
    
    /*{{{  push out from sides*/
    for (i=0;i<ndirection;i++)
    {
      int j, k ;
      
      if (count < side_limit[i])
      {
        #ifdef GRID_DEBUG
        fprintf(stderr,"MIDDLE PUSH\n") ;
        #endif
    
        for (j=0;j<2;j++)
        {
          int start = mid[i] ;
          if (node[start]->edge[orthogonal[i][j]] == unknown)
            continue ;
          start = node[start]->con_proc[node[start]->edge[orthogonal[i][j]]] ;
          
          for (k=1;k<count;k++)
            if (k <= side_limit[orthogonal[i][j]])
            {
              int new ;
              if (!push_out (i, start, &new))
                GRID_MATCH_FAILED("Could not push out")
    
              if (node[start]->edge[orthogonal[i][j]] == unknown)
                break ;
              start = node[start]->con_proc[node[start]->edge[orthogonal[i][j]]] ;
            }
            else
              break ;
        }  
    
        if (!push_out (i, mid[i], &mid[i]))
          GRID_MATCH_FAILED("Could not push out")
      }
    }
    /*}}}*/

    count++ ;
    #ifdef GRID_DEBUG
    for (i=0;i<node_cnt;i++)
      fprintf(stderr,"Node %2d : %2d %2d %2d %2d\n",i,node[i]->edge[0],node[i]->edge[1],node[i]->edge[2],node[i]->edge[3]) ;
    fprintf(stderr,"##################################\n") ;
    #endif
  }
  while (progress) ;

  #ifdef GRID_DEBUG
  fprintf(stderr,"Dropped out of progress loop\n") ;
  #endif
  return(TRUE) ;
}
/*}}}*/

/*{{{  int  next_edge (int n, int i)*/
int  next_edge (int n, int i)
{
  for (;i<ndirection;i++)
    if (node[n]->con_proc[i] != not_used)
      return(i) ;
  return(unknown) ;
}
/*}}}*/

/*{{{  void route_grid (int id1, int x1, int y1, int id2, int x2, int y2, int target, int *dir, int *vir)*/
void route_grid (int id1, int x1, int y1, int id2, int x2, int y2, int target, int *dir, int *vir)
{
  /*{{{  initialisation*/
  int dy  = (y2-y1+ny)%ny ;   /* Number of southerly steps */
  int dx  = (x2-x1+nx)%nx ;   /* Number of easterly steps  */
  int mx1 = (sp == spliced_in_y) ? (x1-offset+nx)%nx : x1 ;
  int my1 = (sp == spliced_in_x) ? (y1-offset+ny)%ny : y1 ;
  int mx2 = (sp == spliced_in_y) ? (x2-offset+nx)%nx : x2 ;
  int my2 = (sp == spliced_in_x) ? (y2-offset+ny)%ny : y2 ;
  int mnx = nx, mny = ny ;
  int mdx, mdy ;
  int i ;
  
  *vir = 0 ;
  *dir = path_end ;
  /*}}}*/
  
  /*{{{  adjust mapping for any splices*/
  if (sp == spliced_in_y)
  {
    for (i=0;i<extra_two;i++)
      if (ID_TO_X(two[i].pnode[0]) == x2)
      {
        int lpy = ID_TO_Y(two[i].pnode[0]) ;
        mny++ ;
        if (y1 > lpy)
          my1++ ;
        if (y2 > lpy)
          my2++ ;
      }
  }
  else if (sp == spliced_in_x)
    for (i=0;i<extra_two;i++)
      if (ID_TO_Y(two[i].pnode[0]) == y2)
      {
        int lpx = ID_TO_X(two[i].pnode[0]) ;
        mnx++ ;
        if (x1 > lpx)
          mx1++ ;
        if (x2 > lpx)
          mx2++ ;
      }
  
  mdy  = (my2-my1+mny)%mny ;
  mdx  = (mx2-mx1+mnx)%mnx ;
  /*}}}*/

  /*{{{  adjust mx2, my2 if routing to spliced node*/
  if (target != not_used)
  {
    int e ;
  
    for (i=0;i<ndirection;i++)
      if (node[target]->con_proc[i] == id2)
        break ;
  
    if (i==ndirection)
    { fprintf(stderr,"Error- Inconsistency in route_torus\n"); exit(-1); }
  
    for (e=0;e<ndirection;e++)
     if (node[node[target]->con_proc[i]]->edge[e] == node[target]->con_link[i])
       break ;
  
    if (e==ndirection)
    { fprintf(stderr,"Error- Inconsistency in route_torus\n"); exit(-1); }
       
    if (sp==spliced_in_y && mx1 == mx2)
    {
      #ifdef GRID_DEBUG
      fprintf(stderr,"Routing to splice from node %d adjusting y coord from %d to %d\n",id1,my2,(my2 - (e==north) + (e==south) + mny) % mny) ;
      #endif
      my2 = (my2 - (e==north) + (e==south) + mny) % mny ;
      mdy++ ;
    }
    if (sp==spliced_in_x && my1 == my2)
    {
      #ifdef GRID_DEBUG
      fprintf(stderr,"Routing from %d (%d,%d) to splice %d via %d (%d,%d): adjusting x coord to %d\n",id1,mx1,my1,target,id2,mx2,my2,(mx2 - (e==west) + (e==east) + mnx) % mnx) ;
      #endif
      mx2 = (mx2 - (e==west)  + (e==east) + mnx) % mnx ;
      mdx++ ;
    }
  }
  /*}}}*/

  if (sp == spliced_in_y)
  {
    if (wrapx == FALSE)
      /*{{{  try grid routing in x*/
      {
        if (x2>x1)
        {
          *dir = east ;
      
          #ifdef GRID_DEBUG
          fprintf(stderr,"GE ") ;
          #endif
      
          return ;
        }
        else if (x2<x1)
        {
          *dir = west ;
      
          #ifdef GRID_DEBUG
          fprintf(stderr,"GW ") ;
          #endif
      
          return ;
        }  
      }
      /*}}}*/
    else
      /*{{{  try torus routing in x*/
      {
        if (dx > 0 && dx <= hnx)
        {
          *dir = east ;
      
          if (mx2>=2 && mx2-mdx<=0)
            *vir = 1 ;
      
          #ifdef GRID_DEBUG
          fprintf(stderr,"TE ") ;
          #endif
            
          return ;
        }
        else if (dx > hnx)
        {
          *dir = west ;
      
          if (mx2<=mnx-2 && mx2+mnx-mdx>=mnx)
            *vir = 1 ;
      
          #ifdef GRID_DEBUG
          fprintf(stderr,"TW ") ;
          #endif
            
          return ;
        }
      }
      /*}}}*/

    if (wrapy == FALSE)
      /*{{{  try grid routing in y*/
      {
        if (y2>y1)
        {
          *dir = south ;
      
          #ifdef GRID_DEBUG
          fprintf(stderr,"GS ") ;
          #endif
      
          return ;
        }
        else if (y2<y1)
        {
          *dir = north ;
      
          #ifdef GRID_DEBUG
          fprintf(stderr,"GN ") ;
          #endif
      
          return ;
        }  
      }
      /*}}}*/
    else
      /*{{{  try torus routing in y*/
      {
        if (dy > 0 && dy <= hny)
        {
          *dir = south ;
      
          if (my2>=2 && my2-mdy<=0)
            *vir = 1 ;
      
          #ifdef GRID_DEBUG
          fprintf(stderr,"TS ") ;
          #endif
            
          return ;
        }
        else if (dy > hny)
        {
          *dir = north ;
      
          if (my2<=mny-2 && my2+mny-mdy>=mny)
            *vir = 1 ;
      
          #ifdef GRID_DEBUG
          fprintf(stderr,"TN ") ;
          #endif
      
          return ;
        }
      }
      /*}}}*/
  }      
  else
  {
    if (wrapy == FALSE)
      /*{{{  try grid routing in y*/
      {
        if (y2>y1)
        {
          *dir = south ;
      
          #ifdef GRID_DEBUG
          fprintf(stderr,"GS ") ;
          #endif
      
          return ;
        }
        else if (y2<y1)
        {
          *dir = north ;
      
          #ifdef GRID_DEBUG
          fprintf(stderr,"GN ") ;
          #endif
          
          return ;
        }  
      }
      /*}}}*/
    else
      /*{{{  try torus routing in y*/
      {
        if (dy > 0 && dy <= hny)
        {
          *dir = south ;
        
          if (my2>=2 && my2-mdy<=0)
            *vir = 1 ;
      
          #ifdef GRID_DEBUG
          fprintf(stderr,"TS ") ;
          #endif
            
          return ;
        }
        else if (dy > hny)
        {
          *dir = north ;
        
          if (my2<=ny-2 && my2+ny-mdy>=ny)
            *vir = 1 ;
      
          #ifdef GRID_DEBUG
          fprintf(stderr,"TN ") ;
          #endif
            
          return ;
        }
      }
      /*}}}*/

    if (wrapx == FALSE)
      /*{{{  try grid routing in x*/
      {
        if (x2>x1)
        {
          *dir = east ;
      
          #ifdef GRID_DEBUG
          fprintf(stderr,"GE ") ;
          #endif
      
          return ;
        }
        else if (x2<x1)
        {
          *dir = west ;
      
          #ifdef GRID_DEBUG
          fprintf(stderr,"GW ") ;
          #endif
      
          return ;
        }  
      }
      /*}}}*/
    else
      /*{{{  try torus routing in x*/
      {
        if (dx > 0 && dx <= hnx)
        {
          *dir = east ;
        
          if (mx2>=2 && mx2-mdx<=0)
            *vir = 1 ;
      
          #ifdef GRID_DEBUG
          fprintf(stderr,"TE ") ;
          #endif
            
          return ;
        }
        else if (dx > hnx)
        {
          *dir = west ;
        
          if (mx2<=nx-2 && mx2+nx-mdx>=nx)
            *vir = 1 ;
      
          #ifdef GRID_DEBUG
          fprintf(stderr,"TW ") ;
          #endif
            
          return ;
        }
      }
      /*}}}*/
  }  
}

/*{{{  COMMENT*/
/* Virtualisation strategy for breaking cycles in rings

   Routing in positive direction:
      All routes for journeys going through connects 0,1,2 start on level 1.
      All level 1 journeys fall to level 0 on passing between 0 and 1
      All others on level 0.
   Routing in negative direction:
      All routes for journeys going through connects 0,n-1,n-2 start on level 1.
      All level 1 journeys fall to level 0 on passing between 0 and n-1
      All others on level 0.

   e.g. For a 7-element ring :

   <- 0 -- 1 -- 2 -- 3 -- 4 -- 5 -- 6 ->

   Journeys starting on virtual level 1 are  012, 0123 and 6012 (positive direction)
   Journeys starting on virtual level 1 are  065, 0654 and 1065 (negative direction)

   This is suffient to prevent deadlock in each ring and, with
   dimension ordered routing, the whole torus is deadlock-free.

   Also maps any spliced links away from the virtualisation jump
*/
/*}}}*/
/*}}}*/

/*{{{  int  find_most (CHECK_MODE mode, int *start, int *number)*/
int  find_most (CHECK_MODE mode, int *start, int *number)
{
  int most = -1, i, j ;
  check_mode = mode ;

  for (i=0;i<ndirection+1;i++)
  {
    start[i] = unknown ;
    number[i] = 0 ;
  }
  
  most = -1 ;
  for (i=0;i<node_cnt;i++)
  {
    int num=0 ;
  
    old_to_new_ids[i] = new_id ;
  
    for (j=0;j<ndirection;j++)
    {
      node[i]->edge[j] = unknown ;
      if (get_node(i,j)!=not_used)
        num++ ;
    }
    
    if (num > most)
      most = num ;
  
    if (start[num] == unknown)
      start[num] = i ;

    number[num]++ ;
  }

  return(most) ;
}
/*}}}*/

/*{{{  NET *map_nodes (NET **oldnode, int *mapping)*/
NET **map_nodes (NET **oldnode, int *mapping)
{
  NET **newnode ;
  int i,j ;

  if ((newnode = (NET **)malloc(node_cnt*sizeof(NET *))) == NULL)
  { fprintf(stderr,"Error- Malloc Failed\n"); exit(-1); }

  for (i=0;i<node_cnt;i++)
    newnode[i] = NULL ;

  for (i=0;i<node_cnt;i++)
    if (mapping[i] >= 0 && mapping[i] < node_cnt)
    {
      #ifdef GRID_DEBUG
      fprintf(stderr,"Mapping %d to %d\n",i,mapping[i]) ;
      #endif

      if (newnode[mapping[i]] != NULL)
      { fprintf(stderr,"Error- Double mapping to id %d\n",mapping[i]); exit(-1); }

      newnode[mapping[i]] = oldnode[i] ;

      for (j=0;j<ndirection;j++)
        if (newnode[mapping[i]]->con_proc[j] != not_used)
          newnode[mapping[i]]->con_proc[j] = mapping[newnode[mapping[i]]->con_proc[j]] ;
    }
    else
      { fprintf(stderr,"Error- Invalid mapping from %d to %d\n",i,mapping[i]); exit(-1); }
    

  return(newnode) ;
}
/*}}}*/

/*{{{  BOOL match_to_grid(BYTE *chosen_dir, BYTE *chosen_vir)*/
BOOL match_to_grid(BYTE *chosen_dir, BYTE *chosen_vir)
{
  int i,j,k ;
  int start[ndirection+1] ;
  int number[ndirection+1] ;
  int most ;

  /*{{{  error check*/
  for (i=0;i<node_cnt;i++)
  {
    for (j=0;j<ndirection;j++)
      if (node[i]->con_proc[j] == i)
        GRID_MATCH_FAILED("Cannot handle nodes with links connected together")
  
    for (j=0;j<ndirection-1;j++)
      if (node[i]->con_proc[j] != not_used)
        for (k=j+1;k<ndirection;k++)
          if (node[i]->con_proc[j] == node[i]->con_proc[k])
            GRID_MATCH_FAILED("Cannot handle nodes with two connecting links")
  }
  /*}}}*/
  
  /*{{{  find start point with most edges*/
  for (i=0;i<ndirection;i++)
    side_limit[i] = INT_MAX ;
  
  most = find_most (grid_check, start, number) ;
  
  #ifdef GRID_DEBUG
  for (i=0;i<ndirection+1;i++)
    fprintf(stderr,"Node %d is first with %d edges\n",start[i],i) ;
  #endif
  
  /*}}}*/

  /*{{{  find initial compass directions*/
  switch (most)
  {
    case 0 :
      /*{{{  recalculate info without grid checking*/
      most = find_most (other_check, start, number) ;
      if (most == 0)
        { fprintf(stderr,"Error- Expected singles to be handled elsewhere\n"); exit(-1); }
      
      #ifdef GRID_DEBUG
      for (i=0;i<ndirection+1;i++)
        fprintf(stderr,"Node %d is first with %d edges\n",start[i],i) ;
      #endif
      /*}}}*/
    case 1 :
    case 2 :
      /*{{{  special case for rings, chains, square*/
      {
        if (number[2]==4 && check_mode==grid_check)
          /*{{{  2-d square*/
          {
            centre = start[most] ;
            old_to_new_ids[centre] = old_id ;
          
            i=next_edge(centre,0) ;
            while (get_node(centre,i) == not_used)
              i=next_edge(centre,++i) ;
            EDGE (centre,east,i) ;
          
            i=next_edge(centre,++i) ;
            while (get_node(centre,i) == not_used)
              i=next_edge(centre,++i) ;
            EDGE (centre,south,i) ;
          
            if (i>=ndirection)
            { fprintf(stderr,"Error- Expected a square\n"); exit(-1); }
          }
          /*}}}*/
        else
          /*{{{  1-d grid or ring*/
          {
            if (start[1] != unknown)
              centre = start[1] ;        /* grid */
            else
              centre = start[2] ;        /* ring */
              
            old_to_new_ids[centre] = old_id ;
          
            check_mode = other_check ; 
            EDGE (centre,south,i=next_edge(centre,0)) ;
          }
          /*}}}*/
        
        break ;
      }
      /*}}}*/
    case 3 :
      /*{{{  special case for thin grids/torii*/
      {
        int n[3], t=-1 ;
      
        check_mode = grid_check ;
        centre = start[most] ;
        old_to_new_ids[centre] = old_id ;
      
        for (i=0;i<3;i++)
        {
          do
          {
            t = next_edge(centre,++t) ;
            if (t==unknown)
              GRID_MATCH_FAILED("Cannot find 3 links from first node") ;
          } while (get_node(centre,t) == not_used) ;
      
          n[i]=t ;
      
          #ifdef GRID_DEBUG
          fprintf(stderr,"3-valent node %d connects via link %d\n",centre,n[i]) ;
          #endif
        }
      
        for (i=0;i<3;i++)
          if (test_square (centre,node[centre]->con_proc[n[i]],
                                  node[centre]->con_proc[n[(i+1)%3]],&j,&k) < 0)
            break ;
      
        if (i==3)
          i=0 ;   /* Break an arbitrary square */
          
        EDGE (centre, north, n[i]) ;
        EDGE (centre, south, n[(i+1)%3]) ;
        EDGE (centre, east, n[3-i-((i+1)%3)]) ;
      
        break ;
      }
      /*}}}*/
    case 4 :
      /*{{{  full-blown grids/torii*/
      {
        BOOL mapped[ndirection] ;
        int d1,d2 ;
      
        for (i=0;i<ndirection;i++)
          mapped[i] = FALSE ;
      
        centre = start[4] ;
        old_to_new_ids[centre] = old_id ;
      
        #ifdef GRID_DEBUG
        fprintf(stderr,"Centre %d : %d %d %d %d\n",centre, get_node(centre,0), get_node(centre,1), get_node(centre,2), get_node(centre,3)) ;
        #endif
      
        if (start[1] == unknown && start[3] == unknown)
          check_mode = torus_check ;
        else
          check_mode = grid_check ;
        
        /*{{{  assume a north edge*/
        EDGE (centre,north,0) ;  
        mapped[0] = TRUE ;
        /*}}}*/
        /*{{{  find an east edge*/
        for (j=1;j<ndirection;j++)
          if (test_square(centre,get_node(centre,0),get_node(centre,j),&d1,&d2) >= 0)
          {
            EDGE(centre,east,j) ;
            mapped[j] = TRUE ;
            break ;
          }
        
        if (j>=ndirection)
          GRID_MATCH_FAILED("Cannot construct first square from centre")
        /*}}}*/
        /*{{{  find a  west edge*/
        for (j=j+1;j<ndirection;j++)
          if (test_square(centre,get_node(centre,0),get_node(centre,j),&d1,&d2) >= 0)
          {
            EDGE(centre,west,j) ;
            mapped[j] = TRUE ;
            break ;
          }
        
        if (j>=ndirection)
          GRID_MATCH_FAILED("Cannot construct second square from centre")
        /*}}}*/
        /*{{{  find a south edge*/
        for (k=1;k<ndirection;k++)
          if (!mapped[k])
            if (test_square(centre,get_node(centre,j),get_node(centre,k),&d1,&d2) >= 0)
            {
              EDGE(centre,south,k) ;
              mapped[k] = TRUE ;
              break ;
            }
        
        if (k>=ndirection)
          GRID_MATCH_FAILED("Cannot construct third square from centre")
        /*}}}*/
      
        break ;
      }
      /*}}}*/
    default :
      { fprintf(stderr,"Error- Valency exceeded 4\n"); exit(-1); }
  }
  /*}}}*/

  if (expand_grid() == FALSE)
  {
    print_grid() ;
    return(FALSE) ;
  }

  print_grid() ;
  
  /*{{{  post-analysis*/
  {
    int origin = centre ;
    BOOL *spliced ;
  
    nx = side_limit[east] + 1 + side_limit[west] ;
    ny = side_limit[north] + 1 + side_limit[south] ;
    hnx = nx/2 ;
    hny = ny/2 ;
    
    if ((spliced = (BOOL *) malloc (sizeof(BOOL) * MAX(nx,ny))) == NULL)
    { fprintf(stderr,"Error- Malloc Failed\n"); exit(-1); }
  
    for (i=0;i<MAX(nx,ny);i++)
      spliced[i] = FALSE ;
    
    /*{{{  label and identify grid*/
    {
      int row, col ;
      int x, y ;
    
      while(node[origin]->edge[north] != unknown)
        origin = get_node(origin,node[origin]->edge[north]) ;
    
      while(node[origin]->edge[west] != unknown)
        origin = get_node(origin,node[origin]->edge[west]) ;
    
      /*{{{  clear old_to_new_ids, so that only proper grid/torii nodes are identified*/
      {
        for (i=0;i<node_cnt;i++)
          old_to_new_ids[i] = new_id ;
      
        col = origin ;
        for (y=0;y<ny;y++)
        {
          row = col ;
          for (x=0;x<nx;x++)
          {
            old_to_new_ids[row] = old_id ;
            row = get_node(row,node[row]->edge[east]) ;
          }
          col = get_node(col,node[col]->edge[south]) ;
        }
      }
      /*}}}*/
      
      /*{{{  relabel nodes to grid numbering*/
      {
        first_id = 0;
        for (i=0;i<node_cnt;i++)
          if (old_to_new_ids[i] == new_id)
          {
            old_to_new_ids[i] = first_id ;
            new_to_old_ids[first_id] = i ;
            first_id++ ;
          }
      
        col = origin ;
        for (y=0;y<ny;y++)
        {
          row = col ;
          for (x=0;x<nx;x++)
          {
            old_to_new_ids[row] = first_id+y*nx+x ;
            new_to_old_ids[first_id+y*nx+x] = row ;
            
            #ifdef GRID_DEBUG
            fprintf(stderr,"Old id %d maps to new id %d (%d,%d), first at %d\n",row,old_to_new_ids[row],x,y,first_id) ;
            #endif
            
            row = get_node(row,node[row]->edge[east]) ;
          }
          col = get_node(col,node[col]->edge[south]) ;
        }
      
        if (old_to_new_ids[0] != 0)
          GRID_MATCH_FAILED("Root processor must map to zero")
      
        {
          NET **oldnode = node ;
          node = map_nodes(oldnode, old_to_new_ids) ;
          free(oldnode) ;
        }
      
        #ifdef GRID_DEBUG
        for (i=0;i<node_cnt;i++)
          for (j=0;j<ndirection;j++)
            if (node[i]->con_proc[j] != not_used)
              fprintf(stderr,"P%d L%d to P%d L%d\n",i,j,node[i]->con_proc[j],node[i]->con_link[j]) ;
        #endif
      }
      /*}}}*/
    
      /*{{{  check for toroidal east-west links*/
      for (y=0;y<ny;y++)
      {
        BOOL used[ndirection] ;
        
        for (i=0;i<ndirection;i++)
          used[i] = FALSE ;
        for (i=0;i<ndirection;i++)
          if (node[XY_TO_ID(0,y)]->edge[i] != unknown)
          {
            #ifdef GRID_DEBUG
            fprintf(stderr,"Node %d link %d is used\n",XY_TO_ID(0,y),i) ;
            #endif
            used[node[XY_TO_ID(0,y)]->edge[i]] = TRUE ;
          }
        for (i=0;i<ndirection;i++)
          if (!used[i])
          {
            #ifdef GRID_DEBUG
            fprintf(stderr,"Try node %d link %d to node %d\n",XY_TO_ID(0,y),i,XY_TO_ID(nx-1,y)) ;
            #endif
            if (get_node(XY_TO_ID(0,y),i) == XY_TO_ID(nx-1,y))
              break ;
          }
      
        for (j=0;j<ndirection;j++)
          used[j] = FALSE ;
        for (j=0;j<ndirection;j++)
          if (node[XY_TO_ID(nx-1,y)]->edge[j] != unknown)
            used[node[XY_TO_ID(nx-1,y)]->edge[j]] = TRUE ;
        for (j=0;j<ndirection;j++)
          if (!used[j])
            if (get_node(XY_TO_ID(nx-1,y),j) == XY_TO_ID(0,y))
              break ;
      
        if (i==ndirection || j==ndirection)
          wrapx = FALSE ;
        else
        {
          EDGE(XY_TO_ID(0,y),west,i) ;
          EDGE(XY_TO_ID(nx-1,y),east,j) ;
        }
      }
      /*}}}*/
      
      /*{{{  check for toroidal north-south links*/
      for (x=0;x<nx;x++)
      {
        BOOL used[ndirection] ;
        
        for (i=0;i<ndirection;i++)
          used[i] = FALSE ;
        for (i=0;i<ndirection;i++)
          if (node[XY_TO_ID(x,0)]->edge[i] != unknown)
          {
            #ifdef GRID_DEBUG
            fprintf(stderr,"Node %d link %d is used\n",XY_TO_ID(x,0),i) ;
            #endif
            used[node[XY_TO_ID(x,0)]->edge[i]] = TRUE ;
          }
        for (i=0;i<ndirection;i++)
          if (!used[i])
          {
            #ifdef GRID_DEBUG
            fprintf(stderr,"Try node %d link %d to node %d\n",XY_TO_ID(x,0),i,XY_TO_ID(x,ny-1)) ;
            #endif
            if (get_node(XY_TO_ID(x,0),i) == XY_TO_ID(x,ny-1))
              break ;
          }
              
        for (j=0;j<ndirection;j++)
          used[j] = FALSE ;
        for (j=0;j<ndirection;j++)
          if (node[XY_TO_ID(x,ny-1)]->edge[j] != unknown)
            used[node[XY_TO_ID(x,ny-1)]->edge[j]] = TRUE ;
        for (j=0;j<ndirection;j++)
          if (!used[j])
          {
            #ifdef GRID_DEBUG
            fprintf(stderr,"Try node %d link %d to node %d\n",XY_TO_ID(x,ny-1), j, XY_TO_ID(x,0)) ;
            #endif
            if (get_node(XY_TO_ID(x,ny-1),j) == XY_TO_ID(x,0))
              break ;
          }
      
        #ifdef GRID_DEBUG
        fprintf(stderr,"%d,%d\n",i,j) ;
        #endif
      
        if (i==ndirection || j==ndirection)
          wrapy = FALSE ;
        else
        {
          EDGE(XY_TO_ID(x,0),north,i) ;
          EDGE(XY_TO_ID(x,ny-1),south,j) ;
        }
      }
      /*}}}*/
    
      #ifdef GRID_DEBUG
      fprintf(stderr,"wrapx : %d, wrapy : %d\n",wrapx,wrapy) ;
      if (wrapx && wrapy)
        fprintf(stderr,"%d by %d torus\n",nx,ny) ;
      else
        fprintf(stderr,"%d by %d grid\n",nx,ny) ;
      #endif
    }
    /*}}}*/
    /*{{{  fill in extra nodes*/
    {
      /*{{{  count extra nodes*/
      {
        BOOL all = TRUE ;
        
        for (i=0;i<first_id;i++)
        {
          int v = valency(i) ;
          if (v==1)
            extra_one++ ;
          else if (v==2)
            extra_two++ ;
          else
            all=FALSE ;
        }
            
        if (!all)
          /*{{{  failed*/
          {
            NET **n = node ;
            node = map_nodes(node, new_to_old_ids) ;
            free(n) ;
            GRID_MATCH_FAILED("Only expected 1 or 2-valent extra")
          }
          /*}}}*/
      }
      /*}}}*/
    
      /*{{{  check extras consistent with topology*/
      if (wrapx && wrapy)
      {
        if (extra_one != 0)
          /*{{{  failed*/
          {
            NET **n = node ;
            node = map_nodes(node, new_to_old_ids) ;
            free(n) ;
            GRID_MATCH_FAILED("Cannot handle one-valent extras in torus")
          }
          /*}}}*/
      }
      else
        if (extra_two != 0)
          /*{{{  failed*/
          {
            NET **n = node ;
            node = map_nodes(node, new_to_old_ids) ;
            free(n) ;
            GRID_MATCH_FAILED("Cannot handle two-valent extras in grid or ring")
          }
          /*}}}*/
      /*}}}*/
    
      /*{{{  allocate extra nodes*/
      if ((extra_one && ((one=(one_node *)malloc(sizeof(one_node)*extra_one)) == NULL)) ||
          (extra_two && ((two=(two_node *)malloc(sizeof(two_node)*extra_two)) == NULL)))
      { fprintf(stderr,"Error- Malloc Failed\n"); exit(-1); }
      /*}}}*/
    
      /*{{{  fill in extra node details*/
      {
        one_node *o = one ;
        two_node *t = two ;
      
        for (i=0;i<first_id;i++)
        {
          int v = valency(i) ;
          if (v==1)
            /*{{{  fill in one_node details*/
            {
              for (j=0;j<ndirection;j++)
                if (node[i]->con_proc[j] != not_used)
                {
                  o->num   = i ;
                  o->pnode = node[i]->con_proc[j] ;
                  o->plink = j ;
                  break ;
                }
            
              o++ ;
            }
            /*}}}*/
          else if (v==2)
            /*{{{  fill in two_node details*/
            {
              t->num = i ;
              for (j=0;j<ndirection;j++)
                if (node[i]->con_proc[j] != not_used)
                {
                  t->pnode[0] = node[i]->con_proc[j] ;
                  t->plink[0] = j ; 
                  break ;
                }
            
              for (j++;j<ndirection;j++)
                if (node[i]->con_proc[j] != not_used)
                {
                  int x0 = ID_TO_X(t->pnode[0]) ;
                  int y0 = ID_TO_Y(t->pnode[0]) ;
                  int x1 = ID_TO_X(node[i]->con_proc[j]) ;
                  int y1 = ID_TO_Y(node[i]->con_proc[j]) ;
                  int lower = 0 ;
            
                  if (x0==x1)
                  {
                    if ((y1+1)%ny == y0)
                      lower = 1 ;
                  }
                  else if (y0==y1)
                  {
                    if ((x1+1)%nx == x0)
                      lower = 1 ;
                  }
                  else
                  { fprintf(stderr,"Error- Extra coords mismatch\n"); exit(-1); }
            
                  if (lower == 0)
                  {
                    t->pnode[1] = node[i]->con_proc[j] ;
                    t->plink[1] = j ;
                  }
                  else
                  {
                    fprintf(stderr,"Swapping parents of extra %d\n",i) ;
                    t->pnode[1] = t->pnode[0] ;
                    t->plink[1] = t->plink[0] ;
                    t->pnode[0] = node[i]->con_proc[j] ;
                    t->plink[0] = j ;
                  }
                  break ;
                }
            
              t++ ;
            }
            
            /* The two neighbouring nodes are stored in t in order */
            /*}}}*/
        }
      }
      /*}}}*/
    
      #ifdef GRID_DEBUG
      fprintf(stderr,"%d extra one nodes, %d extra two nodes\n",extra_one,extra_two) ;
      #endif
    
      /*{{{  find splicing direction and offset to break cycles at*/
      for (i=0;i<extra_two;i++)
      {
        int dir0 = node[two[i].pnode[0]]->edge[node[two[i].num]->con_link[two[i].plink[0]]] ;
        int dir1 = node[two[i].pnode[1]]->edge[node[two[i].num]->con_link[two[i].plink[1]]] ;
      
        if ((dir0==north && dir1==south) || (dir0==south && dir1==north))
          if (sp == none || sp == spliced_in_y)
          {
            spliced[ID_TO_X(two[i].pnode[0])] = TRUE ;
            sp = spliced_in_y ;
          }        
          else
            /*{{{  failed*/
            {
              NET **n = node ;
              node = map_nodes(node, new_to_old_ids) ;
              free(n) ;
              GRID_MATCH_FAILED("Only splice in one dimension")
            }
            /*}}}*/
        else
          if ((dir0==east && dir1==west) || (dir0==west && dir1==east))
            if (sp == none || sp == spliced_in_x)
            {
              spliced[ID_TO_Y(two[i].pnode[0])] = TRUE ;
              sp = spliced_in_x ;
            }
            else
              /*{{{  failed*/
              {
                NET **n = node ;
                node = map_nodes(node, new_to_old_ids) ;
                free(n) ;
                GRID_MATCH_FAILED("Only splice in one dimension")
              }
              /*}}}*/
          else
            /*{{{  failed*/
            {
              NET **n = node ;
              node = map_nodes(node, new_to_old_ids) ;
              free(n) ;
              GRID_MATCH_FAILED("Unusual splice")
            }
            /*}}}*/
      }
      
      {
        int n = (sp==spliced_in_y) ? nx : ny ;
        for (offset=0;offset<n-1;offset++)
          if (spliced[offset]==FALSE && spliced[offset+1]==FALSE)
            break ;
        if (offset==(n-1))
          /*{{{  failed*/
          {
            NET **n = node ;
            node = map_nodes(node, new_to_old_ids) ;
            free(n) ;
            GRID_MATCH_FAILED("Too many splices, cannot deadlock-free route")
          }
          /*}}}*/
      }  
      /*}}}*/
    
      #ifdef GRID_DEBUG
      fprintf(stderr,"Spliced in %c, offset is %d\n",
        (sp==spliced_in_y) ? 'y' : (sp==spliced_in_x) ? 'x' : '-' ,offset) ;
      #endif
    }
    /*}}}*/
    /*{{{  route regular parts of grid/torus*/
    {
      int x1, y1 ;
      int x2, y2 ;
    
      for (x1=0;x1<nx;x1++)
        for (y1=0;y1<ny;y1++)
        {
          int id1 = XY_TO_ID(x1,y1) ;
    
          for (x2=0;x2<nx;x2++)
            for (y2=0;y2<ny;y2++)
            {
              int dir, vir ;
              int id2 = XY_TO_ID(x2,y2) ;
              
              route_grid (id1,x1,y1,id2,x2,y2,not_used,&dir,&vir) ;
              
              #ifdef GRID_DEBUG
              fprintf(stderr,"(%d, %d) to (%d, %d) via %d:%d\n",x1,y1,x2,y2,dir,vir) ;
              #endif
    
              if (dir == path_end)
                CHOSEN_DIR(id1,id2) = path_end ;
              else
                CHOSEN_DIR(id1,id2) = node[id1]->edge[dir] ;
              CHOSEN_VIR(id1,id2) = vir ;
            }
        }
    }          
    /*}}}*/
    /*{{{  route irregular parts*/
    {
      int x,y ;
    
      /*{{{  add extra ones to grid*/
      for (i=0;i<extra_one;i++)
      {
        CHOSEN_DIR(one[i].num,one[i].num) = path_end ;
        CHOSEN_VIR(one[i].num,one[i].num) = 0 ;
        
        for (x=0;x<nx;x++)
          for (y=0;y<ny;y++)
          {
            int id = XY_TO_ID(x,y) ;
      
            CHOSEN_DIR(one[i].num,id) = one[i].plink ;
            CHOSEN_VIR(one[i].num,id) = 0 ;
            #ifdef GRID_DEBUG
            fprintf(stderr,"Route from %d to %d through link %d\n",one[i].num,id,CHOSEN_DIR(one[i].num,id)) ;
            #endif
            
            if (id == one[i].pnode)
              CHOSEN_DIR(id,one[i].num) = node[one[i].num]->con_link[one[i].plink] ;
            else
              CHOSEN_DIR(id,one[i].num) = CHOSEN_DIR(id,one[i].pnode) ;
            CHOSEN_VIR(id,one[i].num) = CHOSEN_VIR(id,one[i].pnode) ;
            #ifdef GRID_DEBUG
            fprintf(stderr,"Route from %d to %d through link %d\n",id,one[i].num,CHOSEN_DIR(id,one[i].num)) ;
            #endif
          }
      
        for (j=0;j<extra_one;j++)
          if (i!=j)
          {
            CHOSEN_DIR(one[i].num,one[j].num) = CHOSEN_DIR(one[i].num,one[j].pnode) ;
            CHOSEN_VIR(one[i].num,one[j].num) = 0 ;
          }      
      }
      /*}}}*/
      /*{{{  add extra twos to torus*/
      for (i=0;i<extra_two;i++)
      {
        int px0 = ID_TO_X(two[i].pnode[0]) ;
        int py0 = ID_TO_Y(two[i].pnode[0]) ;
        int px1 = ID_TO_X(two[i].pnode[1]) ;
        int py1 = ID_TO_Y(two[i].pnode[1]) ;
      
        CHOSEN_DIR(two[i].num,two[i].num) = path_end ;
        CHOSEN_VIR(two[i].num,two[i].num) = 0 ;
        
        for (x=0;x<nx;x++)
          for (y=0;y<ny;y++)
          {
            int id = XY_TO_ID(x,y) ;
            int waytonew, wayfromnew ;
      
            if (py0 == py1)
              /*{{{  choose closest way, if equal route in positive direction*/
              {
                int dx0 = MIN((px0-x+nx)%nx,(x-px0+nx)%nx) ;
                int dx1 = MIN((px1-x+nx)%nx,(x-px1+nx)%nx) ;
              
                if (dx0 < dx1)
                {  
                  waytonew = 0 ;
                  wayfromnew = 0 ;
                }
                else if (dx1 < dx0)
                {  
                  waytonew = 1 ;
                  wayfromnew = 1 ;
                }
                else if (px0-x > 0)
                {
                  waytonew = 0 ;
                  wayfromnew = 1 ;
                }
                else
                {
                  waytonew = 1 ;
                  wayfromnew = 0 ;
                }
              
                #ifdef GRID_DEBUG
                fprintf(stderr,"X-Route from %d to %d through link node %d (dx0=%d,dx1=%d), reverse through %d\n",id,two[i].num,two[i].pnode[waytonew],dx0,dx1,two[i].pnode[wayfromnew]) ;
                #endif
              }
              /*}}}*/
            else if (px0 == px1)
              /*{{{  choose closest way, if equal route in positive direction*/
              {
                int dy0 = MIN((py0-y+ny)%ny,(y-py0+ny)%ny) ;
                int dy1 = MIN((py1-y+ny)%ny,(y-py1+ny)%ny) ;
              
                if (dy0 < dy1)
                {  
                  waytonew = 0 ;
                  wayfromnew = 0 ;
                }
                else if (dy1 < dy0)
                {  
                  waytonew = 1 ;
                  wayfromnew = 1 ;
                }
                else if (py0-y > 0)
                {
                  waytonew = 0 ;
                  wayfromnew = 1 ;
                }
                else
                {
                  waytonew = 1 ;
                  wayfromnew = 0 ;
                }
              
                #ifdef GRID_DEBUG
                fprintf(stderr,"Y-Route from %d to %d through link node %d (dy0=%d,dy1=%d), reverse through %d\n",id,two[i].num,two[i].pnode[waytonew],dy0,dy1,two[i].pnode[wayfromnew]) ;
                #endif
              }
              /*}}}*/
            else
              { fprintf(stderr,"Error- Unexpected condition\n"); exit(-1); }
          
            /*{{{  route from node to new id via plink[waytonew]*/
            if (id == two[i].pnode[0])
            {
              CHOSEN_VIR(id,two[i].num) = 0 ;
              CHOSEN_DIR(id,two[i].num) = node[two[i].num]->con_link[two[i].plink[0]] ;
            }
            else if (id == two[i].pnode[1])
            {
              CHOSEN_VIR(id,two[i].num) = 0 ;
              CHOSEN_DIR(id,two[i].num) = node[two[i].num]->con_link[two[i].plink[1]] ;
            }
            else
            {
              int dir, vir ;
            
              route_grid (id, x, y, two[i].pnode[waytonew],
                          ID_TO_X(two[i].pnode[waytonew]),
                          ID_TO_Y(two[i].pnode[waytonew]), two[i].num, &dir, &vir) ;
            
              CHOSEN_VIR(id,two[i].num) = vir ;
              CHOSEN_DIR(id,two[i].num) = dir ;
            
            /*  
              CHOSEN_VIR(id,two[i].num) = CHOSEN_VIR(id,two[i].pnode[waytonew]) ;
              CHOSEN_DIR(id,two[i].num) = CHOSEN_DIR(id,two[i].pnode[waytonew]) ;
            */
            }
            /*}}}*/
      
            /*{{{  route from new id to node via plink[wayfromnew]*/
            CHOSEN_DIR(two[i].num,id) = two[i].plink[wayfromnew] ;
            CHOSEN_VIR(two[i].num,id) = 0 ;
            /*}}}*/
          }
      /*
        for (j=0;j<extra_two;j++)
          if (i!=j)
          {
            CHOSEN_DIR(two[i].num,two[j].num) = CHOSEN_DIR(two[i].num,two[j].pnode[way]) ;
            CHOSEN_VIR(two[i].num,two[j].num) = 0 ;
          }      */
      }
      /*}}}*/
    }
    /*}}}*/
    /*{{{  free memory*/
    free(spliced) ;
    if (extra_one) free(one) ;
    if (extra_two) free(two) ;
    /*}}}*/
  
    printf("#TOPOLOGY grid ") ;
  
    if (nx==1)
      printf("1 %d %d", ny, wrapy) ;
    else if (ny==1)
      printf("1 %d %d", nx, wrapx) ;
    else
      printf("2 %d %d %d %d", nx, ny, wrapx, wrapy) ;
  
    printf(" %d\n", first_id) ;
  }
  /*}}}*/
  /*{{{  check routing*/
  {
    int start, finish, current ;
    
    for (start=0;start<node_cnt;start++)
      for (finish=0;finish<node_cnt;finish++)
      {
        int hop=0 ;
  
        if (debug)
          fprintf(stderr, "%d to %d :\n", start, finish) ; 
        
        current = start ;
        while (current!=finish)
        {
          int out_link ;
  
          if ((out_link = CHOSEN_DIR(current,finish)) == not_done)
          {
            fprintf(stderr, "\nError-%s(%d)- No direction chosen for %d to %d\n", __FILE__,__LINE__,current,finish) ;
            exit(-1);
          }
  
          if ((out_link = CHOSEN_DIR(current,finish)) == path_end)
          {
            fprintf(stderr, "\nError-%s(%d)- Path ended at wrong node for %d to %d\n", __FILE__,__LINE__,current,finish) ;
            exit(-1);
          }
  
          if (debug) 
            fprintf(stderr, "%d (%d), ", current, out_link) ;
          
          hop++ ;
          current = node[current]->con_proc[out_link] ;
        }
  
        if (debug)
          fprintf(stderr, "%d\n", current) ;
  
        if (CHOSEN_DIR(current,finish) != path_end)
        {
          fprintf(stderr, "\nError-%s(%d)- No path end marker for %d to %d\n", __FILE__,__LINE__,current,finish) ;
          exit(-1);
        }
          
        if (hop>diameter)
        {
          diameter = hop ;
          fprintf(stderr,"Routing does not use shortest path (eg. %d to %d) , diameter now %d\n",start,finish,diameter) ;
        }
      }
  }
/*}}}*/

  return(TRUE) ;
}
/*}}}*/
/*}}}*/

/*{{{  void prune_loops(void)*/
void prune_loops(void)
{
  int i, j ;

  for (i=0;i<node_cnt;i++)
    for (j=0;j<ndirection;j++)
      if (node[i]->con_proc[j] == i)
      {
        node[i]->con_proc[j] = not_used ;
        node[i]->con_link[j] = not_used ;
      }
}

/* This function deletes all links with both ends on same processor */
/*}}}*/

/*{{{  void parse_options (int argc, char **argv)*/
void parse_options (int argc, char **argv)
{
  int p ;

  #ifdef PARSE_DEBUG
  for (p=1;p<argc;p++)
    fprintf(stderr,"ARG %d : !%s!\n",p,argv[p]) ;
  #endif
    
  for (p=1;p<argc;p++)
  {
    if ((argv[p][0] == '-') || (argv[p][0] == '/'))
      /*{{{  check for s,d,w,m,r options*/
      {
        char c = argv[p][1] ;
        BOOL b = TRUE ;
      
        if (c=='n')
        {
          b = FALSE ;
          c = argv[p][2] ;
        }
        
        if (isalpha(c))
        {
          char option = (isupper(c) ? tolower(c) : c) ;
        
          if (option == 'd')
            debug = b ;
          else if (option == 's')
            silent = b ;
          else if (option == 'w')
            wiring_dgm = b ;
          else if (option == 'm')
            show_map = b ;
          else if (option == 'r')
            net_match = b ;
        }
      }
      /*}}}*/
    else
      if (argv[p][0] == 'T')
        if ((atoi(&argv[p][1])<100)||(atoi(&argv[p][1])>999))
        {
          fprintf(stderr,"\nError-%s(%d)- Invalid processor type\n",__FILE__,__LINE__);
          exit(-1);
        }
        else
          cl_part = argv[p] ;
      else
      {
        int num ;
        
        if (sscanf(argv[p], "%d", &num) == 1)
        {
          if ((strchr(argv[p], 'M') != NULL) || (strchr(argv[p], 'm') != NULL))
            cl_memory = num*kilobyte ;
          else
            if ((strchr(argv[p], 'K') != NULL) || (strchr(argv[p], 'k') != NULL))
              cl_memory = num ;
            else
              if (num >= ndirection)
              {
                fprintf(stderr, "Error-%s(%d)- Invalid boot link parameter %d\n", __FILE__,__LINE__, num) ;
                exit(-1) ;
              }
              else
                cl_boot_link = num ;
        }
      }
      
  }
}
/*}}}*/
     
/*{{{  int main(int argc, char *argv[])*/
int main(int argc, char *argv[])
{
  /*{{{  force buffered stdio for icc*/
  #ifdef _ICC
  char bufin[BUFSIZ], bufout[BUFSIZ] ;
  setbuf(stdin,  bufin) ;
  setbuf(stdout, bufout) ;
  #endif
  /*}}}*/
  /*{{{  parse CLI parameters*/
  {
    char *envargs = getenv(ROUTEGEN_ARG) ;
    char **eargv ;
    int  eargc=1 ;
  
    if (envargs != NULL)
    {  
      char *e = envargs ;
  
      while(*e)
      {
        while (*e && isspace(*e)) *e++ ;
        if (*e == 0) break ;
        while (*e && !isspace(*e)) *e++ ;
        eargc++ ;
      }
    
      if ((eargv = (char **) malloc (eargc*sizeof(char*))) == NULL)
      {
        fprintf(stderr, "Error-%s(%d)- Malloc for args array failed\n", __FILE__,__LINE__) ;
        return(-1) ;
      }
    
      e = envargs ;
      eargc=1 ;
    
      while(*e)
      {
        while (*e && isspace(*e)) *e++ ;
        if (*e == 0) break ;
        eargv[eargc] = e ;
        while (*e && !isspace(*e)) *e++ ;
        if (*e == 0) break ;
        *e++ = 0 ;
        eargc++ ;
      }
        
      parse_options(eargc,eargv) ;
    }
    
    parse_options(argc,argv) ;
    
    if (wiring_dgm)
      /*{{{  assert defaults if no parameters given*/
      {
        if (cl_memory < 0)
          cl_memory = memory_default ;
      
        if (cl_boot_link < 0)
          cl_boot_link = boot_link_default ;
          
        if (cl_part == NULL)
          cl_part = part_default ;
      }
      /*}}}*/
  }      
  
  /*}}}*/
  /*{{{  title*/
  if (!silent)
  {
    fprintf(stderr, "NETWORK CONFIGURATION FILE GENERATOR\n");
    fprintf(stderr, "Mark Debbage and Mark B. Hill   Jan 1991 V1.0\n");
    fprintf(stderr, "University Of Southampton, ESPRIT P2701 PUMA\n") ;
  }
  
  if (!silent && wiring_dgm)
    fprintf(stderr, "Wiring diagram input : homogeneous net of %dK %ss booted from link %d\n", cl_memory, cl_part, cl_boot_link) ;
  /*}}}*/
  /*{{{  display options if debug*/
  if (debug)
  {
    fprintf(stderr,"Options: debug (%c) silent (%c) wiring (%c) mappings (%c) recognise nets (%c)\n",
            debug ? 'T' : 'F', silent ? 'T' : 'F', wiring_dgm ? 'T' : 'F',
            show_map ? 'T' : 'F', net_match ? 'T' : 'F') ;
  
    if (wiring_dgm)
      fprintf(stderr,"Processor %s, memory %dK, boot link %d\n",cl_part,cl_memory,cl_boot_link) ;
  }
  /*}}}*/
  
  /*{{{  malloc node pointers */
  if ((node = (NET **) malloc (current_maxnodes * sizeof(NET *))) == NULL)
  {
    fprintf(stderr, "Error-%s(%d)- Malloc for net pointers array failed\n", __FILE__,__LINE__) ;
    return(-1) ;
  }
  /*}}}*/
  /*{{{  get input file*/
  if (wiring_dgm)
  {
    if (LoadWiringDiagram(stdin) < 0)
    {
      fprintf(stderr, "Error-%s(%d)- Wiring diagram file read failed\n", __FILE__,__LINE__) ;
      return(-1) ;
    }
  }
  else
  {
    if (LoadCheckerFile(stdin) < 0)
    {
      fprintf(stderr, "Error-%s(%d)- Checker file read failed\n", __FILE__,__LINE__) ;
      return(-1) ;
    }
  }
  
  prune_loops() ;
  /*}}}*/

  /*{{{  if (!silent) print memory requirement*/
  if (!silent)
    fprintf(stderr, "\nTotal dynamic memory need to route this network is %d bytes\n",
         node_cnt*sizeof(NET) + current_maxnodes*sizeof(NET *) +
         4*node_cnt*node_cnt + node_cnt*node_cnt*ndirection +
         node_cnt*ndirection*sizeof(int)+2*node_cnt*sizeof(int)) ;
  /*}}}*/

  {
    /*{{{  declare arrays bounded by node_cnt*/
    BYTE *distance ;   /* Distance from a to b */
    BYTE *direction ;  /* Ways to go from a to b */
    BYTE *route_ctr ;  /* Number of ways to go from a to b */
    BYTE *chosen_dir ; /* Chosen direction to go from a to b */
    BYTE *chosen_vir ; /* Chosen virtual link to go from a to b */
    int (*usage) [ndirection] ;  /* Count of link usage */
    /*}}}*/
    /*{{{  malloc  arrays bounded by node_cnt*/
    if (((distance = (BYTE *) malloc (node_cnt*node_cnt)) == NULL) ||
        ((direction = (BYTE *) malloc (node_cnt*node_cnt*ndirection)) == NULL) ||
        ((route_ctr = (BYTE *) malloc (node_cnt*node_cnt)) == NULL) ||
        ((usage = (int (*)[ndirection]) malloc (node_cnt*ndirection*sizeof(int))) == NULL) ||
        ((chosen_dir = (BYTE *) malloc (node_cnt*node_cnt)) == NULL) ||    
        ((chosen_vir = (BYTE *) malloc (node_cnt*node_cnt)) == NULL) ||
        ((old_to_new_ids = (int *) malloc (node_cnt*sizeof(int))) == NULL) ||
        ((new_to_old_ids = (int *) malloc (node_cnt*sizeof(int))) == NULL))
    {
      fprintf(stderr, "Error-%s(%d)- Malloc for routegen arrays failed\n", __FILE__,__LINE__) ;
      return(-1) ;
    }
    /*}}}*/

    /*{{{  PHASE I   : Generating shortest distances*/
    {
      int dir ;
      int start,finish, current ;
      int dist = 0;
      BOOL doing = TRUE;
      
      if (!silent)
        fprintf(stderr, "\nPHASE I : Generating shortest distances ") ;
      
      /*{{{  initialise distance/direction arrays*/
      for (start=0;start<node_cnt;start++)
        for (finish=0;finish<node_cnt;finish++)
        {
          DISTANCE(start,finish) = MAX_BYTE ;   /* large flag */
          ROUTE_CTR(start,finish) = ZERO ;
          if (start==finish)
            DISTANCE(start,finish) = ZERO ;
          for (dir=0;dir<ndirection;dir++)
            if (start==finish)
              DIRECTION(start,finish,dir) = path_end ;
            else
              DIRECTION(start,finish,dir) = not_done ;
        }
        
      /*}}}*/
    
      /*{{{  branch out from each node to find shortest distances*/
      while (doing)
      {
        if (!silent)
          fprintf(stderr, ".") ;
        
        doing = FALSE;
      
        dist++;
        
        for (start=0;start<node_cnt;start++) /* Find all nodes dist steps */
          for (current=0;current<node_cnt;current++)
            if (DISTANCE(start,current) == dist-1)
              for (dir=0;dir<ndirection;dir++)
                if (node[current]->con_proc[dir] != not_used)
                  if (DISTANCE(start,node[current]->con_proc[dir]) > dist)
                  {
                    DISTANCE(start,node[current]->con_proc[dir]) = dist ;
                    doing = TRUE;
                  }
      }
      /*}}}*/
    
      diameter = dist-1;
      
      if (!silent)
        fprintf(stderr, "\nNetwork has diameter %d\n", diameter) ;
    }
    /*}}}*/

    if (diameter != 0)
    {
      BOOL matched = FALSE ;
      
      if (net_match)
      {
        if (matched = match_to_grid(chosen_dir, chosen_vir))
        {
          if (!silent)
            fprintf(stderr,"Deadlock Free Net - Matched to grid!!!!!!\n");
        }
        #ifdef CUBE_MATCH
        else if (matched = match_to_cube(chosen_dir, chosen_vir))
        {
          if (!silent)
            fprintf(stderr,"Deadlock Free Net - Matched to binary n-cube!!!!!!\n");
        }
        #endif
      }

      if (!matched)
      {
        /*{{{  declare arrays bounded by diameter*/
        int  *tried_routes ;
        int  *best_route ;
        /*}}}*/
        /*{{{  malloc  arrays bounded by diameter*/
        if ((tried_routes = (int *) malloc (diameter*sizeof(int))) == NULL)
        {
          fprintf(stderr, "Error-%s(%d)- Malloc for tried_routes array failed\n", __FILE__,__LINE__) ;
          return(-1) ;
        }
        
        if ((best_route = (int *) malloc (diameter*sizeof(int))) == NULL)
        {
          fprintf(stderr, "Error-%s(%d)- Malloc for best_route array failed\n", __FILE__,__LINE__) ;
          return(-1) ;
        }
        /*}}}*/
  
        /*{{{  PHASE II  : Generate all shortest paths*/
        {
          int start,finish ;
        
          if (!silent)
            fprintf(stderr, "\nPHASE II : Generating shortest paths ") ;
          
          for (start=0;start<node_cnt;start++)
          {
            int dir;
        
            if (!silent)
              fprintf(stderr, ".") ;
              
            for (finish=0;finish<node_cnt;finish++)
              for (dir=0;dir<ndirection;dir++)
                if (node[start]->con_proc[dir] != not_used)
                  if (DISTANCE(start,finish) == DISTANCE(node[start]->con_proc[dir],finish) + 1)
                    DIRECTION(start,finish,ROUTE_CTR(start,finish)++) = dir ;
          }
        
          if (!silent)
            fprintf(stderr, "\n") ;
        }
        /*}}}*/
        /*{{{  PHASE III : Select routes and link virtualisations*/
        {
          int minimum ;
          int out_link,dir,start,finish,current ;
        
          if (!silent)
            fprintf(stderr, "\nPHASE III : Select routes ") ;
          
          /*{{{  initialization*/
          for (start=0;start<node_cnt;start++)
          {
            for (finish=0;finish<node_cnt;finish++)
              if (start == finish)
                CHOSEN_DIR(start,finish) = path_end ;
              else
                CHOSEN_DIR(start,finish) = not_done ;
              
            for (dir=0;dir<ndirection;dir++)
               usage[start][dir] = 0 ;
          }
          /*}}}*/
        
          /* Strategy modified to allocate shortest routes first
             and to merge convergent routes */
        
          /*{{{  fair rule - choose route with lowest max usage*/
          {
            int d ;
            for (d=1;d<=diameter;d++)
            {
              for (start=0;start<node_cnt;start++)
                for (finish=0;finish<node_cnt;finish++)
                  if (DISTANCE(start,finish) == d)
                    /*{{{  find cheapest route*/
                    {
                      int hop,dist ;
                      int going, cost ;
                    
                      if (debug)
                        fprintf(stderr, "%d to %d :\n", start, finish) ; 
                        
                      /*{{{  error check*/
                      if (start==finish)
                      {
                        fprintf(stderr, "\nError-%s(%d)- No need to select 0 hop route!\n", __FILE__,__LINE__) ;
                        return(-1);
                      }
                      /*}}}*/
                      /*{{{  initialize cheapest route search*/
                      minimum = MAX_VALUE ;
                      going = true ;
                      
                      for (dist=0;dist<diameter;dist++)
                        tried_routes[dist] = 0 ;
                      /*}}}*/
                    
                      while (going)
                      {
                        if (debug)
                          fprintf(stderr, "            Route  : ") ; 
                        
                        /*{{{  do a route*/
                        hop = 0 ;
                        current = start ;
                        cost = 0 ;
                            
                        /*{{{  calculate cost from start to finish using tried_routes*/
                        while (current != finish)
                        {
                          if (hop >= diameter)
                          {
                            fprintf(stderr, "\nError-%s(%d)- Network diameter exceed in route selection\n", __FILE__,__LINE__) ;
                            return(-1);
                          }
                            
                          if ((out_link=CHOSEN_DIR(current,finish)) == not_done)
                            out_link = DIRECTION(current,finish,tried_routes[hop]) ;
                        
                          if (out_link >= ndirection)
                          {
                            fprintf(stderr, "\nError-%s(%d)- Invalid index into usage on routing\n", __FILE__,__LINE__) ;
                            return(-1);
                          }
                          
                          if (usage[current][out_link] > cost)
                            cost = usage[current][out_link] ;
                        
                          if (debug) 
                            fprintf(stderr, "%d (%d), ", current, out_link) ;
                          
                          current = node[current]->con_proc[out_link] ;
                          hop++ ;
                        }
                        /*}}}*/
                        
                        if (debug)
                          fprintf(stderr, "%d : costs %d\n", current, cost) ;
                        
                        /*{{{  record most cost-effective route*/
                        if (cost < minimum)
                        {
                          int dist ;
                          minimum = cost ;        /* Found a cheaper one */
                          for (dist=0;dist<diameter;dist++)
                            if ((best_route[dist] = tried_routes[dist]) >= ndirection)
                            {
                             fprintf(stderr, "\nError-%s(%d)- Invalid best route\n", __FILE__,__LINE__) ;
                             return(-1);
                            }
                        }
                        /*}}}*/
                        /*}}}*/
                        /*{{{  calculate next route to try*/
                        /* This works by considering tried_routes[diameter-1 .. 0] as a counter
                           with each element in a base determined by number of shortest routes
                           coming out of it (or base 1 if direction chosen). This number is
                           incremented each main loop iteration to generate a new route to try.
                        
                           In case of already selected direction that tried_routes entry is
                           invalid and overridden by the selected direction. */
                           
                        hop = 0 ;
                        current = start ;
                        
                        /*{{{  increment tried_routes counter */
                        tried_routes[0]++ ;
                        /*}}}*/
                        
                        while (current != finish)
                        {
                          int base ;
                          
                          if ((out_link=CHOSEN_DIR(current,finish)) == not_done)
                            base = ROUTE_CTR(current,finish) ;
                          else
                            base = 1 ;
                            
                          if (tried_routes[hop] >= base)
                          {
                            tried_routes[hop] = 0 ;
                            if (hop == DISTANCE(start,finish)-1)
                              /*{{{  counter overflow implies search done*/
                              going = false ;
                              /*}}}*/
                            else
                              /*{{{  carry into next digit*/
                              tried_routes[hop+1]++ ;
                              /*}}}*/
                          }
                        
                          if (out_link == not_done)
                            out_link = DIRECTION(current,finish,tried_routes[hop]) ;
                             
                          current = node[current]->con_proc[out_link] ;
                          hop++ ;
                        }
                        /*}}}*/
                      }
                    
                      /*{{{  use best route found*/
                      hop = 0 ;
                      current = start ;
                      if (debug)
                        fprintf(stderr, "            Choose : ") ; 
                      
                      while (current != finish)
                      {
                        if (CHOSEN_DIR(current,finish) == not_done)
                          /*{{{  made new direction choice*/
                          out_link = (CHOSEN_DIR(current,finish) = DIRECTION(current,finish,best_route[hop])) ;
                          /*}}}*/
                        else
                          /*{{{  use already chosen direction*/
                          out_link=CHOSEN_DIR(current,finish) ;
                          /*}}}*/
                      
                        if (debug) 
                          fprintf(stderr, "%d (%d), ", current, out_link) ;
                      
                      
                        if (out_link >= ndirection)
                        {
                          fprintf(stderr, "\nError-%s(%d)- Invalid index into usage on route selection\n", __FILE__,__LINE__) ;
                          return(-1);
                        }   
                          
                        usage[current][out_link]++ ;
                        current = node[current]->con_proc[out_link] ;
                        hop++ ;
                      }
                      
                      if (debug)
                        fprintf(stderr, "%d : costs %d\n", current, cost) ;
                      /*}}}*/
                    }
                    /*}}}*/
          
              if (!silent)
                fprintf(stderr,".") ;
            }
          
            if (!silent)
              fprintf(stderr,"\n") ;
          }
          /*}}}*/
        }
        /*}}}*/
        /*{{{  free heap*/
        free(tried_routes) ;
        free(best_route) ;
        /*}}}*/
        /*{{{  assess link usage*/
        {
          int low, high, sum, count, links ;
          int path, start, finish, current ;
          int high_proc, high_dir ;
          int dir ;
          double mean, variance ;
          
          if (!silent)
            fprintf(stderr, "\nLINK USAGE : Analysis including root transputer\n") ;
        
          path = 0;
          
          for (start=0;start<node_cnt;start++)
            for (dir=0;dir<ndirection;dir++)
              usage[start][dir] = 0 ;
            
          /*{{{  tot up link usage*/
          {
            for (start=0;start<node_cnt;start++)
            {
              for (finish=0;finish<node_cnt;finish++)
              {
                int hop=0 ;
          
                if (debug)
                  fprintf(stderr, "%d to %d :\n", start, finish) ; 
                
                current = start ;
                while (current!=finish)
                {
                  int out_link ;
          
                  if ((out_link = CHOSEN_DIR(current,finish)) == not_done)
                  {
                    fprintf(stderr, "\nError-%s(%d)- No direction chosen for %d to %d\n", __FILE__,__LINE__,current,finish) ;
                    return(-1);
                  }
          
                  if (debug) 
                    fprintf(stderr, "%d (%d), ", current, out_link) ;
                  
                  hop++ ;
                  usage[current][out_link]++ ;
                  current = node[current]->con_proc[out_link] ;
                }
                path += hop;
          
                if (debug)
                  fprintf(stderr, "%d\n", current) ;
              }
            }
          }
          /*}}}*/
            
          low = MAX_VALUE ;
          high = 0 ;
          sum = 0 ;
          links = 0 ;
            
          /*{{{  compute low, mean, high of usages*/
          {
            for (start=0;start<node_cnt;start++)
              for (dir=0;dir<ndirection;dir++)
                if (node[start]->con_proc[dir] != not_used)
                {
                  links++ ;
                  count = usage[start][dir] ;
                  sum += count ;
                  if (count>high)
                  {
                    high = count ;
                    high_proc = start ;
                    high_dir = dir ;
                  }
                  if (count<low)  low = count ;
                }
          
            mean = ((double) sum)/((double) links) ;
          }  
          /*}}}*/
          /*{{{  compute variance of usages*/
          {
            double sum_of_sq_deltas = 0.0;
          
            for (start=0;start<node_cnt;start++)
              for (dir=0;dir<ndirection;dir++)
                if (node[start]->con_proc[dir] != not_used)
                  sum_of_sq_deltas += SQR(usage[start][dir] - mean) ;
          
            variance = (sum_of_sq_deltas)/(links) ;
          }  
          /*}}}*/
            
          if (!silent)
            /*{{{  print results*/
            if (node_cnt != 1)
            {
              fprintf(stderr, "%d of %d hard links used\n", links, node_cnt*ndirection) ;
              fprintf(stderr, "Mean path length is %.2f\n", ((double) path)/((double) (node_cnt*node_cnt))) ;
              
              fprintf(stderr, "High usage is %d (occured on processor %d, link %d)\n", high, high_proc, high_dir) ;
              fprintf(stderr, "Mean usage is %.2f (%d %% of maximum)\n", mean, (sum*100)/(links*high)) ;
              fprintf(stderr, "Low  usage is %d (%d %% of maximum)\n", low, (low*100)/high) ;
              fprintf(stderr, "Variance of usages is %.3f\n", variance) ;
            }  
            /*}}}*/
          
          /*{{{  dump usage to stderr*/
          if (debug)
          {
            int dir ;
          
            fprintf(stderr, "\nLINK USAGE : Individual link usages\n") ;
            
            for (start=0;start<node_cnt;start++)
            {
              fprintf(stderr, "Node %d : ", start) ;
              for (dir=0;dir<ndirection;dir++)
                fprintf(stderr, "%d%s", usage[start][dir], (dir==(ndirection-1) ?  "\n" : ", ")) ;
            }
          }
          /*}}}*/
        }
        /*}}}*/
        /*{{{  generate virtual link numbers (by structured buffer pool)*/
        {
          int proc,finish ;
        
          for (proc=0;proc<node_cnt;proc++)
            for (finish=0;finish<node_cnt;finish++)  
              if (DISTANCE(proc,finish)<2)
                CHOSEN_VIR(proc,finish) = 0 ;
              else
                CHOSEN_VIR(proc,finish) = DISTANCE(proc,finish)-1 ;
        }
        /*}}}*/
        /*{{{  generate identity id mapping*/
        {
          int i ;
          for(i=0;i<node_cnt;i++)
          {
            old_to_new_ids[i] = i ;
            new_to_old_ids[i] = i ;
          }
        }
        /*}}}*/
      }
    }
    else
    {
      CHOSEN_DIR(0,0) = path_end ;
      CHOSEN_VIR(0,0) = 0 ;
    }

    /*{{{  show mapping*/
    if (debug || show_map)
    {
      int i ;
      for (i=0;i<node_cnt;i++)
        fprintf(stderr, "Mapped input proc %d to ncf proc %d\n", node[i]->unmapped_id, i) ;
    }      
    /*}}}*/
    /*{{{  output network configuration file*/
    {
      int proc,finish ;
      
      if (!silent)
        fprintf(stderr, "\nWriting out network configuration file\n") ;
      
      /*{{{  header*/
      fprintf(stdout, "%d %d\n",node_cnt,diameter);
      
      if (free_links==0)
        fprintf(stderr, "Warning-routegen.c : No host connection\n") ;
      /*}}}*/
      /*{{{  processor descriptions*/
      {
        int link, dist ;
      
        if (root==not_used)
        {
          fprintf(stderr, "\nError-%s(%d)- No Host Connection\n", __FILE__,__LINE__) ;
          return(-1);
        }
          
        for (dist=0;dist<=diameter;dist++)
          for (proc=0;proc<node_cnt;proc++)
            if (DISTANCE(new_to_old_ids[root],new_to_old_ids[proc])==dist)
              /*{{{  write nodes dist hops from route*/
              {
                fprintf(stdout, "\n%d\n%s\n", proc, node[proc]->part );
                fprintf(stdout, "%dK\n", node[proc]->memory) ;
              
                for (link=0;link<ndirection;link++)
                {
                  if (node[proc]->boot==link)
                  {
                    if (proc==root)
                    {
                      if (node[proc]->con_proc[link] == not_used)
                        fprintf(stdout,"h ");
                      else
                        {
                          fprintf(stderr,"Root node does not have specified boot link available\n") ;
                          fprintf(stdout, "n%d:%d ", node[proc]->con_proc[link], node[proc]->con_link[link]) ;
                        }          
                    }
                    else
                      fprintf(stdout, "i%d:%d ", node[proc]->con_proc[link], node[proc]->con_link[link]) ;
                    continue;
                  }
              
                  if (node[proc]->con_proc[link]==not_used)
                  {
                    fprintf(stdout,"x ");
                    continue;
                  }
                  
                  {
                    int neighbour = node[proc]->con_proc[link] ;
                    
                    if (node[neighbour]->boot == not_used)
                    {
                      fprintf(stdout, "o%d:%d ", node[proc]->con_proc[link], node[proc]->con_link[link]) ;
                      node[neighbour]->boot = CHOSEN_DIR(neighbour,proc); 
                    }
                    else
                      fprintf(stdout, "n%d:%d ", node[proc]->con_proc[link], node[proc]->con_link[link]) ;
                  }
                }
                fprintf(stdout,"\n");
              }
              /*}}}*/
      }
      /*}}}*/
      /*{{{  routing tables*/
      for (proc=0;proc<node_cnt;proc++)
      {
        fprintf(stdout, "\n%d %d\n",proc, node_cnt*2);
      
        for (finish=0;finish<node_cnt;finish++)  
        {
          if (finish%16==15)
            fprintf(stdout,"\n") ;
      
          if (CHOSEN_DIR(proc,finish)==path_end)
            fprintf(stdout,"%d ", ndirection);
          else
            fprintf(stdout, "%d ", CHOSEN_DIR(proc,finish));
      
          fprintf(stdout, "%d ", CHOSEN_VIR(proc,finish)) ;
        }
        fprintf(stdout,"\n");
      }
      /*}}}*/
    }
    /*}}}*/
    /*{{{  free heap*/
    {
      int i ;
    
      for (i=0;i<node_cnt;i++)
        if (node[i] != NULL) free(node[i]) ;
    
      free(node) ;
      free(distance) ;
      free(direction) ;
      free(route_ctr) ;
      free(usage) ;
      free(chosen_dir) ;
      free(chosen_vir) ;
      free(old_to_new_ids) ;
      free(new_to_old_ids) ;
    }
    /*}}}*/
  }
  /*{{{  finished*/
  if (!silent)
  fprintf(stderr, "Done.\n") ;
  
  return(0) ;
  /*}}}*/
}
/*}}}*/

