/*{{{  File banner*/
/*@(#)=====================================================*\
||@(#)  Project : PUMA ESPRIT P2701
||@(#)  Authors : Mark Debbage and Mark Hill
||@(#)            University of Southampton
||  
||@(#)    Title : Deadlock-Freedom Verifier for NCF files
||@(#)   System : ROUTEGEN
||@(#) Filename : ncfcheck.c
||@(#)  Version : 2.4
||@(#)     Date : 7/13/92
\*@(#)====================================================*/
/*}}}*/

/*{{{  includes*/
#include <stdio.h>
#include <string.h>
#include <math.h>
#include <ctype.h>

#ifdef _ICC
#include <stdlib.h>
#endif
/*}}}*/
/*{{{  definitions*/
typedef unsigned char BYTE ;
typedef unsigned char BOOL;

#define MAX_VALUE    1000000
#define MAX_LINE     100
#define MAX_PART     10

#define ndirection  4             /* no of transputer links */

#define not_done (-1)
#define not_used (-1)
#define not_link (-3)
#define no_arc   (~0)
#define rm_arc   (~1)

#define TRUE  (1)
#define FALSE (0)
#define kilobyte (1024)

#define ll 0
#define vl 1

#define SQR(x) ((x)*(x))
/*}}}*/
/*{{{  typedef LINK_TYPES*/
typedef enum {invalid, inactive, host, virtual} LINK_TYPES ;
/*}}}*/
/*{{{  struct NET*/
typedef struct
{
  int con_proc[ndirection] ; /* Neighbouring processor */
  int con_link[ndirection] ; /* Neighbouring processor's link */
  int boot ;                 /* Boot link */
  char part[MAX_PART];       /* Part Name */
  int memory;                /* Memory in K */
}
NET ;
/*}}}*/

/*{{{  network description statics*/
static NET *node ;
static int node_cnt    = 0;
static int root        = not_done;
static int diameter ;
static int debug       = FALSE ;
static int silent      = FALSE ;
static int num_vlinks   = 0 ;
/*}}}*/
/*{{{  access arrays by pointer expansions*/
#define OUT_LINK(s,d,a)  (*(out_link  + (s)*node_cnt*2 + (d)*2 + (a)))
#define CDG_LIST(v,i)    (*(cdg_list  + (v)*vlinks_per_node + (i)))
/*}}}*/
/*{{{  vlink to vid mapping*/
#define vlinks_per_node (num_vlinks*ndirection)
#define total_vlinks    (vlinks_per_node*node_cnt)
#define VLINK_ID(p,l,v) ((v) + (l)*num_vlinks + (p)*vlinks_per_node)

#define PROC(vid)  (vid / vlinks_per_node)
#define LINK(vid)  ((vid / num_vlinks) % ndirection)
#define VLINK(vid) (vid % num_vlinks)
/*}}}*/

int  linenum = 0 ;
/*{{{  string parsing routines*/
/*{{{  char *skip_non_blanks (char *s)*/
char *skip_non_blanks (char *s)
{
  while ((*s != 0) && (!isspace(*s)))
    s++ ;
  return(s) ;
}

/*}}}*/
/*{{{  char *skip_blanks (char *s)*/
char *skip_blanks (char *s)
{
  while ((*s != 0) && (isspace(*s)))
    s++ ;
  return(s) ;
}

/*}}}*/
/*{{{  char *skip_digits (char *s)*/
char *skip_digits (char *s)
{
  while ((*s != 0) && (isdigit(*s)))
    s++ ;
  return(s) ;
}

/*}}}*/
/*}}}*/
/*{{{  pragma matching*/
#define num_topologies 1
#define grid 0
char topology_text[num_topologies][5] = { "grid" } ;

/*{{{  void extract_topology(char *s)*/
void extract_topology(char *s)
{
  int i ;

  s = skip_blanks(s) ;
  
  for (i=0;i<num_topologies;i++)
    if (strncmp(s,topology_text[i],strlen(topology_text[i])) == 0)
    {
      fprintf(stderr,"Topology is %s\n",topology_text[i]) ;
      break ;
    }      

  switch(i)
  {
    case grid :
       /*{{{  read grid information*/
       {
         int dims, d, num ;
       
         s = skip_non_blanks(s) ;
         s = skip_blanks(s) ;
       
         if (sscanf(s,"%d",&dims) != 1)
         {
           fprintf(stderr, "Error-%s(%d)- Parse failed reading dimensions on line %d\n",__FILE__,__LINE__,linenum) ;
           exit(-1) ;
         }
       
         if (dims<0)
         {
           fprintf(stderr, "Error-%s(%d)- Parse failed, negative dimensions on line %d\n",__FILE__,__LINE__,linenum) ;
           exit(-1) ;
         }
       
         s = skip_non_blanks(s) ;
         s = skip_blanks(s) ;
       
         if (debug) fprintf(stderr,"Dimensions : ") ;
         for (d=0;d<dims;d++)
         {
           if (sscanf(s,"%d",&num) != 1)
           {
             fprintf(stderr, "Error-%s(%d)- Parse failed reading a dimension on line %d\n",__FILE__,__LINE__,linenum) ;
             exit(-1) ;
           }
           if (debug) fprintf(stderr,"%d ",num) ;
           s = skip_non_blanks(s) ;
           s = skip_blanks(s) ;
         }
         if (debug) fprintf(stderr,"\n") ;
         
         if (debug) fprintf(stderr,"Wrap flags : ") ;
         for (d=0;d<dims;d++)
         {
           if (sscanf(s,"%d",&num) != 1)
           {
             fprintf(stderr, "Error-%s(%d)- Parse failed reading a wrap flag on line %d\n",__FILE__,__LINE__,linenum) ;
             exit(-1) ;
           }
           if (debug) fprintf(stderr,"%d ",num) ;
           s = skip_non_blanks(s) ;
           s = skip_blanks(s) ;
         }
         if (debug) fprintf(stderr,"\n") ;
       
         if (sscanf(s,"%d",&num) != 1)
         {
           fprintf(stderr, "Error-%s(%d)- Parse failed reading first id on line %d\n",__FILE__,__LINE__,linenum) ;
           exit(-1) ;
         }
         if (debug) fprintf(stderr,"First id is %d\n",num) ;
         s = skip_non_blanks(s) ;
         s = skip_blanks(s) ;
         
         break ;
       }
       /*}}}*/
    default :
      fprintf(stderr, "Warning-%s(%d)- Unrecognised topology (%s) on line %d\n",__FILE__,__LINE__,s, linenum) ;
  }
}
/*}}}*/

#define num_pragmas 1
#define topology 0
char pragma_text[num_pragmas][10] = { "#TOPOLOGY" } ;

/*{{{  void identify_pragma (char *s)*/
void identify_pragma (char *s)
{
  int i ;

  for (i=0;i<num_pragmas;i++)
    if (strncmp(s,pragma_text[i],strlen(pragma_text[i])) == 0)
    {
      if (debug) fprintf(stderr,"Found %s pragma\n",pragma_text[i]) ;
      break ;
    }      

  switch(i)
  {
    case topology :
      extract_topology(s+strlen(pragma_text[i])) ;
      break ;
  }
}
/*}}}*/
/*}}}*/
/*{{{  file reading primitives*/
#define buflen 160
char linebuf[buflen];
char *linepos = linebuf ;
char *strbuf = NULL ;

/*{{{  char *fgetstring (char *line, int n, FILE *file)*/
/*
   This routine reads a line, strips out # comments and strips
   out lines of just white space. Usage much like fgets.
*/

char *fgetstring (char *line, int n, FILE *file)
{
  char *s ;

  do
  { 
    if ((s=fgets (line, n, file)) == NULL)
      return(NULL) ;
       
    if (!feof(file) && (s[0] != 0))
    {
      char *hash = strchr (s, '#') ;
      int i ;
       
      if ((strlen(s) == n-1) && (s[n-1] != '\n'))
        fprintf(stderr, "Warning-%s(%d)- Line %d too long so wrapped round\n",__FILE__,__LINE__,linenum) ;
      else
        linenum++ ;

      if (hash == s)
        identify_pragma (hash) ;
       
      if (hash != NULL)
        *hash = 0 ;
       
      for (i=0;i<strlen(s);i++)
        if (!isspace(s[i]))
          break ;
       
      if (i==strlen(s))
        s[0] = 0 ;    
    }
  } while (*s == 0) ;

  return(s) ;
}
/*}}}*/
  
/*{{{  int   read_number (FILE *file)*/
int   read_number (FILE *file)
{
  int i ;

  linepos = skip_blanks(linepos) ;
  
  if (*linepos == 0)
  {
    if (fgetstring(linebuf, buflen, file) == NULL)
    {
      fprintf(stderr, "Error-%s(%d)- Parse failed reading number on line %d\n",__FILE__,__LINE__,linenum) ;
      exit(-1) ;
    }
    linepos = linebuf ;
  }

  if (feof(file) || (*linepos == 0))
  {
    fprintf(stderr, "Error-%s(%d)- Parse failed reading number on line %d\n",__FILE__,__LINE__,linenum) ;
    exit(-1) ;
  }

  if (sscanf(linepos,"%d",&i) != 1)
  {
    fprintf(stderr, "Error-%s(%d)- Parse failed reading number on line %d\n",__FILE__,__LINE__,linenum) ;
    exit(-1) ;
  }

  linepos = skip_digits(linepos) ;

  return(i) ;  
}
/*}}}*/
/*{{{  char *read_string (FILE *file)*/
char *read_string (FILE *file)
{
  linepos = skip_blanks(linepos) ;

  if (*linepos == 0)
  {
    if (fgetstring(linebuf, buflen, file) == NULL)
    {
      fprintf(stderr, "Error-%s(%d)- Parse failed reading string on line %d\n",__FILE__,__LINE__,linenum) ;
      exit(-1) ;
    }
    linepos = linebuf ;
  }

  if (feof(file) || (*linepos == 0))
  {
    fprintf(stderr, "Error-%s(%d)- Parse failed reading string on line %d\n",__FILE__,__LINE__,linenum) ;
    exit(-1) ;
  }

  if (strbuf != NULL)
    free(strbuf) ;
    
  strbuf = (char *) malloc(strlen(linepos+1)) ;

  if (sscanf(linepos,"%s",strbuf) != 1)
  {
    fprintf(stderr, "Error-%s(%d)- Parse failed reading string on line %d\n",__FILE__,__LINE__,linenum) ;
    exit(-1) ;
  }

  linepos = skip_non_blanks(linepos) ;

  return(strbuf) ;
}
/*}}}*/
/*{{{  char  read_char   (FILE *file)*/
char  read_char   (FILE *file)
{
  char c ;
  
  if (*linepos == 0)
  {
    if (fgetstring(linebuf, buflen, file) == NULL)
    {
      fprintf(stderr, "Error-%s(%d)- Parse failed reading character on line %d\n",__FILE__,__LINE__,linenum) ;
      exit(-1) ;
    }
    linepos = linebuf ;
  }

  if (feof(file) || (*linepos == 0))
  {
    fprintf(stderr, "Error-%s(%d)- Parse failed reading character on line %d\n",__FILE__,__LINE__,linenum) ;
    exit(-1) ;
  }

  if (sscanf(linepos,"%c",&c) != 1)
  {
    fprintf(stderr, "Error-%s(%d)- Parse failed reading character on line %d\n",__FILE__,__LINE__,linenum) ;
    exit(-1) ;
  }

  linepos++ ;
  
  return(c) ;  
}
/*}}}*/
/*{{{  void  next_field  (FILE *file)*/
void  next_field  (FILE *file)
{
  linepos = skip_blanks(linepos) ;
}
/*}}}*/
/*}}}*/
/*{{{  int LoadNCFfile(FILE *net_fp)*/
int LoadNCFfile(FILE *net_fp)
{
  int index;
  
  /*{{{  read node_cnt diameter*/
  node_cnt = read_number(net_fp) ;
  diameter = read_number(net_fp) ;
  
  if (node_cnt <= 0 || diameter < 0)
  {
    fprintf(stderr, "Error-%s(%d)- Invalid NCF header\n",__FILE__,__LINE__);
    return(-1);
  }
  
  if (debug) fprintf(stderr,"Network contains %d processors, diameter %d\n", node_cnt, diameter) ;
  /*}}}*/
  /*{{{  malloc node array*/
  if ((node = (NET *) malloc (node_cnt * sizeof(NET))) == NULL)
  {
    fprintf(stderr, "Error-%s(%d)- Malloc for nodes array failed\n", __FILE__,__LINE__) ;
    return(-1) ;
  }
  /*}}}*/
  for (index=0;index<node_cnt;index++)
    node[index].boot = not_used;

  for (index=0;index<node_cnt;index++)
  {
    int n;
    
    /*{{{  read n*/
    n = read_number(net_fp) ;
    
    if (n<0 || n>=node_cnt || node[n].boot != not_used)
    {
      fprintf(stderr, "Error-%s(%d)- Bad processor id %d\n",__FILE__,__LINE__,n);
      return(-1) ;
    }
    
    if (debug) fprintf(stderr,"PROCESSOR %d\n", n) ;
    
    /*}}}*/
    /*{{{  read part*/
    {
      char *s = read_string (net_fp) ;
    
      if (strlen(s) >= MAX_PART-1)
        fprintf(stderr, "Warning-%s(%d)- Truncating part string\n",__FILE__,__LINE__);
    
      strncpy (node[n].part, s, MAX_PART-1) ;
      node[n].part[MAX_PART-1] = 0 ;
    
      if (debug) fprintf(stderr,"PART:%s\n",node[n].part);
    }
    /*}}}*/
    /*{{{  read memory*/
    {
      int size  = read_number(net_fp) ;
      char unit = read_char (net_fp) ;
    
      if (debug) fprintf(stderr,"Memory:%d%c\n", size,unit) ;
    
      if (unit=='M')
        node[n].memory = kilobyte*size ;
      else
        if (unit=='K')
          node[n].memory = size ;
        else
        {
          fprintf(stderr, "Error-%s(%d)- Bad memory units ('%c')\n",__FILE__,__LINE__,unit);
          return(-1);
        }
    }
    /*}}}*/
    /*{{{  read links*/
    {
      int dir ;
      char type ;
      
      for (dir=0;dir<ndirection;dir++)
      {
        next_field(net_fp) ;
        type = read_char (net_fp) ;
      
        switch ((int) type) 
        {
          case (int) 'h' :
            /*{{{  host link*/
            if (debug) fprintf(stderr,"host ");
            root = n;
            node[n].boot = dir;
            node[n].con_proc[dir] = not_used;
            node[n].con_link[dir] = not_link;
            break;
            /*}}}*/
          case (int) 'i' :
            if (node[n].boot == not_used)
              node[n].boot = dir ;
            else
              {
                fprintf(stderr, "Error-%s(%d)- Invalid boot tree, two parents\n",__FILE__,__LINE__);
                return(-1) ;
              }
          case (int) 'o' :
          case (int) 'n' :
            /*{{{  used link*/
            {
              node[n].con_proc[dir] = read_number(net_fp) ;
            
              if (read_char(net_fp) != ':')
              {
                fprintf(stderr,"Error-%s(%d)- Bad connector format\n",__FILE__,__LINE__) ;
                return(-1);
              }
              
              node[n].con_link[dir] = read_number(net_fp) ;
            
              if (debug) fprintf(stderr,"virtual(%d:%d) ",node[n].con_proc[dir],node[n].con_link[dir]);
              
              break;
            }
            /*}}}*/
          case (int) 'x' :
            /*{{{  unused link*/
            if (debug) fprintf(stderr,"inactive ");
            node[n].con_proc[dir] = not_used;
            node[n].con_link[dir] = not_link;
            break;
            /*}}}*/
          default :
            /*{{{  catch anything else*/
            {
              fprintf(stderr,"Error-%s(%d)- Unknown link type ('%c')\n",__FILE__,__LINE__,type) ;
              return(-1);
            }
            /*}}}*/
        }
      }
      if (debug) fprintf(stderr,"\n");
    }            
    /*}}}*/
  }
  
  if (root==not_used && node[root].boot==not_used)
    {
      fprintf(stderr,"Error-%s(%d)- No host link declared\n",__FILE__,__LINE__);
      return(-1);
    }
}    
/*}}}*/
/*{{{  int GetRoutes(FILE *in_fp, BYTE *out_link)*/
int GetRoutes(FILE *in_fp, BYTE *out_link)
{
  int proc ;
  int max_vlinks = 0;
  BOOL *done ;

  if ((done = (BOOL *) malloc (node_cnt * sizeof(BOOL))) == NULL)
  {
    fprintf(stderr,"Error-%s(%d)- Out of memory\n",__FILE__,__LINE__);
    exit(-1);
  }
  
  for (proc=0;proc<node_cnt;proc++)
    done[proc] = FALSE ;
    
  for (proc=0;proc<node_cnt;proc++)
  {
    int finish ;
    int n;
    
    /*{{{  processor number*/
    n = read_number(in_fp) ;
    
    if (n<0 || n>=node_cnt || done[n])
    {
      fprintf(stderr, "Error-%s(%d)- Bad processor id %d\n",__FILE__,__LINE__,n);
      return(-1) ;
    }
    
    done[n] = TRUE ;
    /*}}}*/
    /*{{{  routing tables length*/
    if (read_number(in_fp) != 2*node_cnt)
    {
      fprintf(stderr, "Error-%s(%d)- Expected %d bytes of routing data\n",__FILE__,__LINE__,2*node_cnt);
      return(-1) ;
    }
    
    /*}}}*/
    
    for (finish=0;finish<node_cnt;finish++)
    {
      /*{{{  logical link*/
      {
        int num = read_number (in_fp) ;
      
        if (num<0 || num>255)
        {
          fprintf(stderr, "Error-%s(%d)- Routing data must be unsigned bytes\n",__FILE__,__LINE__);
          return(-1) ;
        }
          
        OUT_LINK(proc,finish,ll) = num ;
      }
      /*}}}*/
      /*{{{  virtual link*/
      {
        int num = read_number (in_fp) ;
      
        if (num<0 || num>255)
        {
          fprintf(stderr, "Error-%s(%d)- Routing data must be unsigned bytes\n",__FILE__,__LINE__);
          return(-1) ;
        }
          
        OUT_LINK(proc,finish,vl) = num ;
      
        max_vlinks = ((num > max_vlinks) ? num : max_vlinks) ;
      }
      /*}}}*/
    }    
  }

  num_vlinks = max_vlinks+1 ;
}  
/*}}}*/

int main(int argc, char *argv[])
{
  /*{{{  variables*/
  int CDG_cycle ;
  /*}}}*/
  /*{{{  force buffered stdio for icc*/
  #ifdef _ICC
  char bufin[BUFSIZ], bufout[BUFSIZ] ;
  setbuf(stdin,  bufin) ;
  setbuf(stdout, bufout) ;
  #endif
  /*}}}*/
  /*{{{  parse CLI parameters*/
  {
    int p ;
  
    for (p=1;p<argc;p++)
    {
      if ((argv[p][0] == '-') || (argv[p][0] == '/'))
      {
        char c = argv[p][1] ;
  
        if (isalpha(c))
        {
          char option = (isupper(c) ? tolower(c) : c) ;
  
          if (option == 'd')
            debug = TRUE ;
          else
            if (option == 's')
             silent = TRUE ;
        }
      }
    }
  }      
  
  /*}}}*/
  /*{{{  title*/
  if (!silent)
  {
    fprintf(stderr, "NETWORK CONFIGURATION FILE CHECKER\n");
    fprintf(stderr, "Mark Debbage and Mark B. Hill   Jan 1991 V1.0\n");
    fprintf(stderr, "University Of Southampton, ESPRIT P2701 PUMA\n") ;
  }
  /*}}}*/
  
  /*{{{  load NCF file*/
  if (LoadNCFfile(stdin) < 0)
  {
    fprintf(stderr, "Error-%s(%d)- .ncf file read failed\n", __FILE__,__LINE__) ;
    return(-1) ;
  }
  /*}}}*/

  {
    /*{{{  declare arrays bounded by node_cnt*/
    BYTE *out_link ; /* routing table */
    int (*usage) [ndirection] ;  /* Count of link usage */
    /*}}}*/
    /*{{{  malloc  arrays bounded by node_cnt*/
    if ((out_link = (BYTE *) malloc (node_cnt*node_cnt*2)) == NULL)
    {
      fprintf(stderr, "Error-%s(%d)- Malloc for routing table failed\n", __FILE__,__LINE__) ;
      return(-1) ;
    }
    
    if ((usage = (int (*)[ndirection]) malloc (node_cnt*ndirection*sizeof(int))) == NULL)
    {
      fprintf(stderr, "Error-%s(%d)- Malloc for usage array failed\n", __FILE__,__LINE__) ;
      return(-1) ;
    }
    /*}}}*/

    /*{{{  load routes*/
    if (!silent)
      fprintf(stderr, "Loading routes\n") ;
    
    if (GetRoutes(stdin, out_link) < 0)
    {
      fprintf(stderr,"Error-%s(%d)- Route transfer failed\n",__FILE__,__LINE__);
      return(-1);
    }
    
    if (!silent)
    {
      fprintf(stderr, "Routes loaded\n") ;
      fprintf(stderr, "Network uses %d levels of virtual links\n", num_vlinks) ;
    }  
    /*}}}*/
    /*{{{  assess link usage*/
    if (node_cnt != 1)
    {
      int low, high, sum, count, links ;
      int path, start, finish, current ;
      int high_proc, high_dir ;
      int dir ;
      double mean, variance ;
      
      if (!silent)
        fprintf(stderr, "\nLINK USAGE : Analysis including root transputer\n") ;
    
      path = 0;
      
      for (start=0;start<node_cnt;start++)
        for (dir=0;dir<ndirection;dir++)
          usage[start][dir] = 0 ;
        
      /*{{{  tot up link usage*/
      {
        for (start=0;start<node_cnt;start++)
        {
          for (finish=0;finish<node_cnt;finish++)
          {
            int hop=0 ;
      
            if (debug)
              fprintf(stderr, "%d to %d :\n", start, finish) ; 
            
            current = start ;
            while (current!=finish)
            {
              int out_dir = OUT_LINK(current,finish,ll) ;
              int next ;
      
              if (debug) 
                fprintf(stderr, "%d (%d), ", current, out_dir) ;
              
              hop++ ;
              usage[current][out_dir]++ ;
              next = node[current].con_proc[out_dir] ;
      
              if (next == not_used)
              {
                fprintf(stderr, "Error-%s(%d)- Routing down unconnected link, routing %d to %d via proc %d link %d\n",__FILE__,__LINE__,start,finish,current,out_dir) ;
                return(-1);
              }
              current = next ;
      
              if (hop > diameter)
              {
                fprintf(stderr, "Error-%s(%d)- Exceeded diameter routing %d to %d\n",__FILE__,__LINE__,start, finish) ;
                return(-1);
              }        
            }
            path += hop;
      
            if (debug)
              fprintf(stderr, "%d\n", current) ;
          }
        }
      }
      /*}}}*/
        
      low = MAX_VALUE ;
      high = 0 ;
      sum = 0 ;
      links = 0 ;
        
      /*{{{  compute low, mean, high of usages*/
      {
        for (start=0;start<node_cnt;start++)
          for (dir=0;dir<ndirection;dir++)
            if (node[start].con_proc[dir] != not_used)
            {
              links++ ;
              count = usage[start][dir] ;
              sum += count ;
              if (count>high)
              {
                high = count ;
                high_proc = start ;
                high_dir = dir ;
              }
              if (count<low)  low = count ;
            }
      
        mean = ((double) sum)/((double) links) ;
      }  
      /*}}}*/
      /*{{{  compute variance of usages*/
      {
        double sum_of_sq_deltas = 0.0;
      
        for (start=0;start<node_cnt;start++)
          for (dir=0;dir<ndirection;dir++)
            if (node[start].con_proc[dir] != not_used)
              sum_of_sq_deltas += SQR(usage[start][dir] - mean) ;
      
        variance = (sum_of_sq_deltas)/(links) ;
      }  
      /*}}}*/
        
      if (!silent)
        /*{{{  print results*/
        {
          fprintf(stderr, "%d of %d hard links used\n", links, node_cnt*ndirection) ;
          fprintf(stderr, "Mean path length is %.2f\n", ((double) path)/((double) (node_cnt*node_cnt))) ;
          
          fprintf(stderr, "High usage is %d (occured on processor %d, link %d)\n", high, high_proc, high_dir) ;
          fprintf(stderr, "Mean usage is %.2f (%d %% of maximum)\n", mean, (sum*100)/(links*high)) ;
          fprintf(stderr, "Low  usage is %d (%d %% of maximum)\n", low, (low*100)/high) ;
          fprintf(stderr, "Variance of usages is %.3f\n", variance) ;
        }  
        /*}}}*/
      
      /*{{{  dump usage to stderr*/
      if (debug)
      {
        int dir ;
      
        fprintf(stderr, "\nLINK USAGE : Individual link usages\n") ;
        
        for (start=0;start<node_cnt;start++)
        {
          fprintf(stderr, "Node %d : ", start) ;
          for (dir=0;dir<ndirection;dir++)
            fprintf(stderr, "%d%s", usage[start][dir], (dir==(ndirection-1) ?  "\n" : ", ")) ;
        }
      }
      /*}}}*/
    }
    /*}}}*/

    /*{{{  if (!silent)  print memory requirement*/
    if (!silent)
      fprintf(stderr, "\nTotal dynamic memory need to check this network is %d bytes\n",
           2*node_cnt*node_cnt +
           sizeof(int)*total_vlinks*vlinks_per_node +
           2*total_vlinks ) ;
    /*}}}*/
    {
      /*{{{  declare arrays bounded by total_vlinks*/
      int  *cdg_list ;
      BYTE *clear_col ;
      BYTE *clear_row ;
      /*}}}*/
      /*{{{  malloc  arrays bounded by total_vlinks*/
      if ((cdg_list = (int *) malloc (total_vlinks*vlinks_per_node*sizeof(int))) == NULL)
      {
        fprintf(stderr, "Error-%s(%d)- Malloc for channel dependency list failed\n", __FILE__,__LINE__) ;
        return(-1) ;
      }
      
      if ((clear_col = (BYTE *) malloc (total_vlinks)) == NULL)
      {
        fprintf(stderr, "Error-%s(%d)- Malloc for clear_col failed\n", __FILE__,__LINE__) ;
        return(-1) ;
      }
      
      if ((clear_row = (BYTE *) malloc (total_vlinks)) == NULL)
      {
        fprintf(stderr, "Error-%s(%d)- Malloc for clear_row failed\n", __FILE__,__LINE__) ;
        return(-1) ;
      }
      /*}}}*/

      /*{{{  clear arrays*/
      {
        int vlink, index ;
      
        if (!silent)
          fprintf(stderr, "Clearing arrays\n") ;
        
        for (vlink=0;vlink<total_vlinks;vlink++)
        {
          clear_col[vlink] = FALSE ;
          clear_row[vlink] = FALSE ;
      
          for (index=0;index<vlinks_per_node;index++)
            CDG_LIST  (vlink,index) = no_arc ;
        }
      }
      /*}}}*/
      /*{{{  route all to all and mark dependencies*/
      {
        int source, destn ;
      
        if (!silent)
          fprintf(stderr, "Generating CDG ") ;
        
        for (source=0;source<node_cnt;source++)
        {
          if (!silent)
            fprintf(stderr, ".") ;
      
          for (destn=0;destn<node_cnt;destn++)
            if (source==destn)
              /*{{{  check link number*/
              {
                if (OUT_LINK(source, destn, ll) != ndirection)
                {
                  fprintf(stderr, "Error-%s(%d)- Bad internal route in NCF\n",__FILE__,__LINE__);
                  return(-1);
                }
              }  
              /*}}}*/
            else
              /*{{{  route*/
              {
                int current = source ;
                int link ; 
                int vlink ;
                int vlink_id = -1 ;
                int hops = 0 ;
              
                while (current != destn)
                {
                  int last_vlink_id = vlink_id ;
                  int next ;
              
                  link  = OUT_LINK(current, destn, ll) ;
                  vlink = OUT_LINK(current, destn, vl) ;
                  next  = node[current].con_proc[link] ;
              
                  if (next == not_used)
                  {
                    fprintf(stderr, "Error-%s(%d)- Routing down unconnected link, routing %d to %d via proc %d link %d\n",__FILE__,__LINE__,source,destn,current,link) ;
                    return(-1);
                  }
                 
                  vlink_id = VLINK_ID(current, link, vlink) ;
              
                  /*{{{  check vlink_id*/
                  if (vlink_id>=total_vlinks)
                  {
                    fprintf(stderr, "Error-%s(%d)- Bad vlink_id generated\n",__FILE__,__LINE__);
                    return(-1);
                  }
                  /*}}}*/
              
                  if (current != source)
                    /*{{{  mark in CDG LIST*/
                    {
                      int i ;
                    
                      if (vlink_id < 0)
                      {
                        fprintf(stderr, "Error-%s(%d)- Invalid CDG arc requested\n",__FILE__,__LINE__);
                        return(-1);
                      }
                    
                      for (i=0;i<vlinks_per_node;i++)
                        if ((CDG_LIST(last_vlink_id,i) == no_arc) || (CDG_LIST(last_vlink_id,i)==vlink_id))
                          break ;
                    
                      /*{{{  check overflow*/
                      if (i==vlinks_per_node)
                      {
                        fprintf(stderr, "Error-%s(%d)- Channel dependency list overflow\n",__FILE__,__LINE__);
                        return(-1);
                      }
                      /*}}}*/
                    
                      CDG_LIST(last_vlink_id,i) = vlink_id ;
                    }
                    /*}}}*/
                    
                  current = next ;
                  hops++ ;
              
                  if (hops > diameter)
                  {
                    fprintf(stderr, "Error-%s(%d)- Exceeded diameter routing %d to %d\n",__FILE__,__LINE__,source,destn);
                    return(-1);
                  }
                }
              }
              /*}}}*/
        }
      
        if (!silent)
          fprintf(stderr, "\n") ;
      }
      /*}}}*/
      /*{{{  if (debug) display CDG*/
      if (debug)
      {
        int vlink_id, i ;
      
        fprintf(stderr, "\nFull channel dependencies list\n") ;
      
        for (vlink_id=0;vlink_id<total_vlinks;vlink_id++)
        {
          BOOL first = TRUE ;
      
          for (i=0;i<vlinks_per_node;i++)
          {
            int vid = CDG_LIST(vlink_id,i) ;
      
            if (vid != no_arc)
              /*{{{  display dependency*/
              {
                if (first)
                {
                  fprintf(stderr, "(%d,%d,%d) depends on", PROC(vlink_id), LINK(vlink_id), VLINK(vlink_id)) ;
                  first = FALSE ;
                }
              
                fprintf(stderr, "  (%d,%d,%d)", PROC(vid), LINK(vid), VLINK(vid)) ;
              }
              /*}}}*/
          }
      
          if (!first) fprintf(stderr, "\n") ;
         }
      }
      /*}}}*/
      /*{{{  check CDG for cycles*/
      {
        BOOL reduced = TRUE ;
        int i, j ;
      
        if (!silent)
          fprintf(stderr, "Checking CDG ") ;
        
        while (reduced)
        {
          reduced = FALSE ;      /* No reductions made yet this iteration */
          
          if (!silent) fprintf(stderr, ".") ;
          
          /*{{{  Scan CDG over columns */
          for (j=0; j<total_vlinks; j++ )
            if (!clear_col[j])
              /*{{{  then check for clarity */
              {
                BOOL clear = TRUE ;
              
                for (i=0; i<total_vlinks; i++)
                {
                  int *start  = &CDG_LIST(i,0) ;
                  int *finish = start+vlinks_per_node ;
                  int *dep ;
                  
                  for (dep=start;(dep<finish) && (*dep != no_arc);dep++)
                    if (*dep == j)
                    {
                      clear = FALSE ;
                      break ;
                    }
              
                  if (!clear) break ;
                }
              
                if ( clear )
                  /*{{{  then clear row */
                   {
                    reduced = TRUE  ;
                    clear_col[j] = TRUE ;
                    clear_row[j] = TRUE ;
                  
                    if (debug) fprintf(stderr, "\nEliminated row %d", j) ;
                  
                     for (i=0; i<vlinks_per_node; i++ )
                       if (CDG_LIST(j,i) != no_arc)
                         CDG_LIST(j,i) = rm_arc ;
                       else
                         break ;
                  }
                  /*}}}*/
              }  
              /*}}}*/
          /*}}}*/
          /*{{{  Scan CDG over rows*/
          for (i=0; i<total_vlinks; i++ )
            if (!clear_row[i])
              /*{{{  then check for Clarity */
              {
                BOOL clear = TRUE ;
                
                for ( j=0; j<total_vlinks; j++ )
                  if (CDG_LIST(i,j) == no_arc)
                    break ;
                  else
                    if (CDG_LIST(i,j) != rm_arc)
                    {
                      clear = FALSE ;
                      break ;
                    }
              
                if (clear)
                  /*{{{  then clear column */
                  {
                     reduced = TRUE ;
                     clear_row[i] = TRUE ;
                     clear_col[i] = TRUE ;
                     
                     if (debug) fprintf(stderr, "\nEliminated column %d\n",i) ;
                     
                     for (j=0; j<total_vlinks; j++ )
                     {
                      int *start  = &CDG_LIST(j,0) ;
                      int *finish = start+vlinks_per_node ;
                      int *dep ;
                      
                      for (dep=start;(dep<finish) && (*dep != no_arc);dep++)
                        if (*dep == i)
                          *dep = rm_arc ;
                     }
                  }
                  /*}}}*/
              }
              /*}}}*/
          /*}}}*/
        } 
                  
         /*{{{  if Non Zero Matrix then cycle exists*/
         CDG_cycle = FALSE ;
         for ( i=0; (i<total_vlinks); i++)
         {  if (! clear_col[i]) CDG_cycle = TRUE ;
            if (! clear_row[i]) CDG_cycle = TRUE ;
         } ;
         /*}}}*/
      }
      /*}}}*/
      /*{{{  check for deadlock*/
      if (CDG_cycle)
      {
        int i, count = 0 ;
        
        fprintf(stderr, "\nError-%s(%d)- NCF network is deadlocked!\n",__FILE__,__LINE__);
      
        for (i=0;i<total_vlinks;i++)
          if (!clear_row[i])
            count++ ;
      
        if (!silent) fprintf(stderr, "%d virtual links involved in cycles\n", count) ;
      }
      else
      {
        if (!silent)
          fprintf(stderr, "\nNCF network is deadlock-free!\n") ;
      }        
      /*}}}*/
      /*{{{  if (debug && CDG_cycle) display CDG*/
      if (debug && CDG_cycle)
      {
        int vlink_id, i ;
      
        fprintf(stderr, "\nChannel dependencies giving CDG cycles\n") ;
      
        for (vlink_id=0;vlink_id<total_vlinks;vlink_id++)
        {
          BOOL first = TRUE ;
      
          for (i=0;i<vlinks_per_node;i++)
          {
            int vid = CDG_LIST(vlink_id,i) ;
      
            if ((vid != no_arc) & (vid != rm_arc))
              /*{{{  display dependency*/
              {
                if (first)
                {
                  fprintf(stderr, "(%d,%d,%d) depends on", PROC(vlink_id), LINK(vlink_id), VLINK(vlink_id)) ;
                  first = FALSE ;
                }
              
                fprintf(stderr, "  (%d,%d,%d)", PROC(vid), LINK(vid), VLINK(vid)) ;
              }
              /*}}}*/
          }
      
          if (!first) fprintf(stderr, "\n") ;
         }
      }
      /*}}}*/
    }
  }  
  
  /*{{{  finished*/
  if (!silent)
  fprintf(stderr, "Done.\n") ;
  
  if (CDG_cycle)
    return(-1) ;
  else
    return(0) ;
  /*}}}*/
}
