/*{{{  File banner*/
/*@(#)=====================================================*\
||@(#)  Project : PUMA ESPRIT P2701
||@(#)  Authors : Mark Debbage and Mark Hill
||@(#)            University of Southampton
||  
||@(#)    Title : ncf to inc file convertor (for static loading)
||@(#)   System : SLOADER
||@(#) Filename : ncf2inc.c
||@(#)  Version : 2.3
||@(#)     Date : 7/15/92
\*@(#)====================================================*/
/*}}}*/

/*{{{  includes*/
#include <stdio.h>
#include <string.h>
#include <ctype.h>

#ifdef _ICC
#include <stdlib.h>
#endif
/*}}}*/
/*{{{  constant definitions*/
typedef int BOOL;

#define MAX_LINE     100
#define MAX_PART     10

#define ndirection  4             /* no of transputer links */

#define not_done (-1)
#define not_used (-1)
#define not_link (-3)

#define TRUE  (1)
#define FALSE (0)
#define kilobyte (1024)

#define MAX(a,b) ((a)>(b) ? (a) : (b))
/*}}}*/

/*{{{  typedef LINK_TYPES*/
typedef enum {invalid, inactive, host, virtual} LINK_TYPES ;
/*}}}*/
/*{{{  struct NET*/
typedef struct
{
  int con_proc[ndirection] ; /* Neighbouring processor */
  int con_link[ndirection] ; /* Neighbouring processor's link */
  int boot ;                 /* Boot link */
  char part[MAX_PART];       /* Part Name */
  int memory;                /* Memory in K */
}
NET ;
/*}}}*/

/*{{{  network description statics*/
static NET *node ;
static int node_cnt    = 0;
static int root        = not_done;
static int diameter ;
static int debug       = FALSE ;
static int silent      = FALSE ;
static int cmode       = FALSE ;
static int tpg         = not_done ;
/*}}}*/

int  linenum = 0 ;
/*{{{  string parsing routines*/
/*{{{  char *skip_non_blanks (char *s)*/
char *skip_non_blanks (char *s)
{
  while ((*s != 0) && (!isspace(*s)))
    s++ ;
  return(s) ;
}

/*}}}*/
/*{{{  char *skip_blanks (char *s)*/
char *skip_blanks (char *s)
{
  while ((*s != 0) && (isspace(*s)))
    s++ ;
  return(s) ;
}

/*}}}*/
/*{{{  char *skip_digits (char *s)*/
char *skip_digits (char *s)
{
  while ((*s != 0) && (isdigit(*s)))
    s++ ;
  return(s) ;
}

/*}}}*/
/*}}}*/
/*{{{  pragma matching*/
#define num_topologies 1
#define grid 0
char topology_text[num_topologies][5] = { "grid" } ;

/*{{{  void extract_topology(char *s)*/
void extract_topology(char *s)
{
  int i ;

  s = skip_blanks(s) ;
  
  for (i=0;i<num_topologies;i++)
    if (strncmp(s,topology_text[i],strlen(topology_text[i])) == 0)
    {
      if (debug)
        fprintf(stderr,"Topology is %s\n",topology_text[i]) ;
      if (cmode)
        fprintf(stdout, "#define NETWORK_TPG %s_tpg\n",topology_text[i]) ;
      tpg = i ;
      break ;
    }      

  switch(i)
  {
    case grid :
       /*{{{  read grid information*/
       {
         int dims, d, num ;
       
         /*{{{  ndims*/
         s = skip_non_blanks(s) ;
         s = skip_blanks(s) ;
         
         if (sscanf(s,"%d",&dims) != 1)
         {
           fprintf(stderr, "Error-%s(%d)- Parse failed reading dimensions on line %d\n",__FILE__,__LINE__,linenum) ;
           exit(-1) ;
         }
         
         if (dims<0)
         {
           fprintf(stderr, "Error-%s(%d)- Parse failed, negative dimensions on line %d\n",__FILE__,__LINE__,linenum) ;
           exit(-1) ;
         }
         
         if (cmode)
           fprintf(stdout, "PRIVATE int grid_ndims = %d ;\n",dims) ;
         /*}}}*/
         /*{{{  dims*/
         s = skip_non_blanks(s) ;
         s = skip_blanks(s) ;
         
         if (dims == 0)
         {
           if (cmode) fprintf(stdout, "PRIVATE int *grid_dims = NULL ;") ;
         }
         else
         {
           if (cmode) fprintf(stdout, "PRIVATE int grid_dims[%d] = {",dims) ;
         
           if (debug) fprintf(stderr,"Dimensions : ") ;
           for (d=0;d<dims;d++)
           {
             if (sscanf(s,"%d",&num) != 1)
             {
               fprintf(stderr, "Error-%s(%d)- Parse failed reading a dimension on line %d\n",__FILE__,__LINE__,linenum) ;
               exit(-1) ;
             }
           
             if (cmode) fprintf(stdout, "%d", num) ;
           
             if (d!=dims-1)
               if (cmode) fprintf(stdout, ",") ;
             
             if (debug) fprintf(stderr,"%d ",num) ;
             s = skip_non_blanks(s) ;
             s = skip_blanks(s) ;
           }
         
           if (cmode) fprintf(stdout,"} ;\n") ;
           
           if (debug) fprintf(stderr,"\n") ;
         }
         
         /*}}}*/
         /*{{{  wraps*/
         if (dims == 0)
         {
           if (cmode) fprintf(stdout, "PRIVATE int *grid_wraps = NULL ;") ;
         }
         else
         {
           if (cmode) fprintf(stdout, "PRIVATE int grid_wraps[%d] = {",dims) ;
           
           if (debug) fprintf(stderr,"Wrap flags : ") ;
           for (d=0;d<dims;d++)
           {
             if (sscanf(s,"%d",&num) != 1)
             {
               fprintf(stderr, "Error-%s(%d)- Parse failed reading a wrap flag on line %d\n",__FILE__,__LINE__,linenum) ;
               exit(-1) ;
             }
         
             if (cmode) fprintf(stdout, "%d", num) ;
           
             if (d!=dims-1)
               if (cmode) fprintf(stdout, ",") ;
             
             if (debug) fprintf(stderr,"%d ",num) ;
         
             s = skip_non_blanks(s) ;
             s = skip_blanks(s) ;
           }
         
           if (cmode) fprintf(stdout,"} ;\n") ;
         
           if (debug) fprintf(stderr,"\n") ;
         }
         /*}}}*/
         /*{{{  first*/
         if (sscanf(s,"%d",&num) != 1)
         {
           fprintf(stderr, "Error-%s(%d)- Parse failed reading first id on line %d\n",__FILE__,__LINE__,linenum) ;
           exit(-1) ;
         }
         if (debug) fprintf(stderr,"First id is %d\n",num) ;
         s = skip_non_blanks(s) ;
         s = skip_blanks(s) ;
         
         if (cmode) fprintf(stdout, "PRIVATE int grid_first = %d ;\n",num) ;
         /*}}}*/
         
         break ;
       }
       /*}}}*/
    default :
      fprintf(stderr, "Warning-%s(%d)- Unrecognised topology (%s) on line %d\n",__FILE__,__LINE__,s, linenum) ;
  }
}
/*}}}*/

#define num_pragmas 1
#define topology 0
char pragma_text[num_pragmas][10] = { "#TOPOLOGY" } ;

/*{{{  void identify_pragma (char *s)*/
void identify_pragma (char *s)
{
  int i ;

  for (i=0;i<num_pragmas;i++)
    if (strncmp(s,pragma_text[i],strlen(pragma_text[i])) == 0)
    {
      if (debug) fprintf(stderr,"Found %s pragma\n",pragma_text[i]) ;
      break ;
    }      

  switch(i)
  {
    case topology :
      extract_topology(s+strlen(pragma_text[i])) ;
      break ;
  }
}
/*}}}*/
/*}}}*/
/*{{{  file reading primitives*/
#define buflen 160
char linebuf[buflen];
char *linepos = linebuf ;
char *strbuf = NULL ;

/*{{{  char *fgetstring (char *line, int n, FILE *file)*/
/*
   This routine reads a line, strips out # comments and strips
   out lines of just white space. Usage much like fgets.
*/

char *fgetstring (char *line, int n, FILE *file)
{
  char *s ;

  do
  { 
    if ((s=fgets (line, n, file)) == NULL)
      return(NULL) ;
       
    if (!feof(file) && (s[0] != 0))
    {
      char *hash = strchr (s, '#') ;
      int i ;
       
      if ((strlen(s) == n-1) && (s[n-1] != '\n'))
        fprintf(stderr, "Warning-%s(%d)- Line %d too long so wrapped round\n",__FILE__,__LINE__,linenum) ;
      else
        linenum++ ;

      if (hash == s)
        identify_pragma (hash) ;
       
      if (hash != NULL)
        *hash = 0 ;
       
      for (i=0;i<strlen(s);i++)
        if (!isspace(s[i]))
          break ;
       
      if (i==strlen(s))
        s[0] = 0 ;    
    }
  } while (*s == 0) ;

  return(s) ;
}
/*}}}*/
  
/*{{{  int   read_number (FILE *file)*/
int   read_number (FILE *file)
{
  int i ;

  linepos = skip_blanks(linepos) ;
  
  if (*linepos == 0)
  {
    if (fgetstring(linebuf, buflen, file) == NULL)
    {
      fprintf(stderr, "Error-%s(%d)- Parse failed reading number on line %d\n",__FILE__,__LINE__,linenum) ;
      exit(-1) ;
    }
    linepos = linebuf ;
  }

  if (feof(file) || (*linepos == 0))
  {
    fprintf(stderr, "Error-%s(%d)- Parse failed reading number on line %d\n",__FILE__,__LINE__,linenum) ;
    exit(-1) ;
  }

  if (sscanf(linepos,"%d",&i) != 1)
  {
    fprintf(stderr, "Error-%s(%d)- Parse failed reading number on line %d\n",__FILE__,__LINE__,linenum) ;
    exit(-1) ;
  }

  linepos = skip_digits(linepos) ;

  return(i) ;  
}
/*}}}*/
/*{{{  char *read_string (FILE *file)*/
char *read_string (FILE *file)
{
  linepos = skip_blanks(linepos) ;

  if (*linepos == 0)
  {
    if (fgetstring(linebuf, buflen, file) == NULL)
    {
      fprintf(stderr, "Error-%s(%d)- Parse failed reading string on line %d\n",__FILE__,__LINE__,linenum) ;
      exit(-1) ;
    }
    linepos = linebuf ;
  }

  if (feof(file) || (*linepos == 0))
  {
    fprintf(stderr, "Error-%s(%d)- Parse failed reading string on line %d\n",__FILE__,__LINE__,linenum) ;
    exit(-1) ;
  }

  if (strbuf != NULL)
    free(strbuf) ;
    
  strbuf = (char *) malloc(strlen(linepos+1)) ;

  if (sscanf(linepos,"%s",strbuf) != 1)
  {
    fprintf(stderr, "Error-%s(%d)- Parse failed reading string on line %d\n",__FILE__,__LINE__,linenum) ;
    exit(-1) ;
  }

  linepos = skip_non_blanks(linepos) ;

  return(strbuf) ;
}
/*}}}*/
/*{{{  char  read_char   (FILE *file)*/
char  read_char   (FILE *file)
{
  char c ;
  
  if (*linepos == 0)
  {
    if (fgetstring(linebuf, buflen, file) == NULL)
    {
      fprintf(stderr, "Error-%s(%d)- Parse failed reading character on line %d\n",__FILE__,__LINE__,linenum) ;
      exit(-1) ;
    }
    linepos = linebuf ;
  }

  if (feof(file) || (*linepos == 0))
  {
    fprintf(stderr, "Error-%s(%d)- Parse failed reading character on line %d\n",__FILE__,__LINE__,linenum) ;
    exit(-1) ;
  }

  if (sscanf(linepos,"%c",&c) != 1)
  {
    fprintf(stderr, "Error-%s(%d)- Parse failed reading character on line %d\n",__FILE__,__LINE__,linenum) ;
    exit(-1) ;
  }

  linepos++ ;
  
  return(c) ;  
}
/*}}}*/
/*{{{  void  next_field  (FILE *file)*/
void  next_field  (FILE *file)
{
  linepos = skip_blanks(linepos) ;
}
/*}}}*/
/*}}}*/

/*{{{  boot tree data structures*/
#define NOT_PROC (-1)

typedef struct
{
  int son ;
  int sibling ;
} VERTEX ;

VERTEX *tree ;
/*}}}*/

/*{{{  int LoadNCFfile(FILE *net_fp)*/
int LoadNCFfile(FILE *net_fp)
{
  int index;
  
  /*{{{  read node_cnt diameter*/
  node_cnt = read_number(net_fp) ;
  diameter = read_number(net_fp) ;
  
  if (node_cnt <= 0 || diameter < 0)
  {
    fprintf(stderr, "Error-%s(%d)- Invalid NCF header\n",__FILE__,__LINE__);
    return(-1);
  }
  
  if (debug) fprintf(stderr,"Network contains %d processors, diameter %d\n", node_cnt, diameter) ;
  /*}}}*/
  /*{{{  malloc node array*/
  if (((node = (NET *) malloc (node_cnt * sizeof(NET))) == NULL) ||
      ((tree = (VERTEX *) malloc (node_cnt * sizeof(VERTEX))) == NULL))
  {
    fprintf(stderr, "Error-%s(%d)- Malloc failed\n", __FILE__,__LINE__) ;
    return(-1) ;
  }
  /*}}}*/

  for (index=0;index<node_cnt;index++)
  {
    node[index].boot = not_used;
    tree[index].son  = NOT_PROC ;
    tree[index].sibling  = NOT_PROC ;
  }
    
  for (index=0;index<node_cnt;index++)
  {
    int n;
    
    /*{{{  read n*/
    n = read_number(net_fp) ;
    
    if (n<0 || n>=node_cnt || node[n].boot != not_used)
    {
      fprintf(stderr, "Error-%s(%d)- Bad processor id %d\n",__FILE__,__LINE__,n);
      return(-1) ;
    }
    
    if (debug) fprintf(stderr,"PROCESSOR %d\n", n) ;
    
    /*}}}*/
    /*{{{  read part*/
    {
      char *s = read_string (net_fp) ;
    
      if (strlen(s) >= MAX_PART-1)
        fprintf(stderr, "Warning-%s(%d)- Truncating part string\n",__FILE__,__LINE__);
    
      strncpy (node[n].part, s, MAX_PART-1) ;
      node[n].part[MAX_PART-1] = 0 ;
    
      if (debug) fprintf(stderr,"PART:%s\n",node[n].part);
    }
    /*}}}*/
    /*{{{  read memory*/
    {
      int size  = read_number(net_fp) ;
      char unit = read_char (net_fp) ;
    
      if (debug) fprintf(stderr,"Memory:%d%c\n", size,unit) ;
    
      if (unit=='M')
        node[n].memory = kilobyte*size ;
      else
        if (unit=='K')
          node[n].memory = size ;
        else
        {
          fprintf(stderr, "Error-%s(%d)- Bad memory units ('%c')\n",__FILE__,__LINE__,unit);
          return(-1);
        }
    }
    /*}}}*/
    /*{{{  read links*/
    {
      int dir ;
      char type ;
      
      for (dir=0;dir<ndirection;dir++)
      {
        next_field(net_fp) ;
        type = read_char (net_fp) ;
      
        switch ((int) type) 
        {
          case (int) 'h' :
            /*{{{  host link*/
            if (debug) fprintf(stderr,"host ");
            root = n;
            node[n].boot = dir;
            node[n].con_proc[dir] = not_used;
            node[n].con_link[dir] = not_link;
            break;
            /*}}}*/
          case (int) 'i' :
          case (int) 'o' :
          case (int) 'n' :
            /*{{{  used link*/
            {
              if (type == 'i')
              {
                if (node[n].boot == not_used)
                  node[n].boot = dir ;
                else
                {
                  fprintf(stderr, "Error-%s(%d)- Invalid boot tree, two parents\n",__FILE__,__LINE__);
                  return(-1) ;
                }
              }
            
              /*{{{  get used link details*/
              {
                node[n].con_proc[dir] = read_number(net_fp) ;
              
                if (read_char(net_fp) != ':')
                {
                  fprintf(stderr,"Error-%s(%d)- Bad connector format\n",__FILE__,__LINE__) ;
                  return(-1);
                }
                
                node[n].con_link[dir] = read_number(net_fp) ;
              
                if (debug) fprintf(stderr,"virtual(%d:%d) ",node[n].con_proc[dir],node[n].con_link[dir]);
              }
              /*}}}*/
              
              if (type == 'o')
                /*{{{  add son to boot tree*/
                {
                  int *proc = &tree[n].son ;
                
                  if (*proc == NOT_PROC)
                    *proc = node[n].con_proc[dir] ;
                  else
                  {
                    do
                    {
                      proc = &tree[*proc].sibling ;
                    } while (*proc != NOT_PROC) ;
                    *proc = node[n].con_proc[dir] ;
                  }
                }
                /*}}}*/
            
              break ;
            }
            /*}}}*/
          case (int) 'x' :
            /*{{{  unused link*/
            if (debug) fprintf(stderr,"inactive ");
            node[n].con_proc[dir] = not_used;
            node[n].con_link[dir] = not_link;
            break;
            /*}}}*/
          default :
            /*{{{  catch anything else*/
            {
              fprintf(stderr,"Error-%s(%d)- Unknown link type ('%c')\n",__FILE__,__LINE__,type) ;
              return(-1);
            }
            /*}}}*/
        }
      }
      if (debug) fprintf(stderr,"\n");
    }            
    /*}}}*/
  }
  
  if (root==not_used && node[root].boot==not_used)
    {
      fprintf(stderr,"Error-%s(%d)- No host link declared\n",__FILE__,__LINE__);
      return(-1);
    }
}    
/*}}}*/
/*{{{  int TransferRoutes(FILE *in_fp, FILE *out_fp)*/
int TransferRoutes(FILE *in_fp, FILE *out_fp)
{
  int proc;
  
  /*{{{  val routes*/
  fprintf(out_fp,"val routes {\n\"");
  for (proc=0;proc<node_cnt;proc++)
  {
    int finish ;
    int n ;
    
    /*{{{  processor number*/
    n = read_number(in_fp) ;
    
    if (n<0 || n>=node_cnt)
    {
      fprintf(stderr, "Error-%s(%d)- Bad processor id %d\n",__FILE__,__LINE__,n);
      return(-1) ;
    }
    /*}}}*/
    /*{{{  routing tables length*/
    if (read_number(in_fp) != 2*node_cnt)
    {
      fprintf(stderr, "Error-%s(%d)- Expected %d bytes of routing data\n",__FILE__,__LINE__,2*node_cnt);
      return(-1) ;
    }
    /*}}}*/
    
    if (proc!=0) fprintf(out_fp,"\",\n\"");
  
    for (finish=0;finish<node_cnt;finish++)
    {
      /*{{{  logical link*/
      {
        int num = read_number (in_fp) ;
      
        if (num<0 || num>255)
        {
          fprintf(stderr, "Error-%s(%d)- Routing data must be unsigned bytes\n",__FILE__,__LINE__);
          return(-1) ;
        }
          
        fprintf(out_fp,"\\x%02x",num);
      }
      /*}}}*/
      /*{{{  virtual link*/
      {
        int num = read_number (in_fp) ;
      
        if (num<0 || num>255)
        {
          fprintf(stderr, "Error-%s(%d)- Routing data must be unsigned bytes\n",__FILE__,__LINE__);
          return(-1) ;
        }
          
        fprintf(out_fp,"\\x%02x",num);
      }
      /*}}}*/
    }    
  }
  
  fprintf(out_fp,"\"\n};\n");
  /*}}}*/
}  
/*}}}*/
/*{{{  void OutputDotInc(FILE *out_fp)*/
int OutputDotInc(FILE *out_fp)
{
  int proc,link ;
  int sizeC = 0;
  
  if (!silent)
    fprintf(stderr,"\nWriting out .inc file for static configuration\n");

  /*{{{  val NumProcs*/
  fprintf(out_fp, "val NumProcs %d ;\n",node_cnt);
  /*}}}*/
  /*{{{  val Root */
  if (root==not_done)
  {
    fprintf(stderr, "Error-%s(%d)- No Host Connection\n",__FILE__,__LINE__) ;
    return(-1);
  }
  else
    fprintf(out_fp,"val Root %d ;\n",root);
    
  if (root!=0)
    fprintf(stderr,"Warning-%s(%d)- Root is not processor 0\n",__FILE__,__LINE__);
    
  /*}}}*/
  /*{{{  val BootLink*/
  fprintf(out_fp,"val BootLink %d ;\n",node[root].boot);
  /*}}}*/
  /*{{{  val C*/
  {
    fprintf(out_fp,"val C {\n");
  
    for (proc=0;proc<node_cnt;proc++)
      for (link=0;link<ndirection;link++)
      {
        int neighbour=node[proc].con_proc[link];
        
        if (neighbour!=not_used)
          if (neighbour>proc)
          {
            if (sizeC!=0)
              fprintf(out_fp,",\n");
            
            fprintf(out_fp,"{ %d, %d, %d, %d }", proc, link, neighbour,node[proc].con_link[link]);
            sizeC++;
          }
      }
  
    if (sizeC == 0) printf("0") ; /* prevent empty array */
    fprintf(out_fp,"\n} ; \n");
  }
  /*}}}*/
  /*{{{  val sizeC*/
  fprintf(out_fp,"val sizeC %d ;\n",sizeC);
  /*}}}*/
  /*{{{  val link_types*/
  fprintf(out_fp,"val link_types {\n");
  for (proc=0;proc<node_cnt;proc++)
  {
    if (proc!=0) fprintf(out_fp,",\n");
    fprintf(out_fp,"{");
    for (link=0;link<ndirection;link++)
    {
      LINK_TYPES type;
      
      if (node[proc].con_proc[link]==not_used)
        type=inactive;
      else
        if ((proc==root)&&(link==node[root].boot))
          type=host;
        else
          type=virtual;
      if (link!=0) fprintf(out_fp,",");
      fprintf(out_fp," %d",(int) type);
    }
    fprintf(out_fp,"}");
  }
  fprintf(out_fp,"\n} ; \n");
  /*}}}*/
  /*{{{  val memories*/
  fprintf(out_fp,"val memories { ");
  for (proc=0;proc<node_cnt;proc++)
  {
    if (proc!=0) fprintf(out_fp,"K, ");
    fprintf(out_fp,"%d",node[proc].memory);
  }
  fprintf(out_fp,"K };\n");
  /*}}}*/
  /*{{{  val ptypes*/
  fprintf(out_fp,"val ptypes { ");
  for (proc=0;proc<node_cnt;proc++)
  {
    if (proc!=0) fprintf(out_fp,", ");
    fprintf(out_fp,"\"%.4s\"",node[proc].part);
  }
  fprintf(out_fp," };\n");
  /*}}}*/
}
/*}}}*/
/*{{{  int OutputNetworkInfo(FILE *out_fp)*/
int OutputNetworkInfo(FILE *out_fp)
{
  int i, j ;

  if (tpg == not_done && cmode)
    fprintf(out_fp,"#define NETWORK_TPG unknown_tpg\n") ;
    
  /*{{{  edges*/
  {
    int connects = 0 ;
    int c = 0 ;
  
    for (i=0;i<node_cnt;i++)
      for (j=0;j<ndirection;j++)
        if (node[i].con_proc[j] != not_used &&  i < node[i].con_proc[j])
          connects++ ;
    
    fprintf(stdout, "PRIVATE int network_num_edges = %d ;\n",connects) ;
    
    if (connects == 0)
      fprintf(stdout, "PRIVATE EDGE *network_edge = NULL ;\n") ;
    else
    {
      fprintf(stdout, "PRIVATE EDGE network_edge[%d] = \n", MAX(1,connects)) ;
      fprintf(stdout, "  {") ;
  
      for (i=0;i<node_cnt;i++)
        for (j=0;j<ndirection;j++)
          if (node[i].con_proc[j] != not_used &&  i < node[i].con_proc[j])
          {
            fprintf(stdout,"{%d,%d}",i,node[i].con_proc[j]) ;
            c++ ;
            if (c!=connects)
              fprintf(stdout,",") ;
            if (c%8 == 7)
              fprintf(stdout,"\n   ") ;
          }
  
      fprintf(stdout, "} ;\n") ;
    }
  }
  /*}}}*/
  /*{{{  vertices*/
  fprintf(stdout, "PRIVATE int network_num_vertices = %d ;\n",node_cnt) ;
  
  fprintf(stdout, "PRIVATE VERTEX network_vertex[%d] = \n",node_cnt) ;
  
  fprintf(stdout, "  {") ;
  
  for (i=0;i<node_cnt;i++)
  {
    fprintf(stdout,"{%d,%d}",tree[i].son,tree[i].sibling) ;
    if (i!=node_cnt-1)
      fprintf(stdout,",") ;
    if (i%8 == 7)
      fprintf(stdout,"\n   ") ;
  }
  
  fprintf(stdout, "} ;\n") ;
  /*}}}*/
}
/*}}}*/
  
int main(int argc, char *argv[])
{
  /*{{{  force buffered stdio for icc*/
  #ifdef _ICC
  char bufin[BUFSIZ], bufout[BUFSIZ] ;
  setbuf(stdin,  bufin) ;
  setbuf(stdout, bufout) ;
  #endif
  /*}}}*/
  /*{{{  parse CLI parameters*/
  {
    int p ;
  
    for (p=1;p<argc;p++)
    {
      if ((argv[p][0] == '-') || (argv[p][0] == '/'))
      {
        char c = argv[p][1] ;
  
        if (isalpha(c))
        {
          char option = (isupper(c) ? tolower(c) : c) ;
  
          if (option == 'd')
            debug = TRUE ;
          else
            if (option == 's')
              silent = TRUE ;
            else
              if (option == 'c')
                cmode = TRUE ;
        }
      }
    }
  }      
  
  /*}}}*/
  /*{{{  title*/
  if (!silent)
  {
    fprintf(stderr, "NETWORK CONFIGURATION FILE TO STATIC INCLUDE FILE CONVERTER\n");
    fprintf(stderr, "Mark Debbage and Mark B. Hill   Jan 1991 V1.0\n");
    fprintf(stderr, "University Of Southampton, ESPRIT P2701 PUMA\n") ;
  }
  /*}}}*/
  
  if (LoadNCFfile(stdin) < 0)
  {
    fprintf(stderr, "Error-%s(%d)- .ncf file read failed\n", __FILE__,__LINE__) ;
    return(-1) ;
  }

  if (cmode)
  {
    if (OutputNetworkInfo(stdout)<0)
    {
      fprintf(stderr, "Error-%s(%d)- .inc write failed\n", __FILE__,__LINE__) ;
      return(-1) ;
    }
  }
  else
  {
    if (OutputDotInc(stdout)<0)
    {
      fprintf(stderr, "Error-%s(%d)- .inc write failed\n", __FILE__,__LINE__) ;
      return(-1) ;
    }

    if (TransferRoutes(stdin,stdout)<0)
    {
      fprintf(stderr,"Error-%s(%d)- Route transfer failed\n",__FILE__,__LINE__);
      return(-1);
    }
  }
  
  /*{{{  finished*/
  if (!silent)
    fprintf(stderr, "Done.\n") ;
  
  return(0) ;
  /*}}}*/
}
