/*{{{  File banner*/
/*@(#)=====================================================*\
||@(#)  Project : PUMA ESPRIT P2701
||@(#)  Authors : Mark Debbage and Mark Hill
||@(#)            University of Southampton
||  
||@(#)    Title : Routing kernel
||@(#)   System : UPR
||@(#) Filename : kernel.c
||@(#)  Version : 2.13
\*@(#)====================================================*/
/*}}}*/

static char *_FILE_ = __FILE__ ;
static char *OOM = "Out of Memory";
/*{{{  includes*/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <misc.h>

#include "router.h"
#include "except.h"
#include "uprctrl.h"
#include "input.h"
#include "output.h"
#include "show.h"
#include "pktio.h"
/*}}}*/

/*{{{  define input and output workspace and code requirements*/
/* For secure mode, define (oversized) workspace requirements in words */
#define secure_input_ws_size  200
#define secure_output_ws_size 200

/* For fast mode, define workspace requirements (in words) and code
   size (in bytes) for FAST RAM copy */

#define fast_input_ws_size 120
#define fast_output_ws_size 35
#define fast_input_code_size 450
#define fast_output_code_size 420
/*}}}*/

/*{{{  declare statics (for fast RAM hopefully)*/
#ifndef NO_GLOBALS_PTR
GLOBALS globals ;
#endif

/* Note always declare statics even if not actually used */
BYTE fast_output_code[fast_output_code_size] ;
BYTE fast_input_code [fast_input_code_size] ;
int  fast_output_ws[UPR_num_links][fast_output_ws_size] ;
int  fast_input_ws [UPR_num_links][fast_input_ws_size] ;

UPR_GLOBALS upr_globals ;

#ifndef UPR_RD
static BOOL debug = FALSE ;
#endif
/*}}}*/

/*{{{  PRIVATE void init_olb*/
PRIVATE void init_olb(OLB *newolb)
{
  int j;
  int num_vout = newolb->num_vlinks ;

  newolb->active = TRUE ;
  newolb->lack_OQB.h = (ORB *) NULL ;
  newolb->lack_OQB.t = (ORB *) NULL ;
  newolb->wdesc = NULL ;

  if (num_vout != 0)
  {
    if ((newolb->volb = (VOLB *) malloc (num_vout*sizeof (VOLB))) == NULL)
      Exception (UPR_fatal, _FILE_, __LINE__, "Out of heap space - too many virtual links") ;
  
    for (j=0;j<num_vout;j++)
    {
      (newolb->volb)[j].pkt_count = 0;
      (newolb->volb)[j].oqb.h = (ORB *) NULL ;
      (newolb->volb)[j].oqb.t = (ORB *) NULL ;
    }
  }
}
/*}}}*/
/*{{{  PRIVATE void init_ilb*/
PRIVATE void init_ilb(int id,ILB *newilb, int num_buffers)
{
  int j ;
  int num_vin = newilb->num_vlinks ;

  if (num_vin==0)
  {
    newilb->pkt_limit  = 0 ;
    newilb->srv_limit  = 0 ;
  }
  else
  {
    newilb->pkt_limit  = num_buffers / num_vin ;
    newilb->srv_limit  = max(1+(num_buffers / (2*num_vin)),
                             (num_buffers/num_vin) - 2) ;

    if ((newilb->vilb = (VILB *) malloc (num_vin*sizeof (VILB))) == NULL)
      Exception (UPR_fatal, _FILE_, __LINE__, "Out of heap space - too many virtual links") ;
  }

  for (j=0;j<num_vin;j++)
  {
    (newilb->vilb)[j].pkt_count = 0;
    (newilb->vilb)[j].srv_count = 0;
    
    (newilb->vilb)[j].lack_ORB.list = (ORB *) InputServiced_p;
    (newilb->vilb)[j].lack_ORB.src[ll] = UPR_USER ;
    (newilb->vilb)[j].lack_ORB.src[vl] = UPR_USER ;
    (newilb->vilb)[j].lack_ORB.bffr = (BYTE *) NULL ;
    HdrBuild(id,j,0,&((newilb->vilb)[j].lack_ORB.header));
    (newilb->vilb)[j].lack_ORB.wdesc = NULL ;
  }  
} 
/*}}}*/
/*{{{  PRIVATE void init_trbb*/
PRIVATE void init_trbb(TRBB *trbb)
{
  trbb->orb.list    = (ORB *) InputServiced_p;
  trbb->orb.src[ll] = UPR_USER;
  trbb->orb.src[vl] = UPR_USER;
  trbb->orb.bffr    = trbb->buffer;
  trbb->orb.wdesc   = (WDESC) NULL;
}
/*}}}*/

/*{{{  PRIVATE void init_olbs*/
PRIVATE void init_olbs ( Process *p,
                         Channel **route_in,
                         BYTE    num_out_vlink[UPR_num_links] )
{
  UPR_GLOBALS *g = &upr_globals ;
  BOOL guard[UPR_num_links] ;
  int  link, count=0 ;
  p=p ;

  for (link=0;link<UPR_num_links;link++)
    if (num_out_vlink[link] == UPR_INACTIVE)
    {
      g->olbs[link].used = FALSE;
      guard[link] = FALSE ;
    }
    else
    {
      g->olbs[link].used  = TRUE ;
      guard[link] = TRUE ;
      count++ ;
    }

  while (count != 0)
  {
    link = ProcGuardedAltList ( route_in, guard ) ;

    #ifndef UPR_RD
    if (debug)
      printf("Received from link number %d at %x\n", link, (int) route_in[link]) ;
    #endif

    g->olbs[link].num_vlinks = ChanInInt(route_in[link]) ;
    g->olbs[link].pkt_limit  = ChanInInt(route_in[link]) ;
    g->olbs[link].srv_limit  = ChanInInt(route_in[link]) ;

    if (g->olbs[link].num_vlinks != num_out_vlink[link])
      Exception(UPR_fatal,_FILE_,__LINE__,"Virtual link numbers mismatch");

    init_olb(&(g->olbs[link])) ;

    guard[link] = FALSE ;
    count-- ;
  }
}
/*}}}*/
/*{{{  PRIVATE void init_ilbs*/
PRIVATE void init_ilbs( Process *p,
                        Channel **route_out,
                        BYTE    num_in_vlink[UPR_num_links],
                        int     num_buffers_per_link[UPR_num_links])
{
  UPR_GLOBALS *g = &upr_globals ;
  int i ;
  p=p;

  for (i=0;i<UPR_num_links;i++)
  {
    if (num_in_vlink[i]==UPR_INACTIVE)
      g->ilbs[i].used = FALSE;
    else
      {
        g->ilbs[i].used = TRUE ;
        g->ilbs[i].num_vlinks = num_in_vlink[i] ;
        
        init_ilb(g->proc_id, &(g->ilbs[i]), num_buffers_per_link[i]);

        #ifndef UPR_RD
        if (debug)
          printf("Sending to link number %d at %x\n", i, (int) route_out[i]) ;
        #endif

        ChanOutInt(route_out[i],g->ilbs[i].num_vlinks) ;
        ChanOutInt(route_out[i],g->ilbs[i].pkt_limit) ;
        ChanOutInt(route_out[i],g->ilbs[i].srv_limit) ;
      }
  }   
}
/*}}}*/

/*{{{  PRIVATE SendArgs(Channel *c, int argc, char *argv[]);*/
PRIVATE void SendArgs(Channel *c, int argc, char *argv[])
{ int i;

  ChanOutInt(c,argc);
  for (i=0;i<argc;i++)
  {
    int size = strlen(argv[i])+1;
    ChanOutInt(c,size);
    ChanOut(c,argv[i],size);
  }

}
/*}}}*/
/*{{{  PRIVATE void output_starter */
PRIVATE void output_starter ( Process    *p,
                              Channel    **route_out,
                              LINK_TYPES link_type[UPR_num_links],
                              VLINK      *out_link,
                              Semaphore  *lock,
                              int        *secure,
                              int        *argc,
                              char       **argv[])
{
  int link, dst ;
  p=p ;

  SemWait (lock) ;

  for (link=0;link<UPR_num_links;link++)
    if (link_type[link] == virtual)
    {
      #ifndef UPR_RD
      if (debug)
        printf("Sending on link %x\n", (int) route_out[link]) ;
      #endif

      ChanOutInt(route_out[link], *secure) ;
      SendArgs(route_out[link],*argc,*argv);

      for (dst=0;dst<upr_globals.num_nodes;dst++)
        if (out_link[dst][ll] == link)
        {
          ChanOutInt(route_out[link], dst) ;
          ChanOutInt(route_out[link], out_link[dst][vl]) ;
        }

      ChanOutInt(route_out[link], upr_globals.proc_id) ;
      ChanOutInt(route_out[link], UPR_LACK) ;
    }
    else if (link_type[link] == hybrid)
    {
      #ifndef UPR_RD
      if (debug)
        printf("Sending on link %x\n", (int) route_out[link]) ;
      #endif
      ChanOutInt(route_out[link], *secure) ;
      SendArgs(route_out[link],*argc,*argv);
    }  
}
/*}}}*/
/*{{{  PRIVATE void input_starter*/
PRIVATE void input_starter ( Process    *p,
                             Channel    **route_in,
                             LINK_TYPES link_type[UPR_num_links],
                             BYTE       (*in_vlink)[UPR_num_links],
                             Semaphore  *lock,
                             int        *secure,
                             int        *argc,
                             char       **argv[] )
{
  BOOL guard[UPR_num_links] ;
  int  link, proc, count=0, n,m ;
  BYTE vlink ;
  p=p ;

  for (link=0;link<UPR_num_links;link++)
    switch (link_type[link])
    {
      case hybrid :
        for (proc=0;proc<upr_globals.num_nodes;proc++)
          in_vlink[proc][link] = 0 ;
      case virtual :
        guard[link] = TRUE ;
        count++ ;
        break;
      case hprlink :
        for (proc=0;proc<upr_globals.num_nodes;proc++)
          in_vlink[proc][link] = 0 ;
      default :
        guard[link] = FALSE ;
    }
  
  while (count != 0)
  {
    link = ProcGuardedAltList ( route_in, guard ) ;

    #ifndef UPR_RD
    if (debug)
      printf("Receiving on number %d, link %x\n", link, (int) route_in[link]) ;
    #endif

    n = ChanInInt(route_in[link]) ;
    m = ChanInInt(route_in[link]) ;

    if (*secure == -1)
      /*{{{  assert security*/
      {
        *secure = n ;
        if ((*argc)!=-1) Exception(UPR_fatal,_FILE_,__LINE__,"Security/argc inconsistency");
        *argc=m;
        if ((*argv = (char**)malloc((*argc)*sizeof(char*)))==NULL)
          Exception(UPR_fatal,_FILE_,__LINE__,OOM);
        for  (m=0;m<*argc;m++)
        {
          int size = ChanInInt(route_in[link]);
          if (( (*argv)[m] = (char*)malloc(size))==NULL)
            Exception(UPR_fatal,_FILE_,__LINE__,OOM);
          ChanIn(route_in[link],(*argv)[m],size);
        }
        SemSignal (lock) ;
      }
      /*}}}*/
    else
      /*{{{  check security consistant*/
      {
        if ((*secure != n)||(*argc!=m)) 
          Exception(UPR_fatal,_FILE_,__LINE__,"Security/argc disagreement");
        for (m=0;m<*argc;m++)
        {
          n = ChanInInt(route_in[link]);
          ChanIn(route_in[link],(*argv)[m],n);
        }
      }
      /*}}}*/

    if (link_type[link]==virtual)
    {
      vlink = UPR_INACTIVE ;
      while (vlink != UPR_LACK)
      {
        proc  = ChanInInt(route_in[link]) ;
        vlink = (BYTE) ChanInInt(route_in[link]) ;

        in_vlink[proc][link] = vlink ;
      }
    }
    guard[link] = FALSE ;
    count-- ;
  }  
}
/*}}}*/

/*{{{  PUBLIC void UPR_Kernel*/
PUBLIC void UPR_Kernel
(
int        argc,
char       *argv[],
int        proc_id,
int        num_nodes,
Channel    **route_in,
Channel    **route_out,
LINK_TYPES link_type[UPR_num_links],
BYTE       *routing_table )

{
  VLINK   *out_link = (VLINK *) routing_table ; /* convert routing tables to VLINK type */
  BYTE    (*in_vlink)[UPR_num_links] ;     /* incoming virtual link for each destn and each incoming link */
  BYTE    num_in_vlink [UPR_num_links] ;   /* number of incoming virtual links per physical link */
  BYTE    num_out_vlink[UPR_num_links] ;   /* number of outgoing virtual links per physical link */
  BOOL    df_link[UPR_num_links] ;         /* set if physical link has no virtual links */
  int     num_active_links ;
  TRBB    *trbbs;
  PORT    **ports;
  int  params = 1 ;
  int  secure = -1 ;
  int  uargc = -1;
  char **uargv;
  
  BOOL hpr_router ;

  /*{{{  declare link handler processes*/
  #ifdef UPR_USE_FAST_RAM
  Process input_p [UPR_num_links] ;
  Process output_p[UPR_num_links] ;
  #endif
  /*}}}*/

  /*{{{  initialise globals pointer and structure if not visible*/
  #ifndef NO_GLOBALS_PTR
  {
    int i ;
  
    *((GLOBALS **) GLOBAL_PTR) = &globals ;
  
    for (i=0;i<sizeof(GLOBALS)/sizeof(void *);i++)
      ((void **) &globals)[i] = (void *) NULL ;
  
    globals.upr_gp = &upr_globals ;
  }    
  #endif
  
  /*}}}*/
  /*{{{  initialise basic upr globals */
  upr_globals.proc_id       = proc_id ;
  upr_globals.num_nodes     = num_nodes ;   
  upr_globals.out_link      = out_link;
  upr_globals.error_message = NULL ;
  upr_globals.error_wdesc   = NotProcess_p ;
  upr_globals.thro_count    = 0 ;
  upr_globals.gsb           = (void *) *(&(argc)-1) ;
  upr_globals.lact          = NULL ;
  upr_globals.network       = network ;
      
  UPR_InitExceptions() ;
  
  /*}}}*/
  
  /* Exceptions now valid : fatals and errors halt, warnings are delayed */

  /*{{{  check for routing kernel options*/
  #ifndef UPR_RD
  {
    int p;
    secure = TRUE ;
    
    for (p=params;p<argc;p++)
    {
      if (argv[p][0] == switch_char)
        if ((argv[p][1] == 'k') || (argv[p][1] == 'K'))
        {
          if ((argv[p][2] == 'd') || (argv[p][1] == 'D'))
          {
            int q;
            for (q=p;q>params;q--)
              argv[q] = argv[q-1];
            params++;
            debug = TRUE ;
          }
          else if ((argv[p][2] == 'f') || (argv[p][1] == 'F'))
          {
            int q;
            for (q=p;q>params;q--)
              argv[q] = argv[q-1];
            params++;
            secure = FALSE ;
          }
          else
            Exception(UPR_error,_FILE_,__LINE__,"Unknown command-line option to routing kernel") ;
        }
    }
    params--;
    uargc = argc-params;
    uargv = argv+params;
  }
  #endif
  /*}}}*/
  /*{{{  globals info*/
  #ifndef UPR_RD
  if (debug)
    printf("UPR Globals are at %x to %x\n", (int) &upr_globals,
            ((int) &upr_globals)+sizeof(upr_globals)) ;
  #endif
  /*}}}*/
  /*{{{  check compile time constants*/
  if (UPR_MAX_PKT_SIZE != sizeof(UPR_CTRL_PKT))
    Exception(UPR_fatal, _FILE_, __LINE__, "Compile time constant error - control packet missized") ;
  
  if (num_nodes > HDR_MAX_DESTN)
    Exception(UPR_fatal, _FILE_, __LINE__, "Network too large for header strategy - adjust header.h and recompile") ;
  /*}}}*/
  /*{{{  malloc and clear in_vlink*/
  {
    int n, l ;
    
    if ((in_vlink = (BYTE (*)[UPR_num_links]) malloc (UPR_num_links*num_nodes*sizeof (BYTE))) == NULL)
      Exception (UPR_fatal, _FILE_, __LINE__, "Out of heap space - too many nodes") ;
  
    for (n=0;n<num_nodes;n++)
      for (l=0;l<UPR_num_links;l++)
        in_vlink[n][l] = UPR_INVALID_SRC ;
  
    upr_globals.in_vlink = in_vlink;    
  }
  /*}}}*/
  /*{{{  swap virtual link numbers over active edges*/
  {
    Process *ips, *ops ;
    Semaphore start_lock ;
  
    #ifdef UPR_RD
    SemInit (&start_lock, 0) ;  /* blocked */
    #else
    SemInit (&start_lock, 1) ;  /* unblocked */
    #endif
  
    if ((ops=ProcAlloc(output_starter,0,7,route_out,link_type,out_link,&start_lock,&secure,&uargc,&uargv))==NULL)
      Exception(UPR_fatal,_FILE_,__LINE__,"Alloc for output starter failed");
  
    if ((ips=ProcAlloc(input_starter,0,7,route_in,link_type,in_vlink,&start_lock,&secure,&uargc,&uargv))==NULL)
      Exception(UPR_fatal,_FILE_,__LINE__,"Alloc for input starter failed");
  
    ProcPar (ops, ips, NULL) ;
  
    ProcAllocClean(ips) ;
    ProcAllocClean(ops) ;
  }
  /*}}}*/
  /*{{{  reset processor if my logical link is 255*/
  if (out_link[proc_id][ll]==0xff)
  {
    #ifndef UPR_RD
    exit_terminate(EXIT_SUCCESS) ;
    #endif
    TpReset() ;
  }
  /*}}}*/
  /*{{{  calculate link number bounds*/
  {
    int  link, destn ;
    BYTE num ;
  
    /*{{{  calculate num_in_vlink array*/
    for (link=0;link<UPR_num_links;link++)
    {
      BOOL active = FALSE ;
      
      num=0 ;
      for(destn=0;destn<num_nodes;destn++)
      {
        BYTE vlink = in_vlink[destn][link] ;
        if ((vlink != UPR_LACK) && (vlink != UPR_INVALID_SRC))
        {
          active = TRUE ;
          num = max(num, vlink) ;
        }
      }
    
      if ((link_type[link]==hprlink)||(link_type[link]==hybrid))
      {
        if (num != 0)
          Exception(UPR_fatal,_FILE_,__LINE__,"Only allowed 1 virtual link per HPR link") ;
        else
          active = TRUE ;
      }
    
      if (active)
        num_in_vlink[link] = num+1 ;
      else
        num_in_vlink[link] = UPR_INACTIVE ;
    }
    /*}}}*/
    /*{{{  calculate num_out_vlink array*/
    for (link=0;link<UPR_num_links;link++)
    {
      BOOL active = FALSE ;
      
      num=0 ;
      for(destn=0;destn<num_nodes;destn++)
        if (out_link[destn][ll] == link)
        {
          active = TRUE ;
          num = max(num, out_link[destn][vl]) ;
        }
    
      if ((link_type[link]==hprlink)||(link_type[link]==hybrid))
      {
        if (num != 0)
          Exception(UPR_fatal,_FILE_,__LINE__,"Only allowed 1 virtual link per HPR link") ;
        else
          active = TRUE ;
      }
        
      if (active)
        num_out_vlink[link] = num + 1;
      else
        num_out_vlink[link] = UPR_INACTIVE ;
    }
    /*}}}*/
    /*{{{  ensure handlers are active in pairs and assert df_link*/
    num_active_links = 0 ;
    hpr_router = FALSE ;
    
    for (link=0;link<UPR_num_links;link++)
    {
      if ((link_type[link] == hprlink)||(link_type[link]==hybrid))
        hpr_router = TRUE ;
        
      if ((num_in_vlink[link]==1) && (num_out_vlink[link]==1))
        df_link[link] = TRUE ;
      else
        df_link[link] = FALSE ;
        
      if ((num_out_vlink[link] != UPR_INACTIVE) &&
          (num_in_vlink[link]  == UPR_INACTIVE))
        num_in_vlink[link]  = 0 ;
    
      if ((num_out_vlink[link] == UPR_INACTIVE) &&
          (num_in_vlink[link]  != UPR_INACTIVE))
        num_out_vlink[link] = 0 ;
    
      if ((num_in_vlink[link] !=  UPR_INACTIVE) &&
          (num_in_vlink[link] != 0))
        num_active_links++ ;
    }
    
    /*}}}*/
  }
  /*}}}*/
  /*{{{  dump params if not reduced and debugging*/
  #ifndef UPR_RD
  if (debug)
  {
    int f, g ;
    printf("This is the kernel on processor %d of a %d node %s network\n", proc_id, num_nodes, secure ? "SECURE" : "FAST") ;
  
    /*{{{  route_in*/
    printf("Incoming links are at : ") ;
    for (f=0;f<UPR_num_links;f++)
      printf("%x ", (int) route_in[f]) ;
    printf("\n") ;
    /*}}}*/
    /*{{{  route_out*/
    printf("Outgoing links are at : ") ;
    for (f=0;f<UPR_num_links;f++)
      printf("%x ", (int) route_out[f]) ;
    printf("\n\n") ;
    /*}}}*/
    /*{{{  link types*/
    printf("link types : ") ;
    for (f=0;f<UPR_num_links;f++)
      switch (link_type[f])
      {
        case invalid  : printf("invalid ") ; break ;
        case inactive : printf("inactive ") ; break ;
        case host     : printf("host ") ; break ;
        case virtual  : printf("virtual%s ",df_link[f] ? "(DFL)" : "(UPR)") ; break ;
        case hprlink  : printf("hprlink ") ; break ;
        case hybrid   : printf("hybrid  ") ; break ;
        default       : printf("unknown ") ; break ;
      }
    printf("\n") ;      
    /*}}}*/
    /*{{{  out_link*/
    printf("out_link array : ") ;
    for (f=0;f<num_nodes;f++)
      printf("{%d,%d} ", out_link[f][ll], out_link[f][vl]) ;
    printf("\n\n") ;
    /*}}}*/
    /*{{{  in_vlink*/
    printf("in_vlink array\n") ;
    for (f=0;f<num_nodes;f++)
    {
      printf("{") ;
      for (g=0;g<UPR_num_links;g++)
        printf("%d ", in_vlink[f][g]) ;
      printf("}\n") ;
    }
    printf("\n") ;
    /*}}}*/
    /*{{{  num_in_vlink*/
    printf("num_in_vlink array : ") ;
    for (f=0;f<UPR_num_links;f++)
      printf("%d ", num_in_vlink[f]) ;
    printf("\n\n") ;  
    /*}}}*/
    /*{{{  num_out_vlink*/
    printf("num_out_vlink array : ") ;
    for (f=0;f<UPR_num_links;f++)
      printf("%d ", num_out_vlink[f]) ;
    printf("\n\n") ;  
    /*}}}*/
    /*{{{  num_active_links*/
    printf("Buffers are divided between %d links\n", num_active_links) ;
    /*}}}*/
  }
  #endif
  /*}}}*/

  /*{{{  kernel*/
  {
    UPR_GLOBALS *g = &upr_globals ;
    int i, result ;
  
    #ifndef UPR_RD
    if (debug)
      printf("Kernel spawned\n");
    #endif
  
    if (num_active_links != 0)
    {
      /*{{{  initialise link data structures*/
      {
        Process *inp, *opp;
        Channel *ins[UPR_num_links+1] ;
        Channel *outs[UPR_num_links+1] ;
        Channel c[UPR_num_links] ;
        int num_buffers_per_link[UPR_num_links] ;
        int link ;
      
        #ifdef UPR_FIXED_MEMORY
          /*{{{  number of buffers independent of virtualization*/
          /* Must ensure that a buffer is always available for packets
             destined for this node, even when all virtual links are
             full of packets */
          
          g->num_buffers = UPR_num_buffers ;
          
          for (link=0;link<UPR_num_links;link++)
            if ((num_in_vlink[link] != UPR_INACTIVE) && (num_in_vlink[link] != 0))
              num_buffers_per_link[link] = (UPR_num_buffers/num_active_links)-1 ;
            else
              num_buffers_per_link[link] = 0 ;
          /*}}}*/
        #else
          /*{{{  number of buffers determined by virtualization*/
          {
            g->num_buffers = UPR_num_links ;
          
            for (link=0;link<UPR_num_links;link++)
              if ((num_in_vlink[link] != UPR_INACTIVE) && (num_in_vlink[link] != 0))
              {
                num_buffers_per_link[link] = num_in_vlink[link] * UPR_buffers_per_vlink ;
                g->num_buffers += num_buffers_per_link[link] ;
              }
              else
                num_buffers_per_link[link] = 0 ;    
          }
          /*}}}*/
        #endif
      
        for (link=0;link<UPR_num_links;link++)
          if ((link_type[link] == hprlink)||(link_type[link] == hybrid))
          {
            c[link] = NotProcess_p ;
            outs[link] = ins[link] = &c[link] ;
          }
          else
          {
            ins[link] = route_in[link] ;
            outs[link] = route_out[link] ;
          }
      
        ins[UPR_num_links] = NULL ;
        outs[UPR_num_links] = NULL ;
           
        if ((inp=ProcAlloc(init_ilbs,8000,3,outs,num_in_vlink,num_buffers_per_link))==NULL)
          Exception(UPR_fatal,_FILE_,__LINE__,"Alloc for init_ilbs failed");
      
        if ((opp=ProcAlloc(init_olbs,8000,2,ins,num_out_vlink))==NULL)
          Exception(UPR_fatal,_FILE_,__LINE__,"Alloc for init_olbs failed");
      
        ProcPar (inp, opp, NULL) ;
      
        ProcAllocClean(inp) ;
        ProcAllocClean(opp) ;
      }
      /*}}}*/
      /*{{{  display buffer requirements if debugging*/
      #ifndef UPR_RD
      if (debug)
        printf("%d buffers required\n", g->num_buffers) ;
      #endif
      /*}}}*/
      /*{{{  allocate through routing buffers*/
      if ((trbbs = (TRBB *) malloc (g->num_buffers*sizeof (TRBB))) == NULL)
        Exception (UPR_fatal, _FILE_, __LINE__, "Out of heap space - too many buffers") ;
      /*}}}*/
      /*{{{  initialise through routing buffers*/
      g->trbb = trbbs;
      for (i=0;i<g->num_buffers;i++)
      {
        trbbs[i].next = &trbbs[(i+1)%g->num_buffers];
        init_trbb(&(trbbs[i]));
      }
      /*}}}*/
    }
    
    /*{{{  allocate port pointers*/
    if ((ports = (PORT **) malloc (UPR_MAX_NUM_PORTS*sizeof (PORT *))) == NULL)
      Exception (UPR_fatal, _FILE_, __LINE__, "Out of heap space - too many ports") ;
    /*}}}*/
    /*{{{  initialise port pointers*/
    g->ports = ports ;
    g->next_port = 0 ;
    for (i=0;i<UPR_MAX_NUM_PORTS;i++)
      ports[i] = (PORT *) NULL ;
    /*}}}*/
    /*{{{  allocate control port*/
    #ifdef UPR_CTRL_PORT_ACTIVE
      UPR_CtrlPortAlloc() ;
    
      #ifndef UPR_RD
        if (debug)
          printf("UPR Control port has id %d and is allocated at %x\n",
          g->upr_ctrl_port_id, (int) ports[g->upr_ctrl_port_id]) ;
      #endif
    #endif
    /*}}}*/
    /*{{{  initialise history buffer*/
    g->hcount = 0 ;
    
    #if (UPR_HISTORY != 0)
      if ((g->history=malloc(history_size * sizeof(HREC)))==NULL)
        Exception(UPR_fatal,_FILE_,__LINE__,OOM) ;
    
      #ifndef UPR_RD
      if (debug)
        printf("History buffer allocated at %x\n", (int) g->history) ;
      #endif
    #else
      g->history = (HREC *) NULL ;
    #endif
    /*}}}*/
  
    InitTransportLayer (proc_id, num_nodes,uargc,uargv);
    
    /*{{{  spawn error handler*/
    #ifdef UPR_CTRL_PORT_ACTIVE
    {
      Process *ehp ;
      
      if ((ehp=ProcAlloc(UPR_ExceptionHandler,0,0)) == NULL)
        Exception(UPR_fatal,_FILE_,__LINE__,"Exception Handler allocation failed") ;
      ProcRunHigh(ehp);
    
      #ifndef UPR_RD
      if (debug)
        printf("Spawned error handler\n") ;
      #endif
    }
    #endif
    /*}}}*/
    /*{{{  spawn output and input handlers*/
    {
      BOOL fast_dfl = FALSE ;
    
      if (!secure && (UPR_HISTORY == 0))
        /*{{{  see if worth using fast deadlock-free links*/
        {
          int df_count=0, link ;
        
          for (link=0;link<UPR_num_links;link++)
            df_count += (df_link[link] ? 1 : 0) ;
        
          if (hpr_router || (df_count >= num_active_links/2))
            fast_dfl = TRUE ;
        }
        /*}}}*/
      
      if (!secure && (UPR_HISTORY == 0))
        /*{{{  move code into statics*/
        {
          if (fast_dfl)
          {
            AsmMove (DFL_InputHandler, fast_input_code, fast_input_code_size) ;
            AsmMove (DFL_OutputHandler, fast_output_code, fast_output_code_size) ;
          
            #ifndef UPR_RD
            if (debug)
            {
              printf("DFL Input handler code moved from %x to %x\n",(int) DFL_InputHandler, (int) fast_input_code) ;
              printf("DFL Output handler code moved from %x to %x\n",(int) DFL_OutputHandler, (int) fast_output_code) ;
            }
            #endif
          }
          else
          {
            AsmMove (UPR_InputHandler, fast_input_code, fast_input_code_size) ;
            AsmMove (UPR_OutputHandler, fast_output_code, fast_output_code_size) ;
          
            #ifndef UPR_RD
            if (debug)
            {
              printf("UPR Input handler code moved from %x to %x\n",(int) UPR_InputHandler, (int) fast_input_code) ;
              printf("UPR Output handler code moved from %x to %x\n",(int) UPR_OutputHandler, (int) fast_output_code) ;
            }
            #endif
          }
        }
        /*}}}*/
      
      for(i=0;i<UPR_num_links;i++)
      {
        if (g->ilbs[i].used == TRUE)
        {
          if (secure || (UPR_HISTORY != 0))
            /*{{{  conventional secure input handler*/
            {
              Process *ihp ;
            
              if (df_link[i])
              {      
                if ((ihp = ProcAlloc (DFL_SecureInputHandler,secure_input_ws_size*sizeof(int),3, route_in[i],&g->ilbs[i],i)) == NULL)
                  Exception(UPR_fatal,_FILE_,__LINE__,"ProcAlloc for secure DFL input handler failed") ;
              }
              else
              {      
                if ((ihp = ProcAlloc (UPR_SecureInputHandler,secure_input_ws_size*sizeof(int),3, route_in[i],&g->ilbs[i],i)) == NULL)
                  Exception(UPR_fatal,_FILE_,__LINE__,"ProcAlloc for secure UPR input handler failed") ;
              }
                
              ProcRunHigh (ihp) ;
              
              #ifndef UPR_RD
              if (debug)
                printf("Spawned secure input handler %d\n", i) ;
              #endif
            }
            /*}}}*/
          else
            /*{{{  launch fast input handler*/
            {
              int *slow_ws ;
              void (*input_code)() ;
            
              if (df_link[i] == fast_dfl)
                input_code = (void (*)()) fast_input_code ;
              else
                if (df_link[i])
                  input_code = DFL_InputHandler ;
                else
                  input_code = UPR_InputHandler ;
            
              if ((slow_ws = (int *) malloc(fast_input_ws_size * sizeof(int))) == NULL)
                Exception(UPR_fatal,_FILE_,__LINE__,OOM) ;
                
              ProcInit (&input_p[i], input_code, slow_ws, fast_input_ws_size*sizeof(int),
                        3, route_in[i],&g->ilbs[i],i) ;
                
              ProcMove (&input_p[i], slow_ws, &fast_input_ws[i][0], fast_input_ws_size) ;
                
              #ifndef UPR_RD
              if (debug)
                printf("Input handler %d workspace moved from %x to %x\n",
                       i, (int) slow_ws, (int) &fast_input_ws[i][0]) ;
              #endif
              
              free(slow_ws) ;
                
              ProcRunHigh (&input_p[i]) ;
                
              #ifndef UPR_RD
              if (debug)
                printf("Spawned fast input handler %d\n", i) ;
              #endif
            }
            /*}}}*/
        }
          
        if (g->olbs[i].used == TRUE)
        {
          if (secure || (UPR_HISTORY != 0))
            /*{{{  conventional secure output handler*/
            {
              Process *ohp ;
            
              if (df_link[i])
              {
                if ((ohp = ProcAlloc (DFL_SecureOutputHandler,secure_output_ws_size*sizeof(int),3, route_out[i], &g->olbs[i], i)) == NULL)
                  Exception(UPR_fatal,_FILE_,__LINE__,"ProcAlloc for secure DFL output handler failed") ;
              }
              else
              {
                if ((ohp = ProcAlloc (UPR_SecureOutputHandler,secure_output_ws_size*sizeof(int),3, route_out[i], &g->olbs[i], i)) == NULL)
                  Exception(UPR_fatal,_FILE_,__LINE__,"ProcAlloc for secure UPR output handler failed") ;
              }
              
              ProcRunHigh (ohp) ;
              
              #ifndef UPR_RD
              if (debug)
                printf("Spawned secure output handler %d\n", i) ;
              #endif
            }
            
            /*}}}*/
          else
            /*{{{  launch fast output handler*/
            {
              int *slow_ws ;
              void (*output_code)() ;
            
              if (df_link[i] == fast_dfl)
                output_code = (void (*)()) fast_output_code ;
              else
                if (df_link[i])
                  output_code = DFL_OutputHandler ;
                else
                  output_code = UPR_OutputHandler ;
              
              if ((slow_ws = (int *) malloc(fast_output_ws_size * sizeof(int))) == NULL)
                Exception(UPR_fatal,_FILE_,__LINE__,OOM) ;
                
              ProcInit (&output_p[i], output_code, slow_ws, fast_output_ws_size*sizeof(int),
                        3, route_out[i], &g->olbs[i], i) ;
                
              ProcMove (&output_p[i], slow_ws, &fast_output_ws[i][0], fast_output_ws_size) ;
                
              #ifndef UPR_RD
              if (debug)
                printf("Output handler %d workspace moved from %x to %x\n",
                       i, (int) slow_ws, (int) &fast_output_ws[i][0]) ;
              #endif
              
              free(slow_ws) ;
                
              ProcRunHigh (&output_p[i]) ;
                
              #ifndef UPR_RD
              if (debug)
                printf("Spawned fast output handler %d\n", i) ;
              #endif
            }
            /*}}}*/
        }
      }
    }    
    /*}}}*/
  
    /*{{{  place current process at back of queue*/
    {
      WDESC wdesc ;
      
      ProcDesc  (&wdesc) ;
      ProcAwaken(wdesc) ;
      ProcSleep () ;
    }
    /*}}}*/
    /*{{{  enable full exception gathering*/
    g->eqb.wdesc = NotProcess_p ;
    SemSignal (&g->eqb.lock) ;
    
    /*}}}*/
    
    result = TransportLayer (uargc, uargv) ;
  
    /*{{{  report transport errors*/
    if (result<0)
    {
      char s[80] ;
      sprintf(s, "Transport layer error %d", result) ;
      Exception(UPR_error,_FILE_,__LINE__,s) ;
    }
    /*}}}*/
    /*{{{  record kernel done*/
    #if (UPR_HISTORY != 0)
    UPR_Record (kernel_thread, 0, kernel_done, 0) ;
    #endif
    /*}}}*/
    /*{{{  terminate*/
    #ifndef UPR_RD
    SemWait (&g->eqb.lock) ;
    exit_terminate(EXIT_SUCCESS) ;
    #endif
    
    ProcSleep() ;
    /*}}}*/
  }
  /*}}}*/
}
/*}}}*/
