/*{{{  File banner*/
/*@(#)=====================================================*\
||@(#)  Project : PUMA ESPRIT P2701
||@(#)  Authors : Mark Debbage and Mark Hill
||@(#)            University of Southampton
||  
||@(#)    Title : Exception handler functions
||@(#)   System : UPR
||@(#) Filename : except.c
||@(#)  Version : 2.3
\*@(#)====================================================*/
/*}}}*/


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <misc.h>

#include "router.h"
#include "except.h"

static char *_FILE_ = __FILE__ ;

/*{{{  PUBLIC void Exception( int severity, char file[], int line, char msg[] )*/
PUBLIC void Exception( int severity, char file[], int line, char msg[] )
{
  UPR_GLOBALS *g = &upr_globals ;

  if ((severity==UPR_fatal) || ((severity==UPR_error) && (g->eqb.wdesc==NULL)))
    /*{{{  fatals and early errors*/
    {
      #ifdef UPR_RD
        /*{{{  record in globals and set error*/
        g->error_message = (BYTE *) msg ;
        ProcDesc(&g->error_wdesc) ;
        ProcSetError();
        /*}}}*/
      #else
        /*{{{  print and terminate server*/
        if (line>=0)
          fprintf(stderr,"Fatal-upr-p%0d-%s(%0d) %s\n",UPR_root,file,line,msg);
        else
          fprintf(stderr,"Fatal-upr-p%0d-%s(?) %s\n",UPR_root,file,msg);
        
        exit_terminate( EXIT_FAILURE );
        /*}}}*/
      #endif
      ProcSleep();
    }
    /*}}}*/
  else
    /*{{{  add to pending queue*/
    {
      int priority ;
      
      ProcGetPRI(&priority);
      if (priority==PROC_LOW) ProcToHI() ;
      
      /*{{{  add to error requests*/
      {
        int h = g->eqb.head ;
        int t = g->eqb.tail ;
        ERROR *next = &((g->eqb.errs)[t].p.error);
        
        if (h==t)
          /*{{{  buffers full so throw error away*/
          (g->eqb.errs)[(UPR_error_depth+t-1)%UPR_error_depth].p.error.lost = TRUE ;
          /*}}}*/
        else
          /*{{{  record error occurence in diary*/
          {
            next->status  = severity;
            next->line_no = line;
            next->lost    = FALSE;
            next->proc_no = g->proc_id;
            
            if (strlen(file)>=MAX_FILENAME)
              strcpy(next->filename,&file[strlen(file)-MAX_FILENAME+1]);
            else
              strcpy(next->filename,file);
              
            if (strlen(msg)>=MAX_MESSAGE)
              strcpy(next->message,&msg[strlen(msg)-MAX_MESSAGE+1]);
            else
              strcpy(next->message,msg);
              
            g->eqb.tail = (t+1)%UPR_error_depth ;
            if (h==UPR_no_errors)
            {
              g->eqb.head = t ;
              if (g->eqb.wdesc != NotProcess_p && g->eqb.wdesc != NULL)
                ProcAwaken(g->eqb.wdesc) ;
            }
          }
          /*}}}*/
      }
      /*}}}*/
    
      #if (UPR_HISTORY != 0)
      UPR_Record (error_thread,0,error_queue,(int) msg) ;
      #endif
      
      if (priority==PROC_LOW) ProcToLO() ;
    
      if (severity==UPR_error)
      {
        g->error_message = (BYTE *) msg ;
        ProcDesc(&g->error_wdesc) ;
        ProcSleep() ;
      }
    }
    /*}}}*/
}
/*}}}*/

/*{{{  PUBLIC void UPR_InitExceptions()*/
PUBLIC void UPR_InitExceptions()
{
  UPR_GLOBALS *g = &upr_globals ;
  
  EQB *eqb = &(g->eqb);
  int i ;

  /*{{{  initialise globals*/
  eqb->head = UPR_no_errors;
  eqb->tail = 0;
  eqb->wdesc = NULL ; /* This indicates handler not launched yet */
  /*}}}*/
  /*{{{  initialise and lock semaphore*/
  SemInit (&upr_globals.eqb.lock, 1) ;
  SemWait (&upr_globals.eqb.lock) ;
  /*}}}*/
  /*{{{  malloc error buffers*/
  if ((eqb->errs = (UPR_CTRL_PKT*) malloc(UPR_error_depth*sizeof(UPR_CTRL_PKT)))==NULL)
    Exception(UPR_fatal,_FILE_,__LINE__,"Malloc of error buffers failed");
  /*}}}*/
  /*{{{  tag UPR_CTRL_PKTs as errors*/
  for (i=0;i<UPR_error_depth;i++)
    eqb->errs[i].tag = error ;
  /*}}}*/
  /*{{{  initialise ORB*/
  HdrBuild (UPR_root, g->upr_ctrl_port_id, UPR_MAX_PKT_SIZE,&(eqb->orb.header));
  /*}}}*/
}
/*}}}*/

/*{{{  PUBLIC int UPR_ReadException(char *message)*/
PUBLIC int UPR_ReadException(char *message)
{
  UPR_GLOBALS *g = &upr_globals ;
  
  int h = g->eqb.head ;
  
  if (h == UPR_no_errors)
    /*{{{  await errors*/
    {
      #if (UPR_HISTORY != 0)
      UPR_Record (error_thread,0,error_sleep, 0) ;
      #endif
      
      ProcDesc(&g->eqb.wdesc) ;
      ProcSleep();
      g->eqb.wdesc = NotProcess_p ;
    
      h = g->eqb.head;
    }
    /*}}}*/
  
  {
    ERROR *err ;

    err = &((g->eqb.errs)[h].p.error) ;

    #if (UPR_HISTORY != 0)
    UPR_Record (error_thread,0,error_send,(int) *((int *) &g->eqb.orb.header)) ;
    #endif
  
    /*{{{  build message*/
    {
      char line_str[10] ;
    
      if (err->line_no>0)
        sprintf(line_str,"%0d",err->line_no) ;
      else
      {
        line_str[0] = '?' ;
        line_str[1] = 0 ;
      }
      
      switch( err->status )
      {
        case UPR_warning :
          sprintf(message,"Warning-upr-p%0d-%s(%s)- %s\n",
          err->proc_no,err->filename, line_str, err->message);
          break ;
      
        case UPR_error :
          sprintf(message,"Error-upr-p%0d-%s(%s)- %s\n",
          err->proc_no,err->filename, line_str, err->message);
          break ;
      
        case UPR_fatal :
          sprintf(message,"Fatal-upr-p%0d-%s(%s)- %s\n",
          err->proc_no,err->filename, line_str, err->message);
          break ;
      
        default :
          sprintf(message,"Fatal-upr-%s(%0d)- Unknown Error Status %d\n",
           _FILE_,__LINE__,err->status);
      }
    }
    /*}}}*/
    
    if ((err->lost)==TRUE)
      strcat(message,LOST_MESSAGE);
    
    h = (h+1)% UPR_error_depth ;
    
    if (h == (g->eqb.tail))
      g->eqb.head = UPR_no_errors ;
    else
      g->eqb.head = h;

    return(err->status);
  }
}
/*}}}*/

/*{{{  PUBLIC void UPR_ExceptionHandler( Process *p )*/
PUBLIC void UPR_ExceptionHandler( Process *p )
{
  SemWait   (&upr_globals.eqb.lock) ;
  SemSignal (&upr_globals.eqb.lock) ;

  #ifdef UPR_RD
  /*{{{  on network*/
  {
    UPR_GLOBALS *g = &upr_globals ;
    
    p=p ;
    
    while (TRUE)
    {
      int h ;
  
      #if (UPR_HISTORY != 0)
      UPR_Record (error_thread,0,error_sleep, 0) ;
      #endif
      
      ProcDesc(&(g->eqb.wdesc));
      ProcSleep();
      g->eqb.wdesc = NotProcess_p ;
  
      h = g->eqb.head ;
  
      do
        /*{{{  process all pending errors*/
        {
          #if (UPR_HISTORY != 0)
            UPR_Record (error_thread,0,error_send,(int) *((int *) &g->eqb.orb.header)) ;
          #endif
        
          g->eqb.orb.bffr = (BYTE *) &g->eqb.errs[h] ;
          UPR_Enqueue_BK(&g->eqb.orb) ;
        
          h = (h+1)% UPR_error_depth ;
          /*{{{  update global to release buffer*/
          g->eqb.head = h ;
          /*}}}*/
        }
        /*}}}*/
      while(h != (g->eqb.tail)) ;
      
      g->eqb.head = UPR_no_errors ;
    }
  }  
  /*}}}*/
  #else
  /*{{{  on root*/
  {
    char message[MAX_ERROR_MESSAGE] ;
    
    p=p ;
  
    while (TRUE)
    {
      UPR_ReadException(message);
      SemWait   (&upr_globals.eqb.lock) ;
      fprintf(stderr,"%s",message);
      SemSignal (&upr_globals.eqb.lock) ;
    }
  }  
  /*}}}*/
  #endif
}
/*}}}*/

/*{{{  PUBLIC void QueueError( ERROR *err )*/
PUBLIC void QueueError( ERROR *err )
{
  #ifdef UPR_RD
    Exception(UPR_fatal,_FILE_,__LINE__,"Remote processor received error");
  #else
    /*{{{  root processor received error*/
    UPR_GLOBALS *g = &upr_globals ;
      
    int h = g->eqb.head ;
    int t = g->eqb.tail ;
      
    if (h==t)
      /*{{{  buffers full so throw error away*/
      (g->eqb.errs)[(UPR_error_depth+t-1)%UPR_error_depth].p.error.lost = TRUE ;
      /*}}}*/
    else
      /*{{{  record error occurence in diary*/
      {
        ERROR *next = &((g->eqb.errs)[t].p.error);
        AsmMove( err, next, sizeof(ERROR));
        
        g->eqb.tail = (t+1)%UPR_error_depth ;
      
        if (h==UPR_no_errors)
        {
          g->eqb.head = t ;
          if (g->eqb.wdesc != NotProcess_p && g->eqb.wdesc != NULL)
            ProcAwaken(g->eqb.wdesc) ;
        }
      }
      /*}}}*/
    
    #if (UPR_HISTORY != 0)
    UPR_Record (error_thread,0,error_queue, (int) err->message) ;
    #endif
    /*}}}*/
  #endif
}
/*}}}*/
  
/*{{{  PUBLIC void UPR_Record (int thread, int index, int pos, int value)*/
PUBLIC void UPR_Record (int thread, int index, int pos, int value)
{
  UPR_GLOBALS *g = &upr_globals ;

  int priority ;

  ProcGetPRI(&priority);
  if (priority==PROC_LOW) ProcToHI() ;

  {
    HREC *hist = &g->history[g->hcount] ;

    if (hist != (HREC *) NULL)
    {
      hist->thread = thread ;
      hist->ref    = (pos << 16) | index ;
      hist->value  = value ;
      hist->time   = ProcTime() ;
    }
    else
      Exception(UPR_error,_FILE_,__LINE__,"No history buffer to record in") ;

    g->hcount = (g->hcount+1) % history_size ;
  }      

  if (priority==PROC_LOW) ProcToLO() ;
}  
/*}}}*/
