/*{{{  File banner*/
/*@(#)=====================================================*\
||@(#)  Project : PUMA ESPRIT P2701
||@(#)  Authors : Mark Debbage and Mark Hill
||@(#)            University of Southampton
||  
||@(#)    Title : Network loader
||@(#)   System : DLOADER
||@(#) Filename : netroot.c
||@(#)  Version : 2.6
\*@(#)====================================================*/
/*}}}*/

/*{{{  includes*/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <misc.h>
#include <host.h>

#include "net.h"
#include "netload.h"
#include "netshow.h"
#include "neterr.h"
#include "topology.h"
/*}}}*/

PUBLIC char switch_char ;
PUBLIC char slash_char ;
PUBLIC int  proc_memory ;

PRIVATE NETWORK network_info ;
PUBLIC NETWORK *network = &network_info ;

/*{{{  PRIVATE void get_switch_and_slash_chars()*/
PRIVATE void get_switch_and_slash_chars()
{
  int host, os, board;

  host_info(&host, &os, &board);
  switch(os)
    {
      case _IMS_OS_DOS:
      case _IMS_OS_VMS:
      case _IMS_OS_CMS:
        switch_char = '/';
        slash_char = '\\';
        break;
      case _IMS_OS_HELIOS:
      case _IMS_OS_SUNOS:
      default:
        switch_char = '-';
        slash_char = '/';
        break;
    }
}
/*}}}*/

/*{{{  local definitions*/
static char *_FILE_ = __FILE__ ;
#define proc_id UPR_root
BOOL    debug = FALSE ;
/*}}}*/

PRIVATE int  linenum = 0 ;
/*{{{  string parsing routines*/
/*{{{  PRIVATE char *skip_non_blanks (char *s)*/
PRIVATE char *skip_non_blanks (char *s)
{
  while ((*s != 0) && (!isspace(*s)))
    s++ ;
  return(s) ;
}

/*}}}*/
/*{{{  PRIVATE char *skip_blanks (char *s)*/
PRIVATE char *skip_blanks (char *s)
{
  while ((*s != 0) && (isspace(*s)))
    s++ ;
  return(s) ;
}

/*}}}*/
/*{{{  PRIVATE char *skip_digits (char *s)*/
PRIVATE char *skip_digits (char *s)
{
  while ((*s != 0) && (isdigit(*s)))
    s++ ;
  return(s) ;
}

/*}}}*/
/*}}}*/
/*{{{  pragma matching*/
PUBLIC char text_tpg [first_tpg+num_tpg][8] = { "unknown","grid","hpr" } ;

/*{{{  PRIVATE void extract_topology(char *s)*/
PRIVATE void extract_topology(char *s)
{
  int i ;

  s = skip_blanks(s) ;
  
  for (i=first_tpg;i<num_tpg-first_tpg;i++)
    if (strncmp(s,text_tpg[i],strlen(text_tpg[i])) == 0)
    {
      if (debug) fprintf(stderr,"Topology type is %s\n",text_tpg[i]) ;
      break ;
    }      

  switch(i)
  {
    case grid_tpg :
      /*{{{  read grid information*/
      {
        int d ;
      
        network->topology.tpg = grid_tpg ;
        
        s = skip_non_blanks(s) ;
        s = skip_blanks(s) ;
      
        if (sscanf(s,"%d",&network->topology.grid.ndims) != 1)
        {
          fprintf(stderr, "Error-%s(%d)- Parse failed reading dimensions on line %d\n",__FILE__,__LINE__,linenum) ;
          exit(-1) ;
        }
      
        if (network->topology.grid.ndims<0)
        {
          fprintf(stderr, "Error-%s(%d)- Parse failed, negative dimensions on line %d\n",__FILE__,__LINE__,linenum) ;
          exit(-1) ;
        }
      
        if (((network->topology.grid.dims  = (int *) malloc (network->topology.grid.ndims*sizeof(int))) == NULL) ||
            ((network->topology.grid.wraps = (BOOL *) malloc (network->topology.grid.ndims*sizeof(BOOL))) == NULL))
          LoadError(_FILE_,__LINE__,"Out of memory") ;
        
        s = skip_non_blanks(s) ;
        s = skip_blanks(s) ;
        
        for (d=0;d<network->topology.grid.ndims;d++)
        {
          if (sscanf(s,"%d",&network->topology.grid.dims[d]) != 1)
          {
            fprintf(stderr, "Error-%s(%d)- Parse failed reading a dimension on line %d\n",__FILE__,__LINE__,linenum) ;
            exit(-1) ;
          }
          
          s = skip_non_blanks(s) ;
          s = skip_blanks(s) ;
        }
       
        for (d=0;d<network->topology.grid.ndims;d++)
        {
          if (sscanf(s,"%d",&network->topology.grid.wraps[d]) != 1)
          {
            fprintf(stderr, "Error-%s(%d)- Parse failed reading a wrap flag on line %d\n",__FILE__,__LINE__,linenum) ;
            exit(-1) ;
          }
          
          s = skip_non_blanks(s) ;
          s = skip_blanks(s) ;
        }
        
        if (sscanf(s,"%d",&network->topology.grid.first) != 1)
        {
          fprintf(stderr, "Error-%s(%d)- Parse failed reading first id on line %d\n",__FILE__,__LINE__,linenum) ;
          exit(-1) ;
        }
        
        s = skip_non_blanks(s) ;
        s = skip_blanks(s) ;
        
        break ;
      }
      /*}}}*/
    case hpr_tpg :
      /*{{{  read hpr information*/
      {
        network->topology.tpg = hpr_tpg ;
      
        fprintf(stderr, "Warning-%s(%d)- Found dynamic HPR topology on line %d\n",__FILE__,__LINE__, linenum) ;
      
        s = skip_non_blanks(s) ;
        s = skip_blanks(s) ;
      
        if (sscanf(s,"%d",&network->topology.hpr.first) != 1)
        {
          fprintf(stderr, "Error-%s(%d)- Parse failed reading first id on line %d\n",__FILE__,__LINE__,linenum) ;
          exit(-1) ;
        }
      
        break ;
      }
      /*}}}*/
    default :
      network->topology.tpg = unknown_tpg ;
      fprintf(stderr, "Warning-%s(%d)- Unrecognised topology (%s) on line %d\n",__FILE__,__LINE__,s, linenum) ;
  }
}
/*}}}*/

#define num_pragmas 1

struct
{
  char text[10] ;
  void (*function)() ;
} pragma[num_pragmas] = { "#TOPOLOGY", extract_topology } ;

/*{{{  PRIVATE void identify_pragma (char *s)*/
PRIVATE void identify_pragma (char *s)
{
  int i ;

  for (i=0;i<num_pragmas;i++)
    if (strncmp(s,pragma[i].text,strlen(pragma[i].text)) == 0)
    {
      pragma[i].function (s+strlen(pragma[i].text)) ;
      break ;
    }
}
/*}}}*/
/*}}}*/
/*{{{  file reading primitives*/
#define buflen 160
PRIVATE char linebuf[buflen];
PRIVATE char *linepos = linebuf ;
PRIVATE char *strbuf = NULL ;

/*{{{  PRIVATE char *fgetstring (char *line, int n, FILE *file)*/
/*
   This routine reads a line, strips out # comments and strips
   out lines of just white space. Usage much like fgets.
*/

PRIVATE char *fgetstring (char *line, int n, FILE *file)
{
  char *s ;

  do
  { 
    if ((s=fgets (line, n, file)) == NULL)
      return(NULL) ;
       
    if (!feof(file) && (s[0] != 0))
    {
      char *hash = strchr (s, '#') ;
      int i ;
       
      if ((strlen(s) == n-1) && (s[n-1] != '\n'))
        fprintf(stderr, "Warning-%s(%d)- Line %d too long so wrapped round\n",__FILE__,__LINE__,linenum) ;
      else
        linenum++ ;

      if (hash == s)
        identify_pragma (hash) ;
       
      if (hash != NULL)
        *hash = 0 ;
       
      for (i=0;i<strlen(s);i++)
        if (!isspace(s[i]))
          break ;
       
      if (i==strlen(s))
        s[0] = 0 ;    
    }
  } while (*s == 0) ;

  return(s) ;
}
/*}}}*/
  
/*{{{  PRIVATE int   read_number (FILE *file)*/
PRIVATE int   read_number (FILE *file)
{
  int i ;

  linepos = skip_blanks(linepos) ;
  
  if (*linepos == 0)
  {
    if (fgetstring(linebuf, buflen, file) == NULL)
    {
      fprintf(stderr, "Error-%s(%d)- Parse failed reading number on line %d\n",__FILE__,__LINE__,linenum) ;
      exit(-1) ;
    }
    linepos = linebuf ;
  }

  if (feof(file) || (*linepos == 0))
  {
    fprintf(stderr, "Error-%s(%d)- Parse failed reading number on line %d\n",__FILE__,__LINE__,linenum) ;
    exit(-1) ;
  }

  if (sscanf(linepos,"%d",&i) != 1)
  {
    fprintf(stderr, "Error-%s(%d)- Parse failed reading number on line %d\n",__FILE__,__LINE__,linenum) ;
    exit(-1) ;
  }

  linepos = skip_digits(linepos) ;

  return(i) ;  
}
/*}}}*/
/*{{{  PRIVATE char *read_string (FILE *file)*/
PRIVATE char *read_string (FILE *file)
{
  linepos = skip_blanks(linepos) ;

  if (*linepos == 0)
  {
    if (fgetstring(linebuf, buflen, file) == NULL)
    {
      fprintf(stderr, "Error-%s(%d)- Parse failed reading string on line %d\n",__FILE__,__LINE__,linenum) ;
      exit(-1) ;
    }
    linepos = linebuf ;
  }

  if (feof(file) || (*linepos == 0))
  {
    fprintf(stderr, "Error-%s(%d)- Parse failed reading string on line %d\n",__FILE__,__LINE__,linenum) ;
    exit(-1) ;
  }

  if (strbuf != NULL)
    free(strbuf) ;
    
  strbuf = (char *) malloc(strlen(linepos+1)) ;

  if (sscanf(linepos,"%s",strbuf) != 1)
  {
    fprintf(stderr, "Error-%s(%d)- Parse failed reading string on line %d\n",__FILE__,__LINE__,linenum) ;
    exit(-1) ;
  }

  linepos = skip_non_blanks(linepos) ;

  return(strbuf) ;
}
/*}}}*/
/*{{{  PRIVATE char  read_char   (FILE *file)*/
PRIVATE char  read_char   (FILE *file)
{
  char c ;
  
  if (*linepos == 0)
  {
    if (fgetstring(linebuf, buflen, file) == NULL)
    {
      fprintf(stderr, "Error-%s(%d)- Parse failed reading character on line %d\n",__FILE__,__LINE__,linenum) ;
      exit(-1) ;
    }
    linepos = linebuf ;
  }

  if (feof(file) || (*linepos == 0))
  {
    fprintf(stderr, "Error-%s(%d)- Parse failed reading character on line %d\n",__FILE__,__LINE__,linenum) ;
    exit(-1) ;
  }

  if (sscanf(linepos,"%c",&c) != 1)
  {
    fprintf(stderr, "Error-%s(%d)- Parse failed reading character on line %d\n",__FILE__,__LINE__,linenum) ;
    exit(-1) ;
  }

  linepos++ ;
  
  return(c) ;  
}
/*}}}*/
/*{{{  PRIVATE void  next_field  (FILE *file)*/
PRIVATE void  next_field  (FILE *file)
{
  linepos = skip_blanks(linepos) ;
}
/*}}}*/
/*}}}*/

/*{{{  PRIVATE show_network (NETWORK *network)*/
PRIVATE show_network (NETWORK *network)
{
  int i ;

  printf("Topology name is %s\n",network->name) ;
  printf("NCF file name is %s\n",network->ncf) ;

  printf("Topology type is %s\n",text_tpg[network->topology.tpg]) ;

  switch(network->topology.tpg)
  {
    case grid_tpg :
      printf("grid has %d dimensions, first id is %d\n",
             network->topology.grid.ndims, network->topology.grid.first) ;
      for (i=0;i<network->topology.grid.ndims;i++)
        printf("%d (%s), ", network->topology.grid.dims[i],
          network->topology.grid.wraps[i] ? "wrapped" : "open") ;
      printf("\n") ;
  }
  
  for (i=0;i<network->num_edges;i++)
    printf("Processor %d to processor %d\n",network->edge[i][0],network->edge[i][1]) ;

  for (i=0;i<network->num_vertices;i++)
  {
    int proc = network->vertex[i].son ;

    printf("Processor %d ", i) ;
    
    if (proc != NOT_PROC)
    {
      printf("boots up ") ;

      do
      {
        printf("%d ", proc) ;
        proc = network->vertex[proc].sibling ;
      } while (proc != NOT_PROC) ;
    }
    printf("\n") ;
  }
}
/*}}}*/

int main(int argc, char *argv[])
{
  /*{{{  variables*/
  Channel *raw_out[5] = {LINK0OUT, LINK1OUT, LINK2OUT, LINK3OUT, NULL} ;
  Channel *raw_in [5] = {LINK0IN,  LINK1IN,  LINK2IN,  LINK3IN,  NULL} ;
  FILE    *net_fp, *kernel_fp ;
  BYTE    *kernel_buffer ;
  int     max_kernel_size ;
  int     kernel_size = 0 ;
  int     num_nodes ;
  int     diameter ;
  NODE    *root ;
  NODE    **nodes ;
  BYTE    *routing_table ;
  int     params = 4 ; /* SKIP over fixed dloader parameters */
  /*}}}*/

  /*{{{  startup and get input files handles*/
  fprintf(stderr,"Dynamic loader, MD MBH, %s\n",__DATE__) ;
  
  if (argc < params)
  {
    printf("Need 3 parameters : <network> <kernel> #max_kernel_size\n") ;
    exit_terminate( EXIT_FAILURE );
  }
    
  net_fp=fopen(argv[1], "rb") ;
  if (net_fp==NULL)
  {
    printf("Fatal-netloader-Cannot find network file %s\n", argv[1]) ;
    exit_terminate( EXIT_FAILURE );
  }
  
  kernel_fp=fopen(argv[2], "rb") ;
  if (kernel_fp==NULL)
  {
    printf("Fatal-netloader-Cannot find program file %s\n", argv[2]) ;
    exit_terminate( EXIT_FAILURE );
  }
  
  if ((max_kernel_size=atoi(argv[3])) == 0)
  {
    printf("Fatal-netloader-Invalid maximum kernel size %s\n", argv[3]) ;
    exit_terminate( EXIT_FAILURE );
  }
  
  max_kernel_size++ ;
  /*}}}*/
  /*{{{  check for network loader options*/
  {
    int p;
    get_switch_and_slash_chars() ;
  
    for (p=params;p<argc;p++)
    {
      if (argv[p][0] == switch_char)
        if ((argv[p][1] == 'l') || (argv[p][1] == 'L'))
        {
          if ((argv[p][2] == 'd') || (argv[p][1] == 'D'))
          {
            int q;
            for (q=p;q>params;q--)
              argv[q] = argv[q-1];
            params++;
            debug = TRUE ;
          }
          else
            LoadError(_FILE_,__LINE__,"Unknown command-line option to network loader") ;
        }
    }
    params-- ;
  }
  /*}}}*/
  /*{{{  get and publish topology and ncf name*/
  {
    char *str = strrchr (argv[1], slash_char) ;
    char *end ;
    int len ;
  
    if (str == NULL)
      str = argv[1] ;
    else
      str++ ;
  
    end = strchr (str, '.') ;
  
    if (end == NULL)
      len = strlen(str) ;
    else
      len = end-str ;
  
    if ((network->name=malloc(len+1)) == NULL)
      LoadError (_FILE_, __LINE__, "Out of heap space - allocating topology name") ;
  
    strncpy (network->name, str, len) ;
    network->name[len] = 0 ;
  
    network->ncf = argv[1] ;
  
    if (debug)
    {
      printf("Topology is %s\n", network->name) ;
      printf("NCF file is %s\n", network->ncf) ;
    }
  }    
    
  /*}}}*/
  /*{{{  read information from .ncf*/
  {
    int count ;
    int n = 0 ;
    int max_edges ;
    
    num_nodes = read_number(net_fp) ;
    diameter  = read_number(net_fp) ;
  
    network->num_edges = 0 ;
    network->num_vertices = num_nodes ;
    max_edges = num_nodes*UPR_num_links/2 ;
  
    if (num_nodes <= 0 || diameter < 0)
      LoadError(_FILE_,__LINE__,"Invalid NCF header\n") ;
    
    /*{{{  malloc and initialise memory*/
    if (((nodes = (NODE **) malloc (num_nodes*sizeof (NODE *))) == NULL) ||
        ((network->vertex = (VERTEX *) malloc (num_nodes*sizeof(VERTEX))) == NULL) ||
        ((network->edge = (EDGE *) malloc (max_edges*sizeof(EDGE))) == NULL))
      LoadError (_FILE_, __LINE__, "Out of heap space - too many nodes") ;
    
    for (n=0;n<num_nodes;n++)
    {
      nodes[n] = (NODE *) NULL ;
      network->vertex[n].son = NOT_PROC ;
      network->vertex[n].sibling = NOT_PROC ;
    }
    /*}}}*/
  
    for (count=0;count<num_nodes;count++)
    {
      /*{{{  get processor id*/
      n = read_number(net_fp) ;
      
      if (n<0 || n>=num_nodes)
        LoadError (_FILE_, __LINE__, "Invalid processor id") ;
      
      if (debug)
        printf("PROCESSOR %d\n", n) ;
      /*}}}*/
      /*{{{  malloc memory for node*/
      if (nodes[n] != NULL)
        LoadError (_FILE_, __LINE__, "Processor record appears twice in ncf file") ;
      
      if ((nodes[n] = (NODE *) malloc (sizeof(NODE))) == NULL)
        LoadError (_FILE_, __LINE__, "Out of heap space - too many nodes") ;
      
      ResetNode(nodes[n], n) ;
      /*}}}*/
      /*{{{  decode processor type*/
      {
        char *s = read_string (net_fp) ;
      
        if (debug)
          printf("%s\n", s) ;
        
        if (strncmp("T2", s, 2)==0)
          LoadError (_FILE_, __LINE__, "16 bit transputer types not supported") ;
        
        if (strncmp("T4", s, 2)==0) nodes[n]->proc_type = 4 ;
        if (strncmp("T8", s, 2)==0) nodes[n]->proc_type = 8 ;
        
        if (nodes[n]->proc_type == 0)
          LoadError (_FILE_, __LINE__, "Unknown transputer type") ;
      }
      /*}}}*/
      /*{{{  decode memory size*/
      {
        int size  = read_number(net_fp) ;
        char unit = read_char (net_fp) ;
          
        if (debug)
          printf("%d%c\n", size, unit) ;
      
        if (unit=='M')
          nodes[n]->mem_size = kilobyte*kilobyte*size ;
        else
          if (unit=='K')
            nodes[n]->mem_size = kilobyte*size ;
          else
            nodes[n]->mem_size = size ;
      }
     /*}}}*/
      /*{{{  extend heap to top of memory if this processor*/
      if (n==proc_id)
      {
        int extra, loss ;
      
        proc_memory = nodes[n]->mem_size ;
        extra = (nodes[n]->mem_size) - (((int) _IMS_heap_front)^((int) NotProcess_p));
        loss  = ((int) _IMS_heap_front) - ((int) _IMS_heap_base) ;
        
        init_heap (_IMS_heap_front, extra, 0) ;
      
        if (debug)
        {
          printf("Memory size is %d\n", nodes[n]->mem_size) ;
          printf("Heap extended by %d bytes (cannot reclaim %d bytes)\n", extra, loss) ;
        }
      }  
      /*}}}*/
      /*{{{  decode boot tree*/
      {
        int dir ;
        int con_proc = -1 ;
        int con_link = -1 ;
        char type ;
        
        for (dir=0;dir<UPR_num_links;dir++)
        {
          next_field(net_fp) ;
          type = read_char (net_fp) ;
      
          if (debug)
            fprintf(stderr,"Dir %d, type %c\n",dir,type) ;
          
          switch ((int) type)
          {
            case (int) 'h' :
              /*{{{  host link*/
              nodes[n]->link_type[dir] = host ;
              if (debug)
                printf("host ") ;
              break ;
              /*}}}*/
          
            case (int) 'o' :
              /*{{{  found son*/
              {
                con_proc = read_number(net_fp) ;
              
                if (read_char(net_fp) != ':')
                  LoadError (_FILE_, __LINE__, "Bad connector format") ;
                
                con_link = read_number(net_fp) ;
                
                if (debug)
                  printf("virtual ") ;
                
                nodes[n]->link_type[dir] = virtual ;
                
                /*{{{  add to node structure*/
                nodes[n]->son[dir] = con_proc ;
                /*}}}*/
                /*{{{  add to connections list*/
                if (n <= con_proc)
                {
                  network->edge[network->num_edges][0] = n ;
                  network->edge[network->num_edges][1] = con_proc ;
                  network->num_edges++ ;
                }
                /*}}}*/
                /*{{{  add son to boot tree*/
                {
                  int *proc = &network->vertex[n].son ;
                
                  if (*proc == NOT_PROC)
                    *proc = con_proc ;
                  else
                  {
                    do
                    {
                      proc = &network->vertex[*proc].sibling ;
                    } while (*proc != NOT_PROC) ;
                    *proc = con_proc ;
                  }
                }
                /*}}}*/
              
                if (debug)
                  printf("\nProcessor %d boots up %d via link %d\n", n, con_proc, dir) ;
              
                break ;
              }
              /*}}}*/
          
            case (int) 'i' :       
            case (int) 'n' :
              /*{{{  found parent or normal link*/
              {
                con_proc = read_number(net_fp) ;
              
                if (read_char(net_fp) != ':')
                  LoadError (_FILE_, __LINE__, "Bad connector format") ;
                
                con_link = read_number(net_fp) ;
              
                if (debug)
                  printf("virtual ") ;
              
                /*{{{  add to connections list*/
                if (n <= con_proc)
                {
                  network->edge[network->num_edges][0] = n ;
                  network->edge[network->num_edges][1] = con_proc ;
                  network->num_edges++ ;
                }
                /*}}}*/
                  
                nodes[n]->link_type[dir] = virtual ;
                break ;
              }
              /*}}}*/
          
            case (int) 'x' :
              /*{{{  inactive link*/
              if (debug)
                printf("inactive ") ;
              
              nodes[n]->link_type[dir] = inactive ;
              break ;
              /*}}}*/
          
            default :
              /*{{{  invalid*/
              if (debug)
                printf("invalid ") ;
              LoadError (_FILE_, __LINE__, "Unknown link type") ;
              /*}}}*/
          }
        }
      
        if (debug)
          printf("\n") ;
      }
    /*}}}*/
    }
  
    if ((root = nodes[proc_id]) == NULL)
      LoadError (_FILE_, __LINE__, "Processor missing from file") ;
  
    network->edge = realloc(network->edge, network->num_edges*sizeof(EDGE)) ;  
  
    if (debug)
      show_network(network) ;
    
    if (debug)
    {
      printf("Boot tree is :\n") ;
      ViewTree(nodes, root) ;
    }
  } 
  /*}}}*/
  /*{{{  initialise floating point unit if T8*/
  if (root->proc_type == 8)
    ProcInitFpu() ;
  /*}}}*/
  /*{{{  read code*/
  if ((kernel_buffer = malloc(max_kernel_size)) == NULL)
    LoadError (_FILE_, __LINE__, "Out of heap space - max kernel size too large") ;
  
  if (debug)
    printf("Reading kernel code\n") ;
  
  /*{{{  COMMENT Character based version*/
  /*
  {
    BYTE c ;
  
    c = getc(kernel_fp) ;
    while (!feof(kernel_fp))
    {
      kernel_buffer[kernel_size++] = c ;
  
      if (kernel_size >= max_kernel_size)
        LoadError (_FILE_, __LINE__, "Kernel buffer too small") ;
  
      c = getc(kernel_fp) ;
    }
  }
  */
  /*}}}*/
  /*{{{  Block based version */
  {
    kernel_size = fread(kernel_buffer, sizeof (BYTE), max_kernel_size, kernel_fp) ;
  
    if (kernel_size == max_kernel_size)
      LoadError (_FILE_, __LINE__, "Kernel buffer too small") ;
  }  
  /*}}}*/
  
  if (debug)
    printf("Kernel code loaded\n") ;
  /*}}}*/

  NetworkLoader (raw_out, raw_in, nodes, root, num_nodes, kernel_size, kernel_buffer) ;

  /*{{{  read and distribute routing tables from .ncf*/
  {
    int  count ;
    int  n = 0 ;
    int  size ;
    BOOL got_mine = FALSE ;
    BYTE *tmp_routing_table ;
    
    for (count=0;count<num_nodes;count++)
    {
      /*{{{  get processor id*/
      n = read_number(net_fp) ;
      
      if (n<0 || n>=num_nodes)
        LoadError(_FILE_,__LINE__,"Bad processor id") ;
      
      if (debug)
        printf("PROCESSOR %d\n", n) ;
      /*}}}*/
      /*{{{  get table size*/
      size = read_number(net_fp) ;
      
      if (debug)
        printf("Routing table of %d bytes\n", size) ;
      /*}}}*/
      /*{{{  allocate array to hold table*/
      if ((tmp_routing_table = malloc(size)) == NULL)
        LoadError (_FILE_, __LINE__, "Out of heap space - too many nodes") ;
      /*}}}*/
      /*{{{  get table*/
      {
        int f ;
      
        for (f=0;f<size;f++)
        {
          tmp_routing_table[f] = read_number(net_fp) ;
      
          if (debug)
            printf("%d, ", tmp_routing_table[f]) ;
        }
      
        if (f != size) break ;
        if (debug)
          printf("\n") ;
      }
      /*}}}*/
  
      if (n == proc_id)
        /*{{{  tables for this processor*/
        {
          got_mine = TRUE ;
          routing_table = tmp_routing_table ;
        }
        /*}}}*/
      else
        /*{{{  ship tables to processor n*/
        {
          int link ;
          
          link = NodeInTree(nodes, root, n) ;
        
          if (link == -1)
            LoadError (_FILE_, __LINE__, "Cannot find node in tree") ;
            
          if (debug)
            printf("Shipping tables for %d down link %d\n", n, link) ;
        
          ChanOutInt(raw_out[link], n) ;
          ChanOutInt(raw_out[link], size) ;
          ChanOut   (raw_out[link], tmp_routing_table, size) ;
        
          free(tmp_routing_table) ;
        }
        /*}}}*/
    }
    
    if (count != num_nodes)
      LoadError (_FILE_, __LINE__, "Routing table format error") ;
    
    if (!got_mine)
      LoadError (_FILE_, __LINE__, "Routing tables for root not found") ;
  }
      /*}}}*/
  /*{{{  start up network*/
  {
    int link ;
  
    for (link=0;link<UPR_num_links;link++)
      if (root->son[link] != dangling)
      {
        if (debug)
          printf("Starting down link %d\n",link) ;
        ChanOutInt(raw_out[link], -1) ;
      }
  }  
  /*}}}*/
  /*{{{  close files*/
  fclose(net_fp) ;
  fclose(kernel_fp) ;
  /*}}}*/
  /*{{{  free dynamic memory*/
  {
    int f ;
    for (f=0;f<num_nodes;f++)
      if ((nodes[f] != NULL) && (f != proc_id)) free (nodes[f]) ;
  
    free(nodes) ;
  }
  /*}}}*/
  /*{{{  set kernel priority*/
  #ifdef UPR_KERNEL_AT_HI_PRI
  ProcToHI() ;
  #endif
  /*}}}*/

  UPR_Kernel (argc-params, argv+params, proc_id, num_nodes, raw_in, raw_out, root->link_type, routing_table) ;
}
  

  

          
      
      
    
    
  
