/*{{{  File banner*/
/*@(#)=====================================================*\
||@(#)  Project : PUMA ESPRIT P2701
||@(#)  Authors : Mark Debbage and Mark Hill
||@(#)            University of Southampton
||  
||@(#)    Title :  Network loader
||@(#)   System : DLOADER
||@(#) Filename : netnode.c
||@(#)  Version : 2.3
\*@(#)====================================================*/
/*}}}*/

/*{{{  includes*/
#include <stdlib.h>

#include "net.h"
#include "netload.h"
#include "neterr.h"
#include "topology.h"
/*}}}*/

PUBLIC int  proc_memory ;
PUBLIC NETWORK *network = NULL ;

static char *_FILE_ = __FILE__ ;

int main(int argc, char *argv[])
{
  /*{{{  variables*/
  Channel *raw_out[5] = {LINK0OUT, LINK1OUT, LINK2OUT, LINK3OUT, NULL} ;
  Channel *raw_in [5] = {LINK0IN,  LINK1IN,  LINK2IN,  LINK3IN,  NULL} ;
  Channel *boot_in, boot_out ;
  int     boot_index ;
  BYTE    *kernel_buffer ;
  int     kernel_size = 0 ;
  int     num_nodes ;
  int     proc_id ;
  NODE    *my_root ;
  NODE    **nodes ;
  BYTE    *routing_table ;
  /*}}}*/

  /*{{{  find boot link*/
  {
    boot_index = ProcAltList (raw_in) ;
    boot_in    = raw_in [boot_index] ;
    boot_out   = raw_out[boot_index] ;
  }  
  
  /*}}}*/
  /*{{{  read parameters from link*/
  {
    int n ;
  
    num_nodes   = ChanInInt(boot_in) ;
    kernel_size = ChanInInt(boot_in) ;
    
    /*{{{  malloc memory for node structure*/
    if ((nodes = (NODE **) malloc (num_nodes*sizeof (NODE *))) == NULL)
      LoadError (_FILE_, __LINE__, "Out of heap space - too many nodes") ;
    
    for (n=0;n<num_nodes;n++)
      nodes[n] = (NODE *) NULL ;
    /*}}}*/
    /*{{{  input tree from link*/
    {
      int node_cnt = 0 ;
      int tag = ChanInInt(boot_in) ;
    
      while (tag != tree_done)
      {
        NODE *node ;
    
        /*{{{  malloc memory for node*/
        if ((node = (NODE *) malloc (sizeof(NODE))) == NULL)
          LoadError (_FILE_, __LINE__, "Out of heap space - too many nodes") ;
        /*}}}*/
        /*{{{  read node structure from link*/
        ChanIn (boot_in, node, sizeof(NODE)) ;
        /*}}}*/
    
        nodes[node->id] = node ;
    
        if (node_cnt == 0)
          /*{{{  node is this processor*/
          {
            proc_id = node->id ;
            
            /*{{{  extend heap to top of memory*/
            {
              int extra = (node->mem_size) - (((int) _IMS_heap_front)^((int) NotProcess_p));
            
              proc_memory = node->mem_size ;
              if (extra < 0)
                LoadError (_FILE_, __LINE__, "Loader assumes 256K on node, cannot resize by negative amount") ;
            
              init_heap (_IMS_heap_front, extra, 0) ;
            }  
            /*}}}*/
          }
          /*}}}*/
    
        node_cnt++ ;
        tag = ChanInInt(boot_in) ;
      }
    }  
    /*}}}*/
  
    if ((my_root = nodes[proc_id]) == NULL)
      LoadError (_FILE_, __LINE__, "Processor missing from file") ;
  }  
  /*}}}*/
  /*{{{  initialise floating point unit if T8*/
  if (my_root->proc_type == 8)
    ProcInitFpu() ;
  /*}}}*/
  /*{{{  read code from boot link if not barren*/
  {
    BOOL barren = TRUE ;
    int s ;
  
    for (s=0;s<UPR_num_links;s++)
      if (my_root->son[s] != dangling) barren = FALSE ;
  
    if (!barren)
    {
      if ((kernel_buffer = malloc (kernel_size)) == NULL)
        LoadError (_FILE_, __LINE__, "Out of heap space") ;
      ChanIn(boot_in, kernel_buffer, kernel_size) ;
    }
    else
      kernel_buffer = NULL ;
  }
  /*}}}*/

  NetworkLoader (raw_out, raw_in, nodes, my_root, num_nodes, kernel_size, kernel_buffer) ;
  
  /*{{{  synchronize with booter*/
  ChanOutInt(boot_out, 0) ;
  /*}}}*/
  /*{{{  read routing tables from boot link*/
  {
    int  n ;
    int size ;
    BOOL got_mine = FALSE ;
    BYTE *tmp_routing_table ;
        
    n = ChanInInt(boot_in) ;
  
    while (n != -1)
    {
      /*{{{  get table size*/
      size = ChanInInt(boot_in) ;
      /*}}}*/
      /*{{{  allocate array to hold table*/
      if ((tmp_routing_table = malloc(size)) == NULL)
        LoadError (_FILE_, __LINE__, "Out of heap space - too many nodes") ;
      /*}}}*/
      /*{{{  get table*/
      ChanIn(boot_in, tmp_routing_table, size) ;
      /*}}}*/
      
      if (n == proc_id)
        /*{{{  tables for this processor*/
        {
          got_mine = TRUE ;
          routing_table = tmp_routing_table ;
        }
        /*}}}*/
      else
        /*{{{  ship tables to processor n*/
        {
          int link ;
          
          link = NodeInTree(nodes, my_root, n) ;
        
          if (link == -1)
            LoadError (_FILE_, __LINE__, "Cannot find node in tree") ;
          
          ChanOutInt(raw_out[link], n) ;
          ChanOutInt(raw_out[link], size) ;
          ChanOut   (raw_out[link], tmp_routing_table, size) ;
        
          free(tmp_routing_table) ;
        }
        /*}}}*/
  
      n = ChanInInt(boot_in) ;
    }
  
    if (!got_mine)
      LoadError (_FILE_, __LINE__, "Missing routing table") ;
  }
      /*}}}*/
  /*{{{  propagate start ups*/
  {
    int link ;
  
    for (link=0;link<UPR_num_links;link++)
      if (my_root->son[link] != dangling)
        ChanOutInt(raw_out[link], -1) ;
  }  
  /*}}}*/
  /*{{{  free dynamic memory*/
  {
    int f ;
    for (f=0;f<num_nodes;f++)
      if ((nodes[f] != NULL) && (f != proc_id)) free (nodes[f]) ;
  
    free(nodes) ;
  
    if (kernel_buffer != NULL)
      free(kernel_buffer) ;
  }
  /*}}}*/
  /*{{{  set kernel priority*/
  #ifdef UPR_KERNEL_AT_HI_PRI
  ProcToHI() ;
  #endif
  /*}}}*/
  
  UPR_Kernel (argc, argv, proc_id, num_nodes, raw_in, raw_out, my_root->link_type, routing_table) ;
}  
