/*{{{  File banner*/
/*@(#)=====================================================*\
||@(#)  Project : GPMIMD ESPRIT P5404
||@(#)  Authors : Mark Debbage and Mark Hill
||@(#)            University of Southampton
||  
||@(#)    Title : Process library interface
||@(#)   System : UBIK
||@(#) Filename : threads.h
||@(#)  Version : 1.15
||@(#)     Date : 7/6/92
\*@(#)====================================================*/
/*}}}*/

#ifndef __threads_h
#define __threads_h

#include <stdarg.h>

/*{{{  defines*/
#ifndef PRIVATE
#define PRIVATE static
#define PUBLIC
#define EXTERN extern
#endif
/*}}}*/

#define TRANSPUTER

#ifdef TRANSPUTER
  /*{{{  transputer declarations*/
  #define MICROCODED_SCHEDULER
  #define USER_CODE_AT_LOW_PRI
  
  typedef struct process process ;
    
  struct process
  {
    int     *context ;
    process *next ;
  } ;
  
  typedef struct
  {
    int size ;     /* a thread is held in a pstack followed by int[size] */
  } pstack ;
  
  typedef enum {comm_processes, user_processes, all_processes} mask_processes ;
  typedef struct { int priority; } mstate ;
  
  /*}}}*/
  #ifdef MICROCODED_SCHEDULER
    /*{{{  transputer scheduling in microcode*/
    /*{{{  scheduling*/
    #ifdef USER_CODE_AT_LOW_PRI
    #define init_scheduler()
    #else
    #define init_scheduler() __asm{ ldlabeldiff c-a; ldpi; a: stl -1; ldlp 0; ldc -1;and;runp;ldlabeldiff c-b;ldpi; b: gajw;stopp; align;word 0;c: ;}
    #endif
    
    #define deschedule(me)   __asm{ ldlp 0; ldpri; or; st (me)->context; stopp; }
    #define reschedule(you)  __asm{ ld (you)->context; runp; }
    #define backofqueue()    __asm{ ldlp 0; ldpri; or; runp; stopp; }
    /*}}}*/
    
    /*{{{  channels and synchronization*/
    typedef process *sync ;
    
    #define idle_sync NULL
    
    #define synchronize(s) {if (*(s) == NULL) {process me; *(s) = &me; deschedule(&me);} else {process *you = *(s) ; *(s)=NULL ; reschedule(you)}}
    
    typedef process *channel ;
    #define idle_channel ((channel) 0x80000000)
    
    #define output(c,m,l) { int len=(l) ; __asm{ ld (m) ; ld (c) ; ld (len) ; out ;}}
    #define input(c,m,l)  { int len=(l) ; __asm{ ld (m) ; ld (c) ; ld (len) ; in ; }}
    /*}}}*/
    
    /*{{{  semaphores*/
    #include <semaphor.h>
    
    typedef Semaphore semaphore ;
    
    #define sema_init(s,v) SemInit(s,v)
    #define sema_signal(s) SemSignal(s)
    #define sema_wait(s)   SemWait(s)
    /*}}}*/
    
    /*{{{  process masking*/
    #ifdef USER_CODE_AT_LOW_PRI
    #define BEGIN_MASK(ms,mt) { __asm{ldpri; st ((ms)->priority);} if ((ms)->priority==PROC_LOW) __asm{ ldlabeldiff c-a; ldpi; a: stl -1; ldlp 0; ldc -1;and;runp;ldlabeldiff c-b;ldpi; b: gajw;stopp; align;word 0;c: ;} }
    #define END_MASK(ms,mt)   { if ((ms)->priority==PROC_LOW) __asm{ldlp 0;ldc 1;or;runp;stopp;} }
    #else
    #define BEGIN_MASK(ms,mt)
    #define END_MASK(ms,mt)
    #endif
    /*}}}*/
    /*}}}*/
  #else
    /*{{{  transputer scheduling in software */
    /*{{{  co-routined scheduling*/
    #define active_p ((process *) -1)
    
    EXTERN void init_scheduler (void) ;
    EXTERN void deschedule (process *me) ;
    EXTERN void reschedule (process *you) ;
    EXTERN void backofqueue (void) ;
    
    /* These two calls are peculiar to VCR and not really part of the interface */
    EXTERN void deactivate (void) ;
    EXTERN void reactivate (void) ;
    /*}}}*/
    
    /*{{{  binary syncs and channels*/
    typedef process *sync ;
    
    #define idle_sync NULL
    
    EXTERN void synchronize (sync *s) ;
    
    typedef struct
    {
      process *id ;
      int      length ;
      void    *message ;
    } channel ;
    
    EXTERN channel idle_channel ;
    EXTERN void output (channel *c, void *message, int length) ;
    EXTERN void input  (channel *c, void *message, int length) ;
    /*}}}*/
    
    /*{{{  semaphores*/
    typedef struct
    {
      process *head ;
      process *tail ;
      int     count ;
    } semaphore ;
    
    EXTERN void sema_init (semaphore *sem, int value) ;
    EXTERN void sema_signal (semaphore *sem) ;
    EXTERN void sema_wait (semaphore *sem) ;
    /*}}}*/
    
    /*{{{  process masking*/
    EXTERN void BEGIN_MASK  (mstate *ms, mask_processes mp) ;
    EXTERN void END_MASK    (mstate *ms, mask_processes mp) ;
    /*}}}*/
    /*}}}*/
  #endif
#else
  #error Architecture not implemented yet
#endif

#endif
