#include <stdio.h>
#include <stdlib.h>
#include "vpi.h"
#include "upri.h"
#include "rk.h"

PRIVATE char* _FILE_ = __FILE__;
PRIVATE char* UML    = "Unexpected Message Length";
PRIVATE char* OOM    = "Out Of Memory";

/*{{{  typedef MRPS : Message Receive Port Structure*/
typedef struct
{
  int  length;   /* Incoming Message Length */
  char *buffer;  /* xmalloced buffer pointer */
  BOOL waiting;  /* True when waiting for message */
} MRPS ;
/*}}}*/
/*{{{  typedef MRB : Message Receive Block */
typedef struct MRB MRB;
struct MRB
{
  MRB *next;   /* for enqueueing MRBs */
  int length;  /* message length */
} ;  
/*}}}*/
/*{{{  typedef MSB : Message Send Block*/
typedef struct 
{
  int length;  /* message length */
} MSB;
/*}}}*/

/*{{{  my processor statics*/
PRIVATE int*  port_ids;              /* Pointer for array of port_ids (one per destn) */
PRIVATE PORT* ports;                 /* Pointer for array of ports (one per source) */
PRIVATE int   num_procs;             /* Number of processors */
PRIVATE int   my_proc_id;            /* My processor id [0..(num_procs-1)] */
PRIVATE process *my_process = NULL;  /* Pointer to record of descheduled user thread */
PRIVATE MRB* head = NULL;            /* Head of received messages queue */
PRIVATE MRB* tail = NULL;            /* Tail of received messages queue */

/*}}}*/

EXTERN int usermain(int my_proc_id, int num_procs, int argc, char **argv);

/*{{{  PUBLIC char *xmalloc(int length)*/
PUBLIC char *xmalloc(int length)
{
  MSB *msb = (MSB*) malloc(sizeof(MSB)+length);
  if (length<=0)  Error(_FILE_,__LINE__,"Minimum xmalloc is 1");
  if (msb==NULL)  Error(_FILE_,__LINE__,OOM);
  msb->length = length;
  return((char*)(msb+1));
}

/*}}}*/
/*{{{  PUBLIC void xfree(char *b)*/
PUBLIC void xfree(char *b)
{
  MRB *mrb = (MRB*) b - 1;
  free(mrb);
}

/*}}}*/
/*{{{  PUBLIC int xlength(char *b)*/
PUBLIC int xlength(char *b)
{
  MRB *mrb = (MRB*) b - 1;
  return(mrb->length);
}

/*}}}*/
/*{{{  PUBLIC void xsend(char *buffer, int node)*/
PUBLIC void xsend(char *buffer, int node)
{
  MSB *msb = (MSB*) buffer - 1;
  ORB hdrorb;

  int destn_port = port_ids[node]+my_proc_id;
  HdrBuild(node,destn_port,sizeof(int),&hdrorb.header);
  hdrorb.bffr = (BYTE*) &msb->length;
  UPR_Enqueue_NBK(&hdrorb);
  {
    char* b = buffer;
    ORB orb;
    orb.bffr = (BYTE*)b;
    HdrBuild( node, destn_port, 0, &orb.header);
    UPR_MultipleEnqueue_BK(&orb,msb->length);
  }
  free(msb);
}
/*}}}*/
/*{{{  PUBLIC char *xrecvb()*/
PUBLIC char *xrecvb()
{
  if (head==NULL)
    /*{{{  record MRB in queue on port*/
    {
      process p;
      if (my_process!=NULL)
        Error(_FILE_,__LINE__,"Found other process in this single process implementation");  
      my_process = &p;
      deschedule(&p);
      if (head==NULL)
        Error(_FILE_,__LINE__,"List still empty after restart");
    }
    /*}}}*/
  /*{{{  return top of pending message list*/
  {
    MRB *mrb= head;
    head = mrb->next;
    return((char*)(mrb+1));
  }
  /*}}}*/
}
/*}}}*/
/*{{{  RK_Postaction(PORT *port, int length, BYTE *buffer)*/
PRIVATE void RK_Postaction(PORT *port, int length, char *buffer)
{
  MRPS *mrps = (MRPS*) port->state;
  if (mrps->waiting)
    /*{{{  just rxed msg length so do malloc of MRB and prepare read of rest*/
    {
      MRB *mrb = malloc(sizeof(MRB)+mrps->length);
      if (mrb==NULL)  Error(_FILE_,__LINE__,OOM);
      mrb->length = mrps->length;
      mrb->next = NULL;
      mrps->buffer = (char*)(mrb+1);
      mrps->waiting=FALSE;  
      port->space = mrps->length;
      port->buffer = (BYTE*) mrps->buffer;
    }
    /*}}}*/
  else
    /*{{{  just rxed last packet */
    {
      MRB *mrb = (MRB*) (mrps->buffer - sizeof(MRB));
      if (head==NULL)
        /*{{{  insert MRB into empty list*/
        {
          head = mrb;
          tail = mrb;
          if (my_process!=NULL)
            /*{{{  restart xrecvb thread*/
            {
              reschedule(my_process);
              my_process=NULL;
            }
            /*}}}*/
        }
        /*}}}*/
      else
        /*{{{  add MDB to end of list*/
        {
          tail->next = mrb;
          tail = mrb;
          if (my_process!=NULL)
            Error(_FILE_,__LINE__,"Multiple messages ready but user process not restarted");
        }
        /*}}}*/
      /*{{{  prepare for next message*/
      port->buffer = (BYTE*) &mrps->length;
      port->space = sizeof(int);    
      mrps->waiting = TRUE;
      /*}}}*/
    }
    /*}}}*/
}
/*}}}*/
/*{{{  PUBLIC int main(int argc, char **argv)*/
PUBLIC int main(int argc, char **argv)
{
  MRPS *mrpss;
  num_procs = NumProcs();
  my_proc_id = ProcId();
  init_scheduler();
  if ((ports = (PORT*)malloc(sizeof(PORT)*num_procs))==NULL ||
      (port_ids = (int*)malloc(sizeof(int)*num_procs))==NULL ||
      (mrpss = (MRPS*)malloc(sizeof(MRPS)*num_procs))==NULL ||
      (port_ids[my_proc_id] = MultiplePortGrab(ports, num_procs)) <0 )
    Error(_FILE_,__LINE__,"RK initialisation failed");
  /*{{{  Initialise Ports*/
  {
    PORT *port = ports;
    MRPS *mrps = mrpss;
    int p;
    for (p=0;p<num_procs;p++)
    {
      PortInit(port,NULL,FALSE,RK_Postaction,FALSE,mrps);
      /*{{{  read length of incoming message to MRPS length*/
      port->buffer = (BYTE*)&mrps->length;
      port->space = sizeof(int);
      mrps->waiting=TRUE;
      /*}}}*/
      port++;
      mrps++;
    }
  }
      
  /*}}}*/
  /*{{{  Exchange bases of PORT IDS*/
  {
    int p;
    for (p=0;p<num_procs;p++)
      if (p<my_proc_id)
      {
        ClaimChannel(p);
        ClientOut(p,&port_ids[my_proc_id],sizeof(int));
        if (ClientIn(p,&port_ids[p],sizeof(int))!=sizeof(int))
          Error(_FILE_,__LINE__,UML);
        ReleaseChannel(p);
      }
      else if (p>my_proc_id)
      {
        int other = GrantService();
        if (ServerIn(&port_ids[other],sizeof(int))!=sizeof(int))
          Error(_FILE_,__LINE__,UML);
        ServerOut(&port_ids[my_proc_id],sizeof(int));
        EndService();
      }
  }
  /*}}}*/
  /*{{{  Execute user code then free memory and return*/
  {
    int res=0;
    res = usermain(my_proc_id,num_procs,argc,argv);
    free(mrpss);
    MultiplePortRelease(ports, num_procs);
    free(ports);
    free(port_ids);
    /*{{{  synchronise as returning on p0 terminates iserver*/
    if (num_procs>1)
    {
      if (my_proc_id==0)
      {
        int p;
        for (p=0;p<num_procs;p++)
          {
            GrantService();
            if (ServerIn(NULL,0)!=0)
              Error(_FILE_,__LINE__,UML);
            EndService();
          }
      }
      else
      {
        ClaimChannel(0);
        ClientOut(0,NULL,0);
        ReleaseChannel(0);
      }
    }
    /*}}}*/
    return(res);    
  }      
  /*}}}*/
}
/*}}}*/
